/*
 * Created on Nov 9, 2006
 *
 * @author ie00165h - Alex Lara
 * (c) 2006 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.utils.searchengine;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.ejie.r01f.log.R01FLog;
import com.ejie.r01f.util.StringUtils;
import com.ejie.r01f.util.URLEncoder;
import com.ejie.r01f.xmlproperties.XMLProperties;
import com.ejie.r01m.objects.searchengine.query.R01MBaseSearchedObj;
import com.ejie.r01m.objects.searchengine.query.R01MQueryObject;
import com.ejie.r01m.objects.searchengine.query.R01MSearchedArea;
import com.ejie.r01m.objects.searchengine.query.R01MSearchedDataRepository;
import com.ejie.r01m.objects.searchengine.query.R01MSearchedMetaData;
import com.ejie.r01m.objects.searchengine.query.R01MSearchedServer;
import com.ejie.r01m.utils.R01MConstants;

/**
 * Utilidades para el motor de bsqueda
 * @author ie00165h - Alex Lara
 */
public abstract class R01MSearchEngineUtils {
///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTANTES
///////////////////////////////////////////////////////////////////////////////////////////
	// Constantes para el traceo de logs
    private static String LOG_TYPE_ID = "r01k.session";
    
    private static boolean DEBUG = (R01FLog.getLogLevel(LOG_TYPE_ID).intValue() == Level.OFF.intValue() ? false:true);
	/**
     * MetaDatos excluidos de la presentacion
     */
    private static final List QRY_PRESENTATION_EXCLUDED_METADATA = XMLProperties.getPropertyList(R01MConstants.API_APPCODE,"searchEngine/queryPresentation/exclude/metaDataList/metaData");
    /**
     * Orden en la que se presentan los metaDatos para su visualizacin
     */
    private static final List QRY_PRESENTATION_METADATA_ORDER = XMLProperties.getPropertyList(R01MConstants.API_APPCODE,"searchEngine/queryPresentation/order/metaDataList/metaData");
    /**
     * Patrn para obtener el valor de las queries
     */
    public static final String QUERY_PATTERN = XMLProperties.get(R01MConstants.API_APPCODE,"searchEngine/queryUrlPattern",
                                                                                           ".*\\?.*(searchGUID|r01kCommonSearchQry|r01kStoredQry|fullText|r01kQry|languageAplication)=([^&]+).*");
    /**
     * Atributo para detectar las queries de la versin 1
     */
    private static final String QUERYV1PARAMFILTER = "languageAplication";

    /**
     * Codificacin "UTF-8"  para las url
     */
    public static final String URL_ENCODING_UTF = "UTF-8";

    public static final String URL_ENCODING_ISO_8859 = "ISO-8859-1";
    /**
     * <pre>
     * Parmetro en el que llega la query de bsqueda (codificada en la URL).
     * El parmetro tendr la forma r01kQry=[query codificada en la URL].
     * </pre>
     * @see com.ejie.r01m.utils.searchengine.R01MSearchEngineUtils#encodeQueryObjectInURL(com.ejie.r01m.objects.searchengine.query.R01MQueryObject)
     */
    public transient static final String PARAM_SEARCH_QUERY = "r01kQry";
    /**
     * <pre>
     * Parmetro en el que llega la query de busqueda desde el searchEngine.
     * El parametro tendr la forma fullText=[el texto de bsqueda].
     * </pre>
     */
    public transient static final String PARAM_SEARCH_QUERY_TEXT = "fullText";

///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS AUXILIARES PARA CONVERTIR LA QUERY EN UNA CADENA EN LA URL
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Convierte el objeto {@link R01MQueryObject} en una cadena con todas las condiciones de bsqueda
     * que puede ser utilizada para pasar la query en una URL.<br>
     * La codificacin ser en {@link #URL_ENCODING_UTF}.
     * @param qry la query
     * @return una cadena con las condiciones de bsqueda
     */
    public static String encodeQueryObjectInURL(R01MQueryObject qry) {
        return R01MSearchEngineUtils.encodeQueryObjectInURL(qry,true);
    }
    /**
     * <pre>
     * Convierte el objeto {@link R01MQueryObject} en una cadena con todas las condiciones de bsqueda
     * que puede ser utilizada para pasar la query en una URL.
     * La codificacin ser en {@link #URL_ENCODING_UTF}.
     * </pre>
     * @param qry la query
     * @param includePresentationProps true si hay que incluir las propiedades de presentacin en la query codificada
     * @return una cadena con las condiciones de bsqueda
     */
    public static String encodeQueryObjectInURL(R01MQueryObject qry,boolean includePresentationProps) {
        StringBuffer outQry = new StringBuffer("");
        if (qry == null) {
            return outQry.toString();
        }
        // Identificador de la bsqueda
        if (includePresentationProps && qry.getQueryOid() != null) {
            outQry.append("i:" + qry.getQueryOid() + ";");
        }
        // Almacenamiento
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("sS",qry.getServers()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("sR",qry.getDataRepositories()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("sA",qry.getAreas()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("sW",qry.getWorkAreas()));
        // Tipologia
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("tC",qry.getContentClusters()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("tF",qry.getContentFamilies()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("tT",qry.getContentTypes()));
        // MetaDatos
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("m",qry.getMetaData()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("mO",qry.getORMetaData()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("mA",qry.getORMetaData()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("o",qry.getOrderBy()));
        // Etiquetas de catalogacion
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("cA",qry.getCatalogLabelsAnd()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("cO",qry.getCatalogLabelsOr()));
        // Repositorios de publicacion
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("p",qry.getPublishRepositories()));
        // Propiedades de presentacion
        if (includePresentationProps) {
            outQry.append(R01MSearchEngineUtils._composeQueryPresentationPropertiesQueryString("pp",qry.getPresentationProperties()));
        }

        // /////////////////////////////////////////////////////////////////////////////////////////
        // IMPORTANTE: La codificacin de la query se realiza en UTF-8, se decodifica en las pginas
        //             de presentacin de resultados de R01H.
        // /////////////////////////////////////////////////////////////////////////////////////////
        String qryEncoded;
        try {
            qryEncoded = URLEncoder.encode(outQry.toString(),URL_ENCODING_UTF);
        } catch (UnsupportedEncodingException ex) {
            /** no se trata **/
            qryEncoded = outQry.toString();
        }

        return qryEncoded;
    }
    /**
     * Unifica los distintos valores de un metadato para una query codificada en URL
     * @param mark separador
     * @param objList lista de valores del metadato
     * @return la cadena con los valores del metadato separados por el separador
     */
    private static StringBuffer _composeSearchedObjListQueryString(String mark,List objList) {
        if (objList == null) {
            return new StringBuffer("");
        }
        StringBuffer outQry = new StringBuffer();
        outQry.append(mark);outQry.append(':');
        R01MBaseSearchedObj obj;
        for (Iterator it = objList.iterator(); it.hasNext(); ) {
            obj = (R01MBaseSearchedObj)it.next();
            if ("sR".equals(mark) || "sA".equals(mark) || "sW".equals(mark)) {
                String key = obj.getKey();
                outQry.append(key.substring(key.lastIndexOf('.') + 1));
            } else {
                outQry.append(obj.getKey());
            }
            if (it.hasNext()) {
                outQry.append(',');
            }
        }
        outQry.append(';');
        if (outQry.length() == mark.length() + 2) {
            return new StringBuffer("");
        }
        return outQry;
    }
    /**
     * <pre>
     * Convierte el objeto {@link R01MQueryObject} en una cadena con todas las condiciones de bsqueda
     * </pre>
     * @param qry la query
     * @return una cadena de texto con las condiciones para reconstruir un R01MQueryObject
     */
    public static String encodeQueryObjectInString(R01MQueryObject qry) {
        StringBuffer outQry = new StringBuffer("");  // NOPMD by mcerezue on 8/04/11 10:58
        if (qry == null) {
            return outQry.toString();
        }
        // Identificador de la bsqueda
            outQry.append("i:" + qry.getQueryOid() + ";");
        // Almacenamiento
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("sS",qry.getServers()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("sR",qry.getDataRepositories()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("sA",qry.getAreas()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("sW",qry.getWorkAreas()));
        // Tipologia
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("tC",qry.getContentClusters()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("tF",qry.getContentFamilies()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("tT",qry.getContentTypes()));
        // MetaDatos
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("m",qry.getMetaData()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("o",qry.getOrderBy()));
        // Etiquetas de catalogacion
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("cA",qry.getCatalogLabelsAnd()));
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("cO",qry.getCatalogLabelsOr()));
        // Repositorios de publicacion
        outQry.append(R01MSearchEngineUtils._composeSearchedObjListQueryString("p",qry.getPublishRepositories()));
        // Propiedades de presentacion
       outQry.append(R01MSearchEngineUtils._composeQueryPresentationPropertiesQueryString("pp",qry.getPresentationProperties()));

        return outQry.toString();
    }
    /**
     * Unifica los distintos valores de un metadato para una query codificada en URL
     * @param mark separador
     * @param props mapa de las propiedades
     * @return la cadena con los valores del metadato separados por el separador
     */
    private static StringBuffer _composeQueryPresentationPropertiesQueryString(String mark,Map props) {
        StringBuffer outQry = new StringBuffer("");
        if (props != null && !props.isEmpty()) {
            outQry.append(mark);outQry.append(':');
            Map.Entry me = null;
            for (Iterator it=props.entrySet().iterator(); it.hasNext(); ) {
                me = (Map.Entry)it.next();
                outQry.append((String)me.getKey());outQry.append('.');outQry.append(String.valueOf(me.getValue()));
                if (it.hasNext()) {
                    outQry.append(',');
                }
            }
        }
        return outQry;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS AUXILIARES PARA PASAR UNA QUERY EN CADENA A OBJETO
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Convierte una cadena a un objeto {@link R01MQueryObject}
     * @param queryString el objeto {@link R01MQueryObject} serializado en forma
     *                    de queryString
     * @return el objeto {@link R01MQueryObject} obtenido a partir de la cadena
     */
    public static R01MQueryObject decodeURLInQueryObject(String queryString) {
        if (StringUtils.isEmptyString(queryString)) {
            return null;
        }
        R01MQueryObject outQry = new R01MQueryObject();
        // Se trata cada condicin de la query.
        // Las condiciones vienen separadas por ;
        String[] qryStrSplitted = queryString.split(";");
        if (qryStrSplitted != null && qryStrSplitted.length > 0) {
            for (int i=0; i<qryStrSplitted.length; i++) {
                if(!R01MSearchEngineUtils._extractSearchedObjFromKey(qryStrSplitted[i],outQry)) {
                    //La query no esta bien construida, ponemos el objeto query a null y dejamos de recorrer las condiciones
                    outQry=null;
                    break;
                }
            }
        }
        return outQry;
    }

    /**
     * Rellena el objeto de bsqueda con la condicin pasada (en formato texto).
     * @param key Clave /condicin en formato texto (ej. m:metadataOid.CONDITION.metadataValues)
     * @param ioQry objeto query donde se rellenar la condicin pasada
     */
    private static boolean _extractSearchedObjFromKey(String key,R01MQueryObject ioQry) {
        // i = identificador (oid de la query)
        // sS/sR/sA/sW = storageServer / storageRepository / storageArea / storageWorkArea
        // tC = typoCluster / typoFamily / typoType
        // m = metaData
        // o = order
        // cA/cO = catalog AND / catalog OR
        // p = repositorio de publicacion
        // pp = propiedad de presentacion
        String patternStr = "(i|sS|sR|sA|sW|tC|tF|tT|m|mA|mO|o|cA|cO|p|pp):([^;:]+)";
                      
        Pattern p = Pattern.compile(patternStr);
        Matcher m = p.matcher(key);
        if (m.find()) {
            String type = m.group(1);
            String values = m.group(2);

            // Se excluyen las cadenas vacas y de espacios en blanco
            if(!StringUtils.isEmptyString(values)) {
                // Si el valor es compuesto (ms de un valor separados por comas),
                // en el objeto Query se mete una entrada para cada valor.
                String[] valuesSplitted = values.split(",");
                if ("i".equals(type)) {
                    ioQry.setQueryOid(m.group(2));
                } else if ("sS".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        ioQry.addServerFromKey(valuesSplitted[i]);
                    }
                } else if ("sR".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        if(ioQry.getServers()!=null && !ioQry.getServers().isEmpty()) {
                            for(Iterator itKeys=ioQry.getServers().iterator();itKeys.hasNext();) {
                                ioQry.addDataRepositoryFromKey(((R01MSearchedServer)itKeys.next()).getServerOid() + "." + valuesSplitted[i]);
                            }
                        }
                    }
                } else if ("sA".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        if(ioQry.getServers()!=null && !ioQry.getServers().isEmpty()) {
                            for(Iterator itServerKeys=ioQry.getServers().iterator();itServerKeys.hasNext();) {
                                String serverKey = ((R01MSearchedServer)itServerKeys.next()).getServerOid();
                                if(ioQry.getDataRepositories()!=null && !ioQry.getDataRepositories().isEmpty()) {
                                    for(Iterator itKeys=ioQry.getDataRepositories().iterator();itKeys.hasNext();) {
                                        ioQry.addAreaFromKey(serverKey + "." + ((R01MSearchedDataRepository)itKeys.next()).getDataRepositoryOid() + "." + valuesSplitted[i]);
                                    }
                                }
                            }
                        }
                    }
                } else if ("sW".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        if(ioQry.getServers()!=null && !ioQry.getServers().isEmpty()) {
                            for(Iterator itServerKeys=ioQry.getServers().iterator();itServerKeys.hasNext();) {
                                String serverKey = ((R01MSearchedServer)itServerKeys.next()).getServerOid();
                                if(ioQry.getDataRepositories()!=null && !ioQry.getDataRepositories().isEmpty()) {
                                        for(Iterator itDataRepositoryKeys=ioQry.getDataRepositories().iterator();itDataRepositoryKeys.hasNext();) {
                                            String datarepositoryKey = ((R01MSearchedDataRepository)itDataRepositoryKeys.next()).getDataRepositoryOid();
                                            if(ioQry.getAreas()!=null && !ioQry.getAreas().isEmpty()) {
                                                for(Iterator itKeys=ioQry.getAreas().iterator();itKeys.hasNext();) {
                                                    ioQry.addWorkAreaFromKey(serverKey + "." + datarepositoryKey + "." + ((R01MSearchedArea)itKeys.next()).getAreaOid() + "." + valuesSplitted[i]);
                                                }
                                            }
                                        }
                                }
                            }
                        }
                    }
                } else if ("tC".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        ioQry.addContentClusterFromKey(valuesSplitted[i]);
                    }
                } else if ("tF".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        ioQry.addContentFamilyFromKey(valuesSplitted[i]);
                    }
                } else if ("tT".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        ioQry.addContentTypeFromKey(valuesSplitted[i]);
                    }

                // En el caso de condiciones de tipo metadato, el valor puede tener distintos formatos por lo que
                // es necesario tratarlo de manera particular para comprobar los valores (mtodo _splitMetaDataConditions)
                } else if ("m".equals(type) || "mA".equals(type)) {     // metaDatos AND
                    valuesSplitted = R01MSearchEngineUtils._splitMetaDataConditions(values);
                    for (int i=0; i<valuesSplitted.length; i++) {
                        ioQry.addMetaDataFromKey(valuesSplitted[i]);
                    }
                } else if ("mO".equals(type)) {                         // metaDatos OR
                    valuesSplitted = R01MSearchEngineUtils._splitMetaDataConditions(values);
                    for (int i=0; i<valuesSplitted.length; i++) {
                        ioQry.addORMetaDataFromKey(valuesSplitted[i]);
                    }
                } else if ("o".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        ioQry.addOrderByMetaDataFromKey(valuesSplitted[i]);
                    }
                } else if ("cA".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        ioQry.addANDStructureCatalogFromKey(valuesSplitted[i]);
                    }
                } else if ("cO".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        ioQry.addORStructureCatalogFromKey(valuesSplitted[i]);
                    }
                } else if ("p".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        ioQry.addPublishRepositoryFromKey(valuesSplitted[i]);
                        ioQry.setPublishedItemsOnly(true);
                    }
                } else if ("pp".equals(type)) {
                    for (int i=0; i<valuesSplitted.length; i++) {
                        String[] propertySplitted = valuesSplitted[i].split("\\.");
                        if (propertySplitted.length == 2) {
                            ioQry.setPresentationProperty(propertySplitted[0],propertySplitted[1]);
                        }
                    }
                }
            }
            return true;
        }
        return false;
    }
    /**
     * Obtiene los valores de los metadatos segn el separador que se pueda usar en cada caso
     * Se trata el split de manera particular
     *      se buscan los IN y se sustituye la coma por un ; para luego volver
     *      a la coma
     * @param metaDataConditions las condiciones sobre metaDatos
     * @return un array con cada condicin independientemente
     */
    private static String[] _splitMetaDataConditions(String metaDataConditions) {

        // Se tratan los casos especiales:
        // Viene una entrada del tipo: documentLanguage.EQ.es,procedureStatus.IN.(16,17)
        // Al tratarla debe devolver:
        // [0] - documentLanguage.EQ.es
        // [1] - procedureStatus.IN.(16,17)

        //Comportamientos particulares
        // IN ==>  metadata.IN.(value1,value2,value3)
        // ANY ==> metadata.ANY.(value1,value2,value3)
        // ALL ==> metadata.ALL.(value1,value2,value3)
        // BETWEEN ==> metadata.BETWEEN.value1,value2
        List valuesList = new ArrayList();

        // Patrn para todos los casos
        Pattern pattern = Pattern.compile(

                // CASO IN, ANY o ALL:
                // Formato = metadata.OPERADOR.(value1,value2,value3)
                "([^,.]+."
                + ("(?:") + (R01MSearchEngineConstants.IN_OPERATOR) + ("|")
                              + (R01MSearchEngineConstants.ANY_OPERATOR) + ("|")
                              + (R01MSearchEngineConstants.ALL_OPERATOR) + (")")
                + (".\\([^\\)]+\\)")

                + ("|")

                // CASO BETWEEN:
                // Formato = metadata.BETWEEN.value1,value2
                + ("[^,.]+.") + (R01MSearchEngineConstants.BETWEEN_OPERATOR) + (".[^,]+,[^,]+")

                + ("|")

                // CASO DEFAULT (resto de operadores tienen un slo valor)
                + ("[^,]+)"));

        Matcher match = pattern.matcher(metaDataConditions);
        while (match.find()) {
            String concurrence = match.group();
            valuesList.add(concurrence);
        }
        return (String [])valuesList.toArray(new String [0]);
    }
    /**
     * Mtodo que obtiene un objeto con la informacin de la query a partir de la URL.
     * Posibles URLs:
     *      Bsqueda almacenada 1         : ?searchGUID=r01koidquery&.....
     *      Bsqueda almacenada 2         : ?r01kStoredQry=24122...
     *      Buscador avanzado             : ?r01kCommonSearchQry=st(serverOid#dataRepositoryOid#area#workarea)$$ty(clusteOid1#familyOid1#typeOid1)$$ty(clusteOid2#familyOid2#typeOid2)$$ty(clusteOid3#familyOid3#typeOid3)$$md(metadDataOid1#operationId1#metaDatavalue1)$$md(metadDataOid2#operationId2#metaDatavalue2)$$md(metadDataOid3#operationId3#metaDatavalue3)$$ax(strureId1,labelId1#strureId2,labelId2#strureId3,labelId3)$$pr(publishRepositoryOid1#publishRepositoryOid2)
     *      Buscador de texto libre       : ?fullText=textToSearch&resultsSource=fullText
     *      Busqueda codificada en la URL : ?r01kLang=es&r01kQry=cA:r01eF3362DDC121AE82CDBCABA97312B9CDCEBBD89C1;tF:estadisticas;pp:r01NavBarBlockSize.10,r01PageSize.10&r01kGuide=r01kGuide1&resultsSource=fullText&r01kResetSrch=true
     *      Queries de la versin 1 (solo turismo) : ?languageAplication=es&languageSearch=es&typeSearch=&oneResult=false&repositorySearch=contentRepository&searchAction=search&newContext=contextNew&numberOfElements=1000&link_url=/s11-16112x/es&link_openWindow=false&link_type=&link_height=&link_width=&link_posX=&link_posY=&structureGuide=&labelGuide=&attributeId=publish&attributeName=Publicado&publishOperation=EQ&publish=1&format=String&attributeId=language&attributeName=Idioma&languageOperation=EQ&language=es&typesId=60;&typesName=Alojamiento;&format=String&attributeId=historicTerritoryCode&attributeName=GIPUZKOA&historicTerritoryCodeOperation=LIKE&historicTerritoryCode=20&format=String&attributeId=municipalityCode&attributeName=MUTRIKU&municipalityCodeOperation=LIKE&municipalityCode=200056&resTypesId=60;&resTypesName=Alojamiento;&hasAxes=false
     * @param queryString la url que llega
     * @return el objeto que representa la query
     */
    public static R01MQueryObject obtainQueryObjectFromUrl(String queryString) {

        R01MQueryObject queryObject = new R01MQueryObject();    // Query General

        // Extraer el tipo de query y su valor
        Pattern p = Pattern.compile(QUERY_PATTERN);
        Matcher m = p.matcher(queryString);
        if (m.matches()) {
            // Obtener el tipo de query
            String queryType = m.group(1);
            String queryValue = m.group(2);

            if (queryType.equals(PARAM_SEARCH_QUERY)) {
                queryObject = R01MSearchEngineUtils.decodeURLInQueryObject(queryValue);
            } else if (queryType.equals(PARAM_SEARCH_QUERY_TEXT)) {
                queryObject = new R01MQueryObject();
                queryObject.setFullTextSearch(true);
                queryObject.setFullText(queryValue);
            } else if (queryType.equals(QUERYV1PARAMFILTER)) {
                //Transformar la query de v1 a v2
                StringBuffer newQuery = new StringBuffer();
                newQuery.append("&r01kQry=");
                StringBuffer newURLEncoded = new StringBuffer();

                p = Pattern.compile("&(languageSearch|numberOfElements|typesId|nodeId)=([^&]+)");
                m = p.matcher(queryString);
                while (m.find()) {
                    String valueType = m.group(1);
                    String valueVal = m.group(2);
                    if("languageSearch".equals(valueType)) {
                        //Lenguaje de la bsqueda
                        newURLEncoded.append("&r01kLang=");newURLEncoded.append(valueVal);
                    } else if("numberOfElements".equals(valueType)){
                        //Pginas
                        //newQuery.append("pp:r01NavBarBlockSize." + valueVal + ",r01PageSize." + valueVal + ";");
                        newQuery.append("pp:r01NavBarBlockSize.10,r01PageSize.10;");
                    } else if("typesId".equals(valueType)){
                        //Tipologa
                        //Quitar el ; del final
                        String[] tipos = valueVal.split(";");
                        if(tipos.length > 0) {
                            newQuery.append("tT:");
                            for(int i=0;i<tipos.length;i++) {
                                TypoConversion newTypeId;
                                try {
                                    newTypeId = R01MSearchEngineUtils._convertTypeId(tipos[i]);
                                } catch (IllegalArgumentException ex) {
                                    //Valor por defecto si falla
                                    newTypeId = new TypoConversion("error","error","error");
                                }
                                newQuery.append(newTypeId.typoOid);
                                newQuery.append(';');
                            }
                        }
                    } else if("nodeId".equals(valueType)){
                        //Catalogacin
                        newQuery.append("cA:");newQuery.append(valueVal);newQuery.append(';');
                    }
                }

                ///////////////////////////////////////////////////////////////////////////////////////////
                //  IDENTIFICACION DE METADATOS
                ///////////////////////////////////////////////////////////////////////////////////////////
                p = Pattern.compile("attributeId=([^&]+)&attributeName=[^&]+&\\1Operation=(EQ|LIKE|GT|LT|GTE|LTE)&\\1=([^&]+)");
                m = p.matcher(queryString);
                while (m.find()) {
                    String metadataOid = m.group(1);
                    if("language".equals(metadataOid) || "publish".equals(metadataOid)) {
                        continue;
                    }
                    //Componer metadatos m:metadato.OPERADOR.valor
                    newQuery.append("m:");
                    newQuery.append(metadataOid);
                    newQuery.append('.');
                    newQuery.append(m.group(2));
                    newQuery.append('.');
                    newQuery.append(m.group(3));
                    newQuery.append(';');
                }
                newQuery.append(newURLEncoded);

                if(DEBUG) {
                	R01FLog.to("r01k.session").fine("newQuery=" + newQuery.toString());
                }
                queryObject = R01MSearchEngineUtils.decodeURLInQueryObject(newQuery.toString());
            }
        }

        return queryObject;
    }

    /**
     * Evitar ataques de cross-site scripting (XSS) formateando la query
     * @param qry query
     */
    public static void sanitizeQuery(R01MQueryObject qry) {

        if(qry.getMetaData()!=null && !qry.getMetaData().isEmpty()) {
            for(Iterator itMetadata=qry.getMetaData().iterator();itMetadata.hasNext();) {
                R01MSearchedMetaData srchMeta = (R01MSearchedMetaData)itMetadata.next();
                srchMeta.setValue(R01MSearchEngineUtils.sanitizeParam(srchMeta.getValue()));
            }
        }
    }
    /**
     * Evitar ataques de cross-site scripting para un parametro especfico
     *  (?i) make it case insensitive
     * case 1 : <script> are removed
     * case 2 : javascript: call are removed
     * case 3 : remove any tag. Example: <img>
     * @param sourceText cadena origen
     * @return cadena filtrada
     */
    public static String sanitizeParam(String sourceText) {
        if(StringUtils.isEmptyString(sourceText)) {
            return sourceText;
        }
        String paramFormatted = sourceText;
        /******************* CASE 1 *****************************/
        paramFormatted  = R01MSearchEngineUtils.sanitizeParam(paramFormatted,"(?i)<script.*?>.*?</script.*?>");
        /******************* CASE 2 *****************************/
        paramFormatted  = R01MSearchEngineUtils.sanitizeParam(paramFormatted,"(?i).*javascript:.*");
        /******************* CASE 3 *****************************/
        paramFormatted  = R01MSearchEngineUtils.sanitizeParam(paramFormatted,"<(.|\\n)+?>");

        return paramFormatted;
    }
    /**
     * Evitar ataques de cross-site scripting aplicando un patrn determinado a un parametro especfico.
     * @param sourceText cadena origen
     * @param pattern patrn de filtrado. Ej:Borrar <script> ->  (?i)<script.*?>.*?</script.*?>
     * @return cadena filtrada
     */
    public static String sanitizeParam(String sourceText, String pattern) {

        StringBuffer finalText = new StringBuffer("");

        Pattern p1 = Pattern.compile(pattern);
        Matcher m1 = p1.matcher(sourceText);
        while (m1.find()) {
            m1.appendReplacement(finalText,"");
        }
        m1.appendTail(finalText);

        return finalText.toString();
    }

    /**
     * Convierte el tipo de contenido para la convivencia V1-V2
     * @param v1Type Tipo de contenido V1
     * @return El tipo de contenido para que todo funciones
     * @throws IllegalArgumentException error al convertir el tipo
     */
    private static TypoConversion _convertTypeId(String v1Type) throws IllegalArgumentException {
        // Traduccin de tipos de contenido...
        // En el fichero de properties hay una seccion que indica como se traducen los
        // tipos v1 a tipos v2:
        //      <migratedTypes>
        //          <t[oidTipoV1]>[oidTipoVistaV2]</t[oidTipoV1]>
        //      </migratedTypes>
        // Si el tipo no se traduce:
        //      <t3>3</t3>      (caso de informacion)
        // Si el tipo se traduce:
        //      <t33>3</t33>    (33 = galeria fotografica --> pasa a 3 = informacion)
        Properties p = XMLProperties.getProperties(R01MConstants.SEARCHENGINE_APPCODE,"contentsRepositoryConfig/migratedTypes");
        if (p == null) {
            throw new IllegalArgumentException("Error!!! NO se ha definido en el fichero de properties la tabla de traduccin de tipos para migracin V1->V2!!!!!");
        }
        String effectiveV1Type = p.getProperty("t" + v1Type);
        if (effectiveV1Type == null) {
            effectiveV1Type = v1Type;
            if(DEBUG) {
            	R01FLog.to("r01k.session").fine("\t\t... NO se ha definido la traduccin del tipo " + v1Type + " en la seccion migratedTypes del fichero properties de r01k!!!\r\nSe traduce por si mismo!!!!");
            }
        }
        if(DEBUG) {
        	R01FLog.to("r01k.session").fine("\t\t...el tipo v1 " + v1Type + " se transforma en tipo " + effectiveV1Type);
        }

        TypoConversion typoConversion = new TypoConversion(effectiveV1Type);
        if (typoConversion.clusterOid != null && typoConversion.familyOid != null && typoConversion.typoOid != null) {
            return typoConversion;
        }
        return null;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  Clase auxiliar de almacenamiento de tipologas de v2
///////////////////////////////////////////////////////////////////////////////////////////
  static class TypoConversion {
      public String clusterOid;
      public String familyOid;
      public String typoOid;

      public TypoConversion() {
          _init();
      }
      public TypoConversion(String newClusterOid,String newFamilyOid,String newTypoOid) {
          this();
          this.clusterOid = newClusterOid;
          this.familyOid = newFamilyOid;
          this.typoOid = newTypoOid;
      }
      public TypoConversion(String v1TypeId) {
          this();
          // Obtener cluster familia y tipo de la tabla de traduccin
          String[] clusterFamilyType = (String[])typesConversion.get(v1TypeId);
          if (clusterFamilyType != null) {
              this.clusterOid = clusterFamilyType[0];
              this.familyOid = clusterFamilyType[1];
              this.typoOid = clusterFamilyType[2];
          }
      }
      // ---------------------------------------------------------
      private Map typesConversion = new HashMap();

      private void _init() {
          this.typesConversion.put("1",new String[] {"euskadi","enlaces","enlace"});
          this.typesConversion.put("55",new String[] {"euskadi","enlaces","servicio_online"});

          this.typesConversion.put("2",new String[] {"euskadi","eventos","evento"});
          this.typesConversion.put("29",new String[] {"euskadi","eventos","evento"});
          this.typesConversion.put("30",new String[] {"euskadi","eventos","evento"});
          this.typesConversion.put("31",new String[] {"euskadi","eventos","evento"});
          this.typesConversion.put("32",new String[] {"euskadi","eventos","evento"});

          this.typesConversion.put("3",new String[] {"euskadi","informacion","informacion"});
          this.typesConversion.put("36",new String[] {"euskadi","informacion","informacion"});
          this.typesConversion.put("33",new String[] {"euskadi","informacion","informacion"});
          this.typesConversion.put("53",new String[] {"euskadi","informacion","faq"});
          this.typesConversion.put("54",new String[] {"euskadi","informacion","recurso_tecnico"});

          this.typesConversion.put("4",new String[] {"euskadi","prensa_comunicacion","noticia"});
          this.typesConversion.put("18",new String[] {"euskadi","prensa_comunicacion","nota_prensa"});
          this.typesConversion.put("19",new String[] {"euskadi","prensa_comunicacion","noticia"});
          this.typesConversion.put("50",new String[] {"euskadi","prensa_comunicacion","publicidad"});


          this.typesConversion.put("5",new String[] {"euskadi","procedimientos_administrativos","ayuda_subvencion"});
          this.typesConversion.put("6",new String[] {"euskadi","procedimientos_administrativos","autorizacion"});
          this.typesConversion.put("7",new String[] {"euskadi","procedimientos_administrativos","carnet"});
          this.typesConversion.put("8",new String[] {"euskadi","procedimientos_administrativos","registro"});
          this.typesConversion.put("9",new String[] {"euskadi","procedimientos_administrativos","inspeccion"});
          this.typesConversion.put("10",new String[] {"euskadi","procedimientos_administrativos","multa_sancion"});
          this.typesConversion.put("12",new String[] {"euskadi","procedimientos_administrativos","procedimiento_otro"});
          this.typesConversion.put("13",new String[] {"euskadi","procedimientos_administrativos","premio_concurso"});
          this.typesConversion.put("14",new String[] {"euskadi","procedimientos_administrativos","procedimiento_otro"});
          this.typesConversion.put("15",new String[] {"euskadi","procedimientos_administrativos","arbitraje_denuncia_reclamacion"});
          this.typesConversion.put("17",new String[] {"euskadi","procedimientos_administrativos","contratacion"});

          this.typesConversion.put("11",new String[] {"euskadi","rrhh","rrhh"});

          this.typesConversion.put("35",new String[] {"euskadi","organizacion","institucion"});
          this.typesConversion.put("16",new String[] {"euskadi","organizacion","organo"});

          this.typesConversion.put("21",new String[] {"euskadi","estadisticas","estadistica"});

          this.typesConversion.put("22",new String[] {"euskadi","documentacion","informe_estudio"});
          this.typesConversion.put("28",new String[] {"euskadi","documentacion","libro"});
          this.typesConversion.put("23",new String[] {"euskadi","documentacion","memoria"});
          this.typesConversion.put("24",new String[] {"euskadi","documentacion","plan_programa_proyecto"});
          this.typesConversion.put("25",new String[] {"euskadi","documentacion","plan_programa_proyecto"});
          this.typesConversion.put("26",new String[] {"euskadi","documentacion","plan_programa_proyecto"});
          this.typesConversion.put("20",new String[] {"euskadi","documentacion","informe_estudio"});
          this.typesConversion.put("37",new String[] {"euskadi","documentacion","boletin_revista"});
          this.typesConversion.put("38",new String[] {"euskadi","documentacion","memoria"});
          this.typesConversion.put("51",new String[] {"euskadi","documentacion","inventario"});
          this.typesConversion.put("52",new String[] {"euskadi","documentacion","manual"});

          this.typesConversion.put("27",new String[] {"euskadi","directorios","centro"});

          this.typesConversion.put("34",new String[] {"euskadi","legislacion","normativa"});


          this.typesConversion.put("60",new String[] {"euskadi","turismo","a_alojamiento"});
          this.typesConversion.put("61",new String[] {"euskadi","turismo","b_restauracion"});
          this.typesConversion.put("62",new String[] {"euskadi","turismo","d_destinos_turisticos"});
          this.typesConversion.put("64",new String[] {"euskadi","turismo","c_transporte_y_movilidad"});
          this.typesConversion.put("65",new String[] {"euskadi","turismo","e_oficinas_turisticas"});
          this.typesConversion.put("66",new String[] {"euskadi","turismo","g_naturaleza"});
          this.typesConversion.put("67",new String[] {"euskadi","turismo","h_cultura_y_patrimonio"});
          this.typesConversion.put("68",new String[] {"euskadi","turismo","i_deportes"});
          this.typesConversion.put("69",new String[] {"euskadi","turismo","j_gastronomia"});
          this.typesConversion.put("70",new String[] {"euskadi","turismo","k_ocio"});
          this.typesConversion.put("71",new String[] {"euskadi","turismo","l_compras"});
          this.typesConversion.put("72",new String[] {"euskadi","turismo","n_negocios"});
      }
    /**
     * @return the clusterOid
     */
    public String getClusterOid() {
        return this.clusterOid;
    }
    /**
     * @param theClusterOid The clusterOid to set.
     */
    public void setClusterOid(String theClusterOid) {
        this.clusterOid = theClusterOid;
    }
    /**
     * @return the familyOid
     */
    public String getFamilyOid() {
        return this.familyOid;
    }
    /**
     * @param theFamilyOid The familyOid to set.
     */
    public void setFamilyOid(String theFamilyOid) {
        this.familyOid = theFamilyOid;
    }
    /**
     * @return the typoOid
     */
    public String getTypoOid() {
        return this.typoOid;
    }
    /**
     * @param theTypoOid The typoOid to set.
     */
    public void setTypoOid(String theTypoOid) {
        this.typoOid = theTypoOid;
    }
    /**
     * @return the typesConversion
     */
    public Map getTypesConversion() {
        return this.typesConversion;
    }
    /**
     * @param theTypesConversion The typesConversion to set.
     */
    public void setTypesConversion(Map theTypesConversion) {
        this.typesConversion = theTypesConversion;
    }



  }
}
