/*
 * Created on Jul 9, 2007
 * 
 * @author IE00165H - Alex Lara
 * (c) 2007 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.utils.searchengine;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.ejie.r01f.log.R01FLog;


/**
 * Encapsula el tratamiento de las queries a texto completo
 */
public class R01MFullTextQueryUtils {
///////////////////////////////////////////////////////////////////////////////////////////
//  FUNCIONES GENERALES
///////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * Filtra palabras prohibidas en la query
     * @param searchQueryString la query string
     * @param wordsToRemoveRegExp expresion regular para identificar las palabras a quitar
     * @return la cadena sin las palabras prohibidas
     */
    public static String filterWords(String searchQueryString,String wordsToRemoveRegExp) {
        StringBuffer outFilteredText = new StringBuffer("");
        Pattern p = Pattern.compile(wordsToRemoveRegExp,Pattern.CASE_INSENSITIVE);    // Machea palabras entre comillas ej: "soporte software"
        Matcher m = p.matcher(searchQueryString);
        boolean directiveValid = m.find();
        while (directiveValid) {
            m.appendReplacement(outFilteredText,"");                  
            // Volver a buscar
            directiveValid = m.find();                    
        }
        m.appendTail(outFilteredText);  
        return outFilteredText.toString();
    }     
///////////////////////////////////////////////////////////////////////////////////////////
//  PARSEO DE LA QUERY
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Arregla la query de bsqueda introduciendo la clusula AND
     * entre cada una de las palabras introducidas
     * @param searchQueryString la query de bsqueda
     * @param reservedWordsRegExp expresion regular que machea las palabras reservadas del buscador
     * @param searchEngineFunctionsRegExp expresion regular que machea las funciones del buscador
     * @return la query arreglada encapsulada en un objeto que incluye el nmero de
     *         palabras buscadas
     */
    public static FixedQuery fixSearchQueryString(final String searchQueryString,
                                                  final int maxFullTextStringLength,
                                                  final int maxFullTextWords,
                                                  final String wordsToRemoveRegExp,
                                                  final String reservedWordsRegExp,
                                                  final String searchEngineFunctionsRegExp) {
        
        // Obtener valores por defecto para los parametros
        int theMaxFullTextWords = maxFullTextWords > 0 ? maxFullTextWords : 20;
        int theMaxFullTextStringLength = maxFullTextStringLength > 0 ? maxFullTextStringLength : 255;
        String theWordsToRemoveRegExp = wordsToRemoveRegExp != null ? wordsToRemoveRegExp : "DELETE\\s*=\\s*TRUE";
        String theReservedWordsRegExp = reservedWordsRegExp != null ? reservedWordsRegExp : "\\b(?:OR|NOT|EOR|XOR|NEAR[0-9]{0,2})\\b";
        String theSearchEngineFunctionsRegExp = searchEngineFunctionsRegExp != null ? searchEngineFunctionsRegExp : "(?:DREFUZZY|SOUNDEX)\\s*\\([^)]+\\)";          
        
                
        // Primeros preprocesados        
        String theQueryString = _filterTags(searchQueryString);                 // Filtrar los tags html si los hay
        theQueryString = R01MFullTextQueryUtils.filterWords(theQueryString,theWordsToRemoveRegExp);      // Filtrar palabras prohibidas
        if (theQueryString.length() > theMaxFullTextStringLength) {
            theQueryString = theQueryString.substring(0,theMaxFullTextStringLength);   // truncar la cadena de busqueda a su tamao mximo
        }
        
        Collection extractedWordsCol = _extractQueryWords(theQueryString,theReservedWordsRegExp,theSearchEngineFunctionsRegExp);        
        
        // Construir la query de busqueda y contar las palabras introducidas en la caja
        ExtractedQueryWord currWord = null;
        ExtractedQueryWord prevWord = null;
        int searchedWordsCount = 0;        
        StringBuffer query = new StringBuffer("");
        for (Iterator it=extractedWordsCol.iterator(); it.hasNext(); ) {
            currWord = (ExtractedQueryWord)it.next();
            if ("\"\"".equals(currWord.word) || "''".equals(currWord.word)) {
                continue;
            }
            if (currWord.type == ExtractedQueryWord.WORD_TYPE_WORD) {
                searchedWordsCount++;
            }
            if (searchedWordsCount > theMaxFullTextWords) {
                break;       // Interrumpir
            }
            if (prevWord != null) {
                if (prevWord.type == currWord.type && currWord.type == ExtractedQueryWord.WORD_TYPE_WORD) {
                    query.append(" + ");
                } else if (!"(".equals(prevWord.word) && !")".equals(currWord.word)) {
                    query.append(' ');
                }
            }
            query.append(currWord.word);                   
            prevWord = currWord;                    
        }
        return new FixedQuery(searchedWordsCount,searchQueryString,query);
    }    
    /**
     * Extrae las diferentes palabras introducidas en una query de bsqueda
     * @param queryString query de bsqueda
     * @param reservedWordsRegExp expresion regular que machea las palabras reservadas del buscador
     * @param searchEngineFunctionRegExp expresion regular que machea las funciones del buscador
     * @return una lista con las palabras buscadas.
     */
    private static Collection _extractQueryWords(final String queryString,
                                                 final String reservedWordsRegExp,
                                                 final String searchEngineFunctionRegExp) {       
        String qryStr = new String(queryString);  // NOPMD by co00390i on 27/11/08 17:59
        
        List searchedWords = new ArrayList();       // Lista de palabras encontradas                                                    
        
        Pattern p = null;
        Matcher m = null;
        boolean directiveValid = false;
        String foundPattern = null;        
        StringBuffer queryString2 = null;
        
        // --PASO 1: Identificar las llamadas a funcion
        queryString2 = new StringBuffer("");
        p = Pattern.compile(searchEngineFunctionRegExp,Pattern.CASE_INSENSITIVE); // Machea las funciones DREFUZZY y SOUNDEX
        m = p.matcher(qryStr);       
        directiveValid = m.find();
        while (directiveValid) {
            foundPattern = qryStr.substring(m.start(),m.end());
            searchedWords.add( new ExtractedQueryWord(m.start(),foundPattern,ExtractedQueryWord.WORD_TYPE_FUNC) );
            // remplazar la cadena con la funcion por otra de igual tamao pero vacia
            char[] emptyStr = new char[foundPattern.length()];
            Arrays.fill(emptyStr,' ');
            m.appendReplacement(queryString2,new String(emptyStr));                  
            // Volver a buscar
            directiveValid = m.find();             
        }
        m.appendTail(queryString2);
        qryStr = queryString2.toString();
        //System.out.println(qryStr);
        
        // --PASO 2: Identificar los parntesis
        int pos = qryStr.indexOf('(',0);
        while (pos >= 0) {
            searchedWords.add( new ExtractedQueryWord(pos,"(",ExtractedQueryWord.WORD_TYPE_SYMBOL) );
            pos = qryStr.indexOf("(",pos+1);
        }
        qryStr = qryStr.replaceAll("\\("," ");
        pos = qryStr.indexOf(")",0);
        while (pos >= 0) {
            searchedWords.add( new ExtractedQueryWord(pos,")",ExtractedQueryWord.WORD_TYPE_SYMBOL) );
            pos = qryStr.indexOf(")",pos+1);
        }
        qryStr = qryStr.replaceAll("\\)"," "); 
        //System.out.println(qryStr);
       
        // --PASO 3: extraer las palabras entrecomilladas y sustituirlas 
        //           por una cadena de blancos del mismo tamao
        queryString2 = new StringBuffer("");
        p = Pattern.compile("(?:[\"'])[^\"']*(?:[\"'])",Pattern.CASE_INSENSITIVE);    // Machea palabras entre comillas ej: "soporte software"
        m = p.matcher(qryStr);
        directiveValid = m.find();
        while (directiveValid) {
            foundPattern = qryStr.substring(m.start(),m.end());
            searchedWords.add( new ExtractedQueryWord(m.start(),foundPattern,ExtractedQueryWord.WORD_TYPE_WORD) );
            // remplazar la cadena entrecomillada por otra de igual tamao pero vacia
            char[] emptyStr = new char[foundPattern.length()];
            Arrays.fill(emptyStr,' ');
            m.appendReplacement(queryString2,new String(emptyStr));                  
            // Volver a buscar
            directiveValid = m.find();                    
        }
        m.appendTail(queryString2);
        qryStr = queryString2.toString();        
        //System.out.println(qryStr);
        
        // --PASO 4: extraer las palabras independientes
        //          Nota: la expresion regular es compleja por la posible existencia de acentos... 
        //                para evitar "sorpresas" se pone en la expresion en hexadecimal (para )
        //          [\*\?]*\b(?:\w|[A\0240\0202\0241\0242\0243]|:|/|\*|\?|\.)+\b[\*\?\.]*
        p = Pattern.compile("[\\*\\?]*\\b(?:\\w|[\\xC0\\xC1\\xC7\\xC8\\xC9\\xCC\\xCD\\xD1\\xD2\\xD3\\xD9\\xDA\\xDC\\xE0\\xE1\\xE7\\xE8\\xE9\\xEC\\xED\\xEE\\xF1\\xF2\\xF3\\xF9\\xFA\\xFC]|:|/|\\*|\\?|\\.)+\\b[\\*\\?\\.]*",Pattern.CASE_INSENSITIVE);              // Machea palabras completas 
        m = p.matcher(qryStr);
        directiveValid = m.find();
        while (directiveValid) {
            foundPattern = qryStr.substring(m.start(),m.end());
            // Ignorar la palabra AND
            if (!foundPattern.equalsIgnoreCase("AND")) {
                if (_isSearchEngineReservedWord(foundPattern,reservedWordsRegExp)) {
                    searchedWords.add( new ExtractedQueryWord(m.start(),foundPattern,ExtractedQueryWord.WORD_TYPE_RES) );
                } else {
                    searchedWords.add( new ExtractedQueryWord(m.start(),foundPattern,ExtractedQueryWord.WORD_TYPE_WORD) );
                }
            }
            // Volver a buscar
            directiveValid = m.find();
        } 
        
        // --PASO 5: Ordenar las palabras extraidas por su posicin en la cadena de entrada
        Collections.sort(searchedWords);                       
        
        return searchedWords;
    }
    /**
     * Comprueba si una palabra buscada es una palabra reservada del buscador
     * @param word la palabra buscada
     * @param reservedWordsRegExp expresion regular que machea las palabras reservadas del buscador
     * @return true si la palabra buscada es una palabra reservada del buscador
     */
    private static boolean _isSearchEngineReservedWord(final String word,
                                                       final String reservedWordsRegExp) {
        Pattern p = Pattern.compile(reservedWordsRegExp,Pattern.CASE_INSENSITIVE);
        Matcher m = p.matcher(word);
        if (m.matches()) {
            return true;
        }
        return false;
    }
    /**
     * Filtra los tags html de la queryString para evitar "ataques" de cross-scripting
     * @param searchQueryString la query string
     * @return la cadena sin los tags html
     */
    private static String _filterTags(String searchQueryString) {
        return R01MFullTextQueryUtils.filterWords(searchQueryString,"<(.|\\n)+?>");
    }   
///////////////////////////////////////////////////////////////////////////////////////////
//  CLASES AUXILIARES
///////////////////////////////////////////////////////////////////////////////////////////            
    /**
     * Encapsula una palabra extraida de la cadena de bsqueda, 
     * asocindola con su posicin dentro de la query
     */
    private static class ExtractedQueryWord implements Comparable {
        public static final int WORD_TYPE_SYMBOL = 0;
        public static final int WORD_TYPE_FUNC = 1;
        public static final int WORD_TYPE_RES = 2;
        public static final int WORD_TYPE_WORD = 9;
        
        public int position = -1;       // Posicin de la palabra en la query
        public String word = "";        // Palabra extraida
        public int type = WORD_TYPE_WORD;  // Palabra reservada
        /**
         * Constructor vacio
         */
        public ExtractedQueryWord() {            
        }
        /**
         * Constructor en base a los miembros
         * @param newPosition
         * @param newWord
         * @param newType
         */
        public ExtractedQueryWord(int newPosition,String newWord,int newType) {
            this.position = newPosition;
            this.word = newWord;
            this.type = newType;
        }
        /* (non-Javadoc)
         * @see java.lang.Comparable#compareTo(java.lang.Object)
         */
        public int compareTo(Object o) {
            ExtractedQueryWord otherExtractedWord = (ExtractedQueryWord)o;
            return (new Integer(this.position)).compareTo(new Integer(otherExtractedWord.position));
        }        
    }
    /**
     * Encapsula el resultado del parseo de la query de autonomy
     * (es el objeto que devuelve la llamada a fixSearchQueryString
     */
    public static class FixedQuery {
        public int wordCount = 0;
        public String query = null;
        public StringBuffer fixedQuery = null;
        /**
         * Constructor vacio
         */        
        public FixedQuery() {           
        }
        /**
         * Constructor en base a los miembros
         * @param newWordCount
         * @param newQuery
         * @param newFixedQuery
         */
        public FixedQuery(int newWordCount,String newQuery,StringBuffer newFixedQuery) {
            this.wordCount = newWordCount;
            this.query = newQuery;
            this.fixedQuery = newFixedQuery;
        }
    }
    
////////////////////////////////////////////////////////////////////////////////////////////////
//  METODO MAIN 
////////////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * Main Method
     * @param args
     */
    public static void main(String[] args) {
        try {
            String wordsToRemoveRegExp = "delete\\s*=\\s*true";
            String reservedWordsRegExp = "\\b(?:OR|NOT|EOR|XOR|NEAR[0-9]{0,2})\\b";
            String searchEngineFunctionsRegExp = "(?:DREFUZZY|SOUNDEX)\\s*\\([^)]+\\)";                                                 
            String queryString = "<script>afafd&\"delete=true\"<script src='asdf'>Hola*hola*? (Alex:*/_as) NEAR5 drefuzzy(es el mejor) \"programador java\" del mundo 'mundial' eso es lo c'est la vie _unico_cierto";
            
            long t1 = System.currentTimeMillis();
            int testTimes = 1;
            for (int times = 0; times < testTimes; times++) {                
                R01FLog.to("r01m.test").fine(queryString);                
                R01FLog.to("r01m.test").fine("'" + fixSearchQueryString(queryString,255,20,wordsToRemoveRegExp,reservedWordsRegExp,searchEngineFunctionsRegExp).fixedQuery + "'");                
            }
            long t2 = System.currentTimeMillis();
    
            R01FLog.to("r01m.test").fine("\r\n\r\nTime Elapsed: " + ((t2 - t1) / 1000) + "sg - " + (t2 - t1) + "msg");
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    } 
}
