/*
 * Created on Sep 5, 2005
 *
 * @author ie00165h - Alex Lara
 * (c) 2005 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.utils;

import java.io.IOException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.ejie.r01f.file.ResourcesLoader;
import com.ejie.r01f.guids.GUIDDispenserDef;
import com.ejie.r01f.guids.GUIDDispenserException;
import com.ejie.r01f.guids.GUIDDispenserManager;
import com.ejie.r01f.log.R01FLog;
import com.ejie.r01f.util.StringUtils;
import com.ejie.r01f.xmlproperties.XMLProperties;
import com.ejie.r01m.utils.searchengine.R01MSearchEngineUtils;

/**
 * Utilidades comunes del API
 */
public class R01MUtils implements Serializable {
    private static final long serialVersionUID = -6911454312452044334L;
///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTANTES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Cache de recusos idiomticos
     */
    private static Map _langResourcesCache = new HashMap(5);

///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTORES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Constructor vacio
     */
    public R01MUtils() {
        super();
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Genera un identificador unico de la secuencia por defecto definida en el properties de la
     * aplicacin que se pasa por parametro
     * @param appCode aplicacion
     * @return el identificador
     * @throws GUIDDispenserException si no se puede generar el guid
     */
    public static String generateNewOid(String appCode) throws GUIDDispenserException {
        return R01MUtils.generateNewOid(appCode,"default");
    }
    /**
     * Genera un identificador unico de una secuencia definida en el properties de la
     * aplicacin que se pasa por parametro
     * @param appCode cdigo de la aplicacin
     * @param seq identificador de secuencia, el atributo id del elemento sequence.
     * @return el identificador generado
     * @throws GUIDDispenserException si no se puede generar el guid
     */
    public static String generateNewOid(String appCode,String seq) throws GUIDDispenserException {
        GUIDDispenserDef dispDef = new GUIDDispenserDef();
        if(appCode.startsWith("r01")) {
        	// Engaar al dispenser de oids para que coja la configuracin del dispenser desde R01M
            // pero luego cambiar el codigo de aplicacin para que la raz de los oids generados comience
            // por este codigo de app
        	dispDef.loadConfig(R01MConstants.API_APPCODE,seq);
        } else {
        	dispDef.loadConfig(appCode,seq);
        }
        dispDef.setAppCode(appCode);

        String oid = GUIDDispenserManager.createDispenser(dispDef).getGUID();
        R01FLog.to("r01m.util").finest("Generando oid %%%%%%%% " + oid + " %%%%%%%%");
        return oid;
    }
    /**
     * Minimiza una cadena muy larga hasta una longitud dada, sustituyendo el
     * resto por ...
     * Ej:  "Esto es una cadena muy muy larga"
     *      se sustiye por: "Esto es una caden..."
     * @param str La cadena a minimizar
     * @param len la longitud de la cadena resultante
     * @return La cadena minimizada
     */
    public static String shortString(String str, int len) {
        if (str != null && str.trim().length() > len)  {
            return str.trim().substring(0, len) + "...";
        }
        return str;
    }
    /**
     * Sustituye el apostrofe o comilla simple por valor "&#39;"
     * @param cadena
     * @return cadena sustituida
     */
    public static String encodeApostrophe(String cadena){
        if (cadena == null) {
            return cadena;
        }
        return cadena.replaceAll("'", "&#39;");
    }
    /**
     * Sustituye el apostrofe o comilla simple por valor "&#39;"
     * @param cadena
     * @return cadena sustituida
     */
    public static String obtainEncoding(String cadena) {

        // Como no se puede establecer a priori la codificacin en que le viene, se comprobar si est codificado
        // algn caracter en UTF-8. En caso de que no sea as se tomar la codificacin como ISO-8859-1

        // Caracteres especiales usados para UTF-8
        //     %C3%81     %C3%80     %C3%82     %C3%84     %C3%83     %C3%85
        //     %C3%89     %C3%8A     %C3%8B     %C3%88
        //     %C3%8D     %C3%8C     %C3%8E     %C3%8F
        //     %C3%93     %C3%92     %C3%94     %C3%96     %C3%95
        //     %C3%9A     %C3%99     %C3%9B     %C3%9C
        //
        //     %C3%A1     %C3%A0     %C3%A2     %C3%A4     %C3%A3     %C3%A5
        //     %C3%A9     %C3%A8     %C3%AA     %C3%AB
        //     %C3%AD     %C3%AC     %C3%AE     %C3%AF
        //     %C3%B3     %C3%B2     %C3%B4     %C3%B6     %C3%B5
        //     %C3%BA     %C3%B9     %C3%BB     %C3%BC
        //
        //     %C3%B1     %C3%91
        //     %C3%A7     %C3%87
        //
        // (libra)  %C2%A3  (euro)  %E2%82%AC

        String regEx = ".*(?:%C3%81|%C3%80|%C3%82|%C3%84|%C3%83|%C3%85" +
                       "|%C3%89|%C3%8A|%C3%8B|%C3%88" +
                       "|%C3%8D|%C3%8C|%C3%8E|%C3%8F" +
                       "|%C3%93|%C3%92|%C3%94|%C3%96|%C3%95" +
                       "|%C3%9A|%C3%99|%C3%9B|%C3%9C" +
                       "|%C3%A1|%C3%A0|%C3%A2|%C3%A4|%C3%A3|%C3%A5" +
                       "|%C3%A9|%C3%A8|%C3%AA|%C3%AB" +
                       "|%C3%AD|%C3%AC|%C3%AE|%C3%AF" +
                       "|%C3%B3|%C3%B2|%C3%B4|%C3%B6|%C3%B5" +
                       "|%C3%BA|%C3%B9|%C3%BB|%C3%BC" +
                       "|%C3%B1|%C3%91|%C3%A7|%C3%87|%C2%A3|%E2%82%AC).*";

        if (!StringUtils.isEmptyString(cadena)) {
            Pattern p = Pattern.compile(regEx);
            Matcher m = p.matcher(cadena);
            if (m.matches() || cadena.indexOf('%') < 0) {
                return R01MSearchEngineUtils.URL_ENCODING_UTF;
            }
        }
        return R01MSearchEngineUtils.URL_ENCODING_ISO_8859;
    }
    /**
     * Formateo de la informacin de un Mapa
     * @param mapa
     * @return mapa formateado en String
     */
    public static String composeDebugInfoOfMap(Map mapa){
        String result="Resultado del Mapa:";
        Map.Entry me;
        if (mapa != null) {
          for(Iterator iter=mapa.entrySet().iterator();iter.hasNext();){
                me = (Map.Entry) iter.next();
                result=result + ("\n\t Clave(") + (me.getKey().toString()) + ("), Valor(") + (me.getValue().toString()) + (")");
          }
        }
        return result;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  CARGA DE FICHEROS DE PROPIEDADES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Obtiene un literal de presentacin del buscador
     * El fichero de recursos a cargar se ha de definir en el fichero properties de la aplicacin en una
     * seccin como
     *      <[module]>
     *          <es>path del fichero de recursos en el castellano</es>
     *          <eu>path del fichero de recursos en el euskera</eu>
     *          ....
     *      </[module]>
     * @param appCode el codigo de aplicacion
     * @param module el identificador del recurso (modulo)
     * @param lang el lenguaje
     * @param literalKey la clave del literal
     * @return el literal
     */
    public static String getLiteral(String appCode,String module,
                                    String lang,String literalKey) {
        if (lang.length() == 2 && R01MConstants.DOCUMENT_ALLOWED_LANGS.toString().indexOf(lang) > 0) {
            Properties langResources = (Properties)_langResourcesCache.get(module + "_" + lang);
            if (langResources == null) {
                langResources = _loadLanguageResourceFile(appCode,module,lang);
            }
            //if (langResources == null) return "<!-- [No se han podido cargar los recursos de idioma para " + appCode + "." + module + "-" + lang + ": " + literalKey + "] -->";
            String outLit = langResources != null && literalKey!=null ? langResources.getProperty(literalKey) : null;
            return outLit != null ? outLit:"";
        }
        return "";
    }
    /**
     * Carga el fichero de recursos para un determinado idioma
     * @param appCode el codigo de aplicacion
     * @param module el modulo
     * @param lang el lenguaje
     * @return los recursos de idioma
     */
    private static Properties _loadLanguageResourceFile(String appCode,String module,String lang) {
        try {
            // Primer intentar ver si los recuros ya estn cargados en la cache
            Properties outProps = null;
            if (_langResourcesCache != null) {
                outProps = (Properties)_langResourcesCache.get(module + "_" + lang);
            }
            if (outProps == null) {
                // No se haban cargado an los properties
                outProps = new Properties();
                String languageResource = XMLProperties.getProperty(appCode,module + "/" + lang);
                if (languageResource == null) {
                    R01FLog.to("r01m.util").severe("NO se ha definido la propiedad " + appCode + "/" + module + "/" + lang + " donde se referencia el nombre del fichero de recursos de idioma!!!.\r\nREVISA EL FICHERO DE PROPIEDADES DE " + appCode);
                    languageResource = "/html/datos/" + module + "/" + module + "_" + lang;
                }
                outProps.load( ResourcesLoader.getResourceFileAsStream(languageResource) );
                // Cachear los recursos recien cargados
                _langResourcesCache.put(module + "_" + lang,outProps);
            }
            // Aqui seguro que se tienen los recursos, bien de la cache o bien recien cargado
            return outProps;
        } catch(IOException e) {
            e.printStackTrace(System.out);
            R01FLog.to("r01m.util").severe("Error cargando ficheros de idiomas: "+e.toString());
        }
        return new Properties();
    }
}
