package com.ejie.r01m.objects.searchengine.session;

import java.io.Serializable;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.logging.Level;

import lombok.Getter;
import lombok.experimental.Accessors;
import r01f.httpclient.HttpClient;
import r01f.httpclient.HttpConnectionStatement;
import r01f.httpclient.HttpRequestParameter;
import r01f.httpclient.HttpRequestStringParameter;
import r01f.util.Pager;
import r01f.util.types.Strings;
import r01mo.model.search.query.R01MSearchQuery;
import r01mo.model.search.results.presentation.R01MSearchResultsPresentation;

import com.ejie.r01f.clone.CloneUtils;
import com.ejie.r01f.log.R01FLog;
import com.ejie.r01f.util.StringUtils;
import com.ejie.r01f.xml.marshalling.XOManager;
import com.ejie.r01f.xml.marshalling.XOMarshallerException;
import com.ejie.r01f.xmlproperties.XMLProperties;
import com.ejie.r01m.exceptions.R01MSearchPerformerException;
import com.ejie.r01m.objects.searchengine.query.R01MQueryObject;
import com.ejie.r01m.objects.searchengine.results.R01MSearchResultItem;
import com.ejie.r01m.objects.searchengine.results.R01MSearchSessionResults;
import com.ejie.r01m.objects.searchengine.results.R01MSearchSourceResults;
import com.ejie.r01m.utils.R01MConstants;
import com.ejie.r01m.utils.searchengine.R01MSearchEngineUtils;


/**
 * Sesin de bsqueda que encapsula todos los datos que un CLIENTE del COORDINADOR de bsquedas debe
 * mantener en SESIN para tener paginacin.
 */
@Accessors(prefix="_")
public class R01MSearchSession implements Serializable {
    private static final long serialVersionUID = -3559593268518671473L;
///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTANTES
///////////////////////////////////////////////////////////////////////////////////////////
    // Propiedades para la presentacin de los resultados de paginacin
    /**
     * Tamao de pgina, propiedad para la presentacin de los resultados de paginacin
     */
    private static final String PAGE_SIZE = XMLProperties.get(R01MConstants.SEARCHENGINE_APPCODE,"presentationProperties/pageSize");
    /**
     * Tamao de bloques de pgina, propiedad para la presentacin de los resultados de paginacin
     */
    private static final String NAVBAR_BLOCK_SIZE = XMLProperties.get(R01MConstants.SEARCHENGINE_APPCODE,"presentationProperties/navBarBlockSize");
    /**
     * Logger.
     */
    private static final String LOG_TYPE_ID = "r01m.searchEngine";

    /**
     * Para evitar que se llame siquiera al logger.
     */
    private static final boolean DEBUG = (R01FLog.getLogLevel(R01MSearchSession.LOG_TYPE_ID).intValue() == Level.OFF.intValue() ? false:true);
    /**
     * Origen de resultados por defecto
     */
    private static final String DEFAULT_SOURCE = "contenidos.inter";
    /**
     * Fichero de mapeo para convertir de xml a objetos y vuelta
     */
    private static final String MAP_FILE = "/r01mQueryObjectMapping.xml";

///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Query que se esta ejecutando (sin gua de navegacion)
     */
    @Getter private R01MQueryObject _query = null;
    /**
     * Paginador
     */
    @Getter private Pager<Integer> _pager;;
    /**
     * Resultados de bsqueda
     */
    private R01MSearchSessionResults _searchSessionResults;

///////////////////////////////////////////////////////////////////////////////////////////
//  FLUNT API
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Factora de sesin de bsqueda a partir de un {@link R01MQueryObject}
     * @param qry el objeto query
     * @return la sesin de bsqueda
     */
    public static R01MSearchSession forQuery(final R01MQueryObject qry) throws R01MSearchPerformerException {
    	return new R01MSearchSession(qry);
    }
    /**
     * Factora de sesin de bsqueda a partir de un {@link R01MSearchQuery}
     * @param qry el objeto query
     * @return la sesin de bsqueda
     */
    public static R01MSearchSession forQuery(final R01MSearchQuery qry) throws R01MSearchPerformerException {
    	return R01MSearchSession.forQueryCustomizingResultsPresentation(qry,R01MSearchResultsPresentation.create());
    }
    /**
     * Factora de sessin de bsqueda a partir de un {@link R01MSearchQuery}
     * @param qry el objeto query
     * @param presentation la presentacin de resultados
     * @return la sesin de bsqueda
     */
    public static R01MSearchSession forQueryCustomizingResultsPresentation(final R01MSearchQuery qry,
    																	   final R01MSearchResultsPresentation presentation) throws R01MSearchPerformerException {
		R01MQueryObject qryV2 = R01MSearchQueryConverter.convert(qry,
																 presentation);
		return R01MSearchSession.forQuery(qryV2);
    }
    /**
     * Factora de sesin de bsqueda a partir de una query codificada en la url
     * @param qryURL query codificada en la url
     * @return la sesin de bsqueda
     */
    public static R01MSearchSession forQueryURL(final String qryURL) throws R01MSearchPerformerException {
        // Convertir el XML de bsqueda en objetos
        R01MQueryObject newQry = R01MSearchEngineUtils.decodeURLInQueryObject(qryURL);
        return R01MSearchSession.forQuery(newQry);
    }
    /**
     * Factora de sesin de bsqueda a partir de una query codificada en xml
     * @param qryXML la query codificada en xml
     * @return la sesin de bsqueda
     * @throws R01MSearchPerformerException
     */
    public static R01MSearchSession forQueryXML(final String qryXML) throws R01MSearchPerformerException {
	    R01MSearchSession outSession = null;
        try {
            // Convertir el XML de bsqueda en objetos
            R01MQueryObject newQry = (R01MQueryObject)XOManager.getObject(MAP_FILE,qryXML);
            outSession = new R01MSearchSession(newQry);
        } catch(XOMarshallerException xoEx) {
            throw new R01MSearchPerformerException("Error al convertir el XML de bsqueda a objetos: " + xoEx.getMessage(),xoEx);
        } 
        return outSession;
    }
    /**
     * Constructor en base al leguaje de la sesin de bsqueda y a la query
     * @param newQry la query
     * @throws R01MSearchInitializeException si ocurre un error al inicializar la bsqueda
     */
    private R01MSearchSession(final R01MQueryObject newQry) throws R01MSearchPerformerException {
        // Arreglar los parmetros si estos no llegan correctamente
        _query = newQry;                 	
        newQry.validate();		// Validar la query
        _query = _composeEffectiveQuery(_query);
        this.getSearchResults(1);		// Obtener la primera pgina
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  UTILIDADES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return la query como XML
     */
    public String queryAsXML() {
    	if (_query == null) return null;
    	String outXML = null;
        try {
            // Convertir el XML de bsqueda en objetos
            outXML = XOManager.getXML(MAP_FILE,_query);
        } catch(XOMarshallerException xoEx) {
            xoEx.printStackTrace(System.out);
        } 
        return outXML;
    }
    /**
     * @return la query codificada en la URL
     */
    public String queryAsURL() {
    	if (_query == null) return null;
    	try {
    		return _createSearchEngineConnection(_query,DEFAULT_SOURCE,_pager.getCurrentPageOrderNumber()).debugURL();
    	} catch (Exception ex) {
    		ex.printStackTrace(System.out);
    	}
    	return null;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS DE EJECUCIN DE BSQUEDAS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Obtiene los resultados de bsqueda para cada uno de los origenes
     * @param pageNum nmero de pgina
     * @return un objeto {@link R01MSearchSessionResults} que contiene:<br>
     *              - El objeto Pager con la informacin de paginacin (fase init).<br>
     *              - los items de resutlados
     * @throws R01MSearchPerformerException si hay algun error al ejecutar la bsqueda
     */
    private R01MSearchSessionResults _retrieveSearchSessionResults(final int pageNum) throws R01MSearchPerformerException {
        if (R01MSearchSession.DEBUG) {
            R01FLog.to(R01MSearchSession.LOG_TYPE_ID).info("\r\n\r\n{SEARCH_SESSION}>> Obtener los resultados...");
        }
        R01MSearchSessionResults outSession = _doQueryUsingHttp(_query,DEFAULT_SOURCE,pageNum);
    	_searchSessionResults = outSession;
        return outSession;
    }
    private static HttpConnectionStatement _createSearchEngineConnection(final R01MQueryObject query,
															  			 final String sourceId,
															  			 final int pageNum) throws MalformedURLException {
			// Crear los parametros: origen, pgina y query
			HttpRequestParameter srcParam = new HttpRequestStringParameter("r01kSrchSrcId",sourceId); 
			HttpRequestParameter pagParam = pageNum > 0 ? new HttpRequestStringParameter("r01kTgtPg",Integer.toString(pageNum)) : null;
			HttpRequestParameter pagParam2 = pageNum > 0 ? new HttpRequestStringParameter("r01kPgCmd",Integer.toString(pageNum)) : null;
			HttpRequestParameter qryParam = new HttpRequestStringParameter("r01kQry",R01MSearchEngineUtils.encodeQueryObjectInString(query));
			List<HttpRequestParameter> params = new ArrayList<HttpRequestParameter>(2);
			if (pagParam != null) params.add(pagParam);
			if (pagParam2 != null) params.add(pagParam2);
			params.add(srcParam);
			params.add(qryParam);
			
			// Base url del servicio
			String baseUrl = "http://www.euskadi.net/r01hSearchResultWar/r01hPresentationXML.jsp";
			
			// Hacer la llamada
			HttpConnectionStatement conx = HttpClient.forUrl(baseUrl).withParameters(params)
													 .usingGETRequestMethod();
										//.usingProxy("intercon","8080","xxx","yyy");
			return conx;
    }
	/**
	 * Realiza la query final utilizando HTTP
	 * @param query la query
	 * @param sourceId el identificador del origen
	 * @param pageNum el nmero de pgina
	 * @return los resultados
	 */
	private static R01MSearchSessionResults _doQueryUsingHttp(final R01MQueryObject query,
															  final String sourceId,
															  final int pageNum) throws R01MSearchPerformerException {
		if (R01MSearchSession.DEBUG) {
            R01FLog.to(R01MSearchSession.LOG_TYPE_ID).info("... fetching results from " + sourceId);
		}
		R01MSearchSessionResults outSession = null;
		try {
			// Hacer la llamada
			HttpConnectionStatement conx = _createSearchEngineConnection(query,sourceId,pageNum);
			String response = conx.loadAsString();
			if (response.trim().length() == 0) {
		        try {
		            // Un poco de log
		        	String qryURL = conx.debugURL();
		            String qryXML = XOManager.getXML(MAP_FILE,query);
		            R01FLog.to(R01MSearchSession.LOG_TYPE_ID).severe(qryURL + "\r\n" + qryXML);
		        } catch(XOMarshallerException xoEx) {
		            throw new R01MSearchPerformerException("Error al convertir el XML de bsqueda a objetos: " + xoEx.getMessage(),xoEx);
		        }
				throw new R01MSearchPerformerException("NO se han obtenido resultados para la query: comprueba que la query es correcta!");
			}
			if (R01MSearchSession.DEBUG) {
	            R01FLog.to(R01MSearchSession.LOG_TYPE_ID).finest("... resultXML: " + response);
			}
			// apa hay contenidos muy antiguos cuyo DCR tiene un nombre que empieza con un nmero
			// dado que este nmero en el XML de resultado pasa a ser un nombre de tag, FALLA el parseo del XML
			// Como solucin chapucera, se busca y sustituiye en el XML los tags numerico con una expresin regular.
			String responseFixed = Strings.of(response)
										  .replaceAll("<([0-9])(.*)>(.*?)</\\1\\2>","<A$1$2>$3</A$1$2>")
										  .asString();
//			System.out.println(responseFixed);
			outSession = (R01MSearchSessionResults)XOManager.getObject(MAP_FILE,responseFixed);
		} catch(Exception ex) {
			throw new R01MSearchPerformerException(ex);
		}
		return outSession;
	} 
///////////////////////////////////////////////////////////////////////////////////////////
//  ACCESO A LOS RESULTADOS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Devuelve los resultados de la pgina actual
     * @return objetos {@link R01MSearchResultItem} con los resultados
     * @throws R01MSearchPerformerException si ocurre algn error al ejecutar la bsqueda
     */
    public R01MSearchResultItem[] getCurrentPageSearchResults() throws R01MSearchPerformerException {
    	return this.getSearchResults(_pager.getCurrentPageOrderNumber());
    }
    /**
     * Obtiene los elementos resultado de la bsqueda en un origen especfico.
     * Normalmente a este mtodo se llama despus de llamar a #getSearchResultsLazy() para
     * obtener los resultados de un origen especfico.
     * @param pageNum nmero de pgina
     * @return Un objeto que encapsula los resultados de la bsqueda en el origen (objetos {@link R01MSearchResultItem})
     * @throws R01MSearchPerformerException si ocurre algn error al ejecutar la bsqueda
     */
    public R01MSearchResultItem[] getSearchResults(final int pageNum) throws R01MSearchPerformerException {
        if (R01MSearchSession.DEBUG) {
            R01FLog.to(R01MSearchSession.LOG_TYPE_ID).info("\r\n\r\n{SEARCH_SESSION}>> Obtener los resultados de la pgina " + pageNum + " del origen " + DEFAULT_SOURCE);
        }
        R01MSearchSessionResults session = null;
        if (_searchSessionResults != null && _pager.getCurrentPageOrderNumber() == pageNum) {
        	session = _searchSessionResults;
        } else {
        	session = _retrieveSearchSessionResults(pageNum);
        	if (_pager == null) _pager = _initPager(session,pageNum);
        }
        return session != null && session.getSearchSourceResults(DEFAULT_SOURCE) != null ? session.getSearchSourceResults(DEFAULT_SOURCE).getResults()
        																				 : null;
    }
    private static Pager<Integer> _initPager(final R01MSearchSessionResults session,final int pageNum) {
    	Pager<Integer> pager = null;
    	R01MSearchSourceResults sourceResults = session.getSearchSourceResults(DEFAULT_SOURCE);
    	if (sourceResults != null) {
    		pager = new Pager<Integer>(sourceResults.getNumberOfResultsPerPage(),
    								   sourceResults.getNavBar().getNavBarBlockSize());
    		for (int i=1; i < sourceResults.getNumberOfResults(); i++) pager.addItem(i);
    		pager.goToPage(pageNum);
    	}
    	return pager;
    }
    /**
     * Pasa un array de objetos {@link R01MSearchResultItem} a una coleccin
     * @param items un array de objetos {@link R01MSearchResultItem}
     * @return una coleccin de objetos {@link R01MSearchResultItem}
     */
    public static Collection<R01MSearchResultItem> getSearchResultItemsAsCollection(final R01MSearchResultItem[] items) {
        if (items == null) return null;
        Collection<R01MSearchResultItem> outCol = new ArrayList<R01MSearchResultItem>(items.length);
        for (int i=0; i<items.length; i++) outCol.add(items[i]);
        return outCol;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS DE PAGINACIN
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Mueve el puntero de UN ORIGEN a la PRIMERA pgina y devuelve una lista con los resultados
     * en el origen indicado.
     * A este mtodo se llama una vez que se ha inicializado la bsqueda y se est paginando en un origen.
     * @return objetos {@link R01MSearchResultItem} con los resultados
     * @throws R01MSearchPerformerException si se produce algun error en el ejecutor de la bsqueda
     */
    public R01MSearchResultItem[] goToFirstPage() throws R01MSearchPerformerException {
    	return this.goToPage(1);
    }
    /**
     * Mueve el puntero DE UN ORIGEN a la anterior pgina y devuelve una lista con los resultados en
     * el origen indicado.
     * A este mtodo se llama una vez que se ha inicializado la bsqueda y se est paginando en un origen.
     * @return objetos {@link R01MSearchResultItem} con los resultados
     * @throws R01MSearchPerformerException si se produce algun error en el ejecutor de la bsqueda
     */
    public R01MSearchResultItem[] goToPrevPage() throws R01MSearchPerformerException {
    	_searchSessionResults = null;	// invalidar resultados
    	_pager.goToPrevPage();
    	return this.getSearchResults(_pager.getCurrentPageOrderNumber());
    }
    /**
     * Mueve el puntero de UN ORIGEN a la siguiente pgina y devuelve una lista con los resultados
     * en el origen indicado.
     * A este mtodo se llama una vez que se ha inicializado la bsqueda y se est paginando en un origen.
     * @return objetos {@link R01MSearchResultItem} con los resultados
     * @throws R01MSearchPerformerException si se produce algun error en el ejecutor de la bsqueda
     */
    public R01MSearchResultItem[] goToNextPage() throws R01MSearchPerformerException {
    	_searchSessionResults = null;	// invalidar resultados
    	_pager.goToNextPage();
    	return this.getSearchResults(_pager.getCurrentPageOrderNumber());
    }
    /**
     * Mueve el puntero de UN ORIGEN a la ULTIMA pgina y devuelve una lista con los resultados
     * en el origen indicado.
     * A este mtodo se llama una vez que se ha inicializado la bsqueda y se est paginando en un origen.
     * @return objetos {@link R01MSearchResultItem} con los resultados
     * @throws R01MSearchPerformerException si se produce algun error en el ejecutor de la bsqueda
     */
    public R01MSearchResultItem[] goToLastPage(final String sourceOid) throws R01MSearchPerformerException {
    	_searchSessionResults = null;	// invalidar resultados
    	_pager.goToLastPage();
    	return this.getSearchResults(_pager.getCurrentPageOrderNumber());
    }
    /**
     * Mueve el puntero DE UN ORIGEN a la pgina indicada y devuelve una lista con los resultados
     * en el origen tambin indicado.
     * A este mtodo se llama una vez que se ha inicializado la bsqueda y se est paginando en un origen.
     * @param pageNum Numero de la pgina
     * @return objetos {@link R01MSearchResultItem} con los resultados
     * @throws R01MSearchPerformerException si se produce algun error en el ejecutor de la bsqueda
     */
    public R01MSearchResultItem[] goToPage(final int pageNum) throws R01MSearchPerformerException {
    	if (_searchSessionResults != null && _pager.getCurrentPageOrderNumber() != pageNum) {
    		_searchSessionResults = null;	// invalidar resultados;
    	} 
    	_pager.goToPage(pageNum);
	    return this.getSearchResults(_pager.getCurrentPageOrderNumber());
    }
    /**
     * Mueve el puntero de UN ORIGEN a la ULTIMA pgina del bloque anterior y devuelve una lista con los resultados
     * en el origen indicado.
     * A este mtodo se llama una vez que se ha inicializado la bsqueda y se est paginando en un origen.
     * @return objetos {@link R01MSearchResultItem} con los resultados
     * @throws R01MSearchPerformerException si se produce algun error en el ejecutor de la bsqueda
     */
    public R01MSearchResultItem[] goToPrevBlockLastPage() throws R01MSearchPerformerException {
    	_searchSessionResults = null;	// invalidar resultados
    	_pager.goToPrevBlockLastPage();
    	return this.getSearchResults(_pager.getCurrentPageOrderNumber());
    }
    /**
     * Mueve el puntero de UN ORIGEN a la PRIMERA pgina del bloque siguiente y devuelve una lista con los resultados
     * en el origen indicado.
     * A este mtodo se llama una vez que se ha inicializado la bsqueda y se est paginando en un origen.
     * @return objetos {@link R01MSearchResultItem} con los resultados
     * @throws R01MSearchPerformerException si se produce algun error en el ejecutor de la bsqueda
     */
    public R01MSearchResultItem[] goToNextBlockFirstPage(final String sourceOid) throws R01MSearchPerformerException {
    	_searchSessionResults = null;	// invalidar resultados
    	_pager.goToNextBlockFirstPage();
    	return this.getSearchResults(_pager.getCurrentPageOrderNumber());
    }
    /**
     * <pre>
     * Construye la query que se va a ejecutar realmente
     * @param inQry La query que se ha pasado para ejecutar y que se modifica en este mtodo
     * @return La query real a ejecutar
     * @throws R01MSearchInitializeException error al construir la query
     */
    private static R01MQueryObject _composeEffectiveQuery(final R01MQueryObject inQry) {
        // Clonar la query original... no modificarla!!!!
        // OJO!!    Esto puede ser un punto de bajo rendimiento ya que el clonado por Reflection es lento...
        R01MQueryObject outQry = (R01MQueryObject)CloneUtils.cloneByReflection(inQry);

        // :::::::::::::::: PASO 0: COMPLETAR OPCIONES DE PRESENTACION
        // Asegurarse antes de nada que existen las propiedades por defecto para la presentacion de la paginacion
        // Poner propiedades por defecto para la presentacion de resultados
        if ( outQry.getPresentationProperties() == null || outQry.getPresentationProperty(R01MQueryObject.PRESENTATION_PAGE_SIZE) == null) {
        	outQry.setPresentationProperty(R01MQueryObject.PRESENTATION_PAGE_SIZE,(R01MSearchSession.PAGE_SIZE==null?"10":R01MSearchSession.PAGE_SIZE)); // Tamao de pgina
        }
        if ( outQry.getPresentationProperties() == null || outQry.getPresentationProperty(R01MQueryObject.PRESENTATION_NAVBAR_BLOCK_SIZE) == null) {
            outQry.setPresentationProperty( R01MQueryObject.PRESENTATION_NAVBAR_BLOCK_SIZE,
                                            (R01MSearchSession.NAVBAR_BLOCK_SIZE==null ? "9" : R01MSearchSession.NAVBAR_BLOCK_SIZE)); // Tamao del bloque de pginas de la barra de navegacin
        }
        // IMPORTANTE!!!
        //      Si la query NO contiene una bsqueda de texto libre:
        //          - Marcar isFullTextSearch = false
        //          - NO devolver adjuntos: solo se busca en los metaDatos
        if (StringUtils.isEmptyString(outQry.getFullText())) {
            outQry.setFullTextSearch(false);        // No es busqueda de texto libre
            outQry.setReturnAttachments(false);     // No devolver adjuntos
        }

        // LENGUAJE DE BUSQUEDA
        // Si no se establece el idioma de la query y en la bsqueda se ha expedicificado que se filtre por idioma de usuario
        // en caso de que el objeto bsqueda no lleve ningn idioma (isSearchByDefaultLang = true) se filtra por el idioma del usuario
//        if (outQry.getMetaData(R01MQueryObject.METADATA_LANGUAGE) == null && outQry.isSearchByDefaultLang()) {
//            outQry.addMetaData(R01MQueryObject.METADATA_LANGUAGE,R01MIndexableMetaData.OPERATION_EQUALS,R01MConstants.DEFAULT_LANG);
//        } 
        return outQry;     // La query modificada
    }
    
//    public static void main(String[] args) {
//    	String response = "<a><b><1aa2a>dsaa<dfads>fas</dfads></1aa2a><1aa2a>dsaa<dfads>fas</dfads></1aa2a><b></a>";
//		String responseFixed = Strings.of(response)
//									  .replaceAll("<([0-9])(.*)>(.*?)</\\1\\2>","<A$1$2>$3</A$1$2>")
//									  .asString();
//		System.out.println(response);
//		System.out.println(responseFixed);
//	}
}
