/*
 * Created on Aug 9, 2005
 *
 * @author IE00165H - Alex Lara
 * (c) 2005 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.objects.searchengine.results;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import lombok.Getter;
import lombok.Setter;

import com.ejie.r01f.log.R01FLog;
import com.ejie.r01m.objects.searchengine.guide.R01MSearchGuideElement;
import com.ejie.r01m.objects.searchengine.query.R01MQueryObject;

/**
 * <pre>
 * Objeto que encapsula los resultados de una bsqueda.
 * El objeto tiene:
 *      - Informacion sobre la sesion: numero total de resultados, resultados por origen, etc.
 *      - Resultados por cada uno de los orgenes.
 * </pre>
 */
public class R01MSearchSessionResults implements Serializable {
    private static final long serialVersionUID = -2124784900869445124L;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Identificador de la sesin de bsqueda
     */
    @Getter @Setter private String sessionOid;
    /**
     * El lenguaje de la bsqueda, por defecto castellano.
     */
    @Getter @Setter private String lang = "es";    // Por defecto castellano... aunque se podra cojer de un properties
    /**
     * Objeto query
     */
    @Getter @Setter private R01MQueryObject qry = null;
    /**
     * Nmero total de resultados (la suma de los resultados de todos
     * los orgenes)
     */
    @Getter @Setter private int totalNumberOfResults = -1;
    /**
     * Resultados por origen (elementos {@link R01MSearchSourceResults} indexados por el oid del origen)
     */
    @Getter @Setter private Map<String,R01MSearchSourceResults> searchResultsBySource = null;
    /**
     * Elementos pasados en la guia de navegacin
     * (el elemento actual es el ltimo del array)
     */
    @Getter @Setter private R01MSearchGuideElement[] guidePastElements = null;
    /**
     * Elementos futuros en la guia de navegacin
     */
    @Getter @Setter private R01MSearchGuideElement[] guideFutureElements = null;
///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTOR
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Constructor vaco
     */
    public R01MSearchSessionResults() {
        super();
    }
    /**
     * Constructor a partir de todos los miembros
     * @param newSessionOid Identificador de session
     * @param newLang Lenguaje
     * @param newTotalNumberOfResults Nmero total de resultados
     * @param newQuery objeto {@link R01MQueryObject} que ha dado lugar a estos resultados
     * @param newGuidePastElements Los elementos {@link R01MSearchGuideElement} pasados en la guia de navegacin
     * @param newGuideFutureElements Los elementos {@link R01MSearchGuideElement} futuros en la guia de navegacin
     */
    public R01MSearchSessionResults(String newSessionOid,String newLang,int newTotalNumberOfResults,
                                    R01MQueryObject newQuery,
                                    R01MSearchGuideElement[] newGuidePastElements,
                                    R01MSearchGuideElement[] newGuideFutureElements) {
        this.sessionOid = newSessionOid;
        this.lang = newLang;
        this.qry = newQuery;
        this.totalNumberOfResults = newTotalNumberOfResults;
        this.guidePastElements = newGuidePastElements;
        this.guideFutureElements = newGuideFutureElements;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Obtiene los resultados de la bsqueda en un origen. Devuelve null si el origen
     * no ha devuelto resultados.
     */
    public R01MSearchResultItem[] getSearchSourceResultItems(String searchSourceOid) {
        R01MSearchSourceResults srcResults = this.getSearchSourceResults(searchSourceOid);
        if (srcResults != null) return srcResults.getResults();
        return null;
    }
    /**
     * Obtiene los resultados de la bsqueda en un origen (objetos {@link R01MSearchResultItem})
     * encapsulados en un objeto {@link R01MSearchSourceResults}
     * @param searchSourceOid El identificador del origen de resultados
     * @return El objeto {@link R01MSearchSourceResults} que "encapsula" los elementos resultados de la bsqueda
     */
    public R01MSearchSourceResults getSearchSourceResults(String searchSourceOid) {
        if (this.searchResultsBySource == null) {
            return null;
        }
        R01MSearchSourceResults srcSearchResults = this.searchResultsBySource.get(searchSourceOid);
        if (srcSearchResults == null) {
            //R01FLog.to("r01m.searchEngine").warning("No existen datos para el origen de resultados de bsqueda " + searchSourceOid);
            return null;
        }
        return srcSearchResults;
    }
    /**
     * Establece los resultados de bsqueda de un origen. Es preciso el objeto {@link R01MSearchSourceResults}
     * exista en la coleccin de resultados de inicializaciones de cada origen.
     * @param searchSourceOid El identificador del origen
     * @param srcResults Un objeto que encapsula los elementos de resultado
     */
    public void setSearchSourceResults(String searchSourceOid,R01MSearchSourceResults srcResults) {
        if (searchSourceOid == null || srcResults == null) return;
        if (this.searchResultsBySource == null) {
            R01FLog.to("r01m.searchEngine").warning("No se pueden establecer los resultados del origen " + searchSourceOid + " ya que no se han inicializado los orgenes. se ha pasado la fase init?");
        }
        if(this.searchResultsBySource == null) {
        	this.searchResultsBySource = new HashMap<String,R01MSearchSourceResults>();
        }
        this.searchResultsBySource.put(searchSourceOid,srcResults);
    }

}
