/*
 * Created on 18-abr-2005
 *
 * @author co01556e
 */
package com.ejie.r01m.objects.searchengine.query;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.ejie.r01f.guids.GUIDDispenserException;
import com.ejie.r01f.util.StringUtils;
import com.ejie.r01f.xmlproperties.XMLProperties;
import com.ejie.r01m.utils.R01MConstants;
import com.ejie.r01m.utils.R01MUtils;
import com.ejie.r01m.utils.searchengine.R01MSearchEngineConstants;

/**
 * <pre>
 * <b><U>Modela una consulta al buscador.</U></b>
 * <p>
 * Se establece el correspondiente filtrado de la bsqueda que se quiere realizar y se parametrizan ciertos
 * valores sobre la presentacin de resultados.
 * <ul>
 *  <li>Parmetros generales de la bsqueda.</li>
 *  <li>Condiciones del modelo de CONTENIDOS.</li>
 *  <li>Condiciones de almacenamiento, es decir, filtro sobre el/los cluster,repositorio,area y workarea.</li>
 *  <li>Condiciones de tipologa, filtro sobre el/los cluster,familia,tipo.</li>
 *  <li>Condiciones aplicadas sobre los metadatos.</li>
 *  <li>Condiciones aplicadas sobre la catalogacin, tanto en ejes como geogrfica.</li>
 *  <li>Criterios de ordenacin (metadatos ordenables).</li>
 * </ul>
 * Los criterios de ordenacin que se utilizarn en las bsquedas son los siguientes:
 * <b>CRITERIOS DE ORDENACIN POR TIPOLOGA</b>
 * En principio los resultados sern ordenados por el metadato especificado en los fichero de tipologas (se sigue una herencia establecida
 * entre las distintas jerarquas de tipologa clusterIndependent --> cluster --> familia --> tipo) en todos ellos se puede establecer el
 * criterio de ordenacin mediante un bloque del XML de configuracin, p.e. para la familia <b>eventos</b> tiene especificado el siguiente
 * criterio de ordenacin en el fichero de tipologa/familia <code>"eventos.xml"</code>:
 * <code>
 * &lt;!--////////////////////////////////// ORDENACION ////////////////////////////////////////////////////////--&gt;
 * &lt;!-- Una lista de orderByMetaData --&gt;
 * &lt;inheritsDefaultOrderByMetaData&gt;false&lt;/inheritsDefaultOrderByMetaData&gt;
 * &lt;defaultOrderByMetaData&gt;
 *      &lt;orderByMetaData metaDataOid='eventStartDate' order='DESC'/&gt;
 * &lt;/defaultOrderByMetaData&gt;
 * </code>
 * Este bloque indica que no debe heredar los criterios de ordenacin superiores (inheritsDefaultOrderByMetaData a false) y que la ordenacin
 * por defecto para la familia eventos es por "eventStartDate" en descendente.
 * Esto es en lo que respecta a la ordenacin por defecto (si la familia no tiene definido un criterio de ordenacin se buscara en el
 * fichero de cluster, pero debera heredarlo).
 *
 * <b>CRITERIOS DE ORDENACIN POR DEFECTO</b>
 * El criterio de ordenacin(actual) por defecto es la Fecha de Publicacin del documento, este criterio est especificado en
 *  el fichero <k><b>searchMetadataStorageConfig.xml</b></k>:
 * <ul>
 *  <li>En las bsquedas de Base de Datos se define en:
 *  <code>
 *      &lt;db.defaultOrderBy&gt;R01K08T00.FECHA_PUBLICATIONINFO_08 DESC&lt;/db.defaultOrderBy&gt;
 *      &lt;db.defaultPublishOrderBy&gt;R01K08V00.FECHA_PUBLICATIONINFO_08 DESC&lt;/db.defaultPublishOrderBy&gt;
 *  </code></li>
 *  <li>En las bsquedas contra Autonomy se define en:
 *  <code>
 *      &lt;!-- Criterios de ordenacion por defecto para las busquedas en texto libre y por metaDatos (formularios) --&gt;
 *      &lt;autonomy.fullTextSearchDefaultOrderBy&gt;Relevance&lt;/autonomy.fullTextSearchDefaultOrderBy&gt;
 *      &lt;autonomy.metaDataSearchDefaultOrderBy&gt;pubTargets_$PUBTARGET_OID$:numberdecreasing&lt;/autonomy.metaDataSearchDefaultOrderBy&gt;
 *  </code></li>
 * </ul>
 *
 * <b>CRITERIOS DE ORDENACIN ESPECIFICADOS EN LA BSQUEDA</b>
 * Se puede especificar un metadato para establecer la ordenacin, p.e.:
 * ?r01kQry=tC:euskadi;tT:evento;m:documentPublicationInfo.EQ.1,documentLanguage.EQ.es,eventType.EQ.0100;pp:r01PageSize.1000;o:documentName.DESC
 * Al definir una ordenacin en la propia query, es prioritaria sobre la definida a nivel de los metadatos especficados para la tipologa.
 * </pre>
 */
public class R01MQueryObject implements Serializable {
    /*(non-Javadoc)
     */
    private static final long serialVersionUID = -4850345987691922295L;
///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTANTES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Identificador de las propiedades para la presentacin para definir el tamao de elementos resulltado de cada pgina.
     */
    public static final String PRESENTATION_PAGE_SIZE = "r01PageSize";
    /**
     * Identificador de las propiedades para la presentacin para definir el nmero de bloques(o enlaces a pginas) a mostrar.<br>
     * Aparece en la gua de navegacin como:<br>
     * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<pre>1 | <u>2</u> | <u>3</u> | <u>4</u> ... <u>Siguiente</u></pre><br>
     * En este caso el tamao de bloque es 4.
     */
    public static final String PRESENTATION_NAVBAR_BLOCK_SIZE = "r01NavBarBlockSize";
    /**
     * Nombre del metadato para el idioma de los documentos.
     */
    public static final String METADATA_LANGUAGE = "documentLanguage";
    /**
     * Nombre del metadato para el texto completo.
     */
    public static final String METADATA_FULLTEXT = "fullText";
    

    /**
     * Estado del documento para las nuevas interfaces de usuario.
     * Se corresponde con :
     * 		Estado de revisin: En redaccin o Pendiente de alguna revisin.
     * 		Estado de publicacin: Despublicado o Nunca publicado.
     */
    public static final int DOC_UI_STATE_NOTPUBLISHED = 0;
    /**
     * Estado del documento para las nuevas interfaces de usuario.
     * Se corresponde con :
     * 		Estado de revisin: Aprobado.
     * 		Estado de publicacin: Publicado.
     */
    public static final int DOC_UI_STATE_PUBLISHED = 1;
    /**
     * Estado del documento para las nuevas interfaces de usuario.
     * Se corresponde con :
     * 		Estado de revisin: En redaccin o Pendiente de alguna revisin.
     * 		Estado de publicacin: Publicado.
     */
    public static final int DOC_UI_STATE_PUBLISHED_AND_MODIFIED = 2;
    /**
     * Valor por defecto para el nmero de resultados por pgina
     */
    public static final transient int DEFAULT_RESULTITEMS_PER_PAGE = XMLProperties.getInt("r01m","searchEngine/numberOfResultItemsPerPage",10);
    /**
     * Nmero mximo de resultados por pgina
     */
    public static final transient int MAX_RESULTITEMS_PER_PAGE = XMLProperties.getInt("r01m","searchEngine/maxNumberOfResultItemsPerPage",1000);

///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS: GENERAL
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Identificador de la query, es nico.
     */
    public String queryOid;
    /**
     * Indica si se ha modificado la query, no se debe modificar este valor por parte de aplicaciones cliente.<br>
     * <pre>
     *      Se utiliza en la parte web para "indicar" al request processor que tiene
     *      que reinicializar la session ya que la query se ha modificado.
     * </pre>
     */
    public boolean modified;
    /**
     * Identificador del motor o mtodo de bsqueda con el que ejecutar la query.<br>
     * <pre>
     *      Si un origen de resultados soporta ms de un mtodo para ejecutar las bsquedas
     *      (texto libre, base de datos, buscador en euskara, etc), en este campo, se indica
     *      el motor o el mtodo que se va a utilizar.
     * </pre>
     */
    public String searchEngineId;
    /**
     * Contexto desde el que se est ejecutando la query.
     * El buscador estar instalado en un contexto: intranet, internet, extranet, etc
     * Desde este contexto, se podrn hacer bsquedas en los orgenes de resultados cuya
     * propiedad contexto coincida con la del buscador.
     */
    public String context;
    /**
     * Nombre de la query.
     */
    public String queryName;
    /**
     * Descripcin de la Query.<br>
     * <b>(Tamao mximo de 255 caracteres).</b>
     */
    public String queryDescription;
    /**
     * Propiedades de presentacin.<br>
     * Las propiedades que pueden ser establecidas en estos momentos son dos:<br>
     * <ul>
     * <li>r01NavBarBlockSize,tamao de bloque (nmero de pginas iniciales).</li>
     * <li>r01PageSize,tamao de pgina.</li>
     * </ul>
     */
    public Map presentationProperties = null;
    /**
     * Indica si se guarda la Query o no, en Base de Datos.
     */
    public boolean saveQuery = false;
    /**
     * Indica si la query viene de una bsqueda a texto completo (buscador de texto libre).
     */
    public boolean isFullTextSearch = false;

    /**
     * Indica si queremos paginar o no: true -> NO Paginar, devuelve todos los resultados; Para el resto de los casos paginaremos
     */
    public boolean allResultsInAPage = false;

///////////////////////////////////////////////////////////////////////////////////////////
//  CONDICIONES DEL MODELO
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Indica si la bsqueda debe devolver documentos de los contenidos
     * o nicamente contenidos.
     */
    public boolean returnDocuments = true;
    /**
     * Indica si la bsqueda debe devolver archivos de datos de los documentos.
     */
    public boolean returnDataFiles = true;
    /**
     * Indica si la bsqueda debe devolver adjuntos de los documentos.
     */
    public boolean returnAttachments = true;
    /**
     * Estado del documento.<br>
     * <ul>
     * <li>-5: Pendiente de 5 aprobaciones.</li>
     * <li>....</li>
     * <li>-1: Pendiente de 1 aprobacin.</li>
     * <li> 0: Aprobado.</li>
     * <li> 1: En redaccin.</li>
     * </ul>
     */
    public String status;
    /**
     * Lista de repositorios de publicacion (objetos {@link R01MSearchedPublishRepository}).
     */
    public List publishRepositories;
    /**
     * Indica si hay que buscar elementos NO publicados, por defecto nicamente elementos publicados!!!.
     */
    public boolean publishedItemsOnly = true;    
    /**
     * Indica si hay que buscar elementos nicamente NO publicados.
     * Actualemente este parmetro slo es efectivo contra Base de datos,
     * ya que autonomy actualmente no indexa contenidos NO publicados
     */
    public boolean onlyNotPublished = false;
    /**
     * Estado del documento en la interfaz del usuario (Sin publicar, Publicado, Publicado Modificado).
     * @see R01MDocument#DOC_UI_STATE_NOTPUBLISHED
     * @see R01MDocument#DOC_UI_STATE_PUBLISHED
     * @see R01MDocument#DOC_UI_STATE_PUBLISHED_AND_MODIFIED
     */
    public int documentUIState;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS: ALMACENAMIENTO
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Lista de servidores en los que se busca (objetos {@link R01MSearchedServer}).
     */
    public List servers;
    /**
     * Lista de repositorios de datos en los que se busca (objetos {@link R01MSearchedDataRepository}).
     */
    public List dataRepositories;
    /**
     * Lista de areas en las que se busca (objetos {@link R01MSearchedArea}).
     */
    public List areas;
    /**
     * Lista de workAreas en las que se busca (objetos {@link R01MSearchedWorkArea}).
     */
    public List workAreas;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS: TIPOLOGIA
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Clusters en los que se busca (objetos {@link R01MSearchedCluster}).
     */
    public List contentClusters;
    /**
     * Lista de familias en las que se busca (objetos {@link R01MSearchedFamily}).
     */
    public List contentFamilies;
    /**
     * Lista de tipos de contenido en los que se busca (objetos {@link R01MSearchedContentType}).
     */
    public List contentTypes;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS: METADATA
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Lista de metaDatos AND y las condiciones sobre cada uno de ellos (objetos {@link R01MSearchedMetaData}).
     */
    public List searchedMetaData;
    /**
     * Lista de metaDatos OR y las condiciones sobre cada uno de ellos (objetos {@link R01MSearchedMetaData}).
     */
    public List orSearchedMetaData;
    /**
     * Nmero de palabras que se buscan como texto completo.<br>
     * Este nmero se establece al pre-procesar la query en el objeto {@link com.ejie.r01m.searchengine.servlet.R01MBaseSearchEngineWebSessionManager}.<br>
     * <b>NOTA:</b> Los grupos de palabras entrecomilladas se cuentan como una sola palabra.<br>
     */
    public int fullTextWordCount = -1;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS: CATALOGACION
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Lista de condiciones AND sobre las etiquetas de catalogacion (objetos {@link R01MSearchedStructureCatalog}).
     */
    public List catalogLabelsAnd;
    /**
     * Lista de condiciones OR sobre las etiquetas de catalogacion (objetos {@link R01MSearchedStructureCatalog}).
     */
    public List catalogLabelsOr;
    /**
     * Lista de condiciones AND sobre la catalogacion geografica (objetos {@link R01MSearchedGeoCatalog}).
     */
    public List geoCatalogsAnd;
    /**
     * Lista de condiciones OR sobre la catalogacion geografica (objetos {@link R01MSearchedGeoCatalog}).
     */
    public List geoCatalogsOr;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS: ORDENACION
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Lista de metadatos sobre los que se ordena (objetos tipo {@link R01MSearchedOrderByMetaData}).
     */
    public List orderBy;
// /////////////////////////////////////////////////////////////////////////////////////////
// CONDICIONES DE BSQUEDA POR IDIOMA
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Indica si filtra por idioma por defecto en caso de que el objeto bsqueda no lleve ningn metadatado de idioma.
     * <code>true</code> Filtra por idioma por defecto en caso de que el objeto bsqueda no lleve ningn metadatado de idioma
     * <code>false</code> No filtra por idioma por defecto en caso de que el objeto bsqueda no lleve ningn metadatado de idioma
     * Se utiliza, por ejemplo, para la busqueda simple en el gestor de contenidos que queremos que nos busque los documentos independientemente
     * del idioma por defecto o del usuario.
     * Por defecto inicializado a true para que busque por el idioma por defecto.
     */
    public boolean searchByDefaultLang = true;

///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTORES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Constructor vaco.
     */
    public R01MQueryObject() {
        // Generar un nuevo oid para la query
        try {
            this.queryOid = R01MUtils.generateNewOid(R01MConstants.API_APPCODE);
        } catch (GUIDDispenserException guidEx) {
            guidEx.printStackTrace(System.out);
            this.queryOid = "error";
        }
        this.queryName = null;
        this.queryDescription = null;
        this.saveQuery = false;
        this.publishRepositories = null;
        this.searchByDefaultLang = true;

        this.contentClusters = new ArrayList();
        this.contentFamilies = new ArrayList();
        this.contentTypes = new ArrayList();
        this.searchedMetaData = new ArrayList();
        this.orSearchedMetaData = new ArrayList();
        this.catalogLabelsAnd = new ArrayList();
        this.catalogLabelsOr = new ArrayList();
        this.geoCatalogsAnd = new ArrayList();
        this.geoCatalogsOr = new ArrayList();
        this.orderBy = new ArrayList();

        this.servers = new ArrayList();
        this.dataRepositories = new ArrayList();
        this.areas = new ArrayList();
        this.workAreas = new ArrayList();
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS PARA GENERAR EL OID DE LA QUERY
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Genera un nuevo identificador para la query.
     */
    public void generateNewQueryOid() {
        try {
            this.queryOid = R01MUtils.generateNewOid(R01MConstants.API_APPCODE);
        } catch (GUIDDispenserException guidEx) {
            guidEx.printStackTrace(System.out);
            this.queryOid = "error";
        }
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS AUXILIARES PARA AADIR CONDICIONES DE ALMACENAMIENTO
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Aade un servidor en el que buscar contenidos.
     * @param searchedServer el objeto {@link R01MSearchedServer}
     * @throws IllegalArgumentException Si el oid del servidor no es valido
     */
    public void addServer(final R01MSearchedServer searchedServer) throws IllegalArgumentException {
        if (searchedServer != null) {
            if (StringUtils.isEmptyString(searchedServer.serverOid)) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid del servidor a buscar NO es valido!");
            }
            if (this.servers == null) {
                this.servers = new ArrayList();
            }
            if(!_listContains(this.servers,searchedServer)) {
                this.servers.add(searchedServer);
            }
        }
    }
    /**
     * Aade objeto {@link R01MSearchedServer} en el que buscar contenidos.
     * @param serverOid El identificador del servidor
     * @param serverName El nombre del servidor
     * @throws IllegalArgumentException Si el oid del servidor no es valido
     */
    public void addServer(final String serverOid,final String serverName) throws IllegalArgumentException {
        this.addServer( new R01MSearchedServer(serverOid,serverName) );
    }
    /**
     * Aade un objeto {@link R01MSearchedServer} en el que buscar contenidos.
     * @param serverOid Identificador del servidor
     * @throws IllegalArgumentException Si el oid del servidor no es valido
     */
    public void addServer(final String serverOid) throws IllegalArgumentException {
        this.addServer(serverOid,null);
    }
    /**
     * Aade un objeto {@link R01MSearchedServer} a partir de su clave.
     * @see R01MSearchedServer#fromKey(String)
     * @param serverKey la clave
     * @throws IllegalArgumentException si el objecto a aadir no es valido
     */
    public void addServerFromKey(final String serverKey) throws IllegalArgumentException {
        R01MSearchedServer serverToAdd = new R01MSearchedServer();
        serverToAdd.fromKey(serverKey);
        this.addServer(serverToAdd);
    }
    /**
     * Devuelve el objeto {@link R01MSearchedServer} buscado.
     * @param serverOid identificador del servidor
     * @return el servidor buscado
     */
    public R01MSearchedServer getServer(final String serverOid) {
        return (R01MSearchedServer)_findSearchedStorageObj(this.servers,serverOid);
    }
    /**
     * Elimina un objecto {@link R01MSearchedServer} buscado de la lista de servidores.
     * @param searchedServerOid identificador del servidor a eliminar
     */
    public void removeServer(final String searchedServerOid) {
        _removeSearchedStorageObj(this.servers,searchedServerOid);
    }
    /**
     * Aade un repositorio de datos (objeto {@link R01MSearchedDataRepository}) en el que buscar.
     * @param searchedDataRepository el objeto {@link R01MSearchedDataRepository}
     * @throws IllegalArgumentException Si el oid del repositorio de datos no es valido
     */
    public void addDataRepository(final R01MSearchedDataRepository searchedDataRepository) throws IllegalArgumentException {
        if (searchedDataRepository != null) {
            if (searchedDataRepository.dataRepositoryOid == null || searchedDataRepository.dataRepositoryOid.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid del repositorio de datos a buscar NO es valido!");
            }
            if (this.dataRepositories == null) {
                this.dataRepositories = new ArrayList();
            }
            if(!_listContains(this.dataRepositories,searchedDataRepository)) {
                this.dataRepositories.add(searchedDataRepository);
            }
        }
    }
    /**
     * Aade un repositorio de datos (objeto {@link R01MSearchedDataRepository}) en el que buscar contenidos.
     * @param serverOid El identificador del servidor en el que esta contenido el repositorio
     * @param dataRepositoryOid El identificador del repositorio de datos
     * @param dataRepositoryName El nombre del repositorio de datos
     * @throws IllegalArgumentException Si el oid del repositorio de datos no es valido
     */
    public void addDataRepository(final String serverOid,final String dataRepositoryOid,final String dataRepositoryName) throws IllegalArgumentException {
        this.addDataRepository( new R01MSearchedDataRepository(serverOid,dataRepositoryOid,dataRepositoryName) );
    }
    /**
     * Aade un repositorio de datos (objeto {@link R01MSearchedDataRepository}) en el que buscar contenidos.
     * @param serverOid Identificacin del servidor
     * @param dataRepositoryOid Identificacin del repositorio de datos
     * @throws IllegalArgumentException Si el oid del repositorio de datos no es valido
     */
    public void addDataRepository(final String serverOid,final String dataRepositoryOid) throws IllegalArgumentException {
        this.addDataRepository(serverOid,dataRepositoryOid,null);
    }
    /**
     * Aade un repositorio de datos (objeto {@link R01MSearchedDataRepository}) a partir de su clave.
     * @see R01MSearchedDataRepository#fromKey(String)
     * @param drKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addDataRepositoryFromKey(final String drKey) throws IllegalArgumentException {
        R01MSearchedDataRepository drToAdd = new R01MSearchedDataRepository();
        drToAdd.fromKey(drKey);
        this.addDataRepository(drToAdd);
    }
    /**
     * Devuelve un repositorio de datos (objeto {@link R01MSearchedDataRepository}) buscado.
     * @param dataRepositoryOid identificador del repositorio de datos
     * @return el objeto buscado
     */
    public R01MSearchedDataRepository getDataRepository(final String dataRepositoryOid) {
        return (R01MSearchedDataRepository)_findSearchedStorageObj(this.dataRepositories,dataRepositoryOid);
    }
    /**
     * Elimina un repositorio de datos (objeto {@link R01MSearchedDataRepository}) buscado de la lista de dataRepositories.
     * @param searchedDataRepOid identificador del dataRepository a eliminar
     */
    public void removeDataRepository(final String searchedDataRepOid) {
        _removeSearchedStorageObj(this.dataRepositories,searchedDataRepOid);
    }
    /**
     * Aade un rea (objeto {@link R01MSearchedArea}) en el que buscar contenidos.
     * @param searchedArea el objeto {@link R01MSearchedArea}
     * @throws IllegalArgumentException Si el oid del rea de datos no es valida
     */
    public void addArea(final R01MSearchedArea searchedArea) throws IllegalArgumentException {
        if (searchedArea != null) {
            if (searchedArea.areaOid == null || searchedArea.areaOid.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid del area a buscar NO es valido!");
            }
            if (this.areas == null) {
                this.areas = new ArrayList();
            }
            if(!_listContains(this.areas,searchedArea)) {
                this.areas.add(searchedArea);
            }
        }
    }
    /**
     * Aade un area (objeto {@link R01MSearchedArea}) en la que buscar.
     * @param serverOid El identificador del servidor en el que esta contenido el area
     * @param dataRepositoryOid El identificador del repositorio de datos en el que esta contenido el area
     * @param areaOid El identificador del rea
     * @param areaName El nombre del rea
     * @throws IllegalArgumentException Si el oid del rea de datos no es valida
     */
    public void addArea( final String serverOid,
                         final String dataRepositoryOid,
                         final String areaOid,
                         final String areaName) throws IllegalArgumentException {
        this.addArea( new R01MSearchedArea(serverOid,dataRepositoryOid,areaOid,areaName) );
    }
    /**
     * Aade un area (objeto {@link R01MSearchedArea}) en la que buscar.
     * @param serverOid El identificador del servidor en el que esta contenido el area
     * @param dataRepositoryOid El identificador del repositorio de datos en el que esta contenido el area
     * @param areaOid El identificador del rea
     * @throws IllegalArgumentException Si el oid del rea de datos no es valida
     */
    public void addArea(final String serverOid,
                        final String dataRepositoryOid,
                        final String areaOid) throws IllegalArgumentException {
        this.addArea(serverOid,dataRepositoryOid,areaOid,null);
    }
    /**
     * Aade un area (objeto {@link R01MSearchedArea}) a partir de su clave.
     * @see R01MSearchedArea#fromKey(String)
     * @param areaKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addAreaFromKey(final String areaKey) throws IllegalArgumentException {
        R01MSearchedArea areaToAdd = new R01MSearchedArea();
        areaToAdd.fromKey(areaKey);
        this.addArea(areaToAdd);
    }
    /**
     * Obtiene un area (objeto {@link R01MSearchedArea}) buscada.
     * @param areaOid identificador del area buscada
     * @return el objeto area
     */
    public R01MSearchedArea getArea(final String areaOid) {
        return (R01MSearchedArea)_findSearchedStorageObj(this.areas,areaOid);
    }
    /**
     * Elimina un area (objeto {@link R01MSearchedArea}) buscado de la lista de areas.
     * @param searchedAreaOid identificador del area a eliminar
     */
    public void removeArea(final String searchedAreaOid) {
        _removeSearchedStorageObj(this.areas,searchedAreaOid);
    }
    /**
     * Aade una workArea (objeto {@link R01MSearchedWorkArea}) en la que buscar contenidos.
     * @param searchedWorkArea objeto R01MSearchedWorkArea
     * @throws IllegalArgumentException Si el oid de la workArea no es vlida
     */
    public void addWorkArea(final R01MSearchedWorkArea searchedWorkArea) throws IllegalArgumentException {
        if (searchedWorkArea != null) {
            if (searchedWorkArea.workAreaOid == null || searchedWorkArea.workAreaOid.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid del WorkArea a buscar NO es valido!");
            }
            if (this.workAreas == null) {
                this.workAreas = new ArrayList();
            }
            if(!_listContains(this.workAreas,searchedWorkArea)) {
                this.workAreas.add(searchedWorkArea);
            }
        }
    }
    /**
     * Aade una workarea (objeto {@link R01MSearchedWorkArea}) en la que buscar.
     * @param serverOid El identificador del servidor en el que esta contenido el workArea
     * @param dataRepositoryOid El identificador del repositorio de datos en el que esta contenido el workArea
     * @param areaOid El identificador del area en el que esta contenido el workArea
     * @param workAreaOid Identificador de la workarea
     * @param workAreaName Nombre de la workArea
     * @throws IllegalArgumentException Si el oid de la workArea no es vlida
     */
    public void addWorkArea( final String serverOid,
                             final String dataRepositoryOid,
                             final String areaOid,
                             final String workAreaOid,
                             final String workAreaName) throws IllegalArgumentException {
        this.addWorkArea( new R01MSearchedWorkArea(serverOid,dataRepositoryOid,areaOid,workAreaOid,workAreaName) );
    }
    /**
     * Aade una workarea (objeto {@link R01MSearchedWorkArea}) en la que buscar.
     * @param serverOid El identificador del servidor en el que esta contenido el workArea
     * @param dataRepositoryOid El identificador del repositorio de datos en el que esta contenido el workArea
     * @param areaOid El identificador del area en el que esta contenido el workArea
     * @param workAreaOid Identificador de la workarea
     * @throws IllegalArgumentException Si el oid de la workArea no es vlida
     */
    public void addWorkArea( final String serverOid,
                             final String dataRepositoryOid,
                             final String areaOid,
                             final String workAreaOid) throws IllegalArgumentException {
        this.addWorkArea(serverOid,dataRepositoryOid,areaOid,workAreaOid,null);
    }
    /**
     * Aade una workArea (objeto {@link R01MSearchedWorkArea}) a partir de su clave.
     * @see R01MSearchedWorkArea#fromKey(String)
     * @param waKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addWorkAreaFromKey(final String waKey) throws IllegalArgumentException {
        R01MSearchedWorkArea waToAdd = new R01MSearchedWorkArea();
        waToAdd.fromKey(waKey);
        this.addWorkArea(waToAdd);
    }
    /**
     * Devuelve la workArea (objeto {@link R01MSearchedWorkArea}) buscada cuyo oid coincide con el que se pasa.
     * @param workAreaOid identificador de la workArea
     * @return la workArea encontrada
     */
    public R01MSearchedWorkArea getWorkArea(final String workAreaOid) {
        return (R01MSearchedWorkArea)_findSearchedStorageObj(this.workAreas,workAreaOid);
    }
    /**
     * Elimina un workArea (objeto {@link R01MSearchedWorkArea}) buscado de la lista de areas.
     * @param searchedWorkAreaOid identificador del area a eliminar
     */
    public void removeWorkdArea(final String searchedWorkAreaOid) {
        _removeSearchedStorageObj(this.workAreas,searchedWorkAreaOid);
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS PRIVADOS PARA ACTUAR SOBRE LA COLECCION DE OBJETOS DE ALMACENAMIENTO
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Encuentra un objeto de tipologia.
     * @param stgObjsList la lista de objetos de tipologia buscados
     * @param searchedStgObjKey identificador del objeto de tipologia a eliminar
     * @return el objeto de tipologa buscado
     */
    private R01MBaseSearchedStorageObj _findSearchedStorageObj( final List stgObjsList,
                                                                final String searchedStgObjKey) {
        if (searchedStgObjKey != null && stgObjsList != null) {
            R01MBaseSearchedStorageObj currStgObj = null;
            int index = -1;     // Indice del elemento buscado
            int i=0;
            for (Iterator it=stgObjsList.iterator(); it.hasNext(); i++) {
                currStgObj = (R01MBaseSearchedStorageObj)it.next();
                if (currStgObj.getKey().equals(searchedStgObjKey)) {
                    index = i;
                    break;
                }
            }
            if (index >= 0) {
                return currStgObj;
            }
        }
        return null;
    }
    /**
     * Elimina un objeto de almacenamiento.
     * @param storageObjsList la lista de objetos de almacenamiento buscados
     * @param searchedStorageObjKey identificador del objeto de almacenamiento a eliminar
     * @return el objeto de almacenamiento borrado
     */
    private R01MBaseSearchedStorageObj _removeSearchedStorageObj( final List storageObjsList,
                                                                  final String searchedStorageObjKey) {
        if (searchedStorageObjKey != null && storageObjsList != null) {
            R01MBaseSearchedStorageObj currStorageObj = null;
            int index = -1;     // Indice del elemento buscado
            int i=0;
            for (Iterator it=storageObjsList.iterator(); it.hasNext(); i++) {
                currStorageObj = (R01MBaseSearchedStorageObj)it.next();
                if (currStorageObj.getKey().equals(searchedStorageObjKey)) {
                    index = i;
                    break;
                }
            }
            if (index > 0) {
                storageObjsList.remove(index);  // El servidor buscado exista
                return currStorageObj;
            }
        }
        return null;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS AUXILIARES PARA AADIR CONDICIONES DE TIPOLOGIA
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Aade un cluster de contenidos (objeto {@link R01MSearchedCluster}) en el que buscar.
     * @param searchedCluster objeto {@link R01MSearchedCluster}
     * @throws IllegalArgumentException si el oid del cluster de contenidos no es vlido
     */
    public void addContentCluster(final R01MSearchedCluster searchedCluster) throws IllegalArgumentException {
        if (searchedCluster != null) {
            if (searchedCluster.oid == null || searchedCluster.oid.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid del Cluster a buscar NO es valido!");
            }
            if (this.contentClusters == null) {
                this.contentClusters = new ArrayList();
            }
            if(!_listContains(this.contentClusters,searchedCluster)) {
                this.contentClusters.add(searchedCluster);
            }
        }
    }
    /**
     * Aade una cluster de contenidos (objeto {@link R01MSearchedCluster}) en el que buscar.
     * @param clusterOid Identificador del cluster de contenidos
     * @throws IllegalArgumentException si el oid del cluster de contenidos no es vlido
     */
    public void addContentCluster(final String clusterOid) throws IllegalArgumentException {
        this.addContentCluster(clusterOid,null);
    }
    /**
     * Aade un cluster de contenidos (objeto {@link R01MSearchedCluster}) en el que buscar.
     * @param clusterOid Identificador del cluster de contenidos
     * @param clusterName Nombre del cluster de contenidos
     * @throws IllegalArgumentException Si el oid del cluster de contenidos no es vlido
     */
    public void addContentCluster(final String clusterOid,final String clusterName) throws IllegalArgumentException {
        this.addContentCluster( new R01MSearchedCluster(clusterOid,clusterName) );
    }
    /**
     * Aade un cluster de contenidos (objeto {@link R01MSearchedCluster}) a partir de su clave.
     * @see R01MSearchedCluster#fromKey(String)
     * @param clusterKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addContentClusterFromKey(final String clusterKey) throws IllegalArgumentException {
        R01MSearchedCluster clusterToAdd = new R01MSearchedCluster();
        clusterToAdd.fromKey(clusterKey);
        this.addContentCluster(clusterToAdd);
    }
    /**
     * Elimina un cluster de contenidos (objeto {@link R01MSearchedCluster}) buscado de la lista de clusters.
     * @param searchedClusterOid identificador del cluster a eliminar
     */
    public void removeContentCluster(final String searchedClusterOid) {
        _removeSearchedTypoObj(this.contentClusters,searchedClusterOid);
    }
    /**
     * Devuelve un cluster de contenidos (objeto {@link R01MSearchedCluster}) de la lista de clusters buscados.
     * @param searchedClusterOid identificador del cluster buscada
     * @return el objeto {@link R01MSearchedFamily}
     */
    public R01MSearchedCluster getContentCluster(final String searchedClusterOid) {
        return (R01MSearchedCluster)_findSearchedTypoObj(this.contentClusters,searchedClusterOid);
    }
    /**
     * Aade una familia de contenidos (objeto {@link R01MSearchedFamily}) en el que buscar.
     * @param searchedFamily objeto {@link R01MSearchedFamily}
     * @throws IllegalArgumentException si el oid de la familia de contenidos no es vlida
     */
    public void addContentFamily(final R01MSearchedFamily searchedFamily) throws IllegalArgumentException {
        if (searchedFamily != null) {
            if (searchedFamily.oid == null || searchedFamily.oid.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid de la familia a buscar NO es valido!");
            }
            if (this.contentFamilies == null) {
                this.contentFamilies = new ArrayList();
            }
            if(!_listContains(this.contentFamilies,searchedFamily)) {
                this.contentFamilies.add(searchedFamily);
            }
        }
    }
    /**
     * Aade una familia de contenidos (objeto {@link R01MSearchedFamily}) en la que buscar.
     * @param familyOid Identificador de la familia
     * @throws IllegalArgumentException si el oid de la familia de contenidos no es vlida
     */
    public void addContentFamily(final String familyOid) throws IllegalArgumentException {
        this.addContentFamily(familyOid,null);
    }
    /**
     * Aade una familia de contenidos (objeto {@link R01MSearchedFamily}) en la que buscar.
     * @param familyOid Identificador de la familia
     * @param familyName Nombre de la familia
     * @throws IllegalArgumentException si el oid de la familia de contenidos no es vlida
     */
    public void addContentFamily(final String familyOid,final String familyName) throws IllegalArgumentException {
        this.addContentFamily( new R01MSearchedFamily(familyOid,familyName) );
    }
    /**
     * Aade una familia de contenidos (objeto {@link R01MSearchedFamily}) a partir de su clave.
     * @see R01MSearchedFamily#fromKey(String)
     * @param familyKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addContentFamilyFromKey(final String familyKey) throws IllegalArgumentException {
        R01MSearchedFamily famToAdd = new R01MSearchedFamily();
        famToAdd.fromKey(familyKey);
        this.addContentFamily(famToAdd);
    }
    /**
     * Elimina un familia de contenidos (objeto {@link R01MSearchedFamily}) de la lista de familias.
     * @param searchedFamilyOid identificador del familia a eliminar
     */
    public void removeContentFamily(final String searchedFamilyOid) {
        _removeSearchedTypoObj(this.contentFamilies,searchedFamilyOid);
    }
    /**
     * Devuelve una familia de contendiso (objeto {@link R01MSearchedFamily}) de la lista de familias buscados.
     * @param searchedFamilyOid identificador de la familia buscada
     * @return el objeto {@link R01MSearchedFamily}
     */
    public R01MSearchedFamily getContentFamily(final String searchedFamilyOid) {
        return (R01MSearchedFamily)_findSearchedTypoObj(this.contentFamilies,searchedFamilyOid);
    }
    /**
     * Aade un tipo de contenidos (objeto {@link R01MSearchedContentType}) en el que buscar.
     * @param searchedType objeto {@link R01MSearchedContentType}
     * @throws IllegalArgumentException Si el oid del tipo de contenido no es vlido
     */
    public void addContentType(final R01MSearchedContentType searchedType) throws IllegalArgumentException {
        if (searchedType != null) {
            if (searchedType.oid == null || searchedType.oid.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid del tipo a buscar NO es valido!");
            }
            if (this.contentTypes == null) {
                this.contentTypes = new ArrayList();
            }
            if(!_listContains(this.contentTypes,searchedType)) {
                this.contentTypes.add(searchedType);
            }
        }
    }
    /**
     * Aade un tipo de contenido (objeto {@link R01MSearchedContentType}) en el que buscar.
     * @param typeOid Identificador del tipo de contenido
     * @throws IllegalArgumentException Si el oid del tipo de contenido no es vlido
     */
    public void addContentType(final String typeOid) throws IllegalArgumentException {
        this.addContentType(typeOid,null);
    }
    /**
     * Aade un tipo de contenido (objeto {@link R01MSearchedContentType}) en el que buscar.
     * @param typeOid Identificador del tipo de contenido
     * @param typeName Nombre del tipo de contenido
     * @throws IllegalArgumentException Si el oid del tipo de contenido no es vlido
     */
    public void addContentType(final String typeOid,final String typeName) throws IllegalArgumentException {
        this.addContentType( new R01MSearchedContentType(typeOid,typeName) );
    }
    /**
     * Aade un tipo de contenidos (objeto {@link R01MSearchedContentType}) a partir de su clave.
     * @see R01MSearchedContentType#fromKey(String)
     * @param typeKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addContentTypeFromKey(final String typeKey) throws IllegalArgumentException {
        R01MSearchedContentType typeToAdd = new R01MSearchedContentType();
        typeToAdd.fromKey(typeKey);
        this.addContentType(typeToAdd);
    }
    /**
     * Elimina un tipo de contenidos (objeto {@link R01MSearchedContentType}) buscado de la lista de tipos.
     * @param searchedTypeOid identificador del tipo a eliminar
     */
    public void removeContentType(final String searchedTypeOid) {
        _removeSearchedTypoObj(this.contentTypes,searchedTypeOid);
    }
    /**
     * Devuelve un tipo de contenido (objeto {@link R01MSearchedContentType}) de la lista de tipos de contenido buscados.
     * @param searchedTypeOid identificador del tipo de contenido buscado
     * @return el objeto {@link R01MSearchedContentType}
     */
    public R01MSearchedContentType getContentType(final String searchedTypeOid) {
        return (R01MSearchedContentType)_findSearchedTypoObj(this.contentTypes,searchedTypeOid);
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METADATOS AND
///////////////////////////////////////////////////////////////////////////////////////////
    //TODO REVISAR este codigo (metodo "addMetaData")ya que podra hacerse una busqueda de un mismo metadato con distintos valores y este metodo solo
    //nos buscara por el ultimo metadato sin concatenar los dos metadatos distintos, ya que machaca los metadatos si son iguales
    /**
     * Aade un metaDato (objeto {@link R01MSearchedMetaData}) a buscar.
     * @param theSearchedMetaData objeto {@link R01MSearchedMetaData}
     * @throws IllegalArgumentException Si el oid del metaDato o su valor son null
     */
    public void addMetaData(final R01MSearchedMetaData theSearchedMetaData) throws IllegalArgumentException {
        if (theSearchedMetaData != null) {
            if (theSearchedMetaData.oid == null || theSearchedMetaData.oid.length() == 0
             || theSearchedMetaData.value == null || theSearchedMetaData.value.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el metaDato " + theSearchedMetaData.getOid() + " a buscar tiene un oid o un valor NO son valido (null)!");
            }
            if (this.searchedMetaData == null) {
                this.searchedMetaData = new ArrayList();
            }

            // Si el metaDato ya existe, nicamente cambiar su valor, si no existe,
            // simplemente introducirlo en la lista
            boolean metaDataExists = false;
            R01MSearchedMetaData currSearchedMetaData = null;
            for(Iterator mdIt=this.searchedMetaData.iterator(); mdIt.hasNext(); ) {
                currSearchedMetaData = (R01MSearchedMetaData)mdIt.next();
                if (currSearchedMetaData.getOid().equals(theSearchedMetaData.getOid())) {
                    metaDataExists = true;
                    break;      // salir del bucle
                }
            }
            if (metaDataExists) {
                // El metaDato ya existe... cambiar sus valores
                currSearchedMetaData.setOperationSelected( theSearchedMetaData.getOperationSelected() );
                currSearchedMetaData.setValue( theSearchedMetaData.getValue() );
            } else {
                // Se trata de un metaDato nuevo
                this.searchedMetaData.add(theSearchedMetaData);
            }
        }
    }
    /**
     * Aade un meta dato (objeto {@link R01MSearchedMetaData}) a buscar.
     * @param metaDataOid Identificador del metaDato
     * @param operationSelected Condicin selecccionada: EQ, LIKE, etc
     * @param metaDataValue Valor del metaDato
     * @throws IllegalArgumentException si el oid del metaDato o su valor son null
     */
    public void addMetaData( final String metaDataOid,
                             final String operationSelected,
                             final String metaDataValue) throws IllegalArgumentException {
        this.addMetaData(metaDataOid,null,operationSelected,metaDataValue);
    }
    /**
     * Aade un nuevo metaDato (objeto {@link R01MSearchedMetaData}) por el que buscar.
     * @param metaDataOid Identificador del metaDato
     * @param metaDataName Nombre del metaDato
     * @param operationSelected Operacin seleccionada: LIKE, EQ, NEC, etc
     * @param value Valor del metaDato
     * @throws IllegalArgumentException Si el oid del metaDato o su valor son null
     */
    public void addMetaData( final String metaDataOid,
                             final String metaDataName,
                             final String operationSelected,
                             final String value) throws IllegalArgumentException {
        this.addMetaData( new R01MSearchedMetaData(metaDataOid,metaDataName,operationSelected,value) );
    }
    /**
     * Aade un metaDato (objeto {@link R01MSearchedMetaData}) a partir de su clave.
     * @see R01MSearchedMetaData#fromKey(String)
     * @param mdKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addMetaDataFromKey(final String mdKey) throws IllegalArgumentException {
        R01MSearchedMetaData mdToAdd = new R01MSearchedMetaData();
        mdToAdd.fromKey(mdKey);
        this.addMetaData(mdToAdd);
    }
    /**
     * Elimina un metaDato (objeto {@link R01MSearchedMetaData}) buscado de la lista de metaDatos.
     * @param searchedMetaDataOid identificador del metaDato a eliminar
     */
    public void removeMetaData(final String searchedMetaDataOid) {
        _removeSearchedTypoObj(this.searchedMetaData,searchedMetaDataOid);
    }
    /**
     * Devuelve un metaDato (objeto {@link R01MSearchedMetaData}) de la lista de metaDatos buscados.
     * @param searchedMetaDataOid identificador del metadato buscado
     * @return el objeto {@link R01MSearchedMetaData}
     */
    public R01MSearchedMetaData getMetaData(final String searchedMetaDataOid) {
        return (R01MSearchedMetaData)_findSearchedTypoObj(this.searchedMetaData,searchedMetaDataOid);
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METADATOS OR
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Aade un metaDato (objeto {@link R01MSearchedMetaData}) a buscar.
     * @param theSearchedMetaData objeto {@link R01MSearchedMetaData}
     * @throws IllegalArgumentException Si el oid del metaDato o su valor son null
     */
    public void addORMetaData(final R01MSearchedMetaData theSearchedMetaData) throws IllegalArgumentException {
        if (theSearchedMetaData != null) {
            if (theSearchedMetaData.oid == null || theSearchedMetaData.oid.length() == 0
             || theSearchedMetaData.value == null || theSearchedMetaData.value.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid del metaDato a buscar o su valor NO son validos!");
            }
            if (this.orSearchedMetaData == null) {
                this.orSearchedMetaData = new ArrayList();
            }

            // En el caso de los metaDatos or, no machacar el valor del mismo metaDato si ya existe
            // ya que se pueden especificar mltiples valores OR para el mismo metaDato
            // commDate=10/10/2007 OR commDate=11/10/2007
            this.orSearchedMetaData.add(theSearchedMetaData);
        }
    }
    /**
     * Aade un meta dato (objeto {@link R01MSearchedMetaData}) a buscar
     * @param metaDataOid Identificador del metaDato
     * @param operationSelected Condicin selecccionada: EQ, LIKE, etc
     * @param metaDataValue Valor del metaDato
     * @throws IllegalArgumentException si el oid del metaDato o su valor son null
     */
    public void addORMetaData( final String metaDataOid,
                               final String operationSelected,
                               final String metaDataValue) throws IllegalArgumentException {
        this.addORMetaData(metaDataOid,null,operationSelected,metaDataValue);
    }
    /**
     * Aade un nuevo metaDato (objeto {@link R01MSearchedMetaData}) por el que buscar
     * @param metaDataOid Identificador del metaDato
     * @param metaDataName Nombre del metaDato
     * @param operationSelected Operacin seleccionada: LIKE, EQ, NEC, etc
     * @param value Valor del metaDato
     * @throws IllegalArgumentException Si el oid del metaDato o su valor son null
     */
    public void addORMetaData( final String metaDataOid,
                               final String metaDataName,
                               final String operationSelected,
                               final String value) throws IllegalArgumentException {
        this.addORMetaData( new R01MSearchedMetaData(metaDataOid,metaDataName,operationSelected,value) );
    }
    /**
     * Aade un metaDato (objeto {@link R01MSearchedMetaData}) a partir de su clave
     * @see R01MSearchedMetaData#fromKey(String)
     * @param mdKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addORMetaDataFromKey(final String mdKey) throws IllegalArgumentException {
        R01MSearchedMetaData mdToAdd = new R01MSearchedMetaData();
        mdToAdd.fromKey(mdKey);
        this.addORMetaData(mdToAdd);
    }
    /**
     * Elimina un metaDato (objeto {@link R01MSearchedMetaData}) buscado de la lista de metaDatos
     * @param searchedMetaDataOid identificador del metaDato a eliminar
     */
    public void removeORMetaData(final String searchedMetaDataOid) {
        _removeSearchedTypoObj(this.orSearchedMetaData,searchedMetaDataOid);
    }
    /**
     * Devuelve un metaDato (objeto {@link R01MSearchedMetaData}) de la lista de metaDatos buscados
     * @param searchedMetaDataOid identificador del metadato buscado
     * @return el objeto {@link R01MSearchedMetaData}
     */
    public R01MSearchedMetaData getORMetaData(final String searchedMetaDataOid) {
        return (R01MSearchedMetaData)_findSearchedTypoObj(this.orSearchedMetaData,searchedMetaDataOid);
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS AD-HOC PARA INCLUIR METADATOS "ESPECIALES"
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Establece el lenguaje por el que hay que filtrar los resultados
     * @param lang el lenguaje
     */
    public void setLanguage(final String lang) {
        if (lang != null) {
            this.addMetaData(R01MQueryObject.METADATA_LANGUAGE,
                            R01MSearchEngineConstants.EQUALS_OPERATOR,
                            lang);
        }
    }
    /**
     * Establece el parametro de bsqueda por texto completo. Si se pasa null, lo elimina
     * @param fullText el texto completo
     */
    public void setFullText(final String fullText) {
        if (fullText != null) {
            this.addMetaData(R01MQueryObject.METADATA_FULLTEXT,
                             R01MSearchEngineConstants.LIKE_OPERATOR,
                             fullText);
            this.isFullTextSearch = true;       // es una query que incluye bsqueda a texto completo
        } else {
            // Si pasan null se elimina el metadato de fullText
            this.removeMetaData(R01MQueryObject.METADATA_FULLTEXT);
            this.isFullTextSearch = false;
        }

    }
    /**
     * Devuelve el parametro FullText
     * @return el texto de bsqueda a texto completo
     */
    public String getFullText() {
        R01MSearchedMetaData fullTextMD = this.getMetaData(R01MQueryObject.METADATA_FULLTEXT);
        return fullTextMD != null ? fullTextMD.getValue() : null;
    }
    /**
     * Devuelve el numero de palabras que componen la bsqueda de texto libre
     * Este valor se establece en el preprocesamiento de la query en el objeto {@link com.ejie.r01m.searchengine.servlet.R01MBaseSearchEngineWebSessionManager}
     * El valor se puede utilizar para activar/desactivar el stemming en funcion del nmero de palabras
     * NOTA los grupos de pablabras entrecomilladas se cuentan como una sola palabra
     * @return el nmero de palabras que componen la query de texto libre
     */
    public int getFullTextWordCount() {
        return this.fullTextWordCount;
    }
    /**
     * Devuelve el numero de palabras que componen la bsqueda de texto libre
     * Este valor se establece en el preprocesamiento de la query en el objeto {@link com.ejie.r01m.searchengine.servlet.R01MBaseSearchEngineWebSessionManager}
     * El valor se puede utilizar para activar/desactivar el stemming en funcion del nmero de palabras
     * NOTA los grupos de pablabras entrecomilladas se cuentan como una sola palabra
     * @param thefullTextWordCount el nmero de palabras que componen la query de texto libre
     */
    public void setFullTextWordCount(final int thefullTextWordCount) {
        this.fullTextWordCount = thefullTextWordCount;
    }
    /**
     * Devuelve si es una bsqueda a texto completo.
     * @return true si se incluye el metadato de bsqueda por texto completo
     */
    public boolean isFullTextSearch() {
        return this.isFullTextSearch;
    }
    /**
     * Establece si es una bsqueda a texto completo.
     * @param newIsFullTextSearch si es una bsqueda a texto completo.
     */
    public void setFullTextSearch(final boolean newIsFullTextSearch) {
        this.isFullTextSearch = newIsFullTextSearch;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS PRIVADOS PARA ACTUAR SOBRE LA COLECCION DE OBJETOS DE TIPOLOGIA
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Encuentra un objeto de tipologia.
     * @param typoObjsList la lista de objetos de tipologia buscados
     * @param searchedTypoObjOid identificador del objeto de tipologia a eliminar
     * @return el objeto de tipologa buscado
     */
    private R01MBaseSearchedTypoObj _findSearchedTypoObj(final List typoObjsList,final String searchedTypoObjOid) {
        if (searchedTypoObjOid != null && typoObjsList != null) {
            R01MBaseSearchedTypoObj currTypoObj = null;
            int index = -1;     // Indice del elemento buscado
            int i=0;
            for (Iterator it=typoObjsList.iterator(); it.hasNext(); i++) {
                currTypoObj = (R01MBaseSearchedTypoObj)it.next();
                if (currTypoObj.getOid().equals(searchedTypoObjOid)) {
                    index = i;
                    break;
                }
            }
            if (index >= 0) {
                return currTypoObj;
            }
        }
        return null;
    }
    /**
     * Elimina un objeto de tipologia.
     * @param typoObjsList la lista de objetos de tipologia buscados
     * @param searchedTypoObjOid identificador del objeto de tipologia a eliminar
     * @return el objeto de tipologa borrado
     */
    private R01MBaseSearchedTypoObj _removeSearchedTypoObj(List typoObjsList,String searchedTypoObjOid) {
        if (searchedTypoObjOid != null && typoObjsList != null) {
            R01MBaseSearchedTypoObj currTypoObj = null;
            int index = -1;     // Indice del elemento buscado
            int i=0;
            for (Iterator it=typoObjsList.iterator(); it.hasNext(); i++) {
                currTypoObj = (R01MBaseSearchedTypoObj)it.next();
                if (currTypoObj.getOid().equals(searchedTypoObjOid)) {
                    index = i;
                    break;
                }
            }
            if (index >= 0) {
                typoObjsList.remove(index);  // El objeto buscado exista
                return currTypoObj;
            }
        }
        return null;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS AUXILIARES PARA AADIR CONDICIONES DE ETIQUETAS DE CATALOGACION
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Aade una etiqueta AND (objeto {@link R01MSearchedStructureCatalog}) a las condiciones de bsqueda.
     * @param searchedCatalog la etiqueta buscada
     * @throws IllegalArgumentException si el oid de la etiqueta no es valida
     */
    public void addANDStructureCatalog(R01MSearchedStructureCatalog searchedCatalog) throws IllegalArgumentException {
        if (searchedCatalog != null) {
            if (searchedCatalog.labelOid == null || searchedCatalog.labelOid.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid de la etiqueta and NO es vlida");
            }
            if (this.catalogLabelsAnd == null) {
                this.catalogLabelsAnd = new ArrayList();
            }
            if (!_listContains(this.catalogLabelsAnd,searchedCatalog)) {
                this.catalogLabelsAnd.add( searchedCatalog );
            }
        }
    }
    /**
     * Aade una etiqueta AND (objeto {@link R01MSearchedStructureCatalog}) a las condiciones de bsqueda.
     * @param labelOid Identificador de la etiqueta
     * @throws IllegalArgumentException si el oid de la etiqueta no es valida
     */
    public void addANDStructureCatalog(String labelOid) throws IllegalArgumentException {
        this.addANDStructureCatalog( new R01MSearchedStructureCatalog(labelOid) );
    }
    /**
     * Aade una catalogacin AND (objeto {@link R01MSearchedStructureCatalog}) en estructuras a partir de su clave.
     * @see R01MSearchedStructureCatalog#fromKey(String)
     * @param catKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addANDStructureCatalogFromKey(String catKey) throws IllegalArgumentException {
        R01MSearchedStructureCatalog catToAdd = new R01MSearchedStructureCatalog();
        catToAdd.fromKey(catKey);
        this.addANDStructureCatalog(catToAdd);
    }
    /**
     * Encuentra una etiqueta de catalogacin (objeto {@link R01MSearchedStructureCatalog}) buscada.
     * @param labelOid identificador de la etiqueta buscada
     * @return la etiqueta buscada
     */
    public R01MSearchedStructureCatalog getANDStructureCatalog(String labelOid) {
        return _findSearchedStructureObj(this.catalogLabelsAnd,labelOid);
    }
    /**
     * Elimina una etiqueta de catalogacin (objeto {@link R01MSearchedStructureCatalog}).
     * @param structureOid identificador de la etiqueta
     */
    public void removeANDStructrueCatalog(String structureOid) {
        _removeSearchedStructureLabel(this.catalogLabelsAnd,structureOid);
    }
    /**
     * Aade una etiqueta OR (objeto {@link R01MSearchedStructureCatalog}) a las condiciones de bsqueda.
     * @param searchedCatalog la etiqueta buscada
     * @throws IllegalArgumentException si el oid de la etiqueta no es valida
     */
    public void addORStructureCatalog(R01MSearchedStructureCatalog searchedCatalog) throws IllegalArgumentException {
        if (searchedCatalog != null) {
            if (searchedCatalog.labelOid == null || searchedCatalog.labelOid.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid de la etiqueta and NO es vlida");
            }
            if (this.catalogLabelsOr == null) {
                this.catalogLabelsOr = new ArrayList();
            }
            if (!_listContains(this.catalogLabelsOr,searchedCatalog)) {
                this.catalogLabelsOr.add( searchedCatalog );
            }
        }
    }
    /**
     * Aade una etiqueta OR (objeto {@link R01MSearchedStructureCatalog}) a las condiciones de bsqueda
     * @param labelOid Identificador de la etiqueta
     * @throws IllegalArgumentException si el oid de la etiqueta no es valida
     */
    public void addORStructureCatalog(String labelOid) throws IllegalArgumentException {
        this.addORStructureCatalog( new R01MSearchedStructureCatalog(labelOid) );
    }
    /**
     * Aade una catalogacin OR (objeto {@link R01MSearchedStructureCatalog}) en estructuras a partir de su clave
     * @see R01MSearchedStructureCatalog#fromKey(String)
     * @param catKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addORStructureCatalogFromKey(String catKey) throws IllegalArgumentException {
        R01MSearchedStructureCatalog catToAdd = new R01MSearchedStructureCatalog();
        catToAdd.fromKey(catKey);
        this.addORStructureCatalog(catToAdd);
    }
    /**
     * Encuentra una etiqueta de catalogacin buscada (objeto {@link R01MSearchedStructureCatalog})
     * @param labelOid identificador de la etiqueta
     * @return el objeto de catalogacion
     */
    public R01MSearchedStructureCatalog getORStructureCatalog(String labelOid) {
        return _findSearchedStructureObj(this.catalogLabelsOr,labelOid);
    }
    /**
     * Elimina una etiqueta de catalogacin (objeto {@link R01MSearchedStructureCatalog})
     * @param structureOid identificador de la etiqueta
     */
    public void removeORStructrueCatalog(String structureOid) {
        _removeSearchedStructureLabel(this.catalogLabelsOr,structureOid);
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS PRIVADOS PARA ACTUAR SOBRE LA COLECCION DE CATALOGACIONES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Encuentra un objeto de catalogacin
     * @param structureObjsList la lista de objetos de catalogacin buscadoss
     * @param searchedStructureObjOid identificador del objeto de catalogacin a buscar
     * @return el objeto de catalogacion buscado
     */
    private R01MSearchedStructureCatalog _findSearchedStructureObj(List structureObjsList,String searchedStructureObjOid) {
        if (searchedStructureObjOid != null && structureObjsList != null) {
            R01MSearchedStructureCatalog currStructureObj = null;
            int index = -1;     // Indice del elemento buscado
            int i=0;
            for (Iterator it=structureObjsList.iterator(); it.hasNext(); i++) {
                currStructureObj = (R01MSearchedStructureCatalog)it.next();
                if (currStructureObj.getLabelOid().equals(searchedStructureObjOid)) {
                    index = i;
                    break;
                }
            }
            if (index >= 0) {
                return currStructureObj;
            }
        }
        return null;
    }
    /**
     * Elimina un objeto de tipologia
     * @param labelList la lista de objetos de catalogacion buscados
     * @param searchedLabelOid identificador del objeto de catalogacion a eliminar
     * @return la etiqueta borrada
     */
    private R01MSearchedStructureCatalog _removeSearchedStructureLabel(List labelList,String searchedLabelOid) {
        if (searchedLabelOid != null && labelList != null) {
            R01MSearchedStructureCatalog currLabelObj = null;
            int index = -1;     // Indice del elemento buscado
            int i=0;
            for (Iterator it=labelList.iterator(); it.hasNext(); i++) {
                currLabelObj = (R01MSearchedStructureCatalog)it.next();
                if (currLabelObj.getLabelOid().equals(searchedLabelOid)) {
                    index = i;
                    break;
                }
            }
            if (index >= 0) {
                labelList.remove(index);  // La etiqueta buscada exista
                return currLabelObj;
            }
        }
        return null;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS AUXILIARES PARA AADIR CONDICIONES DE CATALOGACION GEOGRAFICA
///////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS AUXILIARES PARA AADIR METADATOS POR LOS QUE ORDENAR
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Aade un nuevo metaDato (objeto {@link R01MSearchedOrderByMetaData}) por el que ordenar
     * @param orderByMetaData el objeto {@link R01MSearchedOrderByMetaData} por el que se ordena
     * @throws IllegalArgumentException Si el oid del metaDato o su valor son null
     */
    public void addOrderByMetaData(R01MSearchedOrderByMetaData orderByMetaData) throws IllegalArgumentException {
        if (orderByMetaData != null) {
            if (orderByMetaData.oid == null || orderByMetaData.oid.length() == 0
             || orderByMetaData.order == null || orderByMetaData.order.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid del metaDato por el que ordenar NO es valido!");
            }
            if (this.orderBy == null) {
                this.orderBy = new ArrayList();
            }
            if (!_listContains(this.orderBy,orderByMetaData)) {
                this.orderBy.add(orderByMetaData);
            }
        }
    }
    /**
     * Aade un nuevo metaDato por el que ordenar (objeto {@link R01MSearchedOrderByMetaData})
     * @param metaDataOid Identificador del metaDato
     * @param metaDataName Nombre del metaDato
     * @param order si la ordenacin es ascendente o descendente
     * @throws IllegalArgumentException Si el oid del metaDato o su valor son null
     */
    public void addOrderByMetaData(String metaDataOid,String metaDataName,String order) throws IllegalArgumentException {
        this.addOrderByMetaData( new R01MSearchedOrderByMetaData(metaDataOid,metaDataName,order) );
    }
    /**
     * Aade un metaDato de ordenacin (objeto {@link R01MSearchedOrderByMetaData})  a partir de su clave
     * @see R01MSearchedOrderByMetaData#fromKey(String)
     * @param orderKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addOrderByMetaDataFromKey(String orderKey) throws IllegalArgumentException {
        R01MSearchedOrderByMetaData orderToAdd = new R01MSearchedOrderByMetaData();
        orderToAdd.fromKey(orderKey);
        this.addOrderByMetaData(orderToAdd);
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS AUXILIARES PARA AADIR REPOSITORIOS DE PUBLICACIN
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Aade un repositorio de publicacin (objeto {@link R01MSearchedPublishRepository}) en el que ha de estar publicado el contenido a buscar
     * @param publishRepository el repositorio de publicacin
     * @throws IllegalArgumentException si el oid del repositorio o su valor son null
     */
    public void addPublishRepository(R01MSearchedPublishRepository publishRepository) throws IllegalArgumentException {
        if (publishRepository != null) {
            if (publishRepository.oid == null || publishRepository.oid.length() == 0) {
                throw new IllegalArgumentException("Error al establecer la query de bsqueda: el oid del repositorio de bsqueda '" + publishRepository.oid + "' NO es vlido!!!");
            }
            if (this.publishRepositories == null) {
                this.publishRepositories = new ArrayList();
            }
            if (!_listContains(this.publishRepositories,publishRepository)) {
                this.publishRepositories.add(publishRepository);
            }
        }
    }
    /**
     * Aade un nuevo repositorio en el que han de estar publicados los contenidos a buscar
     * (objeto {@link R01MSearchedPublishRepository})
     * @param repositoryOid Identifciador del repositorio
     * @throws IllegalArgumentException si el oid del repositorio o su valor son null
     */
    public void addPublishRepository(String repositoryOid) throws IllegalArgumentException {
        this.addPublishRepository( new R01MSearchedPublishRepository(repositoryOid) );
    }
    /**
     * Aade un nuevo repositorio en el que han de estar publicados los contenidos buscados
     * (objeto {@link R01MSearchedPublishRepository})
     * @param repositoryOid Identificador del repositorio
     * @param repositoryName Nombre del repositorio
     * @throws IllegalArgumentException si el oid del repositorio o su valor son null
     */
    public void addPublishRepository(String repositoryOid,String repositoryName) throws IllegalArgumentException {
        this.addPublishRepository( new R01MSearchedPublishRepository(repositoryOid,repositoryName) );
    }
    /**
     * Aade un repositorio de publicacin (objeto {@link R01MSearchedPublishRepository}) a partir de su clave
     * @see R01MSearchedStructureCatalog#fromKey(String)
     * @param pubKey la clave
     * @throws IllegalArgumentException si el objeto a aadir no es valido
     */
    public void addPublishRepositoryFromKey(String pubKey) throws IllegalArgumentException {
        R01MSearchedPublishRepository pubToAdd = new R01MSearchedPublishRepository();
        pubToAdd.fromKey(pubKey);
        this.addPublishRepository(pubToAdd);
    }
    /**
     * Elimina un repositorio de publicacin (objeto {@link R01MSearchedPublishRepository}) de la lista
     * de repositorios de publicacin en los que buscar
     * @param repOid identificador del repositorio de publicacin
     * @throws IllegalArgumentException si el identificador a borrar es nulo
     */
    public void removePublishRepository(String repOid) throws IllegalArgumentException {
        if (repOid != null && this.publishRepositories != null) {
            R01MSearchedPublishRepository currRep = null;
            int i=0;
            for (Iterator pIt=this.publishRepositories.iterator(); pIt.hasNext(); i++) {
                currRep = (R01MSearchedPublishRepository)pIt.next();
                if (currRep.getOid().equals(repOid)) {
                    this.publishRepositories.remove(i);
                    break;
                }
            }
        }
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS DE DEBUG
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Obtiene informacin de depuracin
     * @return Una cadena con informacin de depuracin
     */
    public StringBuffer composeDebugInfo() {
        StringBuffer sb = new StringBuffer(601);
        sb.append("\r\n[Query] guid:");
        sb.append(this.queryOid);
        sb.append(" context: ");
        sb.append(this.context);
        sb.append(" searchEngineId: ");
        sb.append(this.searchEngineId);
        sb.append(" modified: ");
        sb.append(this.modified);
        sb.append("\r\n\t>> SISTEMA: \r\n\t\tName: ");
        sb.append(this.queryName);
        sb.append("\r\n\t\tDescription: ");
        sb.append(this.queryDescription);
        sb.append("\r\n\t\tSaveQuery: ");
        sb.append(this.saveQuery);
        sb.append("\r\n\t>> TIPOLOGIA:\r\n\t\tClusters\r\n");
        sb.append(_composeQueryElementDebugInfo(this.contentClusters));
        sb.append("\t\tFamilies\r\n");
        sb.append(_composeQueryElementDebugInfo(this.contentFamilies));
        sb.append("\t\tTypes\r\n");
        sb.append(_composeQueryElementDebugInfo(this.contentTypes));

        sb.append("\t>> ALMACENAMIENTO:\r\n\t\tServers\r\n");
        sb.append(_composeQueryElementDebugInfo(this.servers));
        sb.append("\t\tDataStores\r\n");
        sb.append(_composeQueryElementDebugInfo(this.dataRepositories));
        sb.append("\t\tAreas\r\n");
        sb.append(_composeQueryElementDebugInfo(this.areas));
        sb.append("\t\tWorkAreas\r\n");
        sb.append(_composeQueryElementDebugInfo(this.workAreas));

        sb.append("\t>> CATALOGACION:\r\n\t\tAND\r\n");
        sb.append(_composeCatalogDebugInfo(this.catalogLabelsAnd));
        sb.append("\t\tOR\r\n");
        sb.append(_composeCatalogDebugInfo(this.catalogLabelsOr));

        sb.append("\t>> METADATOS:\r\n");
        sb.append(_composeQueryElementDebugInfo(this.searchedMetaData));
        sb.append(_composeQueryElementDebugInfo(this.orSearchedMetaData));

        sb.append("\t>> ORDENACION: \r\n");
        sb.append(_composeQueryElementDebugInfo(this.orderBy));

        sb.append("\t>> REPOSITORIOS DE PUBLICACION: \r\n");
        sb.append(_composeQueryElementDebugInfo(this.publishRepositories));
        sb.append("\t\tBuscar SOLO publicados: ");
        sb.append(this.publishedItemsOnly ? "true":"false");
        sb.append("\r\n\t>> MODELO: \r\n\t\tDevolver ficheros de datos: ");
        sb.append(this.returnDataFiles);
        sb.append("\r\n\t\tDevolver documentos: ");
        sb.append(this.returnDocuments);
        sb.append("\r\n\t\tDevolver adjuntos: ");
        sb.append(this.returnDocuments);
        sb.append("\r\n\t\tstatus: ");
        sb.append(this.status);
        sb.append("\r\n\t>> PROPIEDADES DE PRESENTACION:\r\n");
        sb.append(_composePresentationPropertiesDebugInfo(this.presentationProperties));
        sb.append("\r\n");

        return sb;
    }
    /**
     * Compone la informacion de debug de una lista de objetos de la query
     * @param searchedObjList La lista de objteos de la query
     * @return La informacion de debug
     */
    private StringBuffer _composeQueryElementDebugInfo(List searchedObjList) {
        StringBuffer sb = new StringBuffer("");
        if (searchedObjList != null) {
            R01MBaseSearchedObj so = null;
            for (Iterator it = searchedObjList.iterator(); it.hasNext(); ) {
                so = (R01MBaseSearchedObj)it.next();
                sb.append("\t\t\t");sb.append(so.composeDebugInfo());sb.append("\r\n");
            }
        }
        return sb;
    }
    /**
     * Compone la informacin de debug de la informacion de catalogacion
     * @param catalogList La informacion de catalogacin
     * @return La informacin de debug
     */
    private StringBuffer _composeCatalogDebugInfo(List catalogList) {
        StringBuffer sb = new StringBuffer("");
        if (catalogList != null) {
            R01MSearchedStructureCatalog cat = null;
            for (Iterator it = catalogList.iterator(); it.hasNext(); ) {
                cat = (R01MSearchedStructureCatalog)it.next();
                sb.append("\t\t\t");sb.append(cat.composeDebugInfo());sb.append("\r\n");
            }
        }
        return sb;
    }
    /**
     * Compone informacin de depuracin acerca de las propiedades de presentacion
     * @param props las propiedades de presentacin
     * @return la cadena de depuracion
     */
    private StringBuffer _composePresentationPropertiesDebugInfo(Map props) {
        StringBuffer sb = new StringBuffer("");
        if (props != null && !props.isEmpty()) {
            Map.Entry me = null;
            for (Iterator it=props.entrySet().iterator(); it.hasNext(); ) {
                me = (Map.Entry)it.next();
                sb.append("\t\t");
                sb.append((String)me.getKey());
                sb.append('=');
                sb.append(String.valueOf(me.getValue()));
                if (it.hasNext()) {
                    sb.append("\r\n");
                }
            }
        }
        return sb;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the queryGUID.
     */
    public String getQueryOid() {
        return this.queryOid;
    }
    /**
     * @return true si la query ha sido modificada
     */
    public boolean isModified() {
        return this.modified;
    }
    /**
     * @param theModified true si la query ha sido modificada
     */
    public void setModified(boolean theModified) {
        this.modified = theModified;
    }
    /**
     * @return Returns the searchEngineId.
     */
    public String getSearchEngineId() {
        return this.searchEngineId;
    }
    /**
     * @return Returns the context.
     */
    public String getContext() {
        return this.context;
    }
    /**
     * @param theContext The context to set.
     */
    public void setContext(String theContext) {
        this.context = theContext;
    }
    /**
     * @param theSearchEngineId The searchEngineId to set.
     */
    public void setSearchEngineId(String theSearchEngineId) {
        this.searchEngineId = theSearchEngineId;
    }
    /**
     * @param theQueryGuid The queryGUID to set.
     */
    public void setQueryOid(String theQueryGuid) {
        this.queryOid = theQueryGuid;
    }
    /**
	 * @return Returns the queryName.
	 */
	public String getQueryName() {
		return this.queryName;
	}
	/**
	 * @param theQueryName The queryName to set.
	 */
	public void setQueryName(String theQueryName) {
		this.queryName = theQueryName;
	}

	/**
	 * @return Returns the queryDescription.
	 */
	public String getQueryDescription() {
		return this.queryDescription;
	}
	/**
	 * @param theQueryDescription The queryDescription to set.
	 */
	public void setQueryDescription(String theQueryDescription) {
	    String queryDescriptionTemp = "";
	    if(theQueryDescription!=null && theQueryDescription.length()>255) {
	        queryDescriptionTemp = this.queryDescription.substring(0,255);
	    } else {
	        queryDescriptionTemp = theQueryDescription;
	    }
		this.queryDescription = queryDescriptionTemp;
	}
	/**
	 * @return Returns the saveQuery.
	 */
	public boolean isSaveQuery() {
		return this.saveQuery;
	}
	/**
	 * @param theSaveQuery The saveQuery to set.
	 */
	public void setSaveQuery(boolean theSaveQuery) {
		this.saveQuery = theSaveQuery;
	}
    /**
     * @return the onlyNotPublished
     */
    public boolean isOnlyNotPublished() {
        return this.onlyNotPublished;
    }
    /**
     * @param theOnlyNotPublished The onlyNotPublished to set.
     */
    public void setOnlyNotPublished(boolean theOnlyNotPublished) {
        this.onlyNotPublished = theOnlyNotPublished;
    }
	/**
	 * @return the documentUIState
	 */
	public int getDocumentUIState() {
		return this.documentUIState;
	}
	/**
	 * @param theDocumentUIState The documentUIState to set.
	 */
	public void setDocumentUIState(int theDocumentUIState) {
	    this.documentUIState = theDocumentUIState;
        if (theDocumentUIState==DOC_UI_STATE_NOTPUBLISHED)
            this.setOnlyNotPublished(true);
        else if (theDocumentUIState==DOC_UI_STATE_PUBLISHED)
            this.setOnlyNotPublished(false);
        else if (theDocumentUIState==DOC_UI_STATE_PUBLISHED_AND_MODIFIED)
            this.setOnlyNotPublished(false);
	}
///////////////////////////////////////////////////////////////////////////////////////////
//  OPCIONES DE PRESENTACION
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @param thePresentationProperties The presentationProperties to set.
     */
    public void setPresentationProperties(Map thePresentationProperties) {
        this.presentationProperties = thePresentationProperties;
    }
    /**
     * Devuelve el mapa con las propiedades de presentacion (atributo-valor)
     * @return El mapa con las propiedades
     */
    public Map getPresentationProperties() {
        return this.presentationProperties;
    }
    /**
     * Establece una propiedad de presentacion (atributo-valor)
     * @param thePropertyName Nombre de la propiedad
     * @param thePropertyValue Valor de la propiedad
     */
    public void setPresentationProperty(String thePropertyName,String thePropertyValue) {
        if (this.presentationProperties == null) {
            this.presentationProperties = new HashMap();
        }
        if (thePropertyName != null && thePropertyValue != null) {
            this.presentationProperties.put(thePropertyName,thePropertyValue);
        }
    }
    /**
     * Obtiene el valor de una propiedad de presentacion
     * @param thePropertyName El nombre de la propiedad de presentacion
     * @return El valor de la propiedad
     */
    public String getPresentationProperty(String thePropertyName) {
        if ( this.presentationProperties == null ||
             thePropertyName == null ||
             this.presentationProperties.get(thePropertyName) == null) {
            return null;
        }
        return String.valueOf(this.presentationProperties.get(thePropertyName));

    }
    /**
     * @param theAllResultsInAPage The allResultsInAPage to set.
     */
    public void setAllResultsInAPage(boolean theAllResultsInAPage) {
        this.allResultsInAPage = theAllResultsInAPage;
    }

    /**
     * @return El boolean allResultsInAPage
     */
    public boolean getAllResultsInAPage() {
        return this.allResultsInAPage;
    }


///////////////////////////////////////////////////////////////////////////////////////////
//  CONDICIONES DE SELECCION EN BASE AL MODELO
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the returnDocuments.
     */
    public boolean isReturnDocuments() {
        return this.returnDocuments;
    }
    /**
     * @param theReturnDocuments The returnDocuments to set.
     */
    public void setReturnDocuments(boolean theReturnDocuments) {
        this.returnDocuments = theReturnDocuments;
    }
    /**
     * @return Returns the returnDataFiles.
     */
    public boolean isReturnDataFiles() {
        return this.returnDataFiles;
    }
    /**
     * @param theReturnAttachments The searchDataFiles to set.
     */
    public void setReturnAttachments(boolean theReturnAttachments) {
        this.returnAttachments = theReturnAttachments;
    }
    /**
     * @return Returns the returnDataFiles.
     */
    public boolean isReturnAttachments() {
        return this.returnAttachments;
    }
    /**
     * @param theReturnDataFiles The searchDataFiles to set.
     */
    public void setReturnDataFiles(boolean theReturnDataFiles) {
        this.returnDataFiles = theReturnDataFiles;
    }
    /**
     * @return Returns the status.
     */
    public String getStatus() {
        return this.status;
    }
    /**
     * @param theStatus The status to set.
     */
    public void setStatus(String theStatus) {
        this.status = theStatus;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET: ORDER BY
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the orderBy (objetos {@link R01MSearchedOrderByMetaData})
     */
    public List getOrderBy() {
        return this.orderBy;
    }
    /**
     * @param theOrderByMetaData The orderBy to set (objetos {@link R01MSearchedOrderByMetaData}).
     */
    public void setOrderBy(List theOrderByMetaData) {
        this.orderBy = theOrderByMetaData;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET: CATALOGACION
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the catalogLabelsAnd (objetos {@link R01MSearchedStructureCatalog})
     */
    public List getCatalogLabelsAnd() {
        return this.catalogLabelsAnd;
    }
    /**
     * Devuelve las etiquetas de catalogacin que proceden de
     * condiciones impuestas en base a las guas de navegacin
     * @return una lista de objetos {@link R01MSearchedStructureCatalog}
     */
    public List getGuideCatalogLabelsAnd() {
        return _extractGuideConditions(this.catalogLabelsAnd);
    }
    /**
     * Devuelve las etiquetas de catalogacin que NO proceden de
     * condiciones impuestas en base a las guas de navegacin
     * @return una lista de objetos {@link R01MSearchedStructureCatalog}
     */
    public List getNotGuideCatalogLabelsAnd() {
        return _extractNotGuideConditions(this.catalogLabelsAnd);
    }
    /**
     * @param theCatalogLabelsAnd The catalogLabelsAnd to set (objetos {@link R01MSearchedStructureCatalog}).
     */
    public void setCatalogLabelsAnd(List theCatalogLabelsAnd) {
        this.catalogLabelsAnd = theCatalogLabelsAnd;
    }
    /**
     * @return Returns the catalogLabelsOr (objetos {@link R01MSearchedStructureCatalog}).
     */
    public List getCatalogLabelsOr() {
        return this.catalogLabelsOr;
    }
    /**
     * @param theCatalogLabelsOr The catalogLabelsOr to set (objetos {@link R01MSearchedStructureCatalog}).
     */
    public void setCatalogLabelsOr(List theCatalogLabelsOr) {
        this.catalogLabelsOr = theCatalogLabelsOr;
    }
    /**
	 * @return Returns the geoCatalogAnd.
	 */
	public List getGeoCatalogsAnd() {
		return this.geoCatalogsAnd;
	}
    /**
     * Devuelve las etiquetas de catalogacin que proceden de
     * condiciones impuestas en base a las guas de navegacin
     * @return una lista de objetos {@link R01MSearchedGeoCatalog}
     */
    public List getGuideGeoCatalogsAnd() {
        return _extractGuideConditions(this.geoCatalogsAnd);
    }
    /**
     * Devuelve las etiquetas de catalogacin que NO proceden de
     * condiciones impuestas en base a las guas de navegacin
     * @return una lista de objetos {@link R01MSearchedStructureCatalog}
     */
    public List getNotGuideGeoCatalogsAnd() {
        return _extractNotGuideConditions(this.geoCatalogsAnd);
    }
	/**
	 * @param theGeoCatalogsAnd The geoCatalogAnd to set.
	 */
	public void setGeoCatalogsAnd(List theGeoCatalogsAnd) {
		this.geoCatalogsAnd = theGeoCatalogsAnd;
	}
	/**
	 * @return Returns the geoCatalogOr.
	 */
	public List getGeoCatalogsOr() {
		return this.geoCatalogsOr;
	}
	/**
	 * @param theGeoCatalogsOr The geoCatalogOr to set.
	 */
	public void setGeoCatalogsOr(List theGeoCatalogsOr) {
		this.geoCatalogsOr = theGeoCatalogsOr;
	}
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET: METADATA
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the searchedMetaData (objetos {@link R01MSearchedMetaData})
     */
    public List getMetaData() {
        return this.searchedMetaData;
    }
    /**
     * @param theSearchedMetaData The searchedMetaData to set (objetos {@link R01MSearchedMetaData}).
     */
    public void setMetaData(List theSearchedMetaData) {
        this.searchedMetaData = theSearchedMetaData;
    }
    /**
     * Devuelve la lista de metaDatos buscados
     * @return una lista de objetos {@link R01MSearchedMetaData}
     */
    public List getSearchedMetaData() {
        return this.getMetaData();
    }
    /**
     * @param theSearchedMetaData
     */
    public void setSearchedMetaData(List theSearchedMetaData) {
        this.setMetaData(theSearchedMetaData);
    }
    /**
     * Devuelve la lista de MetaDatos
     * @return Returns the searchedMetaData (objetos {@link R01MSearchedMetaData})
     */
    public List getORMetaData() {
        return this.orSearchedMetaData;
    }
    /**
     * Establece la lista de metaDatos OR
     * @param theORSearchedMetaData The searchedMetaData to set (objetos {@link R01MSearchedMetaData}).
     */
    public void setORMetaData(List theORSearchedMetaData) {
        this.orSearchedMetaData = theORSearchedMetaData;
    }
    /**
     * Devuelve la lista de metaDatos or
     * @return una lista de objetos {@link R01MSearchedMetaData}
     */
    public List getOrSearchedMetaData() {
        return this.getORMetaData();
    }
    /**
     * Establece la lista de metaDatos OR
     * @param theORSearchedMetaData una lista de objetos {@link R01MSearchedMetaData}
     */
    public void setOrSearchedMetaData(List theORSearchedMetaData) {
        this.setORMetaData(theORSearchedMetaData);
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET: TIPOLOGIA
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the contentClusters (objetos {@link R01MSearchedCluster}).
     */
    public List getContentClusters() {
        return this.contentClusters;
    }
    /**
     * Devuelve los clusters que proceden de condiciones impuestas en base
     * a guias de navegacin
     * @return una lista de objetos {@link R01MSearchedCluster}
     */
    public List getGuideContentClusters() {
        return _extractGuideConditions(this.contentClusters);
    }
    /**
     * Devuelve los clusters que proceden de condiciones NO impuestas en
     * base a guias de navegacin
     * @return una lista de objetos {@link R01MSearchedStructureCatalog}
     */
    public List getNotGuideContentClusters() {
        return _extractNotGuideConditions(this.contentClusters);
    }
    /**
     * @param theContentClusters The contentClusters to set (objetos {@link R01MSearchedCluster}).
     */
    public void setContentClusters(List theContentClusters) {
        this.contentClusters = theContentClusters;
    }
    /**
     * @return Returns the contentFamilies (objetos {@link R01MSearchedFamily}).
     */
    public List getContentFamilies() {
        return this.contentFamilies;
    }
    /**
     * Devuelve las familias que proceden de condiciones impuestas en base
     * a guias de navegacin
     * @return una lista de objetos {@link R01MSearchedFamily}
     */
    public List getGuideContentFamilies() {
        return _extractGuideConditions(this.contentFamilies);
    }
    /**
     * Devuelve las familias que proceden de condiciones NO impuestas en base
     * a guias de navegacin
     * @return una lista de objetos {@link R01MSearchedStructureCatalog}
     */
    public List getNotGuideContentFamilies() {
        return _extractNotGuideConditions(this.contentFamilies);
    }
    /**
     * @param theContentFamilies The contentFamilies to set (objetos {@link R01MSearchedFamily}).
     */
    public void setContentFamilies(List theContentFamilies) {
        this.contentFamilies = theContentFamilies;
    }
    /**
     * @return Returns the contentTypes (objetos {@link R01MSearchedContentType}).
     */
    public List getContentTypes() {
        return this.contentTypes;
    }
    /**
     * Devuelve los tipos de contenidos que proceden de condiciones impuestas en
     * base a tipos de contenido
     * @return una lista de objetos {@link R01MSearchedContentType}
     */
    public List getGuideContentTypes() {
        return _extractGuideConditions(this.contentTypes);
    }
    /**
     * Devuelve los tipos de contenidos que NO proceden de condiciones impuestas en
     * base a tipos de contenidos
     * @return una lista de objetos {@link R01MSearchedStructureCatalog}
     */
    public List getNotGuideContentTypes() {
        return _extractNotGuideConditions(this.contentTypes);
    }
    /**
     * @param theContentTypes The contentTypes to set (objetos {@link R01MSearchedContentType}).
     */
    public void setContentTypes(List theContentTypes) {
        this.contentTypes = theContentTypes;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET: STORAGE
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the servers (objetos {@link R01MSearchedServer}).
     */
    public List getServers() {
        return this.servers;
    }
    /**
     * @param theServers The servers to set (objetos {@link R01MSearchedServer}).
     */
    public void setServers(List theServers) {
        this.servers = theServers;
    }
    /**
     * @return Returns the dataRepository (objetos {@link R01MSearchedDataRepository}).
     */
    public List getDataRepositories() {
        return this.dataRepositories;
    }
    /**
     * @param theDataRepository The dataRepository to set (objetos {@link R01MSearchedDataRepository}).
     */
    public void setDataRepositories(List theDataRepository) {
        this.dataRepositories = theDataRepository;
    }
    /**
     * @return Returns the areas (objetos {@link R01MSearchedArea}).
     */
    public List getAreas() {
        return this.areas;
    }
    /**
     * @param theAreas The areas to set (objetos {@link R01MSearchedArea}).
     */
    public void setAreas(List theAreas) {
        this.areas = theAreas;
    }
    /**
     * @return Returns the workAreas (objetos {@link R01MSearchedWorkArea}).
     */
    public List getWorkAreas() {
        return this.workAreas;
    }
    /**
     * @param theWorkAreas The workAreas to set (objetos {@link R01MSearchedWorkArea}).
     */
    public void setWorkAreas(List theWorkAreas) {
        this.workAreas = theWorkAreas;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//   GET & SET: CONDICIONES DE BSQUEDA POR IDIOMA
///////////////////////////////////////////////////////////////////////////////////////////
   /**
    * @return the searchByDefaultLang
    */
   public boolean isSearchByDefaultLang() {
       return this.searchByDefaultLang;
   }
   /**
    * @param theSearchByDefaultLang The searchByDefaultLang to set.
    */
   public void setSearchByDefaultLang(boolean theSearchByDefaultLang) {
       this.searchByDefaultLang = theSearchByDefaultLang;
   }
///////////////////////////////////////////////////////////////////////////////////////////
//  REPOSITORIOS DE PUBLICACION
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the publishRepositories (objetos {@link R01MSearchedPublishRepository}).
     */
    public List getPublishRepositories() {
        return this.publishRepositories;
    }
    /**
     * @param thePublishRepositories The publishRepositories to set (objetos {@link R01MSearchedPublishRepository}).
     */
    public void setPublishRepositories(List thePublishRepositories) {
        this.publishRepositories = thePublishRepositories;
    }
    /**
     * Indica si hay que buscar los documentos no publicados
     * @return true si hay que buscar los documentos no publicados
     */
    public boolean isPublishedItemsOnly() {
        return this.publishedItemsOnly;
    }
    /**
     * Devuelve si hay que buscar nicamente elementos publicados
     * @param thePublishedItemsOnly
     */
    public void setPublishedItemsOnly(boolean thePublishedItemsOnly) {
        this.publishedItemsOnly = thePublishedItemsOnly;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS GENERALES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Valida que el objeto es correcto
     * @return true si el objeto es correcto
     */
    public boolean validate() {
        // Validar las condiciones de presentacion
        if (this.presentationProperties != null
         && this.presentationProperties.get(R01MQueryObject.PRESENTATION_PAGE_SIZE) != null) {
            try {
                int pageSize = Integer.parseInt((String) this.presentationProperties.get(R01MQueryObject.PRESENTATION_PAGE_SIZE));
                if (pageSize >= MAX_RESULTITEMS_PER_PAGE) {
                    this.presentationProperties.put(R01MQueryObject.PRESENTATION_PAGE_SIZE,
                                                    new Integer(MAX_RESULTITEMS_PER_PAGE));
                } else if (pageSize < 0) {
                    this.presentationProperties.remove(R01MQueryObject.PRESENTATION_PAGE_SIZE);
                }
            } catch(NumberFormatException nfEx) {
                this.presentationProperties.remove(R01MQueryObject.PRESENTATION_PAGE_SIZE);
            }
        }
        // De momento devuelve siempre true ya que es capaz de
        // resolver internamente los errores
        return true;
    }


///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS PRIVADOS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Devuelve las condiciones que proceden de condiciones impuestas
     * en base a las guas de navegacin
     * @return una lista de objetos {@link R01MBaseSearchedObj}
     */
    private List _extractGuideConditions(List conds) {
        List outConds = new ArrayList();
        if (conds != null) {
            R01MBaseSearchedObj searchedObj = null;
            for (Iterator it = conds.iterator(); it.hasNext(); ) {
                searchedObj = (R01MBaseSearchedObj)it.next();
                if (searchedObj.guideCondition != null) {       // procede de una guia
                    outConds.add(searchedObj);
                }
            }
        }
        if (outConds.isEmpty()) {
            return null;
        }
        return outConds;
    }
    /**
     * Devuelve las condiciones que NO proceden de condiciones impuestas
     * en base a las guas de navegacin
     * @return una lista de objetos {@link R01MBaseSearchedObj}
     */
    private List _extractNotGuideConditions(List conds) {
        List outConds = new ArrayList();
        if (conds != null) {
            R01MBaseSearchedObj searchedObj = null;
            for (Iterator it = conds.iterator(); it.hasNext(); ) {
                searchedObj = (R01MBaseSearchedObj)it.next();
                if (searchedObj.guideCondition == null) {       // no procede de una guia
                    outConds.add(searchedObj);
                }
            }
        }
        if (outConds.isEmpty()) {
            return null;
        }
        return outConds;
    }
    /**
     * Comprueba si un objeto buscado est ya en una lista de objetos buscados
     * @param theList la lista de objetos buscados
     * @param theObj el nuevo objeto a introducir en la lista
     * @return true si el objeto a introducir ya est en la lista, y false si no es as
     */
    private boolean _listContains(final List theList,final R01MBaseSearchedObj theObj) {
        if (theList != null && !theList.isEmpty()) {
            R01MBaseSearchedObj currSearchedObj = null;
            for (Iterator it=theList.iterator(); it.hasNext(); ) {
                currSearchedObj = (R01MBaseSearchedObj)it.next();
                if(currSearchedObj.equalsTo(theObj)) {
                    return true;
                }
            }
        }
        return false;
    }

}
