/*
 * Created on Sep 8, 2005
 *
 * @author ie00165h - Alex Lara
 * (c) 2005 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.objects.searchengine.indexer;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * <pre>
 * Documento para indexar en el buscador.
 * Contiene toda la informacin relativa al contenido y al propio documento
 * (nombres, objetos padre, catalogaciones, almacenamiento, tipificacion, etc)
 * de forma que es indexable de forma autnoma.
 * </pre>
 */
public class R01MIndexableDocument extends R01MBaseIndexableObject implements Serializable {
    private static final long serialVersionUID = -6759943549116142479L;
///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTANTES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Estado del documento para las nuevas interfaces de usuario.
     * Se corresponde con :
     * 		Estado de revisin: En redaccin o Pendiente de alguna revisin.
     * 		Estado de publicacin: Despublicado o Nunca publicado.
     */
    public static final int DOC_UI_STATE_NOTPUBLISHED = 0;
    /**
     * Estado del documento para las nuevas interfaces de usuario.
     * Se corresponde con :
     * 		Estado de revisin: Aprobado.
     * 		Estado de publicacin: Publicado.
     */
    public static final int DOC_UI_STATE_PUBLISHED = 1;
    /**
     * Estado del documento para las nuevas interfaces de usuario.
     * Se corresponde con :
     * 		Estado de revisin: En redaccin o Pendiente de alguna revisin.
     * 		Estado de publicacin: Publicado.
     */
    public static final int DOC_UI_STATE_PUBLISHED_AND_MODIFIED = 2;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS: DOCUMENTO
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Estado de aprobacin
     */
    private int approvalStatus;
    /**
     * Informacin de publicacion por repositorio: objetos {@link R01MIndexablePublicationInfo}
     */
    private List publicationInfo;
    /**
     * Languaje del documento
     */
    private String language;
    /**
     * Nivel de profundidad
     */
    private String depthLevel;
    /**
     * Lista de ficheros de datos del documento: objetos {@link R01MIndexableDataFile}.
     */
    private Map dataFiles;
    /**
     * Path relativo al contenido del documento
     */
    private String contentRelativePath;
    /**
     * Informacin de vinculaciones de contenidos.
     */
    private R01MIndexableDocumentLinkedData documentLinkedData = new R01MIndexableDocumentLinkedData();
    /**
     * Tags en texto libre separados por comas.
     */
    private String tags;
    /**
     * Datos de portal.Los atributos portalDataInitDate y portalDataEndDate sustituyen al atributo [Fecha Inicio] que tenemos
     * a nivel de contenido (aunque de momento no lo quitaramos) y permitiran poder especificar cundo se quiere que un contenido
     * empiece a aparecer en un contentList y cundo se quiere que desaparezca.
     */
    private Date portalDataInitDate;
    /**
     * Datos de portal.Los atributos portalDataInitDate y portalDataEndDate sustituyen al atributo [Fecha Inicio] que tenemos
     * a nivel de contenido (aunque de momento no lo quitaramos) y permitiran poder especificar cundo se quiere que un contenido
     * empiece a aparecer en un contentList y cundo se quiere que desaparezca.
     */
    private Date portalDataEndDate;
    /**
     * Nombre SEO asociado al documento.
     */
    private String portalDataDocumentSEOName = "";
    /**
     * Descripcion SEO asociada al documento.
     */
    private String portalDataDocumentSEODescription = "";
    /**
     * Alias principal del documento (el que se enlazar en los resultados de bsqueda)
     */
    private String mainFriendlyURL;
    /**
     * URLs al documento publicado organizados por el documento. 
     * Se refiere a los documentos equivalentes al objeto en el resto de idiomas.
     * Ej: 
     * 
     * Documento definido en el objeto: 
     *     /contenidos/informacion/contenido_1/es_tortuga/avanza.html
     * Si existen en esos idiomas, los documentos equivalentes al actual son:
     *     /contenidos/informacion/contenido_1/eu_tortuga/aurrera.html
     *     /contenidos/informacion/contenido_1/en_tortuga/forward.html
     * Este metadato se usa principalmente para cambiar el documento al cambiar el lenguaje de la pgina
     * con el selector de idiomas.
     * 
     * Como puede haber ms de un repositorio de publicacin la estructura del mapa ser:
     *      repo - Mapa{lang, path}
     * siendo lang, repo y path strings simple
     */
    private Map urlsByLanguage;
    /**
     * Mismo caso que en el anterior pero en lugar de almacenar la URL en formato tradicional, 
     * almacena la URL amigable si la tiene
     */
    private Map friendlyUrlsByLanguage;
///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTORES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Constructor vacio
     */
    public R01MIndexableDocument() {
        super();
        this.setObjectType(R01MBaseIndexableObject.OBJECT_TYPE_DOCUMENT);
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET: DOCUMENTO
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the approvalStatus.
     */
    public int getApprovalStatus() {
        return this.approvalStatus;
    }
    /**
     * @param theApprovalStatus The approvalStatus to set.
     */
    public void setApprovalStatus(final int theApprovalStatus) {
        this.approvalStatus = theApprovalStatus;
    }
    /**
     * @return Returns the publicationInfo: objetos R01MIndexablePublicationInfo
     */
    public List getPublicationInfo() {
        return this.publicationInfo;
    }
    /**
     * @param thePublicationInfo The publicationInfo to set: objetos R01MIndexablePublicationInfo
     */
    public void setPublicationInfo(final List thePublicationInfo) {
        this.publicationInfo = thePublicationInfo;
    }
    /**
     * Devuelve true si el documento est publicado en el repositorio que se pasa
     * como parmetro
     * @param repOid identificador del repositorio
     * @return true si est publicado
     */
    public boolean isPublishedInRepository(final String repOid) {
        if (repOid == null) {
            return false;
        }
        boolean isPublished = false;
        List publications = this.getPublicationInfo();
        if (publications != null && !publications.isEmpty()) {
            // Extraer las publicaciones del repositorio
            List targetPublications = new ArrayList();
            for (Iterator itPublications=publications.iterator(); itPublications.hasNext(); ) {
                R01MIndexablePublicationInfo publicacion = (R01MIndexablePublicationInfo)itPublications.next();
                if (publicacion.getRepositoryOid().equals(repOid)) {
                    targetPublications.add(publicacion);
                }
            }
            if (!targetPublications.isEmpty()) {
                for (Iterator itPub1=targetPublications.iterator(); itPub1.hasNext(); ) {
                    R01MIndexablePublicationInfo pub1 = (R01MIndexablePublicationInfo)itPub1.next();
                    if (pub1.getState() == R01MIndexablePublicationInfo.STATE_PUBLISHED) {
                        boolean isUnpublished = false;
                        // Buscar si existe despublicacin
                        for(Iterator itPub2=targetPublications.iterator();itPub2.hasNext();) {
                            R01MIndexablePublicationInfo pub2 = (R01MIndexablePublicationInfo)itPub2.next();
                            if (pub1.getRepositoryOid().equals(pub2.getRepositoryOid()) && pub2.getState() == R01MIndexablePublicationInfo.STATE_UNPUBLISHED) {
                                isUnpublished = true;
                                // Comparar fechas y ver si la despublicacion es posterior... esta despublicado
                                if (pub1.getPublishDate().after(pub2.getPublishDate())) {
                                    isPublished = true;
                                }
                                break;
                            }
                        }
                        // Existe un repositorio con publicacin y sin despublicacin
                        if (!isUnpublished) {
                            isPublished = true;
                            break;
                        }
                    } // if PUBLISHED
                } // for publications
            }
        }
        return isPublished;
    }
    /**
     * @return Returns the depthLevel.
     */
    public String getDepthLevel() {
        return this.depthLevel;
    }
    /**
     * @param theDepthLevel The theDepthLevel to set.
     */
    public void setDepthLevel(final String theDepthLevel) {
        this.depthLevel = theDepthLevel;
    }
    /**
     * @return Returns the language.
     */
    public String getLanguage() {
        return this.language;
    }
    /**
     * @param theLanguage The theLanguage to set.
     */
    public void setLanguage(final String theLanguage) {
        this.language = theLanguage;
    }
    /**
	 * @return the tags
	 */
	public String getTags() {
		return this.tags;
	}
	/**
	 * @param theTags The tags to set.
	 */
	public void setTags(String theTags) {
		this.tags = theTags;
	}
	/**
	 * @return the portalDataInitDate
	 */
	public Date getPortalDataInitDate() {
		return this.portalDataInitDate;
	}
	/**
	 * @param thePortalDataInitDate The portalDataInitDate to set.
	 */
	public void setPortalDataInitDate(Date thePortalDataInitDate) {
		this.portalDataInitDate = thePortalDataInitDate;
	}
	/**
	 * @return the portalDataEndDate
	 */
	public Date getPortalDataEndDate() {
		return this.portalDataEndDate;
	}
	/**
	 * @param thePortalDataEndDate The portalDataEndDate to set.
	 */
	public void setPortalDataEndDate(Date thePortalDataEndDate) {
		this.portalDataEndDate = thePortalDataEndDate;
	}
	/**
	 * @return the portalDataDocumentSEOName
	 */
	public String getPortalDataDocumentSEOName() {
		return this.portalDataDocumentSEOName;
	}
	/**
	 * @param thePortalDataDocumentSEOName The portalDataDocumentSEOName to set.
	 */
	public void setPortalDataDocumentSEOName(String thePortalDataDocumentSEOName) {
		this.portalDataDocumentSEOName = thePortalDataDocumentSEOName;
	}
	/**
	 * @return the portalDataDocumentSEODescription
	 */
	public String getPortalDataDocumentSEODescription() {
		return this.portalDataDocumentSEODescription;
	}
	/**
	 * @param thePortalDataDocumentSEODescription The portalDataDocumentSEODescription to set.
	 */
	public void setPortalDataDocumentSEODescription(String thePortalDataDocumentSEODescription) {
		this.portalDataDocumentSEODescription = thePortalDataDocumentSEODescription;
	}
	/**
     * @return the mainFriendlyURL
     */
    public String getMainFriendlyURL() {
        return this.mainFriendlyURL;
    }
    /**
     * @param theMainFriendlyURL The mainFriendlyURL to set.
     */
    public void setMainFriendlyURL(String theMainFriendlyURL) {
        this.mainFriendlyURL = theMainFriendlyURL;
    }
    /**
     * @return the urlsByLanguage
     */
    public Map getUrlsByLanguage() {
        return this.urlsByLanguage;
    }
    /**
     * @param theUrlsByLanguage The urlsByLanguage to set.
     */
    public void setUrlsByLanguage(Map theUrlsByLanguage) {
        this.urlsByLanguage = theUrlsByLanguage;
    }
    /**
     * @return the friendlyUrlsByLanguage
     */
    public Map getFriendlyUrlsByLanguage() {
        return this.friendlyUrlsByLanguage;
    }
    /**
     * @param theFriendlyUrlsByLanguage The friendlyUrlsByLanguage to set.
     */
    public void setFriendlyUrlsByLanguage(Map theFriendlyUrlsByLanguage) {
        this.friendlyUrlsByLanguage = theFriendlyUrlsByLanguage;
    }
///////////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the dataFiles.
     */
    public Map getDataFiles() {
        return this.dataFiles;
    }
    /**
     * @param theDataFiles The dataFiles to set.
     */
    public void setDataFiles(Map theDataFiles) {
        this.dataFiles = theDataFiles;
    }
    /**
     * Aade un nuevo datafile ad documento
     * @param theDataFile el nuevo datafile
     */
    public void addDataFile(R01MIndexableDataFile theDataFile) {
        if (this.dataFiles == null) {
            this.dataFiles = new HashMap();
        }
        this.dataFiles.put(theDataFile.getOid(),theDataFile);
    }
    /**
     * Devuelve el dataFile cuyo indicador se pasa como parametro
     * @param dataFileOid identificador del dataFile
     * @return el dataFile indexable
     */
    public R01MIndexableDataFile getDataFile(String dataFileOid) {
        return this.dataFiles == null ? null:(R01MIndexableDataFile)this.dataFiles.get(dataFileOid);
    }
    /**
     * Devuelve el dataFile de inicio o principal
     * @return el dataFile de inicio
     */
    public R01MIndexableDataFile getMainDataFile() {
        if (this.dataFiles == null) {
            return null;
        }
        R01MIndexableDataFile mainDataFile = null;
        for (Iterator it=this.dataFiles.values().iterator(); it.hasNext(); ) {
            R01MIndexableDataFile currDF = (R01MIndexableDataFile)it.next();
            if (currDF.isMainDataFile()) {
                mainDataFile = currDF;
                break;
            }
        }
        return mainDataFile;
    }
    /**
     * Devuelve los dataFiles que contienen metaDatos
     * @return un array de objetos {@link R01MIndexableDataFile} con aquellos
     *         dataFiles que contienen metaDatos
     */
    public R01MIndexableDataFile[] getMetaDataContainerDataFiles() {
        if (this.dataFiles == null) return null;
        List metaDataContainerDataFiles = new ArrayList();
        for (Iterator it=this.dataFiles.values().iterator(); it.hasNext(); ) {
            R01MIndexableDataFile currDF = (R01MIndexableDataFile)it.next();
            if (currDF.isMetaDataContainer()) {
                metaDataContainerDataFiles.add(currDF);
            }
        }
        return (R01MIndexableDataFile[])metaDataContainerDataFiles.toArray(new R01MIndexableDataFile[metaDataContainerDataFiles.size()]);
    }
    /**
     * Obtiene el path relativo al documento del dataFile por principal generado con la plantilla
     * por defecto
     * @return el path relativo
     */
    public String getDocumentMainDataFileDefaultGeneratedFileDocumentRelativePath() {
        R01MIndexableDataFile mainDf = this.getMainDataFile();
        if (mainDf == null) {
            return null;
        }
        return mainDf.getDefaultTemplateGeneratedFileDocumentRelativePath();
    }
    /**
     * Devuelve los paths relativos de los ficheros generados a partir del dataFile
     * por defecto
     * @return un mapa indexado por la plantilla de los paths relativos a los ficheros generados
     */
    public Map getDocumentMainDataFileGeneratedFilesDocumentRelativePaths() {
        R01MIndexableDataFile mainDf = this.getMainDataFile();
        if (mainDf == null) {
            return null;
        }
        return mainDf.getGeneratedFilesDocumentRelativePaths();
    }
    /**
     * Obtiene los metaDatos de un documento, para ello recorre todos los dataFiles
     * y obtiene los metaDatos de aquellos dataFiles que son "capaces" de contener
     * metaDatos
     * @return un mapa con metaDatos agregados
     */
    public Map getMetaData() {
        Map metaData = new HashMap();
        Map docDFs = this.getDataFiles();
        if (docDFs != null) {
            R01MIndexableDataFile idxDf = null;
            for (Iterator dfIt = docDFs.values().iterator(); dfIt.hasNext(); ) {
                idxDf = (R01MIndexableDataFile)dfIt.next();
                if (idxDf.isMetaDataContainer()) {
                    Map.Entry mdME = null;
                    if(idxDf.getMetaData() != null && idxDf.getMetaData().size() > 0) {
                        for (Iterator mdIt = idxDf.getMetaData().entrySet().iterator(); mdIt.hasNext(); ) {
                            mdME = (Map.Entry)mdIt.next();
                            metaData.put(mdME.getKey(),mdME.getValue());
                        }
                    }
                }
            }
        }
        return metaData;
    }
    /**
     * @return el path relativo
     */
    public String getContentRelativePath() {
        return this.contentRelativePath;
    }
    /**
     * @param theContentRelativePath
     */
    public void setContentRelativePath(String theContentRelativePath) {
        this.contentRelativePath = theContentRelativePath;
    }
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return the documentLinkedData
     */
    public R01MIndexableDocumentLinkedData getDocumentLinkedData() {
        return this.documentLinkedData;
    }
    /**
     * @param theDocumentLinkedData The documentLinkedData to set.
     */
    public void setDocumentLinkedData(R01MIndexableDocumentLinkedData theDocumentLinkedData) {
        this.documentLinkedData = theDocumentLinkedData;
    }

}
