/*
 * Created on Sep 8, 2005 @author ie00165h - Alex Lara (c) 2005 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.objects.searchengine.indexer;

import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import com.ejie.r01m.utils.R01MConstants;

/**
 * <pre>
 * Objeto base de un objeto complejo indexable en el buscador.
 * Los objetos complejos son: contenidos, documentos y archivos de datos.
 * Para indexar estos objetos complejos se utiliza un objeto que contiene TODA la informacin acerca del objeto (independientemente que est
 * asociada al contenido, al documento o al archivo):
 *      - Nombres y descripciones del propio objeto (R01MIndexable[Object]).
 *      - Almacenamiento (R01MBaseIndexableObject)
 *      - Tipificacin ({@link R01MBaseIndexableObject})
 *      - Catalogacin geogrfica y en ejes ({@link R01MBaseIndexableObject})
 * <p>
 * IMPORTANTE: Los objetos para indexacin de los objetos bsicos del modelo EXTIENDEN de {@link R01MBaseIndexableObject},
 * de forma que AL MENOS todos ellos tienen la informacin anterior (nombres, descripciones, almacenamiento, tipificacin y catalogacin).
 * <p>
 * Adicionalment los objetos para indexacin correspondientes a cada objeto bsico tienen sus propios miembros adicionales:
 *      - Nombres y descripciones del objeto(s) padre (objetos R01MBaseIndexableObject)
 *      - Cualquier otro objeto o propiedad
 *
 * Con esta arquitectura,cualquier objeto de la jerarqua: [contenido > documento > archivo de datos] es INDEXABLE independientemente,
 * pasando al indexador un objeto que contiene TODOS los datos nombres y descripciones, almacenamiento, tipificacion y catalogacin.
 * </pre>
 */
public abstract class R01MBaseIndexableObject implements Serializable {
    private static final long serialVersionUID = -7648660240656764641L;
///////////////////////////////////////////////////////////////////////////////////////////
// CONSTANTES
///////////////////////////////////////////////////////////////////////////////////////////
    // Constantes que se asignan a la propiedad objectType
    public static final int OBJECT_TYPE_UNKNOWN = R01MConstants.OBJECT_UNKNOWN;         //-1
    public static final int OBJECT_TYPE_CONTENT =  R01MConstants.OBJECT_CONTENT;        //0
    public static final int OBJECT_TYPE_DOCUMENT = R01MConstants.OBJECT_DOCUMENT;       //1
    public static final int OBJECT_TYPE_DATAFILE = R01MConstants.OBJECT_DATAFILE;       //2
    public static final int OBJECT_TYPE_PORTALPAGE = R01MConstants.OBJECT_PORTALPAGE;         //4
    public static final int OBJECT_TYPE_PORTALLANGPAGE = R01MConstants.OBJECT_PORTALLANGPAGE; //5
///////////////////////////////////////////////////////////////////////////////////////////
// MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////
//------- GENERALES
    /**
     * Tipo de objeto
     */
    private int objectType = -1;
    /**
     * Identificador
     */
     private String oid = null;
    /**
     * Autor
     */
    private String author;
    /**
     * Fecha de creacin
     */
    private Date createDate;
     /**
      * Nombre
      */
     private String name = null;
     /**
      * Nombre interno
      */
     private String internalName = null;
     /**
      * Descripcion
      */
     private String description = null;
//------- URLs
     /**
      * Url de edicion en diferentes idiomas
      */
     private Map editURLs = null;
  // ------- Paths
     /**
      * Informacin de los archivos Risp, de clave el tipo de documento risp y de valor objetos {@link R01MRispDocument}.
      */
     private Map contentRispDocumentsInfo = null;
     /**
      * Localizacion relativa en la workArea
      */
     private String workAreaRelativePath = null;
// ------- Metas
     /**
      * MetaDatos indexados por el oid
      */
     private Map metas = null;
     /**
      * Palabras clave
      */
     private String keywords = null;
//////////////////////////////////////////////////////////////////////////////////////////
// CONSTRUCTORES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Constructor vacio
     */
    public R01MBaseIndexableObject() {
        super();
        this.objectType = R01MBaseIndexableObject.OBJECT_TYPE_UNKNOWN;
    }
    /**
     * Constructor de un objeto resumen a partir de todos sus datos
     * @param newObjType tipo de objeto
     * @param newOid Identificador del objeto
     * @param newName Nombre del objeto
     * @param newInternalName Nombre interno del objeto
     * @param newDescription descripcion del objeto
     * @param newEditURLs URL de edicin en diferentes idiomas del interfaz de usuario
     * @param newContentRispDocumentsInfo paths a informacion Risp
     * @param newMetas metadatos indexados por el oid del metaDato
     * @param newKeywords palabras clave
     */
    public R01MBaseIndexableObject(final int newObjType,
                                   final String newOid,
                                   final String newName,
                                   final String newInternalName,
                                   final String newDescription,
                                   final Map newEditURLs,
                                   final Map newContentRispDocumentsInfo, 
                                   final Map newMetas,
                                   final String newKeywords) {
        this();
        this.objectType = newObjType;
        this.oid = newOid;
        this.name = newName;
        this.internalName = newInternalName;
        this.description = newDescription;
        this.editURLs = newEditURLs;
        this.contentRispDocumentsInfo = newContentRispDocumentsInfo;
        this.metas = newMetas;
        this.keywords = newKeywords;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET: GENERAL
////////////////////////////////////////////////////////////////////////////////////////// /
    /**
     * @return Returns the oid.
     */
    public String getOid() {
        return this.oid;
    }
    /**
     * @param theOid The oid to set.
     */
    public void setOid(final String theOid) {
        this.oid = theOid;
    }
    /**
     * @return Returns the objectType.
     */
    public int getObjectType() {
        return this.objectType;
    }
    /**
     * @param theObjectType The objectType to set.
     */
    public void setObjectType(final int theObjectType) {
        this.objectType = theObjectType;
    }
    /**
     * @return Returns the author.
     */
    public String getAuthor() {
        return this.author;
    }
    /**
     * @param theAuthor The author to set.
     */
    public void setAuthor(final String theAuthor) {
        this.author = theAuthor;
    }
    /**
     * @return Returns the creationDate.
     */
    public Date getCreateDate() {
        return this.createDate;
    }
    /**
     * @param theCreationDate The theCreationDate to set.
     */
    public void setCreateDate(final Date theCreationDate) {
        this.createDate = theCreationDate;
    }
    /**
     * @return Returns the name.
     */
    public String getName() {
        return this.name;
    }
    /**
     * @param theName The name to set.
     */
    public void setName(final String theName) {
        this.name = theName;
    }
    /**
     * @return Returns the description.
     */
    public String getDescription() {
        return this.description;
    }
    /**
     * @param theDescription The description to set.
     */
    public void setDescription(final String theDescription) {
        this.description = theDescription;
    }
    /**
     * @return Returns the internalName.
     */
    public String getInternalName() {
        return this.internalName;
    }
    /**
     * @param theInternalName The name to set.
     */
    public void setInternalName(final String theInternalName) {
        this.internalName = theInternalName;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET: URLs
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the editURL en cada idioma del interfaz de usuario.
     */
    public Map getEditURLs() {
        return this.editURLs;
    }
    /**
     * Devuelve la url de edicin en un determinado idioma de usuario
     * @param uiLang el lenguaje de usuario
     * @return la url de edicin
     */
    public String getEditURL(final String uiLang) {
        return this.editURLs != null ? (String)this.editURLs.get(uiLang) : null;
    }
    /**
     * @param theEditURLs Las url de edicin en cada idioma del interfaz de usuario
     */
    public void setEditURLs(final Map theEditURLs) {
        this.editURLs = theEditURLs;
    }
    /**
     * Mapa de clave el tipo de documento y de valor objetos {@link R01MRispDocument}.
     * @return the contentRispDocumentsInfo
     */
    public Map getContentRispDocumentsInfo() {
        return this.contentRispDocumentsInfo;
    }
    /**
     * @param theContentRispDocumentsInfo The contentRispDocumentsInfo to set.
     */
    public void setContentRispDocumentsInfo(final Map theContentRispDocumentsInfo) {
        this.contentRispDocumentsInfo = theContentRispDocumentsInfo;
    }
    /**
     * Aade una url de edicin
     * @param uiLang el lenguaje del interfaz de usuario
     * @param editURL la url de edicin
     */
    public void addEditURL(final String uiLang,final String editURL) {
        if (this.editURLs == null) {
            this.editURLs = new HashMap();
        }
        this.editURLs.put(uiLang,editURL);
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET: Paths
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the workAreaRelativePath.
     */
    public String getWorkAreaRelativePath() {
        return this.workAreaRelativePath;
    }
    /**
     * @param theWorkAreaRelativePath The workAreaRelativePath to set.
     */
    public void setWorkAreaRelativePath(final String theWorkAreaRelativePath) {
        this.workAreaRelativePath = theWorkAreaRelativePath;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET: Metas y Keywords
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Devuelve los metas
     * @return un mapa con los metas indexados
     */
    public Map getMetas() {
        return this.metas;
    }
    /**
     * Establece los metas
     * @param theMetas un mapa con los metas
     */
    public void setMetas(final Map theMetas) {
        this.metas = theMetas;
    }
    /**
     * Devuelve el valor de un meta
     * @param metaOid el oid del meta a devolver
     * @return el valor del meta
     */
    public String getMeta(final String metaOid) {
        return this.metas != null ? (String)this.metas.get(metaOid) : null;
    }
    /**
     * Devuelve los keywords
     * @return las plabras clave
     */
    public String getKeywords() {
        return this.keywords;
    }
    /**
     * Establece las keywords
     * @param theKeywords las palabras clave
     */
    public void setKeywords(final String theKeywords) {
        this.keywords = theKeywords;
    }
}
