/*
 * Created on Jan 4, 2006
 *
 * @author IE00165H - Alex Lara
 * (c) 2006 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.objects.searchengine.guide;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.ejie.r01f.util.StringUtils;

/**
 * <b>Nodo</b> de un rbol que define una <b>gua de navegacin</b><br>
 * Ejemplo de una gua:
 * <pre>
 *          >> Ayudas y Subvenciones    <-- [Tipo]
 *                  >> Dpto Hacienda    <-- [Etiqueta de catalogacin]
 *                          >> Direccin de Finanzas            <-- [Etiqueta de catalogacin]
 *                          >> Direccin de Control Econmico   <-- [Etiqueta de catalogacin]
 *                  >> Dpto Cultura     <-- [Etiqueta de catalogacin]
 *                          >> Araba        <-- [Etiqueta de catalogacin geogrfica]
 *                          >> Bizkaia      <-- [Etiqueta de catalogacin geogrfica]
 *                          >> Gipuzkoa     <-- [Etiqueta de catalogacin geogrfica]
 *          >> Autorizaciones
 *                  >> Dpto Industria
 * </pre>
 * <p>
 * Las gua de navegacin es un "aadido" que permite facilitar al usuario una forma de filtrar resultados.<br>
 * <pre>
 *              >> Elemento 1 Guia (x resultados)
 *              >> Elemento 2 Guia (y resultados)
 *              ---- Resultados (x + y) ----
 *           Pinchando en cualquiera de los elementos de la gua se aadir una condicin a la query y
 *           se presentar el siguiente nivel de la gua
 *              >> Elemento 1.1 Guia (a resultados)
 *              >> Elemento 1.2 Guia (b resultados)
 *              ---- Resultados (x = a+b) ----
 * </pre>
 * La gua es una estructura en "arbol" (padre-hijo) que puede contener nodos para filtrar de distintas formas:<br>
 * <ul>
 *      <li>Filtrado por tipologa: un cluster, familia o tipo.</li>
 *      <li>Filtrado por etiquetas de catalogacin.</li>
 *      <li>Filtrado por etiquetas de catalogacin geogrfica.</li>
 * </ul>
 * Cada nodo de la gua es un elemento tipo R01MSearchGuideNode que tiene una referencia al nodo padre y a los nodos
 * hijo. Adems contiene un objeto {@link com.ejie.r01m.objects.searchengine.guide.R01MSearchGuideElement} que describe el elemento de la gua: tipo e identificadores.
 */
public class R01MSearchGuideNode implements Serializable {
    /*(non-Javadoc)
     */
    private static final long serialVersionUID = -3000958207014210919L;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Indica si es un nodo clonado desde la gua maestra.
     */
    private boolean clonedNode;
    /**
     * Elemento padre, raz.
     */
    private R01MSearchGuideNode parentNode;
    /**
     * Elementos "hijo" (elementos {@link R01MSearchGuideNode}).
     */
    private List futureNodes;
    /**
     * El path por hasta llegar al elemento actual (el ltimo del path).<br>
     * Est compuesto por los oids de los elementos de la guia separados por <b>"/"</b>.<br>
     * Si se trata de un eje de catalogacin, ser el identificador de cada elemento, es el oid de la etiqueta.
     */
    private R01MSearchGuideElement nodeElement;
    /**
     * Tipo de nodo.
     * @see com.ejie.r01m.config.objects.searchengine.guide.R01MSearchGuideDef
     */
    private String nodeType = null; // typo / structure / geoStructure

///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTOR
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Constructor vaco
     */
    public R01MSearchGuideNode() {
        super();
    }
    /**
     * Constructor a partir de otro nodo.
     * @param other otro nodo
     */
    public R01MSearchGuideNode(final R01MSearchGuideNode other) {
        parentNode = other.getParentNode();
        futureNodes = other.getFutureNodes();
        nodeElement = other.getNodeElement();
        nodeType = other.getNodeType();
    }
    /**
     * Constructor en base a un elemento de la gua.
     * @param newNodeType tipo de nodo (typo/structure/geoStructure)
     * @param newBranchDefPath path de la rama en la definicion de la gua
     *                         Ej: guide_oid/Branch1_name/Branch2_name
     * @param newNodeElementPathInBranch path compuesto por los identificadores de los elementos recorridos
     *                                   en el branch hasta llegar al actual (que es el ltimo)
     * @param newNodeElementDescs descripciones del elemento en diferentes idiomas
     */
    public R01MSearchGuideNode( final String newNodeType,
                                final String newBranchDefPath,
                                final String newNodeElementPathInBranch,
                                final Map newNodeElementDescs) {
        nodeType = newNodeType;
        nodeElement = new R01MSearchGuideElement(newBranchDefPath,newNodeElementPathInBranch,newNodeElementDescs);
    }
    /**
     * Constructor en base al nodo padre.
     * @param newNodeType tipo de nodo (typo/structure/geoStructure)
     * @param newParentNode nodo padre
     */
    public R01MSearchGuideNode(final String newNodeType,final R01MSearchGuideNode newParentNode) {
        nodeType = newNodeType;
        parentNode = newParentNode;
    }
    /**
     * Constructor en base al tipo de nodo y al elemento.
     * @param newNodeType tipo de nodo (typo/structure/geoStructure
     * @param newElement nodo padre
     */
    public R01MSearchGuideNode(final String newNodeType,final R01MSearchGuideElement newElement) {
        nodeElement = newElement;
        nodeType = newNodeType;
    }
    /**
     * Constructor en base al nodo padre y al elemento.
     * @param newNodeType tipo de nodo (typo/structure/geoStructure)
     * @param newParentNode nodo padre
     * @param newElement elemento correspondiente al nodo
     */
    public R01MSearchGuideNode( final String newNodeType,
                                final R01MSearchGuideNode newParentNode,
                                final R01MSearchGuideElement newElement) {
        this(newNodeType,newParentNode);
        nodeElement = newElement;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Aade un nuevo nodo futuro (hijo) del nodo actual.
     * @param futureNode El nuevo nodo futuro
     */
    public void addFutureGuideNode(final R01MSearchGuideNode futureNode) {
        if (futureNode == null) {
            return;
        }
        if (futureNodes == null) {
            futureNodes = new ArrayList();
        }
        futureNodes.add(futureNode);
    }
    /**
     * Elimina el nodo futuro cuyo path en la guia coincide con el que se pasa.
     * @param nodePathInGuide nodo absoluto del nodo en la guia
     */
    public void removeFutureGuideNode(String nodePathInGuide) {
        if (futureNodes == null) {
            return;
        }
        R01MSearchGuideNode currFutureNode;
        int idx = 0;
        boolean nodeFound = false;
        for (Iterator it=futureNodes.iterator(); it.hasNext(); idx++) {
            currFutureNode = (R01MSearchGuideNode)it.next();
            if (currFutureNode.getPathInGuide().equals(nodePathInGuide)) {
                nodeFound = true;
                break;
            }
        }
        // Si se ha encontrado el nodo en idx est el indice del nodo en la lista
        if (nodeFound) {
            futureNodes.remove(idx);
        }
    }
    /**
     * Devuelve los elementos de los nodos futuros del nodo actual.
     * @return un array de objetos {@link R01MSearchGuideElement}
     */
    public R01MSearchGuideElement[] getFutureElementsArray() {
        if (futureNodes == null) {
            return null;
        }
        List futureGuideElements = new ArrayList();
        R01MSearchGuideNode currFutureNode;
        for (Iterator it=futureNodes.iterator(); it.hasNext(); ) {
            currFutureNode = (R01MSearchGuideNode)it.next();
            if (currFutureNode.getNodeElement() != null) {
                futureGuideElements.add(currFutureNode.getNodeElement());
            }
        }
        return (R01MSearchGuideElement[])futureGuideElements.toArray(new R01MSearchGuideElement[futureGuideElements.size()]);
    }
    /**
     * Obtiene un array con los elementos pasados en la guia de navegacin hasta llegar
     * al nodo actual.
     * @return Un array de elementos {@link R01MSearchGuideElement} en el que el ltimo elemento
     *         es el nodo actual
     */
    public R01MSearchGuideElement[] getPastElementsArray() {
        if (parentNode == null) {
            return null;
        }
        List pastGuideElements = new ArrayList();
        R01MSearchGuideNode currPastNod = parentNode;
        do {
            // OJO!!! Aadir siempre al principio de la lista, de forma que el primer elemento
            //        de la guia siempre quede el primero
            if (currPastNod.getNodeElement() != null) {
                pastGuideElements.add( 0,currPastNod.getNodeElement() );
            }
            currPastNod = currPastNod.getParentNode();        // Ir hacia atrs
        } while (currPastNod != null);
        return (R01MSearchGuideElement[])pastGuideElements.toArray( new R01MSearchGuideElement[pastGuideElements.size()] );
    }
    /**
     * Establece el estado del elemento subyacente del nodo.
     * @param numberOfResults numero de resultados
     * @param highLight si hay que resaltar el nodo o no
     */
    public void setNodeElementState(final int numberOfResults,final boolean highLight) {
        if (nodeElement == null) {
            return;
        }
        nodeElement.setHighLight(highLight);
        nodeElement.setNumberOfResults(numberOfResults);
    }
    /**
     * Aade al estado del elemento actual un nuevo estado
     * (suma el nmero de elementos y comprueba si hay que iluminar el nodo o no).
     * @param numberOfResults numero de resultados a aadir
     * @param highLight si hay que resaltar el nodo o no
     */
    public void addNodeElementState(final int numberOfResults,final boolean highLight) {
        if (nodeElement == null) {
            return;
        }
        nodeElement.setHighLight( nodeElement.isHighLight() | highLight );
        nodeElement.setNumberOfResults( nodeElement.getNumberOfResults() + numberOfResults );
    }
    /**
     * Resetea el estado del elemento (numero de resultados y si hay que resaltar o no).
     */
    public void resetNodeElement() {
        if (nodeElement != null) {
            nodeElement.reset();
        }
    }
    /**
     * Resetea el estado de los nodos futuros (numero de resultados y si hay que resaltar o no).
     */
    public void resetFutureNodeElements() {
        if (futureNodes != null) {
            R01MSearchGuideNode theNod = null;  // Nodo futuro de la guia
            for (Iterator it=futureNodes.iterator(); it.hasNext(); ) {
                theNod = (R01MSearchGuideNode)it.next();
                theNod.resetNodeElement();
            }
        }
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  CLONADO
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Clona el nodo y devuelve un objeto completamente nuevo.
     * @param theParentNode nodo padre
     * @return un objeto {@link R01MSearchGuideNode} clonado
     */
    public R01MSearchGuideNode cloneNode(R01MSearchGuideNode theParentNode) {
        // No volver a clonar el nodo si ya esta clonado...
        if (this.isClonedNode()) {
            return this;
        }

        R01MSearchGuideElement clonedEl = nodeElement.cloneElement();
        R01MSearchGuideNode theClonedNode = new R01MSearchGuideNode(new String(nodeType),theParentNode,clonedEl);
        theClonedNode.setClonedNode(true);     // Es un nodo clonado!!
        if (futureNodes != null) {
            // Inicializar la lista de nodos futuros
            theClonedNode.setFutureNodes( new ArrayList(futureNodes.size()) );
            for (Iterator it=futureNodes.iterator(); it.hasNext(); ) {
                R01MSearchGuideNode futureNode = (R01MSearchGuideNode)it.next();

                // Clonar... funcion recursiva...
                R01MSearchGuideNode clonedFutureNode = futureNode.cloneNode(theClonedNode);        // Poner como padre del nodo futuro el nodo clonado
                // OJO!!!   La relacin padre-hijo en los nodos nicamente se establece entre
                //          NODOS DENTRO DE UN BRANCH, es decir si hay dos branches:
                //              branch_eje
                //                  |_ branch_typos
                //          La relacin padre-hijo entre nodos nicamente se mantiene entre los nodos de cada branch.
                //          Para relacionar un branch con otro, cada nodo terminal del branch padre, tiene como hijos
                //          (nodos futuros) los nodos de primer nivel del branch de tipos, pero NO hay relacin
                //          padre-hijo entre los nodos terminales del branch de ejes y los nodos de primer nivel
                //          del branch de tipos
                if (!clonedFutureNode.getBranchDefPath().equals(theClonedNode.getBranchDefPath())) {
                    clonedFutureNode.setParentNode(null);   // Romper la relacin padre-hijo ya que se cambia de branch
                }

                // Aadir el clon del nodo futuro a la lista de nodos futuros del clon del nodo actual
                theClonedNode.addFutureGuideNode( clonedFutureNode );
            }
        }
        return theClonedNode;
    }
    /**
     * Clona un nodo.
     * @return un objeto {@link R01MSearchGuideNode} clonado en base al actual
     */
    public R01MSearchGuideNode cloneNode() {
        if (this.isClonedNode()) {
            return this;
        }
        R01MSearchGuideElement clonedEl = nodeElement.cloneElement();
        R01MSearchGuideNode theClonedNode = new R01MSearchGuideNode(nodeType,clonedEl);
        theClonedNode.setClonedNode(true);     // Es un nodo clonado!!
        return theClonedNode;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  DEPURACION
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Compone informacion de depuracin del nodo.
     * @param level nivel del nodo
     * @return una cadena de depuracion
     */
    String composeDebugInfo(final int level) {
        StringBuffer dbg = new StringBuffer();
        String tab = new String( StringUtils.getCharArray('\t',level) );
        if (nodeElement != null) {
            dbg.append(nodeElement.composeDebugInfo(level));
            dbg.append("\r\n");
        }
        if (futureNodes != null) {
            if (futureNodes != null) {
                dbg.append(tab);dbg.append("* Nodos Futuros...\r\n");
            }
            for (Iterator it=futureNodes.iterator(); it.hasNext(); ) {
                dbg.append( ((R01MSearchGuideNode)it.next()).composeDebugInfo(level+1) );
                if (it.hasNext()) dbg.append("\r\n");
            }
        }
        return dbg.toString();

    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return _clonedNode
     */
    public boolean isClonedNode() {
        return this.clonedNode;
    }
    /**
     * @param bClonedNode
     */
    public void setClonedNode(boolean bClonedNode) {
        clonedNode = bClonedNode;
    }
    /**
     * @return Returns the _nodeType
     */
    public String getNodeType() {
        return nodeType;
    }
    /**
     * @param theNodeType
     */
    public void setNodeType(String theNodeType) {
        nodeType = theNodeType;
    }
    /**
     * @return Returns the _futureNodes.
     */
    public List getFutureNodes() {
        return futureNodes;
    }
    /**
     * Obtiene un nodo futuro a partir del path absoluto del elemento (path del branch + path del elemento).
     * @param nodeElementPathInBranch path absoluto del elemento
     * @return el nodo si este se encuentra entre la coleccin de nodos futuros o null si no se encuentra
     */
    public R01MSearchGuideNode getFutureNode(String nodeElementPathInBranch) {
        if (futureNodes == null) {
            return null;
        }
        R01MSearchGuideNode outNode = null;
        for (Iterator it=futureNodes.iterator(); it.hasNext(); ) {
            R01MSearchGuideNode currNode = (R01MSearchGuideNode)it.next();
            if (currNode.getPathInGuide().equals(nodeElementPathInBranch)) {
                outNode = currNode;
                break;  // salir del for
            }
        }
        return outNode;
    }
    /**
     * Devuelve un array con los elementos futuros de la gua.
     * @return un array de objetos R01MSearchGuideNode
     */
    public R01MSearchGuideNode[] getFutureNodesArray() {
        return futureNodes == null ? null:(R01MSearchGuideNode[])futureNodes.toArray(new R01MSearchGuideNode[futureNodes.size()]);
    }
    /**
     * @param theChildNodes The _futureNodes to set.
     */
    public void setFutureNodes(List theChildNodes) {
        futureNodes = theChildNodes;
        if (theChildNodes == null) {
            return;
        }
        // Establecer el vinculo de cada nodo hijo con el nodo padre (este nodo)
        for (Iterator it=theChildNodes.iterator(); it.hasNext(); ) {
            R01MSearchGuideNode currNode = (R01MSearchGuideNode)it.next();
            currNode.setParentNode(this);
        }
    }
    /**
     * Establece los nodos futuros a partir de un array de nodos hijo
     * @param theChildNodes un array de objetos R01MSearchGuideNode con los nodos hijo
     */
    public void setFutureNodesFromArray(R01MSearchGuideNode[] theChildNodes) {
        if (theChildNodes == null) {
            futureNodes = null;
            return;
        }
        if (futureNodes == null) {
            futureNodes = new ArrayList(theChildNodes.length);
        }
        // Establecer el vinculo de cada nodo hijo con el nodo padre (este nodo)
        for (int i=0; i<theChildNodes.length; i++) {
            theChildNodes[i].setParentNode(this);
            futureNodes.add(theChildNodes[i]);
        }
    }
    /**
     * @return Returns the _nodeElement.
     */
    public R01MSearchGuideElement getNodeElement() {
        return nodeElement;
    }
    /**
     * Obtiene la descripcion del nodo en todos los idiomas
     * @return la descripcion del nodo
     */
    public Map getDescriptions() {
        return nodeElement == null ? null : nodeElement.getDescription();
    }
    /**
     * Obtiene la descripcin del nodo en un determinado idioma.
     * @param lang el idioma
     * @return la descripcin en el idioma
     */
    public String getDescription(String lang) {
        return this.getDescriptions() == null ? null:(String)this.getDescriptions().get(lang);
    }
    /**
     * @param theNodeElement The _nodeElement to set.
     */
    public void setNodeElement(R01MSearchGuideElement theNodeElement) {
        nodeElement = theNodeElement;
    }
    /**
     * Devuelve el path del elemento relativo al punto de inicio
     * en el branch en la gua.<br>
     * (ver la definicion de la guia {@link com.ejie.r01m.config.objects.searchengine.guide.R01MSearchGuideDef})
     * @return el path relativo del elemento al punto de inicio en el branch
     */
    public String getPathInBranch() {
        return nodeElement == null ? null : nodeElement.getElementPathInBranch();
    }
    /**
     * Devuelve el path del elemento absoluto, es decir, incluyendo el path del branch en la gua.<br>
     * (ver la definicion de la guia {@link com.ejie.r01m.config.objects.searchengine.guide.R01MSearchGuideDef})
     * @return el path absoluto del elemento
     */
    public String getPathInGuide() {
        return nodeElement == null ? null : nodeElement.getPathInGuide();
    }
    /**
     * Obtiene el path del branch al que pertenece este nodo en la definicin de la gua.<br>
     * (ver la definicion de la guia {@link com.ejie.r01m.config.objects.searchengine.guide.R01MSearchGuideDef})
     * @return el path del branch al que pertenece el nodo en la definicion de la guia o null si
     *         el elemento del nodo es nulo
     */
    public String getBranchDefPath() {
        return nodeElement == null ? null : nodeElement.getBranchDefPath();
    }
    /**
     * @return Returns the _parentNode.
     */
    public R01MSearchGuideNode getParentNode() {
        return parentNode;
    }
    /**
     * @param theParentNode The _parentNode to set.
     */
    public void setParentNode(R01MSearchGuideNode theParentNode) {
        parentNode = theParentNode;
    }
}
