/*
 * Created on Oct 11, 2006
 * 
 * @author IE00165H - Alex Lara
 * (c) 2006 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.objects.searchengine.guide;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.ejie.r01f.util.StringUtils;

/**
 * Elemento de la gua de navegacin.
 */
public class R01MSearchGuideElement implements Serializable {
    /*(non-Javadoc)
     */
    private static final long serialVersionUID = -2296964948690384531L;    
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * <pre>
     * Definicin del branch al que pertenece el nodo actual de la gua.
     * Ej: guide_1/branch_name1/branch_name2
     * </pre>
     */
    public String branchDefPath;
    /**
     * <pre>
     * Path del elemento relativo al branch de la gua al que pertenece.
     * Est compuesto por los oids de los elementos de la gua separados por /
     * Si se trata de un eje de catalogacin, ser el identificador de cada elemento
     * es el oid de la etiqueta.
     * Ej: euskadi/informacion
     * </pre>
     */
    public String elementPathInBranch;
    /**
     * Descripcin del elemento de la gua.
     */
    public Map description;
    /**
     * Nmero de resultados del elemento.
     */
    public int numberOfResults = -1;
    /**
     * Indica si hay que resaltar el resultado.
     */
    public boolean highLight = false;
    /**
     * Tiempo que se tarda en computar el nmero de resultados.
     */
    public long computeTime = -1;

///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTORES
///////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * Constructor vacio
     */
    public R01MSearchGuideElement() {
        super();
    }
    /**
     * Constructor a partir de otro elemento de gua.
     * @param other el otro elemento de gua
     */
    public R01MSearchGuideElement(R01MSearchGuideElement other) {
        this.branchDefPath = other.getBranchDefPath();
        this.elementPathInBranch = other.getElementPathInBranch();
        this.description = other.getDescription();
        this.highLight = other.isHighLight();
        this.numberOfResults = other.getNumberOfResults(); 
        this.computeTime = other.getComputeTime();
    }
    /**
     * Constructor 
     * @param newBranchDefPath path de la definicin del branch en el que se encuentra el nodo
     * @param newElementPathInBranch path del nodo en el branch (relativo al branch: comienza en el branch)
     * @param descs descripciones del elemento en diferentes idiomas
     */
    public R01MSearchGuideElement(String newBranchDefPath,String newElementPathInBranch,Map descs) {
        this.branchDefPath = newBranchDefPath;
        this.elementPathInBranch = newElementPathInBranch;
        this.description = descs;
    }
    /**
     * Constructor
     * @param newBranchDefPath path de la definicin del branch en el que se encuentra el nodo
     * @param newElementPathInBranch path del nodo relativo al branch
     * @param descs descripciones del elemento en diferentes idiomas
     * @param newNumberOfResults numero de resultados del elemento
     * @param newHighLight indica si hay que resaltar el resultado
     */
    public R01MSearchGuideElement(String newBranchDefPath,String newElementPathInBranch,Map descs,
                                  int newNumberOfResults,boolean newHighLight) {
        this(newBranchDefPath,newElementPathInBranch,descs);
        this.numberOfResults = newNumberOfResults;
        this.highLight = newHighLight;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Obtiene el identificador de la gua a partir del path del branch ya que
     * este tiene el formato guideOid/branch_name1/branch_name2...
     * @return el identificador de la gua
     */
    public String getGuideOid() {
        String[] branchDefPathSplitted = this.branchDefPath.split("/");
        return branchDefPathSplitted[0];    // El identificador de la gua siempre es el primer elemento del path
    }
    /**
     * Obtiene la clave el elemento de la gua que se compone concatenando el path
     * de la definicin del branch y el path del elemento.
     * @return branchDefPath/elementPath
     */
    public String getPathInGuide() {
        return this.branchDefPath + "/" + this.elementPathInBranch;
    }
    /**
     * Resetea el estado del elemento: nmero de resultados y si hay que resaltar o no
     */
    public void reset() {
        this.numberOfResults = 0;
        this.highLight = false;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  CLONADO
///////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * Clona el elemento y devuelve un nuevo objeto completamente nuevo.
     * @return nuevo elemento de la gua de navegacin con la misma informacin
     */
    public R01MSearchGuideElement cloneElement() { 
        // Devuelve un clon pero con el nmero de elementos a -1 y highligth=false
        R01MSearchGuideElement clonedEl = new R01MSearchGuideElement(new String(this.branchDefPath),new String(this.elementPathInBranch),
                                                                     _cloneStringMap(this.description));
        return clonedEl;
    }    
    /**
     * Clona un mapa de strings
     * @param theMap mapa string(key)-string(value)
     * @return nuevo mapa de strings
     */
    private Map _cloneStringMap(Map theMap)  {
        Map outMap = null;
        if (theMap != null) {
            outMap = new HashMap(theMap.size());
            for (Iterator it=theMap.entrySet().iterator(); it.hasNext(); ) {
                Map.Entry me = (Map.Entry)it.next();
                outMap.put(new String((String)me.getKey()),new String((String)me.getValue())); // NOPMD by co00390i on 21/11/08 10:28
            }
        }
        return outMap;
    }    

///////////////////////////////////////////////////////////////////////////////////////////
//  DEPURACION
///////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * Obtiene informacin de depuracin acerca del elemento
     * @param level nivel del nodo al que pertenece el elemento 
     * @return una cadena con la informacion
     */
    String composeDebugInfo(final int level) {
        StringBuffer dbg = new StringBuffer(80);
        String tab = new String( StringUtils.getCharArray('\t',level) );
        dbg.append(tab);dbg.append("-path in guide   : ");dbg.append(this.getPathInGuide());dbg.append("\r\n");
        dbg.append(tab);dbg.append("-description (es): ");dbg.append(this.getDescription("es"));dbg.append("\r\n");
        dbg.append(tab);dbg.append("-numberOfResults : ");dbg.append(this.getNumberOfResults());dbg.append(" - highLight=");dbg.append(this.isHighLight());
        
        return dbg.toString();
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @return Returns the branchDefPath
     */
    public String getBranchDefPath() {
        return this.branchDefPath;
    }
    /**
     * @param theBranchDefPath
     */
    public void setBranchDefPath(String theBranchDefPath) {
        this.branchDefPath = theBranchDefPath;
    }
    /**
     * @return Returns the elementPathInBranch
     */
    public String getElementPathInBranch() {
        return this.elementPathInBranch;
    }
    /**
     * @param theElementPath
     */
    public void setElementPathInBranch(String theElementPath) {
        this.elementPathInBranch = theElementPath;
    }
    /**
     * @return Returns the description
     */
    public Map getDescription() {
        return this.description;
    }
    /**
     * Obtiene la descripcion en un idioma
     * @param lang el idioma
     * @return la descripcion en el idioma
     */
    public String getDescription(String lang) {
        return this.description == null ? null:(String)this.description.get(lang);
    }
    /**
     * @param theDescription
     */
    public void setDescription(Map theDescription) {
        this.description = theDescription;
    }
    /**
     * @return Returns the highLight
     */
    public boolean isHighLight() {
        return this.highLight;
    }
    /**
     * @param theHighLight
     */
    public void setHighLight(boolean theHighLight) {
        this.highLight = theHighLight;
    }
    /**
     * @return Returns the numberOfResults
     */
    public int getNumberOfResults() {
        return this.numberOfResults;
    }
    /**
     * @param theNumberOfResults
     */
    public void setNumberOfResults(int theNumberOfResults) {
        this.numberOfResults = theNumberOfResults;
    }
    /**
     * @return Returns the computeTime
     */
    public long getComputeTime() {
        return this.computeTime;
    }
    /**
     * @param theComputeTime
     */
    public void setComputeTime(long theComputeTime) {
        this.computeTime = theComputeTime;
    }
}
