/*
 * Created on Oct 18, 2006
 * 
 * @author IE00165H - Alex Lara
 * (c) 2006 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.objects.searchengine.guide;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * <pre>
 * Objeto que modela una gua de navegacin
 * 
 * Una gua de navegacin es un objeto en rbol compuesto por objetos {@link R01MSearchGuideNode}
 * que se obtiene a partir de la definicin de la gua representada por el objeto
 * {@link com.ejie.r01m.config.objects.searchengine.guide.R01MSearchGuideDef}
 * El manager de guias de navegacin "expande" la definicin de la gua y crea todos los nodos
 * de la misma en el objeto {@link R01MSearchGuide}
 * <p>
 * IMPORTANTE:  La definicin de una gua de navegacin (objeto {@link com.ejie.r01m.config.objects.searchengine.guide.R01MSearchGuideDef}) 
 *              es la base para construir una gua de navegacin (objeto {@link com.ejie.r01m.objects.searchengine.guide.R01MSearchGuide})
 *              que simplemente es la "expansin" de las branches (ramas = {@link com.ejie.r01m.config.objects.searchengine.guide.R01MSearchGuideDefBranch})
 *              en los nodos correspondientes (objetos {@link R01MSearchGuideNode})
 * <p>
 * Cada nodo de la gua (objeto {@link R01MSearchGuideNode}) tiene asociado un elemento
 * (objeto {@link R01MSearchGuideElement} con la descripcin, nmero de resultados, etc
 * Cada nodo de la gua tiene un path UNICO como: 
 *      oidGuia/branch_name1/branch_name2/..../branch_nameN/elementOid1/elementOid2/.../elementOidN
 * es decir, el path del elemento se compone concatenando el path del branch en la definicin
 * de la gua y el path del elemento en dicho branch.
 * 
 * Para acceder de forma rpida a los elementos/nodos de la gua a partir de su path, 
 * este objeto indexa los nodos/elementos en funcin de su path.
 * </pre>
 */
public class R01MSearchGuide implements Serializable {
    private static final long serialVersionUID = 3623968304877854630L;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Identificador de la definicin de la gua
     */
    private String guideDefOid;
    /**
     * Nombre de la gua
     */
    private Map name;
    /**
     * Descripcion de la gua
     */
    private Map description;
    /**
     * Nodos de primer nivel de la gua de navegacin
     */
    private R01MSearchGuideNode[] firstLevelNodes;
    /**
     * Elemento actual de la gua de navegacin
     * (en el momento inicial es el elemento raz)
     */
    private R01MSearchGuideNode currentGuideNode = null;    
    /**
     * <pre>
     * Indice de los nodos de la gua.
     * Indexa cada uno de los nodos en base a su path en la gua y permite
     * el acceso rpido a cualquier nodo de la gua.
     * </pre>
     */
    private Map guideNodesIndex;
    
///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTORES
///////////////////////////////////////////////////////////////////////////////////////////       
    /**
     * Construcor vacio
     */
    public R01MSearchGuide() {
        super();
        guideNodesIndex = new HashMap();     // Indice de nodos
    }
    /**
     * Constructor en base al nombre de la gua
     * @param newName nombre de la gua en diferentes idiomas
     * @param newDescription descripcion de la gua en diferentes idiomas
     */
    public R01MSearchGuide(final String newGuideDefOid,final Map newName,final Map newDescription) {
        this();
        guideDefOid = newGuideDefOid;
        description = newDescription;
        name = newName;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Devuelve el identificador nico de la gua
     * @return el identificador nico de la gua
     */
    public String getGuideDefOid() {
        return this.guideDefOid;
    }
    /**
     * Establece el identificador nico de la gua
     * @param theGuideDefOid el identificador nico de la gua
     */
    public void setGuideDefOid(final String theGuideDefOid) {
        guideDefOid = theGuideDefOid;
    }    
    /**
     * Devuelve el nombre de la gua en forma de mapa idioma-descripcion
     * @return el nombre
     */
    public Map getName() {
        return name;
    }
    /**
     * Obtiene el nombre de una gua en un determinado idioma
     * @param lang el idioma
     * @return el nombre de la gua en el lenguaje especificado
     */
    public String getName(final String lang) {
        return name == null ? null : (String)name.get(lang);
    }
    /**
     * Establece el nombre de la gua en forma de un mapa idoma-descripcion
     * @param theName el nombre de la gua
     */
    public void setName(final Map theName) {
        name = theName;
    } 
    /**
     * Devuelve las descripciones de la gua en forma de un mapa idioma-descripcion
     * @return las descripciones
     */
    public Map getDescription() {
        return description;
    }
    /**
     * Obtiene la descripcion de una gua en un determinado idioma
     * @param lang el idioma
     * @return la descripcion de la gua en el lenguaje especificado
     */
    public String getDescription(final String lang) {
        return description == null ? null : (String)description.get(lang);
    }
    /**
     * Establece las descripciones de la gua en forma de un mapa idioma-descripcion
     * @param theDescription la descripcion
     */
    public void setDescription(final Map theDescription) {
        description = theDescription;
    }        
    /**
     * Devuelve los nodos de primer nivel de la gua
     * @return un array de nodos {@link R01MSearchGuideNode}
     */
    public R01MSearchGuideNode[] getFirstLevelNodes() {
        return firstLevelNodes;
    }
    /**
     * Establece los nodos de primer nivel de la gua
     * @param theFirstLevelNodes un array de objetos {@link R01MSearchGuideNode}
     */
    public void setFirstLevelNodes(final R01MSearchGuideNode theFirstLevelNodes[]) {
        firstLevelNodes = theFirstLevelNodes;
    }
    /**
     * Devuelve el nodo actual de la gua
     * @return el nodo actual
     */
    public R01MSearchGuideNode getCurrentGuideNode() {
        return currentGuideNode;
    }
    /**
     * Establece el nodo actual de la gua
     * @param theCurrentGuideNode el nodo actual
     */
    public void setCurrentGuideNode(R01MSearchGuideNode theCurrentGuideNode) {
        currentGuideNode = theCurrentGuideNode;
    }    
    /**
     * Obtiene el indice de nodos (elementos {@link R01MSearchGuideNode} indexados por su path absoluto en la gua)
     * @return un mapa de nodos indexados por su path en la gua
     */
    public Map getGuideNodesIndex() {
        return guideNodesIndex;
    }
    /**
     * Establece el indice de nodos (elementos {@link R01MSearchGuideNode} indexados por su path absoluto en la gua)
     * @param theNodeElementsIndex un mapa de nodos indexados por su path en la gua
     */
    public void setGuideNodesIndex(Map theNodeElementsIndex) {
        guideNodesIndex = theNodeElementsIndex;
    }
    /**
     * Obtiene el indice de nodos de un branch (elementos {@link R01MSearchGuideNode} indexados por su path absoluto en la gua)
     * @return el indice de nodos del branch
     */
    public Map getBranchNodesIndex(final String branchDefPath) {
        Map outBranchNodesIndex = null;
        // Recorrer todo el ndice in "extraer" los nodos que pertenecen al branch
        Map index = this.getGuideNodesIndex();
        if (index == null) {
            return null;
        }
        Map.Entry me = null;
        R01MSearchGuideNode currNode = null;
        String currNodeGuidePath = null;
        for (Iterator it=index.entrySet().iterator(); it.hasNext(); ) {
            me = (Map.Entry)it.next();
            currNodeGuidePath = (String)me.getKey();
            currNode = (R01MSearchGuideNode)me.getValue();
            if (currNodeGuidePath.startsWith(branchDefPath)) {
                if (outBranchNodesIndex == null) {
                    outBranchNodesIndex = new HashMap(); // NOPMD by co00390i on 21/11/08 10:25
                }
                outBranchNodesIndex.put(currNodeGuidePath,currNode);
            }
        }
        return outBranchNodesIndex;
    }
    
    /**
     * Devuelve los elementos de los nodos de primer nivel de la gua
     * @return un array de objetos {@link R01MSearchGuideElement}
     */
    public R01MSearchGuideElement[] getFirstLevelNodesElements() {
        R01MSearchGuideElement[] outEls = null;        
        if (firstLevelNodes != null) {
            outEls = new R01MSearchGuideElement[firstLevelNodes.length];
            for (int i=0; i<firstLevelNodes.length; i++) {
                outEls[i] = firstLevelNodes[i].getNodeElement();
            }
        }
        return outEls;
    }
    /**
     * Obtiene un nodo futuro a partir del path absoluto del elemento (path del branch + path del elemento)
     * @param nodeElementPathInBranch path absoluto del elemento
     * @return el nodo si este se encuentra entre la coleccin de nodos futuros o null si no se encuentra
     */
    public R01MSearchGuideNode getNextLevelNode(String nodeElementPathInBranch) {
        R01MSearchGuideNode[] futureNodes = (currentGuideNode != null ? currentGuideNode.getFutureNodesArray() : firstLevelNodes);
        if (futureNodes == null) {
            return null;
        }
        R01MSearchGuideNode outNode = null;
        for (int i=0; i<futureNodes.length; i++) {
            if (futureNodes[i].getPathInGuide().equals(nodeElementPathInBranch)) {
                outNode = futureNodes[i];
                break;  // salir del for
            }
        }
        return outNode;
    }
    /**
     * Devuelve los elementos de los nodos futuros del nodo actual
     * @return un array de objetos {@link R01MSearchGuideElement}
     */
    public R01MSearchGuideElement[] getCurrentNodeFutureElementsArray() {
        // Si el nodo actual de la gua es null, se est en la raz de la gua...
        // ... devolver entonces los elementos de primer nivel
        if (currentGuideNode == null) {
            return this.getFirstLevelNodesElements();
        }
        // ... en otro caso, devolver los elementos futuros del nodo actual
        return currentGuideNode.getFutureElementsArray();
    }
    /**
     * Obtiene un array con los elementos pasados en la gua de navegacin hasta llegar
     * al nodo actual  
     * @return Un array de elementos {@link R01MSearchGuideElement} en el que el ltimo elemento
     *         es el nodo actual
     */
    public R01MSearchGuideElement[] getCurrentNodePastElementsArray() {
        // Si el nodo actual de la gua es null, se est en la raz de la gua...
        // ... devolver entonces null (no hay elementos pasados)
        if (currentGuideNode == null) {
            return null;
        }
        // ... en otro caso, devolver los elementos pasados del nodo actual
        return currentGuideNode.getPastElementsArray();        
    }    
    /**
     * Resetea el estado de los nodos futuros de la gua
     * (pone a -1 el nmero de resultados del elemento y a false la propiedad highLight)
     */
    public void resetFutureNodesState() {
        if (currentGuideNode != null) {
            // Resetear los elementos futuros del nodo actual
            currentGuideNode.resetFutureNodeElements();
        } else {
            // Resetear los elementos del primer nivel
            R01MSearchGuideElement[] firstLevelEls = this.getFirstLevelNodesElements();
            if (firstLevelEls != null) {
                for (int i=0; i<firstLevelEls.length; i++) {
                    firstLevelEls[i].reset();
                }
            }
        }
    }
    
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS
///////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * Encuentra un nodo en la gua a partir de su indice en la gua
     * @param nodePathInGuide path absoluto del nodo en la gua
     * @return el objeto {@link R01MSearchGuideNode} de la gua
     */
    public R01MSearchGuideNode findNode(String nodePathInGuide) {
        return guideNodesIndex == null ? null : (R01MSearchGuideNode)guideNodesIndex.get(nodePathInGuide);
    }
    /**
     * Elimina un nodo de la gua de navegacin, para ello, se quita de:
     *      - El indice
     *      - De todos los nodos donde pueda estar referenciado
     * @param nodePathInGuide path del nodo en la gua
     */
    public void removeNode(String nodePathInGuide) {
        if (nodePathInGuide == null) {
            return;
        }
        
        
        // 1.- Eliminar el nodo del indice de nodos
        guideNodesIndex.remove(nodePathInGuide);  
        
        // 2.- Eliminar el nodo de todos los sitios
        Set nodesToRemove = new HashSet();        
        for (int i=0; i<firstLevelNodes.length; i++) {            
            _removeGuideNode(nodePathInGuide,firstLevelNodes[i]);
            // Ver si hay que eliminar este nodo
            if (firstLevelNodes[i].getPathInGuide().equals(nodePathInGuide)) {
                nodesToRemove.add(new Integer(i));             // NOPMD by co00390i on 21/11/08 10:26
            }
        }
        // Eliminar los nodos marcados
        if (!nodesToRemove.isEmpty()) {
            List newFirstLevelNodes = new ArrayList();
            for (int i=0; i<firstLevelNodes.length; i++) {
                if ( !nodesToRemove.contains(new Integer(i)) ) { // NOPMD by co00390i on 21/11/08 10:27
                    newFirstLevelNodes.add(firstLevelNodes[i]);
                }
            }
            firstLevelNodes = (R01MSearchGuideNode[])newFirstLevelNodes.toArray(new R01MSearchGuideNode[newFirstLevelNodes.size()]);
        }
    }
    private void _removeGuideNode(String nodeToRemoveGuidePath,R01MSearchGuideNode node) {
        if (node.getFutureNodes() != null && node.getFutureNodes().size() > 0) {
            List nodesToRemove = new ArrayList(); 
            R01MSearchGuideNode currNode = null;
            int i=0;            
            for (Iterator it=node.getFutureNodes().iterator(); it.hasNext(); i++) {
                currNode = (R01MSearchGuideNode)it.next();
                // Llamada iterativa
                _removeGuideNode( nodeToRemoveGuidePath,currNode );
                // Ver si hay que eliminar este nodo
                if (currNode.getPathInGuide().equals(nodeToRemoveGuidePath)) {
                    nodesToRemove.add(new Integer(i));                 // NOPMD by co00390i on 21/11/08 10:27
                }
            }
            // Eliminar los nodos marcados...
            for (Iterator it=nodesToRemove.iterator(); it.hasNext(); ) {
                int index = ((Integer)it.next()).intValue();
                node.getFutureNodes().remove( index );
            }
        }
    }    
    /**
     * Sigue un elemento de la gua de navegaci
     * @param futureNodePathInGuide Clave del elemento de la gua de navegacin
     * @return true si se ha conseguido ir al elemento indicado
     */
    public boolean followGuide(String futureNodePathInGuide) {
        return this.gotoNode(futureNodePathInGuide);
    }
    /**
     * "Rebobina" una gua de navegacin hasta el elemento que se indica
     * @param pastNodePathInGuide Clave del elemento al cual hay que "rebobinar"
     * @return true si se ha conseguido ir al elemento inidicado
     */
    public boolean rewindGuide(String pastNodePathInGuide) {
        return this.gotoNode(pastNodePathInGuide);
    }  
    /**
     * Lleva gua de navegacin hasta el elemento que se indica
     * @param nodePathInGuide Clave del elemento al cual hay que ir
     * @return true si se ha conseguido ir al elemento inidicado
     */
    public boolean gotoNode(String nodePathInGuide) {
        R01MSearchGuideNode node = this.findNode(nodePathInGuide);
        if (node == null) {
            return false;
        }
        // Hacer que el nodo actual de la gua sea el nodo futuro localizado
        if (node.getNodeElement() != null) {
            this.setCurrentGuideNode(node);                  
        }
        return true;
    }       
///////////////////////////////////////////////////////////////////////////////////////////
//  CLONADO
///////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * <pre>
     * Clonar la gua de navegacin y todos sus nodos.
     * La nica diferencia respecto a la gua de navegacin original es que los elementos
     * de los nodos tienen el nmero de elementos a -1 y highlight=false.
     * </pre>
     * @return la gua de navegacin clonada
     */
    public R01MSearchGuide cloneGuide() { 
        // Crear una nueva gua
        R01MSearchGuide clonedGuide = new R01MSearchGuide( new String(guideDefOid),
                                                          _cloneStringMap(name),
                                                          _cloneStringMap(description));
//        if (_firstLevelNodes != null) {
//            R01MSearchGuideNode clonedFirstLevelNodes[] = new R01MSearchGuideNode[_firstLevelNodes.length];
//            for (int i=0; i < _firstLevelNodes.length; i++) {
//                clonedFirstLevelNodes[i] = _firstLevelNodes[i].cloneNode(null);     // no hay padre...
//            }
//            clonedGuide.setFirstLevelNodes(clonedFirstLevelNodes);
//            // Componer el ndice de la gua
//            _composeClonedNodesIndex(clonedGuide.getGuideNodesIndex(),clonedFirstLevelNodes); 
//        }
//        return clonedGuide;
        
        // Clonar en base al ndice
        Map clonedGuideIndex = clonedGuide.getGuideNodesIndex();
        R01MSearchGuideNode clonedNode = null;
        for (Iterator it=guideNodesIndex.values().iterator(); it.hasNext(); ) {
            clonedNode = ((R01MSearchGuideNode)it.next()).cloneNode();
            clonedGuideIndex.put(clonedNode.getPathInGuide(),clonedNode);
        }
        // Reconstruir la jerarqua padre-hijo en base a la gua maestro
        R01MSearchGuideNode masterNode = null;
        for (Iterator it=guideNodesIndex.values().iterator(); it.hasNext(); ) {
            masterNode = (R01MSearchGuideNode)it.next();
            // Buscar el nodo clonado del master
            clonedNode = clonedGuide.findNode(masterNode.getPathInGuide());
            
            // Copiar el nodo padre
            if (masterNode.getParentNode() != null) {
                clonedNode.setParentNode( clonedGuide.findNode(masterNode.getParentNode().getPathInGuide()) );
            }
            // Copiar los nodos hijo
            if (masterNode.getFutureNodes() != null) {
                for (Iterator fit=masterNode.getFutureNodes().iterator(); fit.hasNext(); ) {
                    clonedNode.addFutureGuideNode( clonedGuide.findNode(((R01MSearchGuideNode)fit.next()).getPathInGuide()) );
                }
            }
        }   
        // Copiar los nodos de primer nivel
        if (firstLevelNodes != null) {
            R01MSearchGuideNode[] clonedGuideFirstLevelNodes = new R01MSearchGuideNode[firstLevelNodes.length];
            for (int i=0; i < firstLevelNodes.length; i++) {
                clonedGuideFirstLevelNodes[i] = clonedGuide.findNode(firstLevelNodes[i].getPathInGuide());
            }
            clonedGuide.setFirstLevelNodes(clonedGuideFirstLevelNodes);            
        }
        return clonedGuide;        
    }      
    /**
     * Funcin recursiva que compone un ndice con los elementos de un branch
     * (al final de la ejecucin en index est el nidice del branch)
     * @param index el indice compuesto
     * @param nodes los nodos del branch actual
     */
    /*private static void _composeClonedNodesIndex(Map index,R01MSearchGuideNode[] nodes) {
        for (int i=0; i<nodes.length; i++) {
            index.put(nodes[i].getPathInGuide(),nodes[i]);
            //System.out.println("guide_clon>" + nodes[i].getNodeElementPathInGuide());
            if (nodes[i].getFutureNodes() != null) _composeClonedNodesIndex(index,nodes[i].getFutureNodesArray());
        }
    }*/
    /**
     * Clona un mapa de strings
     * @param theMap mapa string(key)-string(value)
     * @return un mapa clonado
     */
    private Map _cloneStringMap(Map theMap)  {
        Map outMap = null;
        if (theMap != null) {
            outMap = new HashMap(theMap.size());
            for (Iterator it=theMap.entrySet().iterator(); it.hasNext(); ) {
                Map.Entry me = (Map.Entry)it.next();
                outMap.put(new String((String)me.getKey()),new String((String)me.getValue())); // NOPMD by co00390i on 21/11/08 10:27
            }
        }
        return outMap;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  DEPURACION
///////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * Obtiene informacin de depuracin sobre la gua de navegacin
     * @return Una cadena con la informacion de depuracion
     */
    public String composeDebugInfo() {
        StringBuffer dbg = new StringBuffer(28);
        dbg.append("GUIA: ");dbg.append(name.get("es"));dbg.append(" guideDefOid=");dbg.append(guideDefOid);dbg.append("\r\n");
        if (firstLevelNodes != null) {
            for (int i=0; i < firstLevelNodes.length; i++) {
                dbg.append(firstLevelNodes[i].composeDebugInfo(1));dbg.append("\r\n");
            }
        }
        return dbg.toString();
    }

    
}
