/*
 * Created on Jul 21, 2005
 *
 * @author ie00165h - Alex Lara
 * (c) 2005 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.objects.cataloguing.structures;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.ejie.r01f.util.StateMap;
import com.ejie.r01m.objects.R01MPrintableInterface;


/**
 * Estructura de catalogacin (eje).Define toda la estructura en rbol que cuelga de una estructura.
 */
public class R01MStructureItem implements Serializable,
                                          R01MPrintableInterface {

    private static final long serialVersionUID = -8805143727227093981L;
///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTANTES
///////////////////////////////////////////////////////////////////////////////////////////
    public final static String SALTO_LINEA = "\r\n";
	public final static String NULL_VALUE = "null";

///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////
    /*
     * Identificador nico del structureItem
     */
    private int relationId;
    /*
     * Identificador nico del padre structureItem
     */
    private int relationParentId;
    /*
     * Identificador numrico de la etiqueta padre de la relacin.
     */
    private int rootLabelId;
    /**
     * Identificador de la etiqueta padre de la relacin.
     */
    private String rootLabelOid;
    /*
     * Identificador numrico de la etiqueta hija de la relacin.
     */
    private int childLabelId;
    /**
     * Identificador de la estructura
     */
	private String structureGuid;
	/**
     * Identificador numrico de la estructura
     */
    private int structureId;
	/**
	 * Item padre, para los items de role 0 ser <code>null</code>.
	 */
	private R01MStructureItem parent = null;
	/**
	 * Elementos de tipo {@link com.ejie.r01m.objects.cataloguing.structures.R01MStructureItem} hijos de la estructura.
	 * Indexados por por el id numrico de la relacin. Puede estar vaco para realizar las cargas de forma ligera,
	 * el atributo {@link #hasChildren} indica si realmente tiene hijos o no.
	 */
	private Map children = new StateMap();
	/**
     * Role.
     */
    private int role = -1;
	/**
	 * Lista con objetos {@link com.ejie.r01m.objects.cataloguing.structures.R01MStructureItem} de similitud.
	 */
	private List similarLabels;
	/**
	 * Ruta desde la estructura a la etiqueta hija
	 */
	private String childLabelPath;
	/**
	 * Label asociado al item, se especificar en el API el idioma del label, de lo contrario no tiene valor.
	 */
	private String standarizedTermInLanguage;
	/**
	 * Descripcin de la etiqueta (utilizacion, criterios, etc).
	 */
	private String labelDescription;
	/**
     * Indica si el hijo de la relacin tambin tiene hijos.
     */
    private boolean hasChildren = false;

///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTORES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Constructor vaco
     */
    public R01MStructureItem() {
        super();
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET
///////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * @return Returns the parent.
	 */
	public R01MStructureItem getParent() {
		return this.parent;
	}

	/**
	 * @param theParent The parent to set.
	 */
	public void setParent(R01MStructureItem theParent) {
		this.parent = theParent;
	}
	/**
	 * @return Returns the childreen.
	 */
	public Map getChildren() {
		return this.children;
	}
	/**
	 * @param theChildren The children to set.
	 */
	public void setChildren(Map theChildren) {
		this.children = theChildren;
	}

	/**
	 * @return Returns the structureGuid.
	 */
	public String getStructureGuid() {
		return this.structureGuid;
	}
	/**
	 * @param theStructureGuid The structureDef to set.
	 */
	public void setStructureGuid(String theStructureGuid) {
		this.structureGuid = theStructureGuid;
	}

	/**
     * @return the structureId
     */
    public int getStructureId() {
        return this.structureId;
    }
    /**
     * @param theStructureId The structureId to set.
     */
    public void setStructureId(int theStructureId) {
        this.structureId = theStructureId;
    }
    /**
	 * @return Returns the role.
	 */
	public int getRole() {
		return this.role;
	}
	/**
	 * @param theRole The role to set.
	 */
	public void setRole(int theRole) {
		this.role = theRole;
	}
	/**
	 * @return Returns the similarLabels.
	 */
	public List getSimilarLabels() {
		return this.similarLabels;
	}
	/**
	 * @param theSimilarLabels The similarLabels to set.
	 */
	public void setSimilarLabels(List theSimilarLabels) {
		this.similarLabels = theSimilarLabels;
	}
	   /**
     * @return the childLabelPath
     */
    public String getChildLabelPath() {
        return this.childLabelPath;
    }

    /**
     * @param theChildLabelPath The childLabelPath to set.
     */
    public void setChildLabelPath(String theChildLabelPath) {
        this.childLabelPath = theChildLabelPath;
    }

////////////////////////////////////////////////////////////////////////////////////////
//		METODOS DE CARGA DE MAPAS
///////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Metodo que comprueba si existen etiquetas similares
	 * @return true si existen etiquetas similares
	 */
	public boolean hasSimilarLabels() {
		if (this.similarLabels.isEmpty()) {
			return false;
		}
		return true;
	}

	/**
	 * Aadir un hijo a una componente de una estructura
	 * @param child
	 */
	public void addChild(R01MStructureItem child) {
		this.children.put(child.getRootLabelOid(),child);
	}

	/**
	 * Aadir una etiqueta a una componente de una estructura
	 * @param similar
	 */
	public void addSimilarLabel(R01MStructureItem similar) {
		this.similarLabels.add("" + similar.getRootLabelId());
	}

	/**
	 * @return childrenGUIDs
	 */
	public List getChildrenGUIDs() {
		ArrayList childrenGUIDs=new ArrayList();
		Iterator childrenIterator=this.getChildren().keySet().iterator();
		childrenGUIDs.add(this.getRootLabelOid());

		if (this.hasChildren()) {
			while (childrenIterator.hasNext()){
				R01MStructureItem st=(R01MStructureItem) this.getChildren().get(childrenIterator.next());
				childrenGUIDs.addAll(st.getChildrenGUIDs());
			}
		}
		return childrenGUIDs;
	}

	/**
	 * @return childrenRelations
	 */
	public List getChildrenRelations() {
		ArrayList childrenRelations=new ArrayList();

		if (this.hasChildren()) {
			Iterator childrenIterator = this.getChildren().keySet().iterator();
			while (childrenIterator.hasNext()){
				R01MStructureItem st=(R01MStructureItem) this.getChildren().get(childrenIterator.next());
				childrenRelations.addAll(st.getChildrenRelations());
			}
			if (this.getParent() == null) {
				childrenRelations.add(new R01MRelation(0,this.getRootLabelId(),R01MRelation.RELATION_HIERARCHY_CODE));
			} else {
				childrenRelations.add(new R01MRelation(this.getParent().getRootLabelId(),this.getRootLabelId(),R01MRelation.RELATION_HIERARCHY_CODE));
			}
		} else {
			if (this.getParent()==null) {
				childrenRelations.add(new R01MRelation(0,this.getRootLabelId(),R01MRelation.RELATION_HIERARCHY_CODE));
			} else {
				childrenRelations.add(new R01MRelation(this.getParent().getRootLabelId(),this.getRootLabelId(),R01MRelation.RELATION_HIERARCHY_CODE));
			}
		}
		return childrenRelations;
	}
    /**
     * @return the relationId
     */
    public int getRelationId() {
        return this.relationId;
    }
    /**
     * @param theRelationId The relationId to set.
     */
    public void setRelationId(int theRelationId) {
        this.relationId = theRelationId;
    }
    /**
     * @return the relationParentId
     */
    public int getRelationParentId() {
        return this.relationParentId;
    }
    /**
     * @param theRelationParentId The relationParentId to set.
     */
    public void setRelationParentId(int theRelationParentId) {
        this.relationParentId = theRelationParentId;
    }

    /**
     * @return the rootLabelId
     */
    public int getRootLabelId() {
        return this.rootLabelId;
    }

    /**
     * @param theRootLabelId The rootLabelId to set.
     */
    public void setRootLabelId(int theRootLabelId) {
        this.rootLabelId = theRootLabelId;
    }

    /**
     * @return the rootLabelOid
     */
    public String getRootLabelOid() {
        return this.rootLabelOid;
    }

    /**
     * @param theRootLabelOid The rootLabelOid to set.
     */
    public void setRootLabelOid(String theRootLabelOid) {
        this.rootLabelOid = theRootLabelOid;
    }

    /**
     * @return the childLabelId
     */
    public int getChildLabelId() {
        return this.childLabelId;
    }

    /**
     * @param theChildLabelId The childLabelId to set.
     */
    public void setChildLabelId(int theChildLabelId) {
        this.childLabelId = theChildLabelId;
    }

    /**
     * Obtiene el trmino para el idioma de presentacin.
	 * @return el trmino para el idioma de presentacin.
	 */
	public String getStandarizedTermInLanguage() {
		return this.standarizedTermInLanguage;
	}

	/**
	 * Establece el trmino para el idioma de presentacin.
	 * @param theStandarizedTermInLanguage el trmino para el idioma de presentacin.
	 */
	public void setStandarizedTermInLanguage(String theStandarizedTermInLanguage) {
		this.standarizedTermInLanguage = theStandarizedTermInLanguage;
	}

	/**
	 * Obtiene la descripcin de la etiqueta (utilizacion, criterios, etc).
	 * @return the labelDescription la descripcin de la etiqueta (utilizacion, criterios, etc).
	 */
	public String getLabelDescription() {
		return this.labelDescription;
	}

	/**
	 * Establece la descripcin de la etiqueta (utilizacion, criterios, etc).
	 * @param theLabelDescription la descripcin de la etiqueta (utilizacion, criterios, etc).
	 */
	public void setLabelDescription(String theLabelDescription) {
		this.labelDescription = theLabelDescription;
	}

	/**
	 * @return the hasChildren
	 */
	public boolean hasChildren() {
		return this.hasChildren;
	}

	/**
	 * @param theHasChildren The hasChildren to set.
	 */
	public void setHasChildren(boolean theHasChildren) {
		this.hasChildren = theHasChildren;
	}
////////////////////////////////////////////////////////////////////////////////////
//  DEBUG
////////////////////////////////////////////////////////////////////////////////// //
    /**
     * Devuelve una representacion en formato cadena del objeto
     * @param prefix prefijo a ser concatenado a todas las lneas
     * @return El resumen de la pagina
     *              Propiedades
     *              Areas Visuales
     */
    public StringBuffer debugInfo(String prefix) {
        String tabPrefix = prefix + "\t";
        StringBuffer sb = new StringBuffer(163);
        sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("======== R01MStructureItem  ========\r\n");
        sb.append(prefix);sb.append("\tStructureId:");sb.append(this.getStructureId());sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("\tStructureGuid:");sb.append(this.getStructureGuid() == null ? "null" : this.getStructureGuid());sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("\tStandarizedTermInLanguage:");sb.append(this.getStandarizedTermInLanguage() == null ? "null" : this.getStandarizedTermInLanguage());sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("\tLabelDescription:");sb.append(this.getLabelDescription() == null ? "null" : this.getLabelDescription());sb.append(SALTO_LINEA);

        // Similar labels
        if (this.getSimilarLabels() != null && !this.getSimilarLabels().isEmpty()) {
            sb.append(prefix);sb.append("\tSimilarLabels:\r\n");
            for (Iterator it = this.getSimilarLabels().iterator(); it.hasNext(); ) {
            	Map.Entry me = (Map.Entry)it.next();

                sb.append(tabPrefix);
                sb.append(((R01MStructureItem)me.getValue()).debugInfo(tabPrefix));
                sb.append(SALTO_LINEA);
            }
        }

        sb.append(prefix);sb.append("\tRelationParentId:");sb.append(this.getRelationParentId());sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("\tRelationId:");sb.append(this.getRelationId());sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("\tRootLabelId:");sb.append(this.getRootLabelId());sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("\tRootLabelOid:");sb.append(this.getRootLabelOid() == null ? "null" : this.getRootLabelOid());sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("\tRole:");sb.append(this.getRole());sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("\tParent:");sb.append(this.getParent() == null ? "null" : this.getParent().debugInfo(tabPrefix).toString());sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("\tHasChildren:");sb.append(this.hasChildren());sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("\tChildLabelId:");sb.append(this.getChildLabelId());sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("\tChildLabelPath:");sb.append(this.getChildLabelPath() == null ? "null" : this.getChildLabelPath());sb.append(SALTO_LINEA);

        // Hijos
        if (this.getChildren() != null && !this.getChildren().isEmpty()) {
            sb.append(prefix);sb.append("\tChildren (");sb.append(this.getChildren().size());sb.append("):\r\n");
            for (Iterator it = this.getChildren().entrySet().iterator(); it.hasNext(); ) {
            	Map.Entry me = (Map.Entry)it.next();
                sb.append(tabPrefix);
                sb.append(((R01MStructureItem)me.getValue()).getStandarizedTermInLanguage());
                sb.append(SALTO_LINEA);
            }
        }

        return sb;
    }

}
