package com.ejie.r01m.objects.cataloguing.structures;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.ejie.r01f.util.StateMap;
import com.ejie.r01m.objects.R01MPrintableInterface;

/**
 * Objeto que modela una estructura (eje) de catalogacin.<br>
 * Una estructura de catalogacin es un conjunto de etiquetas de catalogacin y sus relaciones jerrquicas.
 */
public class R01MStructure implements Serializable,
                                      R01MPrintableInterface {

	private static final long serialVersionUID = -658010022470457242L;

///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTANTES
///////////////////////////////////////////////////////////////////////////////////////////
    public final static String SALTO_LINEA = "\r\n";
	public final static String NULL_VALUE = "null";
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Definicin de la estructura.
	 */
	private R01MStructureDefinition sDefinition = null;
	/**
	 * Objetos {@link R01MStructureItem} con los miembros de la estructura indexados por el identificador de la etiqueta.
	 */
	private Map structureItems = new StateMap();

// /////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTORES
// /////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Constructor vacio
	 */
	public R01MStructure() {
		super();
	}
	/**
	 * Constructor de la estructura a partir de su definicin
	 * @param sDef
	 */
	public R01MStructure(R01MStructureDefinition sDef) {
		super();
		sDefinition = sDef;
	}
///////////////////////////////////////////////////////////////////////////////////////////
//  GET && SET
///////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Metodo que aade descendientes de una estructura, indexados por el oid de la etiqueta descendiente.
     * @param newStructureItem
	 */
	public void addStructureItem(R01MStructureItem newStructureItem){
		structureItems.put(newStructureItem.getRootLabelOid(),newStructureItem);
	}
	/**
	 * Definicin de la estructura.
	 * @return Returns the sDefinition.
	 */
	public R01MStructureDefinition getSDefinition() {
		return sDefinition;
	}
	/**
	 * Establece la definicin de la estructura.
	 * @param definition The sDefinition to set.
	 */
	public void setSDefinition(R01MStructureDefinition definition) {
	    sDefinition = definition;
	}
	/**
	 * Miembros de la estructura.
	 * @return Returns the structuresItem.
	 * @deprecated usar {@link #getStructureItems()}
	 */
	public Map getStructuresItem() {
		return structureItems;
	}
	/**
	 * Establece los miembros de la estructura.
	 * @param newStructuresItem The structuresItem to set.
	 * @deprecated usar {@link #setStructureItems(Map)}
	 */
	public void setStructuresItem(Map newStructuresItem) {
		structureItems = newStructuresItem;
	}
	/**
     * @return the structureItems
     */
    public Map getStructureItems() {
        return this.structureItems;
    }
    /**
     * @param theStructureItems The structureItems to set.
     */
    public void setStructureItems(Map theStructureItems) {
        this.structureItems = theStructureItems;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS PUBLICOS
///////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Busca una etiqueta dentro de la estructura a partir del oid de la etiqueta.
	 * @param labelOid oid de la etiqueta.
	 * @return Una coleccion de objetos {@link R01MStructureItem} cada uno de los cuales
	 *         es el elemento inicio de la rama que contiene la etiqueta buscada
	 *         Devuelve <code>null</code> si la estructura esta vacia o un array vaco si la etiqueta
	 *         no se encuentra en la estructura.
	 */
	public Collection findLabel(String labelOid) {
	    if (structureItems == null) return null;
        List outStructureItems = new ArrayList();   // Lista de salida
        R01MStructureItem currItem = null;
        R01MStructureItem foundItem = null;
        for (Iterator it=structureItems.values().iterator(); it.hasNext(); ) {
            currItem = (R01MStructureItem)it.next();
            foundItem = this.findLableInSubBranch(currItem,labelOid);
            if (foundItem != null) outStructureItems.add(currItem);    // Aadir el elemento de primer nivel!!!
        }
        return outStructureItems;
	}
    /**
     * Encuentra una etiqueta con un oid dado en una sub-rama de la estructura
     * que comienza en el objeto {@link R01MStructureItem} que se pasa como parametro.
     * @param parentItem objeto {@link R01MStructureItem} que da inicio a la sub-rama
     * @param labelOid identificador de la etiqueta.
     * @return el objeto {@link R01MStructureItem} con la etiqueta encontrada.
     * @deprecated
     */
	public R01MStructureItem findLableInSubBranch(R01MStructureItem parentItem,String labelOid) {
       return findLabelInSubBranch(parentItem,labelOid);
	}
	
	/**
     * Encuentra una etiqueta con un oid dado en una sub-rama de la estructura
     * que comienza en el objeto {@link R01MStructureItem} que se pasa como parametro.
     * @param parentItem objeto {@link R01MStructureItem} que da inicio a la sub-rama
     * @param labelOid identificador de la etiqueta.
     * @return el objeto {@link R01MStructureItem} con la etiqueta encontrada.
     */
	public R01MStructureItem findLabelInSubBranch(R01MStructureItem parentItem,String labelOid) {
        // Ver si el propio padre, es el objeto buscado
        if (parentItem.getRootLabelOid().equals(labelOid)) return parentItem;
        // Buscar en los descendientes recurrentemente
        R01MStructureItem outItem = null;
        for (Iterator it=parentItem.getChildren().values().iterator(); it.hasNext(); ) {
            outItem = this.findLableInSubBranch((R01MStructureItem)it.next(),labelOid);
            if (outItem != null) return outItem;   // detener la bsqueda
        }
        // No se ha encontrado la etiqueta
        return null;
    }
////////////////////////////////////////////////////////////////////////////////////
//  DEBUG
////////////////////////////////////////////////////////////////////////////////// //
    /**
     * Devuelve una representacion en formato cadena del objeto
     * @param prefix prefijo a ser concatenado a todas las lneas
     * @return una cadena con el detalle de la estructura
     */
    public StringBuffer debugInfo(String prefix) {
        String tabPrefix = prefix + "\t";
        StringBuffer sb = new StringBuffer(163);
        sb.append(SALTO_LINEA);
        sb.append(prefix);sb.append("======== R01MStructure  ========\r\n");
        sb.append(prefix);sb.append("\tStructure definition:");sb.append(this.getSDefinition().debugInfo(tabPrefix));sb.append(SALTO_LINEA);

        // Items
        if (this.getStructureItems() != null && !this.getStructureItems().isEmpty()) {
            sb.append(prefix);sb.append("\tStructureItems (");sb.append(this.getStructureItems().size());sb.append("):\r\n");
            Map.Entry me = null;
            for (Iterator it = this.getStructureItems().entrySet().iterator(); it.hasNext(); ) {
                me = (Map.Entry)it.next();

                sb.append(tabPrefix);
                sb.append(((R01MStructureItem)me.getValue()).debugInfo(tabPrefix));
                sb.append(SALTO_LINEA);
            }
        }

        return sb;
    }

}
