/*
 * Created on 20-may-2005
 *
 * @author IE00165H
 * (c) 2005 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.objects.cataloguing.structures;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import com.ejie.r01f.guids.GUIDDispenserException;
import com.ejie.r01f.util.StringUtils;
import com.ejie.r01m.exceptions.R01MCatalogException;
import com.ejie.r01m.utils.R01MConstants;
import com.ejie.r01m.utils.R01MUtils;

/**
 * Representa una etiqueta de catalogacin.<br>
 * Se entiende por etiqueta de catalogacin a un elemento que califica un contenido dotndole de un significado.
 */
public class R01MLabel implements Serializable {
    private static final long serialVersionUID = 3064906250948960952L;
///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTANTES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Literal de etiqueta sin trmino.
     */
    public static final String WITHOUT_TERM = "**Sin t&eacute;rmino**";
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Identificador numrico de la etiqueta en Base de Datos.
     */
    private int labelId;
    /**
     * Identificador de la etiqueta.
     */
    private String oid;
     /**
      * Identificador de la bolsa a la que pertenece la etiqueta
      */
     private int bagId;
     /**
      * Descripcin de la etiqueta (utilizacion, criterios, etc).
      */
     private String description;
     /**
      * Sinnimos de la etiqueta indexados por el idioma.
      * Es un mapa por idiomas de "Arrays de trminos ({@link com.ejie.r01m.objects.cataloguing.structures.R01MTerm})".
      */
     private Map languageSynonyms = new HashMap();
     /**
      * Trminos normalizados indexados por idioma.
      * Es un mapa por idiomas de objetos trmino {@link com.ejie.r01m.objects.cataloguing.structures.R01MTerm}.
      */
     private Map standardizedTerms = new HashMap();
     /**
      * Identificador externo.
      */
     private String extOID;

///////////////////////////////////////////////////////////////////////////////////////////
//   CONSTRUCTORES
////////////////////////////////////////////////////////////////////////////////////////// /
     /**
      * Constructor vacio
      */
     public R01MLabel() {
        super();
        // Generar automaticamente un oid para la etiqueta
        try {
            this.oid = R01MUtils.generateNewOid(R01MConstants.STRUCTURES_APPCODE);
        } catch (GUIDDispenserException guidDispEx) {
            guidDispEx.printStackTrace(System.out);
            this.oid = null;
        }
     }
///////////////////////////////////////////////////////////////////////////////////////////
//   GETTERS & SETTERS
////////////////////////////////////////////////////////////////////////////////////////// /
     /**
      * Identificador numrico de la etiqueta.
      * @return Returns the labelId.
      */
     public int getLabelId() {
         return this.labelId;
     }
     /**
      * Establece el identificador numrico de la etiqueta.
      * @param theLabelId The labelId to set.
      */
     public void setLabelId(int theLabelId) {
         this.labelId = theLabelId;
     }
     /**
      * Identificador numrico de la bolsa.
      * @return el identificador numrico de la bolsa.
      */
     public int getBagId() {
         return this.bagId;
     }
     /**
      * Establece el identificador numrico de la bolsa.
      * @param theBagId el identificador numrico de la bolsa.
      */
     public void setBagId(int theBagId) {
         this.bagId = theBagId;
     }
     /**
      * Identificador de la etiqueta.
      * @return Returns the oid.
      */
     public String getOid() {
         return this.oid;
     }
     /**
      * Establece el identificador de la etiqueta.
      * @param theOid The oid to set.
      */
     public void setOid(String theOid) {
         this.oid = theOid;
     }
    /**
     * Descripcin de la etiqueta.
     * @return Returns the description.
     */
    public String getDescription() {
        return this.description;
    }
    /**
     * Establece la descripcin de la etiqueta
     * @param theDescription The description to set.
     */
    public void setDescription(String theDescription) {
        this.description = theDescription;
    }
    /**
     * Sinnimos de la etiqueta indexados por el idioma.
     * @return Returns the synonyms.
     */
    public Map getLanguageSynonyms() {
        return this.languageSynonyms;
    }
    /**
     * Sinnimos de la etiqueta para un idioma.
     * @param language el idioma.
     * @return los sinnimos para ese idioma.
     */
	public Collection getLanguageSynonyms(String language)
	{
		return ((Collection)this.languageSynonyms.get(language));
	}
    /**
     * Establece los sinnimos de la etiqueta indexados por el idioma.
     * @param theSynonyms The synonyms to set.
     */
    public void setLanguageSynonyms(Map theSynonyms) {
        this.languageSynonyms = theSynonyms;
    }
	/**
	 * Identificador externo.
	 * @return Returns the extOID.
	 */
	public String getExtOID() {
		return this.extOID;
	}
	/**
	 * Establece el identificador externo.
	 * @param theExtOID The extOID to set.
	 */
	public void setExtOID(String theExtOID) {
		this.extOID = theExtOID;
	}
	/**
	 * Trminos normalizados indexados por idioma.
	 * @return Returns the standardizedTerms.
	 */
	public Map getStandardizedTerms() {
		return this.standardizedTerms;
	}
	/**
     * Obtiene el trmino estandar de una etiqueta en un idioma.
	 * @param language el idioma.
	 * @return el objeto {@link com.ejie.r01m.objects.cataloguing.structures.R01MTerm}.
	 * @throws R01MCatalogException si no existe el trmino estandar en el idioma.
	 */
	public R01MTerm getStandardizedTerm(String language)throws R01MCatalogException {
		try {
			return (R01MTerm)this.standardizedTerms.get(language);
		} catch(NullPointerException nullEx) {
			throw new R01MCatalogException("No hay trmino en ese idioma");
		}
	}
    /**
     * Obtiene el valor (descripcin) del trmino estandar, una etiqueta en un idioma.
     * @param language el idioma.
     * @return una cadena con el trmino.
     */
    public String getStandarizedTermValue(String language){
        String value = null;
        if (this.standardizedTerms!=null && !this.standardizedTerms.isEmpty() && this.standardizedTerms.containsKey(language)) {
            R01MTerm mterm;
            try {
                mterm = this.getStandardizedTerm(language);
                value = mterm.getValue();
            } catch (R01MCatalogException ex) {
                value = WITHOUT_TERM;
            }
            if (value == null) {
                value = WITHOUT_TERM;
            }
        } else {
            value = WITHOUT_TERM;
        }
        return value;
    }
	/**
	 * Establece los trminos normalizados indexados por idioma.
	 * @param theStandardizedTerms The standardizedTerms to set.
	 */
	public void setStandardizedTerms(Map theStandardizedTerms) {
		this.standardizedTerms = theStandardizedTerms;
	}

	/**
	 * Metodo que aade un termino al mapa de standars de una etiqueta
	 * @param term objeto R01MTerm termino que se aade
	 */
	public void addStandardizedTerm(R01MTerm term){
		String language= term.getLanguage();
		//Se comprueba si hay ya un Stardard y si es asi se trasforma en sinonimo
		if(this.standardizedTerms.containsKey(language))
		{
		    term.setType(R01MTerm.TERM_TYPE_SYNONYMOUS);
		    this.addLanguageSynonymous(term);
		}else {
		    this.standardizedTerms.put(language,term);
		}
	}

	/**
	 * Metodo que aade el termino al mapa de sinonimos de una etiqueta
	 * @param term
	 */
	public void addLanguageSynonymous(R01MTerm term){
			String language= term.getLanguage();
			Collection col = null;
			if (this.languageSynonyms.containsKey(language)){
				col=(Collection)this.languageSynonyms.get(language);
			} else {
				col = new ArrayList();
			}
			col.add(term);
			this.languageSynonyms.put(language,col);
	}

    /**
     * Metodo que escribe en un StringBuffer si no existe la etiqueta en la bolsa
     */
    public  StringBuffer composeDebugInfo() {
        StringBuffer sb = new StringBuffer(29);
        sb.append("Oid=");sb.append(StringUtils.isEmptyString(this.getOid())?"":this.getOid());sb.append(" no exists in the bag");sb.append("\r\n");

        return sb;
    }

}
