package com.ejie.r01m.objects.cataloguing.structures;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.ejie.r01f.guids.GUIDDispenserException;
import com.ejie.r01m.utils.R01MConstants;
import com.ejie.r01m.utils.R01MUtils;

/**
 * Objeto de definicin de un grupo de bolsas de catalogacin.<br>
 * Un grupo de bolsas de catalogacin es una agrupacin de bolsas de catalogacin.
 */
public class R01MBagGroup implements Serializable {
    private static final long serialVersionUID = -3008144987381636462L;

// /////////////////////////////////////////////////////////////////////////////////////////
// CONSTANTES
// /////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Identificador por defecto cuando no se establece un valor al id numrico.
     */
    public static final int EMPTY_ID = -1;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Identificador del grupo de bolsa de catalogacin (-1 indica que no se ha dado valor)
     */
    private int id = EMPTY_ID;
    /**
     * Identificador del grupo de bolsa de catalogacin para la obtencin de los nombres en los distintos idiomas
     */
    private String guid;
    /**
     * Mapa de Descripciones del grupo de la bolsa.
     */
    private Map descriptions;
    /**
     * Mapa de nombres del grupo de la bolsa por idiomas.
     */
    private Map names;

    /**
     * Lista de las bolsas pertenecientes al grupo de bolsa.
     */
    private List belongBags;

///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTOR
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Constructor vacio
     */
	public R01MBagGroup() {
		super();
		// Generar automaticamente un oid para la bolsa
        try {
            this.guid = R01MUtils.generateNewOid(R01MConstants.STRUCTURES_APPCODE);
        } catch (GUIDDispenserException guidDispEx) {
            guidDispEx.printStackTrace(System.out);
            this.guid = null;
        }
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS GET/SET
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Nombres del grupo de bolsas
     * @return Returns the names.
     */
    public Map getNames() {
        return this.names;
    }
    /**
     * Devuelve el nombre de un grupo de bolsas de catalogacin
     * @param lang el lenguaje
     * @return nombre de un grupo de bolsas de catalogacin
     */
    public String getName(String lang) {
        return this.names != null ? (String)this.names.get(lang) : null;
    }
    /**
     * Establece el nombre del grupo de bolsas
     * @param theNames The names to set.
     */
    public void setNames(Map theNames) {
        this.names = theNames;
    }
    /**
     * Establece el nombre del grupo de bolsas a traves del idioma
     * @param lang idioma (es, eu..).
     * @param theName The names to set.
     */
    public void setName(String lang,String theName) {
        if (this.names == null) {
            this.names = new HashMap();
        }
        this.names.put(lang,theName);
    }
    /**
     * Bolsas pertenecientes al grupo
     * @return Returns the belongBags.
     */
    public List getBelongBags() {
        return this.belongBags;
    }

    /**
     * Establece las bolsas pertenecientes al grupo de bolsa
     * @param theBelongBags The bagGroupNames to set.
     */
    public void setBelongBags(List theBelongBags) {
        this.belongBags = theBelongBags;
    }

    /**
     * Establece una bolsa perteneciente al grupo de bolsa.
     * @param theBagId The BagId to set.
     * @return objeto R01MBag
     */
    public R01MBag getBelongBag(String theBagId) {

        if (this.belongBags == null) {
            this.belongBags = new ArrayList();
        }
        Iterator bags = this.belongBags.iterator();
        R01MBag bagResult = null;

        while (bags.hasNext()){
          R01MBag bag = (R01MBag) bags.next();
          if (bag.getBagId() ==  Integer.parseInt(theBagId)) {
              bagResult = bag;
              break;
          }
        }
        return bagResult;
    }
	/**
	 * Descripcin del grupo de bolsas
	 * @param lang el lenguaje
	 * @return Returns the Description.
	 */
	public String getDescription(String lang) {
	    return this.descriptions != null ? (String)this.descriptions.get(lang) : null;
	}

    /**
     * Establece la descripcin del grupo de bolsas para el idioma especificado
     * @param lang el lenguaje
     * @param theDescription The theDescription to set.
     */
    public void setDescription(String lang,String theDescription) {
        if (this.descriptions == null) {
            this.descriptions = new HashMap();
        }
        this.descriptions.put(lang,theDescription);
    }
    /**
     * Establece las descripciones del grupo de bolsas
     * @param theDescriptions The descriptions to set.
     */
    public void setDescriptions(Map theDescriptions) {
        this.descriptions = theDescriptions;
    }
    /**
     * Descripciones del grupo de bolsas
     * @return Returns the descriptions.
     */
    public Map getDescriptions() {
        return this.descriptions;
    }
	/**
	 * Id del grupo de bolsas
	 * @return Returns the groupOid.
	 */
	public int getId() {
		return this.id;
	}
	/**
	 * Establece el Id del grupo de bolsa
	 * @param theId The groupOid to set.
	 */
	public void setId(int theId) {
		this.id = theId;
	}
	/**
     * Id del grupo de bolsa para la obtencin de los nombres en los distintos idiomas
     * @return Returns the groupOid.
     */
    public String getGuid() {
        return this.guid;
    }
    /**
     * Establece el id del grupo de bolsa para la obtencin de los nombres en los distintos idiomas
     * @param theGUID The bagGroupGUID to set.
     */
    public void setGuid(String theGUID) {
        this.guid = theGUID;
    }

}
