/*
 * Created on Jul 21, 2005
 *
 * @author ie00165h - Alex Lara
 * (c) 2005 EJIE: Eusko Jaurlaritzako Informatika Elkartea
 */
package com.ejie.r01m.objects.cataloguing;

import java.util.StringTokenizer;

import com.ejie.r01m.objects.R01MPrintableInterface;
import com.ejie.r01m.objects.contentmodel.R01MBaseContentModelObject;

/**
 * <pre>
 * Catalogacin en una estructura (eje).
 * Toda catalogacin en un eje se compone de:
 *      - Eje en el que se ha realizado la catalogacin.
 *      - Etiqueta de catalogacin.
 *      - Rol que ocupa la etiqueta de catalogacin en el eje.
 *          Una etiqueta en un eje SIEMPRE desempea un rol ya que la etiqueta
 *          puede pertenecer a ms de un eje, e incluso aparecer ms de una vez
 *          en el mismo eje.
 *          Para "distinguir" la etiqueta en el eje, se asocia un "rol" a la
 *          etiqueta en el eje.
 *              Por ejemplo, la "viceconsejeria de turismo" ocupa el rol de
 *              "viceconsejera" en el eje de estructura de gobierno.
 *
 * La estructura de clases puede verse en el siguiente diagrama de clases:
 * <a href="#" target="_blank" onClick="window.open('../../../../../doc-files/r01-Diagramas de Clases/R01M - Contenidos.ucls.pdf', this.target, 'width= 750 ,height= 550 ,resizable=yes'); return false;">Diagrama de clases de contenidos.</a>
 * </pre>
 */
public class R01MStructureCatalog extends R01MBaseContentModelObject implements R01MPrintableInterface, 
																				Comparable {
    private static final long serialVersionUID = -2328414711758952555L;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Identificador de la estructura en la que se ha catalogado.
     */
    public String structureOid;
    /**
     * Identificador de la etiqueta que se ha asociado.
     */
    public String oid;
    /**
     * Rol de la etiqueta en el eje.
     */
    public String role;
    

    ///////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTORES
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Constructor vaco
     */
    public R01MStructureCatalog() {
        super();
    }
    /**
     * Constructor a partir de la estructura, la etiqueta y el rol de la etiqueta en la estructura
     * @param newStructureOid Identificador de la estructura
     * @param newLabelOid Identificador de la etiqueta
     * @param newRole Rol de la etiqueta en la estructura    
     */
    public R01MStructureCatalog(String newStructureOid,String newLabelOid,String newRole) {
        this.structureOid = newStructureOid;
        this.oid = newLabelOid;
        this.role = newRole;
    
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  METODOS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Obtiene un cdigo nemotcnico concatenando cada uno de los objetos separados por ":".
     * @return estructura,rol y etiqueta.
     */
    public String composeMemoCode() {
        return this.structureOid + ";" + this.role + ";" + this.oid;
    }
    /**
     * Carga la localizacin del cdigo nemotcnico.
     * @param memoCode el cdigo memotcnico
     */
    public void loadFromMemoCode(String memoCode) {
        if (memoCode == null) return;
        StringTokenizer st = new StringTokenizer(memoCode,";");
        if (st.countTokens() != 3) return;
        int i=0;
        while (st.hasMoreTokens()) {
            switch(i) {
                case 0: this.structureOid = st.nextToken(); break;
                case 1: this.role = st.nextToken(); break;
                case 2: this.oid = st.nextToken(); break;
                default:
            }
            i++;
        }
    }
///////////////////////////////////////////////////////////////////////////////////////////
//  GET & SET
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Metodo para ser utilizado como ndice en el comparador JVTComparator
     * @return clave del objeto
     */
    public String getKey() {
        return this.getOid() + ":" + this.getStructureOid();
    }
    /**
     * @param theKey
     */
    public void setKey(String theKey) {
        /* No se hace nada */
    }
    /**
     * @return Returns the structureOid.
     */
    public String getStructureOid() {
        return structureOid;
    }
    /**
     * @param theStructureOid The structureOid to set.
     */
    public void setStructureOid(String theStructureOid) {
        this.structureOid = theStructureOid;
    }
    /**
     * @return Returns the labelOid.
     */
    public String getOid() {
        return oid;
    }
    /**
     * @param theOid The oid to set.
     */
    public void setOid(String theOid) {
        this.oid = theOid;
    }
    /**
     * @return Returns the role.
     */
    public String getRole() {
        return role;
    }
    /**
     * @param theRole The role to set.
     */
    public void setRole(String theRole) {
        this.role = theRole;
    }
// /////////////////////////////////////////////////////////////////////////////////////////
// Interfaz java.lang.Comparable
// /////////////////////////////////////////////////////////////////////////////////////////
        /**
         * Compara un objeto de catalogacin con otro
         * @param catalog objeto de catalogacin con el que comparar
         * @return 0 si son iguales, -1 si son distintos
         */
        public int compareTo(Object catalog) {
            R01MStructureCatalog otherCatalog = (R01MStructureCatalog) catalog;
            if(this.getStructureOid().equals(otherCatalog.getStructureOid())
                    && this.getOid().equals(otherCatalog.getOid())
                    && this.getRole().equals(otherCatalog.getRole())) {
                return 0;
            }
            return -1;
        }
////////////////////////////////////////////////////////////////////////////////////
//    DEBUG
//////////////////////////////////////////////////////////////////////////////////   //
        /**
         * Devuelve una representacion en formato cadena del objeto
         * @param prefix prefijo a ser concatenado a todas las lneas
         * @return cadena
         */
        public StringBuffer debugInfo(String prefix) {
            StringBuffer sb = new StringBuffer(76);
            sb.append("\r\n");
            sb.append(prefix);sb.append("======== Label ========\r\n");
            sb.append(prefix);sb.append("\tStructureOid:");sb.append(this.getStructureOid() == null?"null":this.getStructureOid());sb.append("\r\n");
            sb.append(prefix);sb.append("\tLabelOid:");sb.append(this.getOid() == null?"null":this.getOid());sb.append("\r\n");
            sb.append(prefix);sb.append("\tRole:");sb.append(this.getRole() == null?"null":this.getRole());sb.append("\r\n");

            return sb;
        }
}
