package r01mo.model.url.alias;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.Accessors;
import r01f.aspects.dirtytrack.ConvertToDirtyStateTrackable;
import r01f.debug.Debuggable;
import r01f.util.types.Strings;
import r01mo.model.oids.R01MURLAliasOID;
import r01mo.model.url.R01MUrl;
import r01mo.model.url.R01MUrlModelObject;
import r01mo.paths.R01MAliasPath;

/**
 * Alias de un recurso: permite asociar URLs alternativas ms usables / semnticas a recursos del gestor de contenidos/portales
 * Se definen para los elementos:
 * <pre>
 *  - Portales
 *    ---------------------------------------------------------------------------------------------
 *    Se puede definir un alias por dominio de manera que se pueda definir, por ejemplo:
 *    
 *       PORTAL s11
 *       	www.paysbasquetourisme.com/bilbao ==> www.bilbao.net
 *       	www.paisvascoturismo.com/bilbao   ==> /s11-12375/es/contenidos/d_destinos_turisticos/0000004981_d2_rec_turismo/es_4981/4981-ficha.html
 *      	www.euskaditurismoa.net/bilbao    ==> /s11-12375/eu/contenidos/d_destinos_turisticos/0000004981_d2_rec_turismo/eu_4981/4981-ficha.html
 *       	www.euskaditurismoa.net/fiestas   ==> /s11-15521/eu/
 *     
 *     Se puede redireccionar a URLs distintas segn el idioma en el que se est navegando (Cookie).
 *     www.euskadi.net/estadisticasSalud ==> [es] /r33-cont/es/contenidos/estadistica/es_def/ficha.html
 *                                           [eu] /r33-cont/eu/contenidos/estadistica/eu_def/ficha.html
 *                                           [en] www.osakidetza.net/datosEstadisticosWar/en
 *  - Pginas de portal
 *    ---------------------------------------------------------------------------------------------
 *    El alias se usa dentro del concepto de URL amigable. Se genera automticamente aunque puede ser modificada por el usuario.
 *    Se puede definir un alias por idioma (recogido de la cookie). Por ejemplo:
 *       
 *       PAGINA s11-jaiak
 *       	Idioma:............................(es)....................(eu)
 *       	Valor Alias (configurable)         fiestas                 jaiak
 *       	URL destino......................../s11-jaiak/es/........../s11-jaiak/eu/
 *        
 *  - Versiones Idiomticas
 *    ---------------------------------------------------------------------------------------------
 *    El alias se usa dentro del concepto de URL amigable. Se genera automticamente aunque puede ser modificada por el usuario.
 *    
 *       Versin idiomtica: /0000004981_d2_rec_turismo/es_4981
 *       	Valor Alias (configurable)         conoce_bilbao
 *       	URL destino......................../contenidos/d_destinos_turisticos/0000004981_d2_rec_turismo/es_4981/4981-ficha.html
 *       
 *       Versin idiomtica: /0000004981_d2_rec_turismo/eu_4981
 *       	Valor Alias (configurable)         ezagutu_bilbo
 *       	URL destino......................../contenidos/d_destinos_turisticos/0000004981_d2_rec_turismo/eu_4981/4981-ficha.html
 * </pre>
 */
@ConvertToDirtyStateTrackable
@Accessors(prefix="_")
@NoArgsConstructor @AllArgsConstructor
public abstract class R01MUrlAliasBase implements R01MUrlModelObject,
												  Debuggable {
    private static final long serialVersionUID = -1620552707887191546L;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * oid del alias
     */
    @Getter @Setter private R01MURLAliasOID _oid;
    /**
      * Alias que se usa (valor configurable por el usuario)
      */
    @Getter @Setter private R01MAliasPath _aliasPath;
    /**
     * Url a la que se redirecciona
     */
    @Getter @Setter private R01MUrl _redirectURL;
     
////////////////////////////////////////////////////////////////////////////////////
//  DEBUG
////////////////////////////////////////////////////////////////////////////////////
    @Override
    public StringBuffer debugInfo() {
        return Strings.of("URL Alias:\r\n\t-oid:{}\r\n-alias:{}\r\n\t-redirectURL:{}")
        			  .customizeWith(_oid,_aliasPath,_redirectURL)
        			  .asStringBuffer();
    }
}
