package r01mo.model.search.query;

import java.util.Arrays;
import java.util.Set;

import lombok.AccessLevel;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import r01f.util.types.collections.CollectionUtils;
import r01mo.model.R01MStoreLocation;
import r01mo.model.R01MTypology;
import r01mo.model.editionstatus.R01MApprovalStatus;
import r01mo.model.editionstatus.R01MUserInterfaceStatus;
import r01mo.model.oids.publish.R01MPublishDestinationRepositoryOID;
import r01mo.model.oids.storage.R01MStorageAreaOID;
import r01mo.model.oids.storage.R01MStorageServerOID;
import r01mo.model.oids.storage.R01MStorageStoreOID;
import r01mo.model.oids.storage.R01MStorageWorkAreaOID;
import r01mo.model.oids.structures.R01MStructureLabelOID;
import r01mo.model.oids.typo.R01MTypoClusterOID;
import r01mo.model.oids.typo.R01MTypoFamilyOID;
import r01mo.model.oids.typo.R01MTypoTypeOID;
import r01mo.model.search.query.metadata.R01MSearchQueryMetaDataBase;
import r01mo.model.search.query.metadata.R01MSearchQueryMetaDataCondition;
import r01mo.model.search.query.metadata.R01MSearchQueryModelObject;
import r01mo.model.search.query.metadata.R01MSearchQueryOrderByMetaData;

import com.google.common.collect.Sets;


/**
 * Modela una query que se lanza contra el buscador de euskadi.net
 * <code>
 * R01MSearchQuery qry = R01MSearchQuery.create()
 * 			   						    .mustHaveStructureLabel("myLabel1","myLabel2")
 * 			   						    .typedInAnyOfTheeseFamilies("ayudas")
 * 			   						    .mustMeetTheeseMetaDataConditions(R01MSearchQueryDateMetaData.forMetaData("myDateMetaDate")
 * 					   																			     .usingCondition(R01MSearchQueryDateMetaDataCondition.AFTER)
 * 					   																			     .with(new Date()),
 * 					   												      R01MSearchQueryNumberMetaData.forMetaData("myNumberMetaData")
 * 					   							 							  						   .usingCondition(R01MSearchQueryNumberMetaDataCondition.BETWEEN)
 * 					   							 							  						   .with(2,3))
 * 					   				    .publishedItemsOnly();
 * </code>
 */
@Accessors(prefix="_")
@NoArgsConstructor(access=AccessLevel.PRIVATE)
@Slf4j
public class R01MSearchQuery implements R01MSearchQueryModelObject {
	private static final long serialVersionUID = -724318808843468426L;
///////////////////////////////////////////////////////////////////////////////////////////
//  MIEMBROS: GENERAL
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Nombre de la query.
     */
    @Getter @Setter(AccessLevel.PRIVATE) private String _name;
    /**
     * Descripcin de la Query.<br>
     * <b>(Tamao mximo de 255 caracteres).</b>
     */
    @Getter @Setter(AccessLevel.PRIVATE) private String _description;
/////////////////////////////////////////////////////////////////////////////////////////
//  ESTADO DE APROBACIN
/////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * Estado de aprobacin
     */
    @Getter @Setter(AccessLevel.PRIVATE) private R01MApprovalStatus _editionApprovalStatus;
    /**
     * Estado del documento en la interfaz del usuario (Sin publicar, Publicado, Publicado Modificado).
     */
    @Getter @Setter(AccessLevel.PRIVATE) private R01MUserInterfaceStatus _userInterfaceStatus;
/////////////////////////////////////////////////////////////////////////////////////////
//  PUBLICACIN
/////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * Lista de repositorios de publicacion
     */
    @Getter @Setter(AccessLevel.PRIVATE) private Set<R01MPublishDestinationRepositoryOID> _publishRepositories;
    /**
     * Indica si hay que buscar elementos NO publicados, por defecto nicamente elementos publicados!!!.
     */
    @Getter @Setter(AccessLevel.PRIVATE) private boolean _publishedItemsOnly = true;    
/////////////////////////////////////////////////////////////////////////////////////////
//  ALMACENAMIENTO
/////////////////////////////////////////////////////////////////////////////////////////    
    /**
     * Lista de servidores en los que se busca
     */
    @Getter @Setter(AccessLevel.PRIVATE) private Set<R01MStorageServerOID> _storageServers;
    /**
     * Lista de repositorios de datos en los que se busca
     */
    @Getter @Setter(AccessLevel.PRIVATE) private Set<R01MStorageStoreOID> _storageStores;
    /**
     * Lista de areas en las que se busca (objetos {@link R01MSearchedArea}).
     */
    @Getter @Setter(AccessLevel.PRIVATE) private Set<R01MStorageAreaOID> _storageAreas;
    /**
     * Lista de workAreas en las que se busca (objetos {@link R01MSearchedWorkArea}).
     */
    @Getter @Setter(AccessLevel.PRIVATE) private Set<R01MStorageWorkAreaOID> _storageWorkAreas;
///////////////////////////////////////////////////////////////////////////////////////////
//  TIPOLOGIA
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Clusters en los que se busca
     */
    @Getter @Setter(AccessLevel.PRIVATE) private Set<R01MTypoClusterOID> _typoClusters;
    /**
     * Lista de familias en las que se busca 
     */
    @Getter @Setter(AccessLevel.PRIVATE) private Set<R01MTypoFamilyOID> _typoFamilies;
    /**
     * Lista de tipos de contenido en los que se busca
     */
    @Getter @Setter(AccessLevel.PRIVATE) private Set<R01MTypoTypeOID> _typoTypes;
/////////////////////////////////////////////////////////////////////////////////////////
//  CATALOGACIN EN ETIQUETAS
/////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Lista de condiciones AND sobre las etiquetas de catalogacion 
     */
    @Getter @Setter(AccessLevel.PRIVATE) private Set<R01MStructureLabelOID> _structureLabelsAND;
    /**
     * Lista de condiciones OR sobre las etiquetas de catalogacion 
     */
    @Getter @Setter(AccessLevel.PRIVATE) public Set<R01MStructureLabelOID> _structureLabelsOR;
/////////////////////////////////////////////////////////////////////////////////////////
//  METADATA
/////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Lista de metaDatos AND y las condiciones sobre cada uno de ellos
     */
    @Getter @Setter(AccessLevel.PRIVATE) private Set<R01MSearchQueryMetaDataBase<? extends R01MSearchQueryMetaDataCondition<?>,?>> _metaDataAND;
    /**
     * Lista de metaDatos OR y las condiciones sobre cada uno de ellos.
     */
    @Getter @Setter(AccessLevel.PRIVATE) private Set<R01MSearchQueryMetaDataBase<? extends R01MSearchQueryMetaDataCondition<?>,?>> _metaDataOR;
///////////////////////////////////////////////////////////////////////////////////////////
//  ORDENACION
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Lista de metadatos sobre los que se ordena 
     */
    @Getter @Setter(AccessLevel.PRIVATE) public Set<R01MSearchQueryOrderByMetaData> _orderByMetadata;

    
/////////////////////////////////////////////////////////////////////////////////////////
//  CREACIN DE OBJETOS: FLUENT API
/////////////////////////////////////////////////////////////////////////////////////////
    public static R01MSearchQuery create() {
    	return new R01MSearchQuery();
    }
    public static R01MSearchQuery create(final String name,final String description) {
    	R01MSearchQuery outQuery = new R01MSearchQuery();
    	outQuery.setName(name);
    	outQuery.setDescription(description);
    	return outQuery;
    }
/////////////////////////////////////////////////////////////////////////////////////////
//  FLUENT API: ESTADO DE EDICIN
/////////////////////////////////////////////////////////////////////////////////////////
    public R01MSearchQuery inEditionStatus(final R01MApprovalStatus status) {
    	this.setEditionApprovalStatus(status);
    	return this;
    }
    public R01MSearchQuery inUserInterfaceStatus(final R01MUserInterfaceStatus status) {
    	this.setUserInterfaceStatus(status);
    	return this;
    }
/////////////////////////////////////////////////////////////////////////////////////////
//  FLUENT API: PUBLICACIN
/////////////////////////////////////////////////////////////////////////////////////////
    public R01MSearchQuery publishedIn(final R01MPublishDestinationRepositoryOID... publishRepositories) {
    	if (publishRepositories == null) return this;
    	_publishedItemsOnly = true;		// Cuando se especifican repositorios de publicacin, se fija a buscar SOLO entre elementos publicados
    	if (_publishRepositories == null) _publishRepositories = Sets.newHashSet();
    	_publishRepositories.addAll(Arrays.asList(publishRepositories));
    	return this;
    }
    public R01MSearchQuery publishedItemsOnly() {
    	_publishedItemsOnly = true;
    	return this;
    }
/////////////////////////////////////////////////////////////////////////////////////////
//  FLUENT API: STORAGE
/////////////////////////////////////////////////////////////////////////////////////////
    public R01MSearchQuery storedIn(final R01MStorageServerOID serverOid,
    								final R01MStorageStoreOID storeOid,
    								final R01MStorageAreaOID areaOid,
    								final R01MStorageWorkAreaOID workAreaOid) {
    	if (!CollectionUtils.isNullOrEmpty(_storageServers)) _storageServers.clear();
    	if (!CollectionUtils.isNullOrEmpty(_storageStores)) _storageStores.clear();
    	if (!CollectionUtils.isNullOrEmpty(_storageAreas)) _storageAreas.clear();
    	if (!CollectionUtils.isNullOrEmpty(_storageWorkAreas)) _storageWorkAreas.clear();
    	
    	this.storedInAnyOfTheeseServers(serverOid);
    	this.storedInAnyOfTheeseStores(storeOid);
    	this.storedInAnyOfTheeseAreas(areaOid);
    	this.storedInAnyOfTheeseWorkAreas(workAreaOid);
    	
    	return this;
    }
    public R01MSearchQuery storedIn(final R01MStoreLocation location) {
    	return this.storedIn(location.getServerOid(),
    					     location.getStoreOid(),
    					     location.getAreaOid(),
    					     location.getWorkAreaOid()); 
    }
    public R01MSearchQuery storedInAnyOfTheeseServers(final R01MStorageServerOID... serversOids) {
    	if (serversOids == null) return this;
    	if (_storageServers == null) _storageServers = Sets.newHashSet();
    	_storageServers.addAll(Arrays.asList(serversOids));
    	return this;
    }
    public R01MSearchQuery storedInAnyOfTheeseServers(final String... serversOids) {
    	if (serversOids == null) return this;
    	R01MStorageServerOID[] oids = new R01MStorageServerOID[serversOids.length];
    	for (int i=0; i<serversOids.length; i++) oids[i] = R01MStorageServerOID.forId(serversOids[i]); 
    	return this.storedInAnyOfTheeseServers(oids);
    }
    public R01MSearchQuery storedInAnyOfTheeseStores(final R01MStorageStoreOID... storesOids) {
    	if (storesOids == null) return this;
    	if ((_storageServers != null && _storageServers.size() > 1)) {
    		log.warn("StorageStores cannot be specified in the query if there are also more than one server");
    		return this;
    	}
    	if (_storageStores == null) _storageStores = Sets.newHashSet();
    	_storageStores.addAll(Arrays.asList(storesOids));
    	return this;
    }
    public R01MSearchQuery storedInAnyOfTheeseStores(final String... storesOids) {
    	if (storesOids == null) return this;
    	R01MStorageStoreOID[] oids = new R01MStorageStoreOID[storesOids.length];
    	for (int i=0; i<storesOids.length; i++) oids[i] = R01MStorageStoreOID.forId(storesOids[i]); 
    	return this.storedInAnyOfTheeseStores(oids);
    }
    public R01MSearchQuery storedInAnyOfTheeseAreas(final R01MStorageAreaOID... areasOids) {
    	if (areasOids == null) return this;
    	if ((_storageServers != null && _storageServers.size() > 1) || (_storageStores != null && _storageStores.size() > 1)) {
    		log.warn("StorageAreas cannot be specified in the query if there are also more than one server or store");
    		return this;
    	}
    	if (_storageAreas == null) _storageAreas = Sets.newHashSet();
    	_storageAreas.addAll(Arrays.asList(areasOids));
    	return this;
    }
    public R01MSearchQuery storedInAnyOfTheeseAreas(final String... areasOids) {
    	if (areasOids == null) return this;
    	R01MStorageAreaOID[] oids = new R01MStorageAreaOID[areasOids.length];
    	for (int i=0; i<areasOids.length; i++) oids[i] = R01MStorageAreaOID.forId(areasOids[i]); 
    	return this.storedInAnyOfTheeseAreas(oids);
    }
    public R01MSearchQuery storedInAnyOfTheeseWorkAreas(final R01MStorageWorkAreaOID... workAreasOids) {
    	if (workAreasOids == null) return this;
    	if ((_storageServers != null && _storageServers.size() > 1) || (_storageStores != null && _storageStores.size() > 1) || (_storageAreas != null && _storageAreas.size() > 1)) {
    		log.warn("StorageWorkAreas cannot be specified in the query if there are also more than one server, store or area");
    		return this;
    	}
    	if (_storageWorkAreas == null) _storageWorkAreas = Sets.newHashSet();
    	_storageWorkAreas.addAll(Arrays.asList(workAreasOids));
    	return this;
    }
    public R01MSearchQuery storedInAnyOfTheeseWorkAreas(final String... workAreasOids) {
    	if (workAreasOids == null) return this;
    	R01MStorageWorkAreaOID[] oids = new R01MStorageWorkAreaOID[workAreasOids.length];
    	for (int i=0; i<workAreasOids.length; i++) oids[i] = R01MStorageWorkAreaOID.forId(workAreasOids[i]); 
    	return this.storedInAnyOfTheeseWorkAreas(oids);
    }
/////////////////////////////////////////////////////////////////////////////////////////
//  FLUENT API TIPOLOGA
/////////////////////////////////////////////////////////////////////////////////////////
    public R01MSearchQuery typedIn(final R01MTypoClusterOID clusterOid,
    							   final R01MTypoFamilyOID familyOid,
    							   final R01MTypoTypeOID typeOid) {
    	if (!CollectionUtils.isNullOrEmpty(_typoClusters)) _typoClusters.clear();
    	if (!CollectionUtils.isNullOrEmpty(_typoFamilies)) _typoFamilies.clear();
    	if (!CollectionUtils.isNullOrEmpty(_typoTypes)) _typoTypes.clear();
    	
    	this.typedInAnyOfTheeseClusters(clusterOid);
    	this.typedInAnyOfTheeseFamilies(familyOid);
    	this.typedInAnyOfTheeseTypes(typeOid);
    	
    	return this;
    }
    public R01MSearchQuery typedIn(final R01MTypology typo) {
    	return this.typedIn(typo.getClusterOid(),
    						typo.getFamilyOid(),
    						typo.getTypeOid());
    }
    public R01MSearchQuery typedInAny(final R01MTypology... typos) {
    	for (R01MTypology typo : typos) {
    		this.typedInAnyOfTheeseClusters(typo.getClusterOid());
    		this.typedInAnyOfTheeseFamilies(typo.getFamilyOid());
    		this.typedInAnyOfTheeseTypes(typo.getTypeOid());
    	}
    	return this;
    }
    public R01MSearchQuery typedInAnyOfTheeseClusters(final R01MTypoClusterOID... clusterOids) {
    	if (clusterOids == null) return this;
    	if (_typoClusters == null) _typoClusters = Sets.newHashSet();
    	_typoClusters.addAll(Arrays.asList(clusterOids));
    	return this;
    }
    public R01MSearchQuery typedInAnyOfTheeseClusters(final String... clusterOids) {
    	if (clusterOids == null) return this;
    	R01MTypoClusterOID[] oids = new R01MTypoClusterOID[clusterOids.length];
    	for (int i=0; i<clusterOids.length; i++) oids[i] = R01MTypoClusterOID.forId(clusterOids[i]); 
    	return this.typedInAnyOfTheeseClusters(oids);
    }
    public R01MSearchQuery typedInAnyOfTheeseFamilies(final R01MTypoFamilyOID... familiesOids) {
    	if (familiesOids == null) return this;
    	if (_typoFamilies == null) _typoFamilies = Sets.newHashSet();
    	_typoFamilies.addAll(Arrays.asList(familiesOids));
    	return this;
    }
    public R01MSearchQuery typedInAnyOfTheeseFamilies(final String... familiesOids) {
    	if (familiesOids == null) return this;
    	R01MTypoFamilyOID[] oids = new R01MTypoFamilyOID[familiesOids.length];
    	for (int i=0; i<familiesOids.length; i++) oids[i] = R01MTypoFamilyOID.forId(familiesOids[i]); 
    	return this.typedInAnyOfTheeseFamilies(oids);
    }
    public R01MSearchQuery typedInAnyOfTheeseTypes(final R01MTypoTypeOID... typesOids) {
    	if (typesOids == null) return this;
    	if (_typoTypes == null) _typoTypes = Sets.newHashSet();
    	_typoTypes.addAll(Arrays.asList(typesOids));
    	return this;
    }
    public R01MSearchQuery typedInAnyOfTheeseTypes(final String... typesOids) {
    	if (typesOids == null) return this;
    	R01MTypoTypeOID[] oids = new R01MTypoTypeOID[typesOids.length];
    	for (int i=0; i<typesOids.length; i++) oids[i] = R01MTypoTypeOID.forId(typesOids[i]); 
    	return this.typedInAnyOfTheeseTypes(oids);
    }
/////////////////////////////////////////////////////////////////////////////////////////
//  FLUENT API: CATALOGACIN
/////////////////////////////////////////////////////////////////////////////////////////
    public R01MSearchQuery mustHaveStructureLabel(final R01MStructureLabelOID... labelsOids) {
    	if (labelsOids == null) return this;
    	if (_structureLabelsAND == null) _structureLabelsAND = Sets.newHashSet();
    	_structureLabelsAND.addAll(Arrays.asList(labelsOids));
    	return this;
    }
    public R01MSearchQuery mustHaveStructureLabel(final String... labelsOids) {
    	if (labelsOids == null) return this;
    	R01MStructureLabelOID[] oids = new R01MStructureLabelOID[labelsOids.length];
    	for (int i=0; i<labelsOids.length; i++) oids[i] = R01MStructureLabelOID.forId(labelsOids[i]); 
    	return this.mustHaveStructureLabel(oids);
    }
    public R01MSearchQuery canHaveStructureLabel(final R01MStructureLabelOID... labelOids) {
    	if (labelOids == null) return this;
    	if (_structureLabelsOR == null) _structureLabelsOR = Sets.newHashSet();
    	_structureLabelsOR.addAll(Arrays.asList(labelOids));
    	return this;
    }
    public R01MSearchQuery canHaveStructureLabel(final String... labelsOids) {
    	if (labelsOids == null) return this;
    	R01MStructureLabelOID[] oids = new R01MStructureLabelOID[labelsOids.length];
    	for (int i=0; i<labelsOids.length; i++) oids[i] = R01MStructureLabelOID.forId(labelsOids[i]); 
    	return this.canHaveStructureLabel(oids);
    }
/////////////////////////////////////////////////////////////////////////////////////////
//  FLUENT API: METADATA
/////////////////////////////////////////////////////////////////////////////////////////
    public R01MSearchQuery mustMeetTheeseMetaDataConditions(final R01MSearchQueryMetaDataBase<? extends R01MSearchQueryMetaDataCondition<?>,?>... metaData) {
    	if (metaData == null) return this;
    	if (_metaDataAND == null) _metaDataAND = Sets.newHashSet();
    	_metaDataAND.addAll(Arrays.asList(metaData));
    	return this;
    }
    public R01MSearchQuery canMeetTheeseMetaDataConditions(final R01MSearchQueryMetaDataBase<? extends R01MSearchQueryMetaDataCondition<?>,?>... metaData) {
    	if (metaData == null) return this;
    	if (_metaDataOR == null) _metaDataAND = Sets.newHashSet();
    	_metaDataOR.addAll(Arrays.asList(metaData));
    	return this;
    }
///////////////////////////////////////////////////////////////////////////////////////////
//	FLUENT API: ORDENAR
///////////////////////////////////////////////////////////////////////////////////////////
    public R01MSearchQuery orderedBy(final R01MSearchQueryOrderByMetaData... orderByMetaData) throws IllegalArgumentException {
    	if (orderByMetaData == null) return this;
    	if (_orderByMetadata == null) _orderByMetadata = Sets.newHashSet();
    	_orderByMetadata.addAll(Arrays.asList(orderByMetaData));
    	return this;
    }
}
