package r01mo.model.editionstatus;

import r01f.enums.EnumExtended;
import r01f.enums.EnumExtendedWrapper;
import r01mo.model.publish.R01MPublishInfo;

/**
 * Estados de edicin / publicacin para la interfaz de usuario
 * Mezcla la informacin del estado de publicacin y edicin en un nico estado:
 * <pre>
 * 		- NO PUBLICADO: nunca se ha publidado (an est en redaccin o pendiente de aprobacin)
 * 		- PUBLICADO Y NO MODIFICADO: en algn momento se ha publicado y NO se ha modificado desde ese momento
 * 		- PUBLICADO Y MODIFICADO: en algn momento se ha publicado, pero POSTERIORMENTE se han hecho cambios que
 * 								  NO se han publicado
 * </pre>
 */
public enum R01MUserInterfaceStatus implements EnumExtended<R01MUserInterfaceStatus> {
	NOT_PUBLISHED,								// NO est publicado 
	PUBLISHED_AND_NOT_MODIFIED_AFTERWARDS,		// Publicado y NO modificado desde ese momento
	PUBLISH_AND_MODIFIED_AFTERWARDS;			// Publicado y modificado despus pero SIN publicar estas modificaciones
												// (modificaciones pendientes de publicar)
/////////////////////////////////////////////////////////////////////////////////////////
//  INTERFAZ EnumExtended
/////////////////////////////////////////////////////////////////////////////////////////
	private static EnumExtendedWrapper<R01MUserInterfaceStatus> _enums = new EnumExtendedWrapper<R01MUserInterfaceStatus>(R01MUserInterfaceStatus.values());
	
	@Override
	public boolean isIn(R01MUserInterfaceStatus... status) {
		return _enums.isIn(this,status);
	}
	@Override
	public boolean is(R01MUserInterfaceStatus other) {
		return _enums.is(this,other);
	}	
/////////////////////////////////////////////////////////////////////////////////////////
//  METODOS
/////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Comprueba si el estado es de despublicacin confirmada
	 * @return true si se trata de un estado de despublicacin
	 */
	public boolean isNeverPublished() {
		return this.is(NOT_PUBLISHED);
	}
	/**
	 * Comprueba si la operacin de de publicacin en curso
	 * @return
	 */
	public boolean isPublishedAndNotModifiedAfterwards() {
		return this.is(PUBLISHED_AND_NOT_MODIFIED_AFTERWARDS);
	}
	/**
	 * Comprueba si el estado es de publicacin confirmada 
	 * @return true si se trata de un estado de despublicacin
	 */
	public boolean areThereModificationsPendingToBePublished() {
		return this.is(PUBLISH_AND_MODIFIED_AFTERWARDS);
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCCION EN BASE AL ESTADO DE PUBLICACION Y MODIFICACIN
/////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Obtiene el estado del interfaz de usuario a partir de la informacin de publicacin 
	 * y el estado de edicin
	 * @param pubInfo informacin de publicacin
	 * @param editStatus estado de edicin
	 * @return el estado del interfaz de usuario
	 */
	public static R01MUserInterfaceStatus guessFrom(final R01MPublishInfo pubInfo,final R01MEditionStatusInfo editStatus) {
		R01MUserInterfaceStatus outStatus = null;
		if (pubInfo == null || !pubInfo.isPublished()) {
			outStatus = NOT_PUBLISHED;
		} else if (editStatus != null && editStatus.isInApprovedStatus()) {
			outStatus = PUBLISHED_AND_NOT_MODIFIED_AFTERWARDS;		// NO ha sido modificado despus de publicar
		} else {
			outStatus = PUBLISH_AND_MODIFIED_AFTERWARDS;			// Ha sido modificado despus de publicar
		}
		return outStatus;
	}
}
