package r01mo.model.assets.gallery;

import java.util.HashMap;
import java.util.Map;

import lombok.NoArgsConstructor;
import lombok.experimental.Accessors;
import r01f.aspects.dirtytrack.ConvertToDirtyStateTrackable;
import r01f.reflection.ReflectionUtils;
import r01f.types.Path;
import r01f.util.types.collections.CollectionUtils;
import r01mo.model.asset.R01MAssetBase;
import r01mo.model.asset.R01MAssetLocated;
import r01mo.model.asset.R01MDocumentAsset;
import r01mo.model.asset.R01MImageAsset;
import r01mo.model.asset.R01MVideoAsset;
import r01mo.model.asset.features.R01MAssetFeatures;
import r01mo.model.oids.assets.R01MAssetsGalleryItemOID;

import com.google.common.base.Predicate;
import com.google.common.collect.Maps;

/**
 * Galera de assets de diferente tipo:
 * <pre>
 * 		- {@link R01MAssetsGalleryImageItem}<R01MAssetsGalleryImageItem | R01MAssetGalleryDocumentItem | R01MAssetGalleryVideoItem>
 * 		- {@link R01MAssetsGalleryDocumentItem}<R01MAssetsGalleryImageItem | R01MAssetGalleryDocumentItem | R01MAssetGalleryVideoItem>
 * 		- {@link R01MAssetsGalleryVideoItem}<R01MAssetsGalleryImageItem | R01MAssetGalleryDocumentItem | R01MAssetGalleryVideoItem>
 * </pre>
 * Ejemplo de uso:
 * <code>
 *		// [1] Crear el Asset localizado en un contenido
 *		R01MAssetContentLocated<R01MDocumentAsset> docAssetInContent = new R01MAssetContentLocated<R01MDocumentAsset>();
 *		docAssetInContent.setAssetData(...);
 *		docAssetInContent.setPath(R01MContentRelativePath.of("/resources/attach/myDoc.pdf"));
 *
 *		R01MAssetContentLocated<R01MImageAsset> imgAssetInContent = new R01MAssetContentLocated<R01MImageAsset>();
 *		imgAssetInContent.setAssetData(...);
 *		imgAssetInContent.setPath(R01MContentRelativePath.of("/resources/images/myImage.jpg"));
 *
 *		R01MAssetContentLocated<R01MVideoAsset> videoAssetInContent = new R01MAssetContentLocated<R01MVideoAsset>();
 *		videoAssetInContent.setAssetData(...);
 *		videoAssetInContent.setPath(R01MContentRelativePath.of("/resources/videos/myVideo.flv"));
 *	
 *		// [2] Crear los Item de la galera
 *		R01MAssetsGalleryDocumentItem<R01MAssetContentLocated<R01MDocumentAsset>> docItem = new R01MAssetsGalleryDocumentItem<R01MAssetContentLocated<R01MDocumentAsset>>();
 *		docItem.setAsset(docAssetInContent);
 *
 *		R01MAssetsGalleryImageItem<R01MAssetContentLocated<R01MImageAsset>> imgItem = new R01MAssetsGalleryImageItem<R01MAssetContentLocated<R01MImageAsset>>();
 *		imgItem.setAsset(imgAssetInContent);
 *
  *		R01MAssetsGalleryVideoItem<R01MAssetContentLocated<R01MVideoAsset>> videoItem = new R01MAssetsGalleryVideoItem<R01MAssetContentLocated<R01MVideoAsset>>();
 *		videoItem.setAsset(videoAssetInContent);
 *	
 *		// [3] Crear la galera y aadir los items
 *		R01MMixedAssetsGallery gallery = new R01MMixedAssetsGallery();
 *		gallery.addImageAsset(imgItem);
 *		gallery.addDocumentAsset(docItem);
 *		gallery.addVideoAsset(videoItem);	
 * </code>
 */
@ConvertToDirtyStateTrackable
@Accessors(prefix="_")
@NoArgsConstructor
public class R01MMixedAssetsGallery extends R01MAssetsGalleryBase<R01MAssetsGalleryItemBase<? extends R01MAssetLocated<? extends R01MAssetBase<? extends R01MAssetFeatures>,
																													   ? extends Path>>> {
	private static final long serialVersionUID = 7605874837617985741L;
/////////////////////////////////////////////////////////////////////////////////////////
//  METODOS PARA AADIR Assets
/////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Aade un asset de tipo DOCUMENTO a la galera
	 * @param theDocument el documento
	 * @return la galera
	 */
	public R01MMixedAssetsGallery addDocumentAsset(final R01MAssetsGalleryDocumentItem<? extends R01MAssetLocated<R01MDocumentAsset,? extends Path>> theDocument) {
		Map<R01MAssetsGalleryItemOID,R01MAssetsGalleryItemBase<? extends R01MAssetLocated<? extends R01MAssetBase<? extends R01MAssetFeatures>,? extends Path>>> items = this.getItems();
		if (items == null) items = new HashMap<R01MAssetsGalleryItemOID,R01MAssetsGalleryItemBase<? extends R01MAssetLocated<? extends R01MAssetBase<? extends R01MAssetFeatures>,? extends Path>>>();
		items.put(R01MAssetsGalleryItemOID.supply(),theDocument);
		return this;
	}
	/**
	 * Aade un asset de tipo documento a la galera
	 * @param theDocument el documento
	 * @return la galera
	 */
	public R01MMixedAssetsGallery addDocumentAsset(final R01MAssetLocated<R01MDocumentAsset,? extends Path> theDocument) {
		R01MAssetsGalleryDocumentItem<R01MAssetLocated<R01MDocumentAsset,? extends Path>> item = new R01MAssetsGalleryDocumentItem<R01MAssetLocated<R01MDocumentAsset,? extends Path>>(theDocument);
		this.addDocumentAsset(item);
		return this;
	}
	/**
	 * Aade un asset de tipo IMAGEN a la galera
	 * @param theImage la imagen
	 * @return la galera
	 */
	public R01MMixedAssetsGallery addImageAsset(final R01MAssetsGalleryImageItem<? extends R01MAssetLocated<R01MImageAsset,? extends Path>> theImage) {
		Map<R01MAssetsGalleryItemOID,R01MAssetsGalleryItemBase<? extends R01MAssetLocated<? extends R01MAssetBase<? extends R01MAssetFeatures>,? extends Path>>> items = this.getItems();
		if (items == null) items = new HashMap<R01MAssetsGalleryItemOID,R01MAssetsGalleryItemBase<? extends R01MAssetLocated<? extends R01MAssetBase<? extends R01MAssetFeatures>,? extends Path>>>();
		items.put(R01MAssetsGalleryItemOID.supply(),theImage);
		return this;
	}
	/**
	 * Aade un asset de tipo imagen a la galera
	 * @param theImage la imagen
	 * @return la galera
	 */
	public R01MMixedAssetsGallery addImageAsset(final R01MAssetLocated<R01MImageAsset,? extends Path> theImage) {
		R01MAssetsGalleryImageItem<R01MAssetLocated<R01MImageAsset,? extends Path>> item = new R01MAssetsGalleryImageItem<R01MAssetLocated<R01MImageAsset,? extends Path>>(theImage);
		this.addImageAsset(item);
		return this;
	}
	/**
	 * Aade un asset de tipo VIDEO a la galera
	 * @param theVideo el video
	 * @return la galera
	 */
	public R01MMixedAssetsGallery addVideoAsset(final R01MAssetsGalleryVideoItem<? extends R01MAssetLocated<R01MVideoAsset,? extends Path>> theImage) {
		Map<R01MAssetsGalleryItemOID,R01MAssetsGalleryItemBase<? extends R01MAssetLocated<? extends R01MAssetBase<? extends R01MAssetFeatures>,? extends Path>>> items = this.getItems();
		if (items == null) items = new HashMap<R01MAssetsGalleryItemOID,R01MAssetsGalleryItemBase<? extends R01MAssetLocated<? extends R01MAssetBase<? extends R01MAssetFeatures>,? extends Path>>>();
		items.put(R01MAssetsGalleryItemOID.supply(),theImage);
		return this;
	}
	/**
	 * Aade un asset de tipo video a la galera
	 * @param theVideo el video
	 * @return la galera
	 */
	public R01MMixedAssetsGallery addVideoAsset(final R01MAssetLocated<R01MVideoAsset,? extends Path> theVideo) {
		R01MAssetsGalleryVideoItem<R01MAssetLocated<R01MVideoAsset,? extends Path>> item = new R01MAssetsGalleryVideoItem<R01MAssetLocated<R01MVideoAsset,? extends Path>>(theVideo);
		this.addVideoAsset(item);
		return this;
	}	
/////////////////////////////////////////////////////////////////////////////////////////
//  METODOS
/////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Devuelve todos los items que son de tipo imagen
     * @return los items de tipo imagen
     */
    @SuppressWarnings("unchecked")
    public Map<R01MAssetsGalleryItemOID,R01MAssetsGalleryImageItem<? extends R01MAssetLocated<R01MImageAsset,? extends Path>>> getImageItemsSorted() {
    	if (CollectionUtils.isNullOrEmpty(this.getItems())) return null;
		@SuppressWarnings("rawtypes")
		Map filteredValues = Maps.filterValues(this.getItems(),new AssetTypeFilter(R01MImageAsset.class));
		return filteredValues != null ? new HashMap<R01MAssetsGalleryItemOID,R01MAssetsGalleryImageItem<? extends R01MAssetLocated<R01MImageAsset,? extends Path>>>(filteredValues) 
									  : null;
    }
    /**
     * Devuelve todos los items que son de tipo video
     * @return los items de tipo video
     */
    @SuppressWarnings("unchecked")
	public Map<R01MAssetsGalleryItemOID,R01MAssetsGalleryVideoItem<? extends R01MAssetLocated<R01MVideoAsset,? extends Path>>> getVideoItemsSorted() {
    	if (CollectionUtils.isNullOrEmpty(this.getItems())) return null;
		@SuppressWarnings("rawtypes")
		Map filteredValues = Maps.filterValues(this.getItems(),new AssetTypeFilter(R01MVideoAsset.class));
		return filteredValues != null ? new HashMap<R01MAssetsGalleryItemOID,R01MAssetsGalleryVideoItem<? extends R01MAssetLocated<R01MVideoAsset,? extends Path>>>(filteredValues) 
									  : null;
    }
    /**
     * Devuelve todos los items que son de tipo documento
     * @return los items de tipo documento
     */
    @SuppressWarnings("unchecked")
    public Map<R01MAssetsGalleryItemOID,R01MAssetsGalleryDocumentItem<? extends R01MAssetLocated<R01MDocumentAsset,? extends Path>>> getDocItemsSorted() {
    	if (CollectionUtils.isNullOrEmpty(this.getItems())) return null;
		@SuppressWarnings("rawtypes")
		Map filteredValues = Maps.filterValues(this.getItems(),new AssetTypeFilter(R01MDocumentAsset.class));
		return filteredValues != null ? new HashMap<R01MAssetsGalleryItemOID,R01MAssetsGalleryDocumentItem<? extends R01MAssetLocated<R01MDocumentAsset,? extends Path>>>(filteredValues) 
									  : null;
    }
    private class AssetTypeFilter implements Predicate<R01MAssetsGalleryItemBase<? extends R01MAssetLocated<?,? extends Path>>> {
    	Class<? extends R01MAssetBase<?>> _assetClass;
    	public AssetTypeFilter(final Class<? extends R01MAssetBase<?>> theClass) {
    		_assetClass = theClass;
    	}
		@Override
		public boolean apply(R01MAssetsGalleryItemBase<? extends R01MAssetLocated<?,? extends Path>> item) {
			return ReflectionUtils.isSameClassAs(item.getAsset().getAssetData(),_assetClass);
		}
    }
}
