package r01mo.model.assets.gallery;

import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.Accessors;
import r01f.aspects.dirtytrack.ConvertToDirtyStateTrackable;
import r01f.types.Path;
import r01f.util.types.collections.CollectionUtils;
import r01mo.model.asset.R01MAssetBase;
import r01mo.model.asset.R01MAssetLocated;
import r01mo.model.asset.R01MAssetModelObject;
import r01mo.model.asset.features.R01MAssetFeatures;
import r01mo.model.oids.R01MResourcesContainerOID;
import r01mo.model.oids.assets.R01MAssetsGalleryItemOID;
import r01mo.model.oids.assets.R01MAssetsGalleryOID;

import com.google.common.base.Predicate;
import com.google.common.collect.Maps;

/**
 * Galera de recursos asociados a un documento.
 * @param <A>
 */
@ConvertToDirtyStateTrackable
@Accessors(prefix="_")
@NoArgsConstructor
abstract class R01MAssetsGalleryBase<A extends R01MAssetsGalleryItemBase<? extends R01MAssetLocated<? extends R01MAssetBase<? extends R01MAssetFeatures>,
																									? extends Path>>> implements R01MAssetModelObject {
    private static final long serialVersionUID = -84965107673862586L;
// /////////////////////////////////////////////////////////////////////////////////////////
// MIEMBROS
///////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Identificador de la galera.
     */
    @Getter @Setter private R01MAssetsGalleryOID _oid;
    /**
     * Tipo de galera (de imgenes, de videos, de documentos, etc).
     */
    @Getter @Setter private R01MAssetsGalleryContentType _type = R01MAssetsGalleryContentType.MIXED;
    /**
     * Nombre de la galera.
     */
    @Getter @Setter private String _name;
    /**
     * Descripcin de la galera.
     */
    @Getter @Setter private String _description;
    /**
     * Nmero de recursos que se van a visualizar por pgina.
     */
    @Getter @Setter private int _resourcesPerPage = 4;
    /**
     * Nombre del contenedor al que pertenece el recurso
     */
    @Getter @Setter private R01MResourcesContainerOID _containerOid;
    /**
     * Recursos de la galera.
     */
    @Getter @Setter private Map<R01MAssetsGalleryItemOID,A> _items;

/////////////////////////////////////////////////////////////////////////////////////////
//  METODOS
/////////////////////////////////////////////////////////////////////////////////////////
    /**
     * Ordena todos los items por el orden de presentacin
     * @return los items de la galera ordenados
     */
    public Map<R01MAssetsGalleryItemOID,A> getItemsSortedByPresentationOrder() {
    	if (CollectionUtils.isNullOrEmpty(_items)) return _items;
    	
    	R01MAssetsGalleryItemComparatorByPresentationOrder comparator = new R01MAssetsGalleryItemComparatorByPresentationOrder(_items);
    	SortedMap<R01MAssetsGalleryItemOID,A> sorted = new TreeMap<R01MAssetsGalleryItemOID,A>(comparator);
    	sorted.putAll(_items);
    	return sorted;
    }
    /**
     * Devuelve todos los items taggeados con una determinada etiqueta
     * @param tag la etiqueta
     * @return una vista de los items que solo contiene aquellos tageados con la etiqueta
     */
    public Map<R01MAssetsGalleryItemOID,A> getItemsTaggedBy(final String tag) {
    	if (CollectionUtils.isNullOrEmpty(_items)) return _items;
    	
		Predicate<A> filter = new Predicate<A>() {
									@Override
									public boolean apply(A item) {
										return item.getAsset() != null 
											&& item.getAsset().getAssetData() != null 
											&& item.getAsset().getAssetData().getTags() != null
											&& item.getAsset().getAssetData().getTags().containsTag(tag);
									}
							  };
		return Maps.filterValues(_items,filter);
    }
}
