/*
 * Created By: malonsoh
 * Created On: 24-mar-2017
 * File: PifHandleUtils.java
 */
package com.ejie.ac09j.utils.pif;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import com.ejie.ac09j.utils.constant.PifConstant;
import com.ejie.ac09j.utils.pif.exception.PifCopyFileException;
import com.ejie.ac09j.utils.pif.exception.PifExportFileException;
import com.ejie.ac09j.utils.pif.exception.PifImportFileException;
import com.ejie.ac09j.utils.pif.exception.PifInfoFileException;
import com.ejie.ac09j.utils.pif.exception.PifListFileException;
import com.ejie.ac09j.utils.pif.exception.PifMoveFileException;
import com.ejie.y31.factory.Y31JanoServiceAbstractFactory;
import com.ejie.y31.service.Y31JanoService;
import com.ejie.y31.vo.Y31AttachmentBean;

import n38a.exe.N38APISesion;

/**
 * The Class PifHandleUtils.
 */
public class PifHandleUtils {

	/** The Constant logger. */
	private static final Logger logger = LoggerFactory.getLogger(PifHandleUtils.class);

	private static final String app = "ac09j";

	/**
	 * Gets the file.
	 *
	 * @param path
	 *            the path
	 * @return the file
	 * @throws PifImportFileException
	 *             the pif import file exception
	 * @throws IOException
	 */
	public InputStream getFile(String path) throws PifImportFileException, IOException {

		Y31JanoService y31JanoService;
		InputStream is = null;
		try {
			y31JanoService = Y31JanoServiceAbstractFactory.getInstance();

			// Se verifica que el fichero existe
			Y31AttachmentBean fileInfo = y31JanoService.info(getAppSessionXlnet(), path);

			logger.trace("Se solicita el fichero: " + fileInfo.getFileName());
			is = y31JanoService.get(getAppSessionXlnet(), path);

		} catch (Exception e) {
			throw new PifImportFileException("Error al recuperar el fichero " + path + " del PIF", e);
		}

		logger.trace("fichero " + path + " descargado con exito.");

		return is;
	}

	/**
	 * Delete file.
	 *
	 * @param path
	 *            the path
	 */
	public void deleteFile(String path) {
		logger.trace("Se procede al borrado del fichero: " + path);

		try {
			Y31JanoServiceAbstractFactory.getInstance().delete(getAppSessionXlnet(), path);

		} catch (Exception e) {
			logger.warn("No se ha podido borrar el fichero del PIF: " + path, e);
		}

		logger.trace("fichero borrado " + path);
	}

	/**
	 * Put file. Sube un fichero basico de forma temporal (10 min)
	 *
	 * @param directorio
	 *            the directorio
	 * @param nombreArchivo
	 *            the nombre archivo
	 * @param pathDestino
	 *            the path destino
	 * @return the string
	 * @throws PifExportFileException
	 *             the pif export file exception
	 * @throws IOException
	 */
	public String putFile(String directorio, String nombreArchivo, String pathDestino)
			throws PifExportFileException, IOException {

		// Se invoca al metodo mas generico
		return putFile(directorio, nombreArchivo, pathDestino,
				Long.valueOf(PifConstant.ConfiguracionBasicaPif.TIEMPOBASEDEVIDAFICHEROSUBIDOPIF), Boolean.FALSE);
	}

	/**
	 * Put file.
	 *
	 * @param directorio
	 *            the directorio
	 * @param nombreArchivo
	 *            the nombre archivo
	 * @param pathDestino
	 *            the path destino
	 * @param tiempoDeVidaSg
	 *            the tiempo de vida sg
	 * @param preserveName
	 *            the preserve name
	 * @return the string
	 * @throws PifExportFileException
	 *             the pif export file exception
	 * @throws IOException
	 */
	public String putFile(String directorio, String nombreArchivo, String pathDestino, Long tiempoDeVidaSg,
			boolean preserveName) throws PifExportFileException, IOException {
		Y31JanoService service = null;
		Y31AttachmentBean resul = null;
		FileInputStream file = null;

		String destino = "";

		try {
			service = Y31JanoServiceAbstractFactory.getInstance();
			file = new FileInputStream(pathFormat(directorio) + nombreArchivo);
			destino = pathFormat(pathDestino) + nombreArchivo;

			logger.trace("Se sube el fichero " + destino + " al PIF.");

			resul = service.put(getAppSessionXlnet(), file, destino, preserveName, tiempoDeVidaSg);

		} catch (Exception e) {
			throw new PifExportFileException("Error al enviar el archivo " + destino + " al PIF", e);
		} finally {
			file.close();
		}

		logger.trace("Fichero " + destino + " subido con exito al PIF.");

		return resul.getFilePath();
	}

	/**
	 * Move file.
	 *
	 * @param pathOrigen
	 *            Ruta l�gica desde donde se va a mover el fichero
	 * @param pathDestino
	 *            Ruta l�gica donde se va a mover el fichero
	 * @param sobreescribir
	 *            Determina si se sobrescribe, si existe, el fichero de destino
	 *            o se conservaran los dos(al nuevo le hace un cambio num�rico
	 *            al nombre)
	 * @return the y31AttachmentBean
	 * @throws PifMoveFileException
	 *             the pif move file exception
	 */
	public Y31AttachmentBean moveFile(String pathOrigen, String pathDestino, boolean sobreescribir)
			throws PifMoveFileException {

		Y31AttachmentBean resul = null;

		try {
			Y31JanoService service = Y31JanoServiceAbstractFactory.getInstance();

			resul = service.move(getAppSessionXlnet(), pathOrigen, pathDestino, sobreescribir);

		} catch (Exception e) {
			throw new PifMoveFileException("Error al mover en el PIF el archivo " + pathOrigen + " a " + pathDestino,
					e);
		}

		logger.trace("El archivo " + pathOrigen + " ha sido movido, correctamente, a " + pathDestino);

		return resul;
	}

	/**
	 * Copy file.
	 *
	 * @param pathOrigen
	 *            Ruta l�gica desde donde se va a copiar el fichero
	 * @param pathDestino
	 *            Ruta l�gica donde se va a copiar el fichero
	 * @param sobreescribir
	 *            Determina si se sobrescribe, si existe, el fichero de destino
	 *            o se conservaran los dos(al nuevo le hace un cambio num�rico
	 *            al nombre)
	 * @return the y31AttachmentBean
	 * @throws PifCopyFileException
	 *             the pif copy file exception
	 */
	public Y31AttachmentBean copyFile(String pathOrigen, String pathDestino, boolean sobreescribir)
			throws PifCopyFileException {

		Y31AttachmentBean resul = null;

		try {
			Y31JanoService service = Y31JanoServiceAbstractFactory.getInstance();

			resul = service.copy(getAppSessionXlnet(), pathOrigen, pathDestino, sobreescribir);

		} catch (Exception e) {
			throw new PifCopyFileException("Error al copiar en el PIF el archivo " + pathOrigen + " a " + pathDestino,
					e);
		}

		logger.trace("El archivo " + pathOrigen + " ha sido copiado, correctamente, a " + pathDestino);

		return resul;
	}

	/**
	 * List file.
	 *
	 * @param pathListado
	 *            El path listado
	 * @return list de Y31AttachmentBean
	 * @throws PifListFileException
	 *             the pif list file exception
	 */
	@SuppressWarnings("unchecked")
	public List<Y31AttachmentBean> listadoUbicacion(String pathListado) throws PifListFileException {

		List<Y31AttachmentBean> resul = null;

		try {
			Y31JanoService service = Y31JanoServiceAbstractFactory.getInstance();

			resul = service.list(getAppSessionXlnet(), pathListado);

		} catch (Exception e) {
			throw new PifListFileException("Error al listar el path " + pathListado + " del PIF", e);
		}

		logger.trace("Se ha listado el contenido del path" + pathListado + " en el PIF");

		return resul;
	}

	/**
	 * Info file.
	 *
	 * @param pathDelFichero
	 *            the path del fichero
	 * @return the y31AttachmentBean
	 * @throws PifInfoFileException
	 *             the pif info file exception
	 */
	public Y31AttachmentBean infoFichero(String pathDelFichero) throws PifInfoFileException {

		Y31AttachmentBean resul = null;

		try {
			Y31JanoService service = Y31JanoServiceAbstractFactory.getInstance();

			resul = service.info(getAppSessionXlnet(), pathDelFichero);

		} catch (Exception e) {
			throw new PifInfoFileException("Error al obtener informacion del path " + pathDelFichero + " en el PIF", e);
		}

		logger.trace("Se ha obtenido informacion del archivo con path" + pathDelFichero + " en el PIF");

		return resul;
	}

	/* CLASES PRIVADAS */

	/**
	 * Path format.
	 *
	 * @param pathDestino
	 *            the path
	 * @return the string
	 */
	private String pathFormat(String path) {
		if (File.separator.equals(path.charAt(path.length() - 1))) {
			return path;
		} else {

			if ('/' == path.charAt(path.length() - 1)) {
				return path;
			} else {
				if (path.contains("/")) {
					return path + "/";
				} else {
					return path + File.separator;
				}
			}

		}
	}

	/**
	 * Obtener sesion de aplicacion XLNETS
	 * 
	 * @return sesion de aplicacion xlnets en formato Document
	 */
	private final static Document getAppSessionXlnet() {
		N38APISesion n38apiSesion = new N38APISesion();
		final Document SesionDocument = n38apiSesion.n38APISesionCrearApp(app);
		return SesionDocument;
	}

}
