/*
 * Created By: malonsoh
 * Created On: 24-mar-2017
 * File: JaxBConverterManager.java
 */
package com.ejie.ac09j.utils.jaxB;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.regex.Pattern;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.sax.SAXSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.InputSource;
import org.xml.sax.XMLReader;

/**
 * The Class JaxBConverterManager.
 *
 * @param <T>
 *            the generic type
 */
public final class JaxBConverterManager<T> {

	private static final Logger LOGGER = LoggerFactory.getLogger(JaxBConverterManager.class);

	/**
	 * Instantiates a new jaxBConverterManager.
	 */
	public JaxBConverterManager() {
		// Create a new JaxBConverterManager
	}

	/**
	 * Serialize object to XML.
	 *
	 * @param object
	 *            the object
	 * @return the string
	 * @throws Exception
	 * @throws Exception
	 *             the w63JaxBConverter exceptions
	 */
	public String serializeObjectToXML(T object) throws Exception {
		StringWriter stringWriter = new StringWriter();

		try {
			// Se convierte el objeto a un xml
			JAXBContext jaxbContext = JAXBContext.newInstance(object.getClass());
			Marshaller jaxbMarshaller;
			jaxbMarshaller = jaxbContext.createMarshaller();
			jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
			jaxbMarshaller.marshal(object, stringWriter);

		} catch (JAXBException e) {
			JaxBConverterManager.LOGGER.error(e.getLocalizedMessage(), e);
			throw new Exception("Error al transformar el objeto, de tipo " + object.getClass().getName() + ", a XML",
					e.getCause());

		}

		return stringWriter.toString();
	}

	/**
	 * Serialize fragment to XML.
	 *
	 * @param object
	 *            the object
	 * @param headboard
	 *            the headboard
	 * @return the string
	 * @throws Exception
	 *             the w63 jaxB converter exceptions
	 */
	public String serializeFragmentToXML(T object, boolean headboard) throws Exception {
		StringWriter stringWriter = new StringWriter();

		try {
			// Se convierte el objeto a un xml
			JAXBContext jaxbContext = JAXBContext.newInstance(object.getClass());
			Marshaller jaxbMarshaller;
			jaxbMarshaller = jaxbContext.createMarshaller();
			jaxbMarshaller.setProperty(Marshaller.JAXB_FRAGMENT, headboard);
			// jaxbMarshaller.setProperty(Marshaller.JAXB_SCHEMA_LOCATION, "");
			jaxbMarshaller.marshal(object, stringWriter);

		} catch (JAXBException e) {
			JaxBConverterManager.LOGGER.error(e.getLocalizedMessage(), e);
			throw new Exception("Error al transformar el objeto, de tipo " + object.getClass().getName() + ", a XML",
					e.getCause());
		}

		return stringWriter.toString();
	}

	/**
	 * Serialize fragment to XML file.
	 *
	 * @param object
	 *            the object
	 * @param filePathWithName
	 *            the file path with name
	 * @param headboard
	 *            the headboard
	 * @throws Exception
	 *             the w 63 jax B converter exceptions
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public void serializeFragmentToXMLFile(T object, String filePathWithName, boolean headboard)
			throws Exception, IOException {

		String filePathWithNameNaturalized = filePathWithName.replace('\\', '/');

		if (!("/".equals(filePathWithNameNaturalized.substring(filePathWithNameNaturalized.length() - 1)))) {
			String[] filePathWithNameNaturalizedChopped = filePathWithNameNaturalized.split("/");

			String fileName = normalizarNombreXml(
					filePathWithNameNaturalizedChopped[filePathWithNameNaturalizedChopped.length - 1]);
			StringBuilder filePath = new StringBuilder(filePathWithNameNaturalized.replace(
					"/" + filePathWithNameNaturalizedChopped[filePathWithNameNaturalizedChopped.length - 1], ""));

			// Se valida que exista el directorio
			File dir = new File(filePath.toString());

			if (!dir.exists()) {
				dir.mkdir();
			}

			// Se crea el fichero
			filePath.append("/").append(fileName);
			File newFile = new File(filePath.toString());

			if (newFile.exists() && !newFile.isDirectory()) {
				newFile.delete();
			}

			newFile.createNewFile();

			// Una vez creado el fichero se procede a guardar el contenido en el
			FileOutputStream newFileOutputStream = new FileOutputStream(newFile);

			try {
				// Se convierte el objeto a un xml
				JAXBContext jaxbContext = JAXBContext.newInstance(object.getClass());
				Marshaller jaxbMarshaller;
				jaxbMarshaller = jaxbContext.createMarshaller();
				jaxbMarshaller.setProperty(Marshaller.JAXB_FRAGMENT, headboard);
				jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);

				jaxbMarshaller.marshal(object, newFileOutputStream);

			} catch (JAXBException e) {
				JaxBConverterManager.LOGGER.error(e.getLocalizedMessage(), e);
				throw new Exception(
						"Error al transformar el objeto, de tipo " + object.getClass().getName() + ", a XML",
						e.getCause());

			} finally {
				newFileOutputStream.flush();
				newFileOutputStream.close();

			}

		} else {
			throw new FileNotFoundException(
					"El nombre del fichero no se ha pasado correctamente. Es necesario que se especifique el nombre tambien no solo la ruta del mismo. El nombre no puede acabar en una barra (\"\\\" o \"/\")");
		}
	}

	/**
	 * Deserialize XML string to object.
	 *
	 * @param clase
	 *            the clase
	 * @param xml
	 *            the xml
	 * @return the t
	 * @throws Exception
	 *             the w63JaxBConverter exceptions
	 */
	@SuppressWarnings("unchecked")
	public T deserializeXMLStringToObject(Class<T> clase, String xml) throws Exception {
		T objeto = null;

		try {
			// Se convierte el xml a un objeto
			JAXBContext jaxbContext = JAXBContext.newInstance(clase);
			Unmarshaller jaxbUnmarshaller = jaxbContext.createUnmarshaller();

			StringReader reader = new StringReader(xml);

			objeto = (T) jaxbUnmarshaller.unmarshal(reader);

		} catch (JAXBException e) {
			JaxBConverterManager.LOGGER.error(e.getLocalizedMessage(), e);
			throw new Exception("Error al pasear el XML a un objeto de tipo " + clase.getName(), e.getCause());
		}

		return objeto;
	}

	/**
	 * Deserialize XML file to object.
	 *
	 * @param clase
	 *            the clase
	 * @param xmlPath
	 *            the xml path
	 * @return the t
	 * @throws Exception
	 *             the w63JaxBConverter exceptions
	 */
	@SuppressWarnings("unchecked")
	public T deserializeXMLFileToObject(Class<T> clase, String xmlPath) throws Exception {
		T objeto = null;

		try {
			// Se convierte el xml a un objeto
			JAXBContext jaxbContext = JAXBContext.newInstance(clase);
			Unmarshaller jaxbUnmarshaller = jaxbContext.createUnmarshaller();

			SAXParserFactory spf = SAXParserFactory.newInstance();
			XMLReader xmlReader = spf.newSAXParser().getXMLReader();
			InputSource inputSource = new InputSource(new FileReader(xmlPath));
			SAXSource source = new SAXSource(xmlReader, inputSource);

			objeto = (T) jaxbUnmarshaller.unmarshal(source);

		} catch (Exception e) {
			JaxBConverterManager.LOGGER.error(e.getLocalizedMessage(), e);
			throw new Exception("Error al deserializar el fichero XML a un objeto de tipo " + clase.getName(),
					e.getCause());
		}

		return objeto;
	}

	/**
	 * Deserialize input stream file to object.
	 *
	 * @param clase
	 *            the clase
	 * @param inputStream
	 *            the input stream
	 * @return the t
	 * @throws Exception
	 *             the w63JaxBConverter exceptions
	 */
	@SuppressWarnings("unchecked")
	public T deserializeInputStreamFileToObject(Class<T> clase, InputStream inputStream) throws Exception {
		T objeto = null;

		try {
			// Se convierte el xml a un objeto
			JAXBContext jaxbContext = JAXBContext.newInstance(clase);
			Unmarshaller jaxbUnmarshaller = jaxbContext.createUnmarshaller();

			objeto = (T) jaxbUnmarshaller.unmarshal(inputStream);

		} catch (Exception e) {
			JaxBConverterManager.LOGGER.error(e.getLocalizedMessage(), e);
			throw new Exception("Error al deserializar el InputStream del XML a un objeto de tipo " + clase.getName(),
					e.getCause());
		}

		return objeto;
	}

	/**
	 * Normalizar nombre xml.
	 *
	 * @param pathFile
	 *            the path file
	 * @return the string
	 */
	private static String normalizarNombreXml(String pathFile) {

		// Se comprueba si se ha pasado el nombre de la plantilla con el tipo de
		// fichero especificado
		if (!(pathFile.contains(".xml"))) {
			String[] nombrePlantillaChopped = pathFile.split(Pattern.quote("."));

			return nombrePlantillaChopped[0].concat(".xml");
		} else {
			return pathFile;
		}
	}

}
