package com.ejie.ac09j.service.mensaje;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;

import javax.mail.Address;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMultipart;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.mail.MailException;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Service;

import com.ejie.ac09j.common.Ac09jConstants;
import com.ejie.ac09j.common.Ac09jException;
import com.ejie.ac09j.common.Ac09jUtils;
import com.ejie.ac09j.dao.SuscriptorDao;
import com.ejie.ac09j.model.Suscriptor;
import com.ejie.ac09j.model.SuscriptorFilter;
import com.ejie.ac09j.model.mensaje.Afectado;
import com.ejie.ac09j.model.mensaje.EnlacesEs;
import com.ejie.ac09j.model.mensaje.EnlacesEu;
import com.ejie.ac09j.model.mensaje.Evolucion;
import com.ejie.ac09j.model.mensaje.Informe;
import com.ejie.ac09j.model.mensaje.Mensaje;
import com.ejie.ac09j.model.mensaje.Seguimiento;
import com.ejie.ac09j.model.mensaje.Suceso;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

@Service(value = "ac09jMensajeService")
public class Ac09jMensajeServiceImpl implements Ac09jMensajeService {

    private static final String COMUN_ASUNTO_MENSAJE = "comun.asuntoMensaje";

    private static final String ASUNTO_MENSAJE_ERROR = "comun.asuntoMensajeError";

    private static final String MENSAJE_ERROR_NUM_AVISO = "mensaje.errorNumAviso";

    private static final String MENSAJE_ERROR_FECHA_PREDICCION = "mensaje.errorFechaPrediccion";

    private static final String MENSAJE_ERROR_LITERAL_COLOR = "mensaje.errorLiteralColor";

    private static final String MENSAJE_ERROR_LITERAL_ID_METEORO = "mensaje.errorLiteralIdMeteoro";

    private static final String MENSAJE_ERROR_EVOLUCION = "mensaje.errorEvolucion";

    private static final String MENSAJE_ERROR_FECHA_CREACION = "mensaje.errorFechaCreacion";

    private static final String MENSAJE_ERROR_FECHA_ACTUALIZACION = "mensaje.errorFechaActualizacion";

    private static final String MENSAJE_MENSAJE_ADJUNTOS = "mensaje.mensajeAdjuntos";

    private static final String MENSAJE_ERROR_ID_ZONA = "mensaje.errorIdZona";

    private static final String MENSAJE_ERROR_SUCESOS = "mensaje.errorSucesos";

    private static final String MENSAJE_ERROR_SEGUIMIENTOS = "mensaje.errorSeguimientos";

    private static final String MENSAJE_ERROR_DATOS_PREDICCIONES = "mensaje.errorDatos";

    private static final String MENSAJE_ERROR_LITERAL_FIN = "mensaje.errorLiteralFin";

    private static final Logger LOGGER = LoggerFactory.getLogger(Ac09jMensajeServiceImpl.class);

    @Autowired
    private Properties appConfiguration;

    @Autowired
    private SuscriptorDao suscriptorDao;

    @Autowired
    private JavaMailSender mailSender;

    @Autowired
    private ReloadableResourceBundleMessageSource appMessageSource;

    SimpleDateFormat sdf = new SimpleDateFormat("dd/MM/yyyy");

    private static final String LANG_CASTELLANO = "es";
    private static final String LANG_EUSKERA = "eu";

    private Boolean errorControlado = false;
    private Map<String, String> erroresControladosEs = new HashMap<String, String>();
    private Map<String, String> erroresControladosEu = new HashMap<String, String>();

    /**
     * 
     */
    @Override
    public void leerMensaje(String mensaje, String fileName) {
        // Declaramos las variables
        Mensaje mensajeNew = new Mensaje();
        String mensajeGenerado = null;
        List<Suscriptor> suscritoresAfectados = new ArrayList<Suscriptor>();
        StringBuilder detalleError = new StringBuilder();
        erroresControladosEs = new HashMap<String, String>();
        erroresControladosEu = new HashMap<String, String>();

        String tipoError;
        try {
            // Se parsea el json al objeto Mensaje New
            mensajeNew = new ObjectMapper().readValue(mensaje, Mensaje.class);
            if (mensajeNew.getAfectados() != null) {
                suscritoresAfectados = leerMensajeFindAfectados(mensaje, fileName, mensajeNew, mensajeGenerado,
                        suscritoresAfectados, detalleError);
                if (!suscritoresAfectados.isEmpty()) {
                    // Si hay suscriptores, se crea el mensaje.
                    crearEnviarMensaje(suscritoresAfectados, mensajeNew, mensaje, fileName);
                } else {
                    enviarMensajeError(
                            crearMensajeError(mensajeGenerado, suscritoresAfectados,
                                    this.appMessageSource.getMessage("mensaje.errorNoSuscriptores", null,
                                            new Locale(LANG_EUSKERA)),
                                    new Locale(LANG_EUSKERA))
                                    + crearMensajeError(mensajeGenerado, suscritoresAfectados,
                                            this.appMessageSource.getMessage("mensaje.errorNoSuscriptores", null,
                                                    new Locale(LANG_CASTELLANO)),
                                            new Locale(LANG_CASTELLANO)),
                            mensaje, fileName, mensajeGenerado);

                }

            } else {
                enviarMensajeError(
                        crearMensajeError(mensajeGenerado, suscritoresAfectados,
                                this.appMessageSource.getMessage("mensaje.errorNoAfectados", null,
                                        new Locale(LANG_EUSKERA)),
                                new Locale(LANG_EUSKERA))
                                + crearMensajeError(mensajeGenerado, suscritoresAfectados,
                                        this.appMessageSource.getMessage("mensaje.errorNoAfectados", null,
                                                new Locale(LANG_CASTELLANO)),
                                        new Locale(LANG_CASTELLANO)),
                        mensaje, fileName, mensajeGenerado);
            }

        } catch (JsonParseException e) {
            Ac09jMensajeServiceImpl.LOGGER.error(e.getLocalizedMessage(), e);
            detalleError.append(e.getMessage());
            tipoError = Ac09jConstants.ERROR_NO_GENERADO;
            createMensajeException(mensaje, fileName, mensajeGenerado, suscritoresAfectados, detalleError, tipoError);
        } catch (JsonMappingException e) {
            detalleError.append(e.getMessage());
            tipoError = Ac09jConstants.ERROR_NO_GENERADO;
            Ac09jMensajeServiceImpl.LOGGER.error(e.getLocalizedMessage(), e);
            createMensajeException(mensaje, fileName, mensajeGenerado, suscritoresAfectados, detalleError, tipoError);
        } catch (IOException e) {
            detalleError.append(e.getMessage());
            tipoError = Ac09jConstants.ERROR_NO_GENERADO;
            Ac09jMensajeServiceImpl.LOGGER.error(e.getLocalizedMessage(), e);
            Ac09jMensajeServiceImpl.LOGGER.error(e.getLocalizedMessage(), e);
            createMensajeException(mensaje, fileName, mensajeGenerado, suscritoresAfectados, detalleError, tipoError);
        }

    }

    private List<Suscriptor> leerMensajeFindAfectados(String mensaje, String fileName, Mensaje mensajeNew,
            String mensajeGenerado, List<Suscriptor> suscritoresAfectados, StringBuilder detalleError) {
        String tipoError;
        List<Suscriptor> suscritoresAfectadosAux = suscritoresAfectados;

        try {
            // Se buscan los suscriptores a los que afecta la alerta
            // recibida
            suscritoresAfectadosAux = getSuscriptoresAfectados(mensajeNew.getAfectados());
        } catch (Ac09jException e) {
            Ac09jMensajeServiceImpl.LOGGER.error(e.getLocalizedMessage(), e);
            detalleError.append(e.getMessage());
            tipoError = Ac09jConstants.ERROR_NO_GENERADO;
            createMensajeException(mensaje, fileName, mensajeGenerado, suscritoresAfectados, detalleError, tipoError);

        }
        return suscritoresAfectadosAux;
    }

    private void createMensajeException(String mensaje, String fileName, String mensajeGenerado,
            List<Suscriptor> suscritoresAfectados, StringBuilder detalleError, String tipoError) {
        enviarMensajeError(
                crearMensajeError(tipoError, suscritoresAfectados, detalleError.toString(), new Locale(LANG_EUSKERA))
                        + crearMensajeError(tipoError, suscritoresAfectados, detalleError.toString(),
                                new Locale(LANG_CASTELLANO)),
                mensaje, fileName, mensajeGenerado);
    }

    /**
     * 
     * @param mensaje
     * @param suscritoresAfectados
     * @param detalleError
     * @return
     */
    private String crearMensajeError(String errorGenerado, List<Suscriptor> suscritoresAfectados, String detalleError,
            Locale locale) {
        StringBuilder mensajeError = new StringBuilder();

        mensajeError.append(Calendar.getInstance().getTime().toString());
        mensajeError.append(Ac09jConstants.SALTO_LINEA);
        mensajeError.append(Ac09jConstants.SALTO_LINEA);
        if (!suscritoresAfectados.isEmpty()) {
            if (null != errorGenerado && Ac09jConstants.ERROR_NO_GENERADO.equals(errorGenerado)) {
                crearMensajeErrorNoGenerado(suscritoresAfectados, detalleError, locale, mensajeError);
            } else if (null != errorGenerado && Ac09jConstants.ERROR_CONTROLADO.equals(errorGenerado)) {
                crearMensajeErrorControlado(suscritoresAfectados, locale, mensajeError);
            }

        } else {
            mensajeError.append(this.appMessageSource.getMessage("mensaje.mensajeSinSuscriptores", null, locale)
                    + Ac09jConstants.SALTO_LINEA);
            if (null != errorGenerado && Ac09jConstants.ERROR_NO_GENERADO.equals(errorGenerado)) {
                mensajeError.append(this.appMessageSource.getMessage("mensaje.mensajeNoGenerado", null, locale)
                        + Ac09jConstants.SALTO_LINEA);

                mensajeError.append(detalleError);

                mensajeError.append(Ac09jConstants.SALTO_LINEA);
                mensajeError.append(Ac09jConstants.SALTO_LINEA);

                mensajeError.append(this.appMessageSource.getMessage(MENSAJE_MENSAJE_ADJUNTOS, null, locale)
                        + Ac09jConstants.SALTO_LINEA);

            }

        }

        mensajeError.append(Ac09jConstants.SALTO_LINEA);
        return mensajeError.toString();

    }

    private void crearMensajeErrorControlado(List<Suscriptor> suscritoresAfectados, Locale locale,
            StringBuilder mensajeError) {
        mensajeError.append(this.appMessageSource.getMessage("mensaje.mensajeErrorControlado", null, locale)
                + Ac09jConstants.SALTO_LINEA);
        mensajeError.append(Ac09jConstants.UL);
        Map<String, String> erroresControlados;
        if (LANG_CASTELLANO.equals(locale.getLanguage())) {
            erroresControlados = erroresControladosEs;
        } else {
            erroresControlados = erroresControladosEu;
        }
        for (String detalleErroControlado : erroresControlados.keySet()) {
            mensajeError.append(Ac09jConstants.LI);
            mensajeError.append(detalleErroControlado);
            mensajeError.append(Ac09jConstants.LI_CIERRE);
        }

        mensajeError.append(Ac09jConstants.UL_CIERRE);

        mensajeError.append(this.appMessageSource.getMessage("mensaje.mensajecomunicadosC", null, locale)
                + Ac09jConstants.SALTO_LINEA);
        mensajeError.append(Ac09jConstants.SALTO_LINEA);
        mensajeError.append(Ac09jConstants.UL);
        for (Suscriptor suscriptor : suscritoresAfectados) {
            mensajeError.append(Ac09jConstants.LI);
            mensajeError.append(suscriptor.getEmail01());
            mensajeError.append(Ac09jConstants.LI_CIERRE);
        }
        mensajeError.append(Ac09jConstants.UL_CIERRE);

        mensajeError.append(this.appMessageSource.getMessage(MENSAJE_MENSAJE_ADJUNTOS, null, locale));
    }

    private void crearMensajeErrorNoGenerado(List<Suscriptor> suscritoresAfectados, String detalleError, Locale locale,
            StringBuilder mensajeError) {
        mensajeError.append(this.appMessageSource.getMessage("mensaje.mensajeNoGenerado", null, locale)
                + Ac09jConstants.SALTO_LINEA);

        mensajeError.append(detalleError);
        mensajeError.append(Ac09jConstants.SALTO_LINEA);

        mensajeError.append(this.appMessageSource.getMessage("mensaje.mensajecomunicados", null, locale)
                + Ac09jConstants.SALTO_LINEA);
        mensajeError.append(Ac09jConstants.SALTO_LINEA);
        mensajeError.append(Ac09jConstants.UL);
        for (Suscriptor suscriptor : suscritoresAfectados) {
            mensajeError.append(Ac09jConstants.LI);
            mensajeError.append(suscriptor.getEmail01());
            mensajeError.append(Ac09jConstants.LI_CIERRE);
        }
        mensajeError.append(Ac09jConstants.UL_CIERRE);

        mensajeError.append(Ac09jConstants.SALTO_LINEA);
        mensajeError.append(Ac09jConstants.SALTO_LINEA);

        mensajeError.append(
                this.appMessageSource.getMessage(MENSAJE_MENSAJE_ADJUNTOS, null, locale) + Ac09jConstants.SALTO_LINEA);
    }

    private Map<String, String> crearEnviarMensajeErrorControl(String mensajeCastellano, String mensajeEuskera,
            StringBuilder detalleError, String mensajeGenerado, String tipoError) {
        Map<String, String> rdo = new HashMap<String, String>();
        String mensajeGeneradoAux = mensajeGenerado;
        String tipoErrorAux = tipoError;

        if (Ac09jConstants.ERROR_NO_GENERADO.equals(mensajeCastellano)
                || Ac09jConstants.ERROR_NO_GENERADO.equals(mensajeEuskera)) {
            mensajeGeneradoAux = Ac09jConstants.ERROR_NO_GENERADO;
            detalleError.append(
                    this.appMessageSource.getMessage("mensaje.errorNoGenerado", null, new Locale(LANG_CASTELLANO)));
            detalleError.append(

                    this.appMessageSource.getMessage("mensaje.errorNoGenerado", null, new Locale(LANG_EUSKERA)));
            tipoErrorAux = Ac09jConstants.ERROR_NO_GENERADO;
        } else {
            mensajeGeneradoAux = mensajeCastellano + mensajeEuskera;
            if (errorControlado) {
                tipoErrorAux = Ac09jConstants.ERROR_CONTROLADO;
            }

        }

        rdo.put("tipoError", tipoErrorAux);
        rdo.put("mensajeGenerado", mensajeGeneradoAux);

        return rdo;
    }

    /**
     * 
     * @param suscritoresAfectados
     * @param mensajeNew
     * @return
     */
    private void crearEnviarMensaje(List<Suscriptor> suscritoresAfectados, Mensaje mensajeNew, String mensaje,
            String fileName) {

        String tipoError = Ac09jConstants.EMPTY_STRING;

        List<Suscriptor> suscriptoresEuskera = new ArrayList<Suscriptor>();
        List<Suscriptor> suscriptoresCastellano = new ArrayList<Suscriptor>();
        List<Suscriptor> suscriptoresBilingue = new ArrayList<Suscriptor>();

        // Se clasifican los suscriptores según el idioma en el que van a
        // recibir la alerta
        crearEnviarMensajeFillSuscriptores(suscritoresAfectados, suscriptoresEuskera, suscriptoresCastellano,
                suscriptoresBilingue);
        String mensajeCastellano = Ac09jConstants.EMPTY_STRING;
        String mensajeEuskera = Ac09jConstants.EMPTY_STRING;
        String mensajeGenerado = Ac09jConstants.EMPTY_STRING;
        StringBuilder detalleError = new StringBuilder(Ac09jConstants.EMPTY_STRING);

        // Se pasa a la creacion del mensaje para cada idioma
        try {
            if (!suscriptoresCastellano.isEmpty() || !suscriptoresBilingue.isEmpty()) {
                mensajeCastellano = crearMensaje(new Locale(LANG_CASTELLANO), mensajeNew);
            }
            if (!suscriptoresEuskera.isEmpty() || !suscriptoresBilingue.isEmpty()) {
                mensajeEuskera = crearMensaje(new Locale(LANG_EUSKERA), mensajeNew);
            }
        } catch (Ac09jException e) {
            detalleError.append(e.getMessage());
            tipoError = Ac09jConstants.ERROR_NO_GENERADO;
            Ac09jMensajeServiceImpl.LOGGER.error(e.getLocalizedMessage(), e);
        }

        Map<String, String> mErrores = crearEnviarMensajeErrorControl(mensajeCastellano, mensajeEuskera, detalleError,
                mensajeGenerado, tipoError);
        tipoError = mErrores.get("tipoError");
        mensajeGenerado = mErrores.get("mensajeGenerado");

        try {
            enviarMensajes(suscriptoresEuskera, mensajeEuskera, suscriptoresCastellano, mensajeCastellano,
                    suscriptoresBilingue);
        } catch (Exception e) {
            detalleError.append(e.getMessage());
            tipoError = Ac09jConstants.ERROR_NO_GENERADO;
            Ac09jMensajeServiceImpl.LOGGER.error(e.getLocalizedMessage(), e);
        }

        if (errorControlado || !Ac09jConstants.EMPTY_STRING.equals(detalleError.toString())) {
            createMensajeException(mensaje, fileName, mensajeGenerado, suscritoresAfectados, detalleError, tipoError);
        }

    }

    private void crearEnviarMensajeFillSuscriptores(List<Suscriptor> suscritoresAfectados,
            List<Suscriptor> suscriptoresEuskera, List<Suscriptor> suscriptoresCastellano,
            List<Suscriptor> suscriptoresBilingue) {
        for (Suscriptor suscriptorAfectado : suscritoresAfectados) {
            if (Ac09jConstants.EUSKERA.equals(suscriptorAfectado.getIdioma01())) {
                suscriptoresEuskera.add(suscriptorAfectado);
            } else if (Ac09jConstants.CASTELLANO.equals(suscriptorAfectado.getIdioma01())) {
                suscriptoresCastellano.add(suscriptorAfectado);
            } else if (Ac09jConstants.AMBOS.equals(suscriptorAfectado.getIdioma01())) {
                suscriptoresBilingue.add(suscriptorAfectado);
            }
        }
    }

    /**
     * 
     * @param mensajeError
     * @param inputStream
     */
    private void enviarMensajeError(String mensajeError, String mensajeJson, String fileName, String mensajeGenerado) {

        String remitente = this.appConfiguration.getProperty("mail.mailRemitente");
        String responsable = this.appConfiguration.getProperty("mail.mailResponsable");

        Address[] arrayOfAddress = new Address[Ac09jConstants.UNO];
        try {
            arrayOfAddress[0] = new InternetAddress(responsable, false);
            final Multipart parts = new MimeMultipart();
            MimeBodyPart body = new MimeBodyPart();
            body.addHeader(Ac09jConstants.CONTENT_TYPE, Ac09jConstants.TEXT_HTML);
            body.addHeader(Ac09jConstants.CHARSET, Ac09jConstants.UTF_8);
            body.setText(mensajeError, Ac09jConstants.UTF_8);
            body.setContent(mensajeError, Ac09jConstants.TEXT_HTML);
            parts.addBodyPart(body);

            MimeBodyPart attachment1 = new MimeBodyPart();
            attachment1.setText(mensajeJson, Ac09jConstants.UTF_8);
            attachment1.setFileName(fileName);
            attachment1.setDisposition(Part.ATTACHMENT);
            attachment1.addHeader(Ac09jConstants.CHARSET, Ac09jConstants.UTF_8);
            parts.addBodyPart(attachment1);

            if (!Ac09jUtils.isEmptyOrNull(mensajeGenerado)) {
                MimeBodyPart attachment2 = new MimeBodyPart();
                attachment2.setText(mensajeGenerado, Ac09jConstants.UTF_8);
                attachment2.setFileName("mensajeGenerado.html");
                attachment2.setDisposition(Part.ATTACHMENT);
                attachment2.addHeader(Ac09jConstants.CONTENT_TYPE, Ac09jConstants.TEXT_HTML);
                attachment2.addHeader(Ac09jConstants.CHARSET, Ac09jConstants.UTF_8);
                parts.addBodyPart(attachment2);
            }

            Ac09aMimeMessagePreparator preparatorerror = new Ac09aMimeMessagePreparator(mensajeError, arrayOfAddress,
                    remitente,
                    this.appMessageSource.getMessage(ASUNTO_MENSAJE_ERROR, null, new Locale(LANG_EUSKERA)) + "/"
                            + this.appMessageSource.getMessage(ASUNTO_MENSAJE_ERROR, null, new Locale(LANG_CASTELLANO)),
                    parts);

            sendEmail(preparatorerror);

        } catch (Exception e) {
            Ac09jMensajeServiceImpl.LOGGER.error(e.getLocalizedMessage(), e);
        }

    }

    /**
     * 
     * @param preparatorerror
     */
    private void sendEmail(Ac09aMimeMessagePreparator preparatorerror) {
        try {
            mailSender.send(preparatorerror);
        } catch (MailException ex) {
            Ac09jMensajeServiceImpl.LOGGER.error(ex.getLocalizedMessage(), ex);
        } catch (Exception ex) {
            Ac09jMensajeServiceImpl.LOGGER.error(ex.getLocalizedMessage(), ex);
        }
    }

    /**
     * @param list List to be chopped
     * @param l    max number of elements per list
     * @return list of sub-lists
     */
    private <T> List<List<T>> chopped(List<T> list, final int l) {
        List<List<T>> parts = new ArrayList<List<T>>();
        final int n = list.size();
        for (int i = 0; i < n; i += l) {
            parts.add(new ArrayList<T>(list.subList(i, Math.min(n, i + l))));
        }
        return parts;
    }

    /**
     * 
     * @param suscriptoresEuskera
     * @param mensajeEuskera
     * @param suscriptoresCastellano
     * @param mensajeCastellano
     * @param suscriptoresBilingue
     * @throws Exception
     * 
     */
    private void enviarMensajes(List<Suscriptor> suscriptoresEuskera, String mensajeEuskera,
            List<Suscriptor> suscriptoresCastellano, String mensajeCastellano, List<Suscriptor> suscriptoresBilingue)
            throws Ac09jException {
        String destinatario = this.appConfiguration.getProperty("mail.mailDestinatario");
        String remitente = this.appConfiguration.getProperty("mail.mailRemitente");

        try {
            if (!suscriptoresCastellano.isEmpty()) {
                List<List<Suscriptor>> suscriptoresCastellanoEnvio = this.chopped(suscriptoresCastellano,
                        Ac09jConstants.MAX_SUSCRIPTORES);
                for (List<Suscriptor> suscriptoresCastellanoEnvioPart : suscriptoresCastellanoEnvio) {
                    enviarMensajeCastellano(suscriptoresCastellanoEnvioPart, mensajeCastellano, destinatario,
                            remitente);
                }
            }
            if (!suscriptoresEuskera.isEmpty()) {
                List<List<Suscriptor>> suscriptoresEuskeraEnvio = this.chopped(suscriptoresEuskera,
                        Ac09jConstants.MAX_SUSCRIPTORES);
                for (List<Suscriptor> suscriptoresEuskeraEnvioPart : suscriptoresEuskeraEnvio) {
                    enviarMensajeEuskera(suscriptoresEuskeraEnvioPart, mensajeEuskera, destinatario, remitente);
                }
            }
            if (!suscriptoresBilingue.isEmpty()) {
                List<List<Suscriptor>> suscriptoresBilingueEnvio = this.chopped(suscriptoresBilingue,
                        Ac09jConstants.MAX_SUSCRIPTORES);
                for (List<Suscriptor> suscriptoresBilingueEnvioPart : suscriptoresBilingueEnvio) {
                    enviarMensajeBilingue(mensajeEuskera, mensajeCastellano, suscriptoresBilingueEnvioPart,
                            destinatario, remitente);
                }
            }

        } catch (Exception e) {
            Ac09jMensajeServiceImpl.LOGGER.error(e.getLocalizedMessage(), e);
            throw new Ac09jException(e.getLocalizedMessage());
        }

    }

    private void enviarMensajeBilingue(String mensajeEuskera, String mensajeCastellano,
            List<Suscriptor> suscriptoresBilingue, String responsable, String remitente) throws MessagingException {
        // Preparar mensaje bilingue
        Address[] arrayOfAddressBilingue = new Address[suscriptoresBilingue.size() + 1];

        arrayOfAddressBilingue[0] = new InternetAddress(responsable, false);
        int k = 1;
        for (Suscriptor suscriptor : suscriptoresBilingue) {
            arrayOfAddressBilingue[k] = new InternetAddress(suscriptor.getEmail01(), false);
            k++;
        }

        final Multipart partsBi = new MimeMultipart();
        MimeBodyPart bodyBi = new MimeBodyPart();
        bodyBi.addHeader(Ac09jConstants.CONTENT_TYPE, Ac09jConstants.TEXT_HTML);
        bodyBi.addHeader(Ac09jConstants.CHARSET, Ac09jConstants.UTF_8);
        bodyBi.setText(mensajeEuskera + mensajeCastellano, Ac09jConstants.UTF_8);
        bodyBi.setContent(mensajeEuskera + mensajeCastellano, Ac09jConstants.TEXT_HTML);
        partsBi.addBodyPart(bodyBi);

        Ac09aMimeMessagePreparator preparatorBi = new Ac09aMimeMessagePreparator(mensajeEuskera + mensajeCastellano,
                arrayOfAddressBilingue, remitente,
                this.appMessageSource.getMessage(COMUN_ASUNTO_MENSAJE, null, new Locale(LANG_EUSKERA)) + "/"
                        + this.appMessageSource.getMessage(COMUN_ASUNTO_MENSAJE, null, new Locale(LANG_CASTELLANO)),
                partsBi);
        mailSender.send(preparatorBi);
        for (Suscriptor suscriptorEnvio : suscriptoresBilingue) {
            this.suscriptorDao.addNumEnvios(suscriptorEnvio);
        }
        LOGGER.info("MENSAJE BILINGUE ENVIADO");
    }

    private void enviarMensajeEuskera(List<Suscriptor> suscriptoresEuskera, String mensajeEuskera, String responsable,
            String remitente) throws MessagingException {
        // Preparar mensaje en euskera
        Address[] arrayOfAddressEu = new Address[suscriptoresEuskera.size() + 1];

        arrayOfAddressEu[0] = new InternetAddress(responsable, false);
        int j = 1;
        for (Suscriptor suscriptor : suscriptoresEuskera) {
            arrayOfAddressEu[j] = new InternetAddress(suscriptor.getEmail01(), false);
            j++;
        }

        final Multipart partsEu = new MimeMultipart();
        MimeBodyPart bodyEu = new MimeBodyPart();
        bodyEu.addHeader(Ac09jConstants.CONTENT_TYPE, Ac09jConstants.TEXT_HTML);
        bodyEu.addHeader(Ac09jConstants.CHARSET, Ac09jConstants.UTF_8);
        bodyEu.setText(mensajeEuskera, Ac09jConstants.UTF_8);
        bodyEu.setContent(mensajeEuskera, Ac09jConstants.TEXT_HTML);
        partsEu.addBodyPart(bodyEu);

        Ac09aMimeMessagePreparator preparatorEus = new Ac09aMimeMessagePreparator(mensajeEuskera, arrayOfAddressEu,
                remitente, this.appMessageSource.getMessage(COMUN_ASUNTO_MENSAJE, null, new Locale(LANG_EUSKERA)),
                partsEu);
        mailSender.send(preparatorEus);
        for (Suscriptor suscriptorEnvio : suscriptoresEuskera) {
            this.suscriptorDao.addNumEnvios(suscriptorEnvio);
        }
        LOGGER.info("MENSAJE EUSKERA ENVIADO");
    }

    private void enviarMensajeCastellano(List<Suscriptor> suscriptoresCastellano, String mensajeCastellano,
            String responsable, String remitente) throws MessagingException {
        // Prepara mensaje en castellano
        Address[] arrayOfAddressCas = new Address[suscriptoresCastellano.size() + 1];

        arrayOfAddressCas[0] = new InternetAddress(responsable, false);
        int i = 1;
        for (Suscriptor suscriptor : suscriptoresCastellano) {
            arrayOfAddressCas[i] = new InternetAddress(suscriptor.getEmail01(), false);
            i++;
        }

        final Multipart parts = new MimeMultipart();
        MimeBodyPart body = new MimeBodyPart();
        body.addHeader(Ac09jConstants.CONTENT_TYPE, Ac09jConstants.TEXT_HTML);
        body.addHeader(Ac09jConstants.CHARSET, Ac09jConstants.UTF_8);
        body.setText(mensajeCastellano, Ac09jConstants.UTF_8);
        body.setContent(mensajeCastellano, Ac09jConstants.TEXT_HTML);
        parts.addBodyPart(body);

        Ac09aMimeMessagePreparator preparatorCas = new Ac09aMimeMessagePreparator(mensajeCastellano, arrayOfAddressCas,
                remitente, this.appMessageSource.getMessage(COMUN_ASUNTO_MENSAJE, null, new Locale(LANG_CASTELLANO)),
                parts);
        mailSender.send(preparatorCas);
        for (Suscriptor suscriptorEnvio : suscriptoresCastellano) {
            this.suscriptorDao.addNumEnvios(suscriptorEnvio);
        }
        LOGGER.info("MENSAJE CASTELLANO ENVIADO");
    }

    /**
     * 
     * @param locale
     * @param mensajeNew
     * @return
     */
    private String crearMensaje(Locale locale, Mensaje mensajeNew) throws Ac09jException {

        // Se recoge la plantilla general y se van reemplazando todas las partes
        String plantillaPrincipal = leerInputStream(
                Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_PLANTILLA_PRINCIPAL));
        plantillaPrincipal = plantillaPrincipal.replace(Ac09jConstants.LOGOS_GV, getLogos());

        if (null != mensajeNew.getInforme()) {
            Informe informeRecibido = mensajeNew.getInforme();
            if (locale.getLanguage().equals(LANG_CASTELLANO)) {
                plantillaPrincipal = plantillaPrincipal.replace(Ac09jConstants.TITULO_MAIL,
                        getTitulo(informeRecibido.getTituloEs()));
            } else {
                plantillaPrincipal = plantillaPrincipal.replace(Ac09jConstants.TITULO_MAIL,
                        getTitulo(informeRecibido.getTituloEu()));
            }
            plantillaPrincipal = plantillaPrincipal.replace(Ac09jConstants.NUMERO_AVISOS,
                    getNumeroAvisos(informeRecibido.getNumAviso(), informeRecibido.getFechaCreacion(),
                            informeRecibido.getFechaActualizacionProgramada(), locale));
            if (informeRecibido.getSeguimiento().isEmpty()) {
                this.errorControlado = true;
                erroresControladosEu.put(
                        this.appMessageSource.getMessage(MENSAJE_ERROR_SEGUIMIENTOS, null, new Locale(LANG_EUSKERA)),
                        this.appMessageSource.getMessage(MENSAJE_ERROR_SEGUIMIENTOS, null, new Locale(LANG_EUSKERA)));
                erroresControladosEs.put(
                        this.appMessageSource.getMessage(MENSAJE_ERROR_SEGUIMIENTOS, null, new Locale(LANG_CASTELLANO)),
                        this.appMessageSource.getMessage(MENSAJE_ERROR_SEGUIMIENTOS, null,
                                new Locale(LANG_CASTELLANO)));
            } else {
                plantillaPrincipal = plantillaPrincipal.replace(Ac09jConstants.CAUSAS_ALERTAS,
                        getCausasAlertas(mensajeNew.getAfectados(), locale));
                StringBuilder seguimientosPrediccion = new StringBuilder();
                plantillaPrincipal = plantillaPrincipal.replace(Ac09jConstants.TITULO_PREDICCION,
                        this.appMessageSource.getMessage("mensaje.tituloPrediccion", null, locale));
                for (Seguimiento seguimiento : informeRecibido.getSeguimiento()) {
                    String seguimientoPrediccion = getPrediccion(seguimiento, locale);
                    seguimientosPrediccion.append(seguimientoPrediccion);
                }
                plantillaPrincipal = plantillaPrincipal.replace(Ac09jConstants.PREDICCION,
                        seguimientosPrediccion.toString());
                plantillaPrincipal = plantillaPrincipal.replace(Ac09jConstants.FENOMENOS,
                        getFenomenos(locale, informeRecibido));
                plantillaPrincipal = plantillaPrincipal.replace(Ac09jConstants.ENLACES,
                        getEnlaces(locale, informeRecibido));
                plantillaPrincipal = plantillaPrincipal.replace(Ac09jConstants.TWITTER, getTwitter(locale));

            }
        } else {
            return Ac09jConstants.ERROR_NO_GENERADO;
        }

        return plantillaPrincipal;
    }

    private CharSequence getEnlaces(Locale locale, Informe informeRecibido) throws Ac09jException {

        String textoEnlace = leerInputStream(
                Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_TITULO_ENLACE));
        textoEnlace = textoEnlace.replace(Ac09jConstants.TITULO_ENLACES,
                this.appMessageSource.getMessage("mensaje.tituloEnlace", null, locale));

        StringBuilder lineasEnlaces = new StringBuilder();

        if (locale.getLanguage().equals(LANG_CASTELLANO)) {
            for (EnlacesEs enlaceEs : informeRecibido.getEnlacesEs()) {
                lineasEnlaces.append(getLineaEnlace(enlaceEs, locale));
            }
        }

        if (locale.getLanguage().equals(LANG_EUSKERA)) {
            for (EnlacesEu enlaceEu : informeRecibido.getEnlacesEu()) {
                lineasEnlaces.append(getLineaEnlace(enlaceEu, locale));
            }
        }
        textoEnlace = textoEnlace.replace(Ac09jConstants.LINEAS_ENLACES, lineasEnlaces);

        return textoEnlace;
    }

    private String getLineaEnlace(EnlacesEu enlaceEu, Locale locale) throws Ac09jException {
        String lineaEnlace = leerInputStream(
                Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_LINEA_ENLACE));
        lineaEnlace = lineaEnlace.replace(Ac09jConstants.SRC_IMAGEN_ENLACE,
                getUrImages() + Ac09jConstants.ICO_ENLACE + Ac09jConstants.EXTENSION_PNG);
        lineaEnlace = lineaEnlace.replace(Ac09jConstants.URL_ENLACE, enlaceEu.getUrl());
        lineaEnlace = lineaEnlace.replace(Ac09jConstants.NOMBRE_ENLACE, enlaceEu.getNombre());
        return lineaEnlace;
    }

    private String getLineaEnlace(EnlacesEs enlaceEs, Locale locale) throws Ac09jException {
        String lineaEnlace = leerInputStream(
                Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_LINEA_ENLACE));
        lineaEnlace = lineaEnlace.replace(Ac09jConstants.SRC_IMAGEN_ENLACE,
                getUrImages() + Ac09jConstants.ICO_ENLACE + Ac09jConstants.EXTENSION_PNG);
        lineaEnlace = lineaEnlace.replace(Ac09jConstants.URL_ENLACE, enlaceEs.getUrl());
        lineaEnlace = lineaEnlace.replace(Ac09jConstants.NOMBRE_ENLACE, enlaceEs.getNombre());
        return lineaEnlace;
    }

    /**
     * 
     * @return
     */
    private String getTwitter(Locale locale) throws Ac09jException {
        String twitter = leerInputStream(
                Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_TWITTER));
        twitter = twitter.replace(Ac09jConstants.SRC_IMAGEN_TWITTER,
                getUrImages() + Ac09jConstants.ICO_TWITTER + Ac09jConstants.EXTENSION_JPJ);
        twitter = twitter.replace(Ac09jConstants.TEXTO_TWITTER,
                this.appMessageSource.getMessage("mensaje.textoTwitter", null, locale));
        return twitter;
    }

    private String getUrImages() {
        return this.appConfiguration.getProperty("statics.path.ac09a") + Ac09jConstants.URL_IMG;

    }

    /**
     * 
     * @param locale
     * @param informeRecibido
     * @return
     */
    private String getFenomenos(Locale locale, Informe informeRecibido) throws Ac09jException {

        String sucesos = leerInputStream(
                Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_FEM_ADVERSO));
        sucesos = sucesos.replace(Ac09jConstants.TITULO_FEM_ADVERSO,
                this.appMessageSource.getMessage("mensaje.tituloFemAdverso", null, locale));

        StringBuilder lineasSuceso = new StringBuilder();
        if (informeRecibido.getSucesos().isEmpty()) {
            this.errorControlado = true;
            erroresControladosEu.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_SUCESOS, null, new Locale(LANG_EUSKERA)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_SUCESOS, null, new Locale(LANG_EUSKERA)));
            erroresControladosEs.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_SUCESOS, null, new Locale(LANG_CASTELLANO)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_SUCESOS, null, new Locale(LANG_EUSKERA)));
            return Ac09jConstants.EMPTY_STRING;
        }
        for (Suceso suceso : informeRecibido.getSucesos()) {
            lineasSuceso.append(getFemAdverso(suceso, locale));
            lineasSuceso.append("<tr ><td height='20'></td></tr>");
        }

        sucesos = sucesos.replace(Ac09jConstants.LINEAS_FEM_ADVERSOS, lineasSuceso);

        return sucesos;
    }

    /**
     * 
     * @param numAviso
     * @param fechaCreacion
     * @param fechaActualizacionProgramada
     * @param locale
     * @return
     */
    private String getNumeroAvisos(String numAviso, String fechaCreacion, String fechaActualizacionProgramada,
            Locale locale) throws Ac09jException {
        String numeroAvisos = leerInputStream(
                Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_NUMAVISOS));

        numeroAvisos = numeroAvisos.replace(Ac09jConstants.LABEL_NUM_AVISO,
                this.appMessageSource.getMessage("mensaje.labelNumAviso", null, locale));

        numeroAvisos = replaceData(numAviso, numeroAvisos, Ac09jConstants.NUM_AVISO, Ac09jConstants.CAMPO_NUM_AVISO);
        numeroAvisos = replaceData(getLiteralFechaCreacion(fechaCreacion, locale), numeroAvisos,
                Ac09jConstants.FECHA_CREACION, Ac09jConstants.CAMPO_FECHA_CREACION);
        numeroAvisos = replaceData(getLiteralFechaActualizacion(fechaActualizacionProgramada, locale), numeroAvisos,
                Ac09jConstants.FECHA_ACTUALIZACION, Ac09jConstants.CAMPO_FECHA_ACTUALIZACION);

        return numeroAvisos;
    }

    /**
     * 
     * @param replacingString
     * @param numeroAvisos
     * @param replacedString
     * @return
     */
    private String replaceData(String replacingString, String numeroAvisos, String replacedString, String campo) {
        String numeroAvisos1;
        if (!Ac09jUtils.isEmptyOrNull(replacingString)) {
            numeroAvisos1 = numeroAvisos.replace(replacedString, replacingString);
        } else {
            numeroAvisos1 = numeroAvisos.replace(replacedString, Ac09jConstants.EMPTY_STRING);
            Object[] paramsNumAvisos = new Object[1];
            paramsNumAvisos[0] = campo;
            this.errorControlado = true;
            erroresControladosEu.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_NUM_AVISO, paramsNumAvisos,
                            new Locale(LANG_EUSKERA)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_NUM_AVISO, paramsNumAvisos,
                            new Locale(LANG_EUSKERA)));
            erroresControladosEs.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_NUM_AVISO, paramsNumAvisos,
                            new Locale(LANG_CASTELLANO)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_NUM_AVISO, paramsNumAvisos,
                            new Locale(LANG_CASTELLANO)));

        }
        return numeroAvisos1;
    }

    private List<Suscriptor> getSuscriptoresAfectados(List<Afectado> afectados) throws Ac09jException {

        List<String> comarcasGipuzkoa = new ArrayList<String>();
        List<String> comarcasAraba = new ArrayList<String>();
        List<String> comarcasBizkaia = new ArrayList<String>();
        List<String> provinciasSinComarc = new ArrayList<String>();

        List<SuscriptorFilter> filterSuscriptores = new ArrayList<SuscriptorFilter>();

        // Por afectado se recoge la conbinacion de provincias/comarcas y se
        // guardan los datos en listas para cada provincia
        for (Afectado afectado : afectados) {
            comarcasGipuzkoa = new ArrayList<String>();
            comarcasAraba = new ArrayList<String>();
            comarcasBizkaia = new ArrayList<String>();
            provinciasSinComarc = new ArrayList<String>();

            for (String provinciaComarca : afectado.getComarcas()) {

                String[] arrayProvComarcas = provinciaComarca.split(",");
                for (String provComarca : arrayProvComarcas) {
                    String provincia = provComarca.split("-")[0];

                    getSuscriptoresAfectadosProvComarca(comarcasGipuzkoa, comarcasAraba, comarcasBizkaia,
                            provinciasSinComarc, provComarca, provincia);
                }
            }
            SuscriptorFilter suscriptorfilter = new SuscriptorFilter(getIdTipoAviso(afectado), comarcasGipuzkoa,
                    comarcasBizkaia, comarcasAraba, provinciasSinComarc);

            filterSuscriptores.add(suscriptorfilter);

        }

        return suscriptorDao.findSuscriptores(filterSuscriptores);

    }

    private void getSuscriptoresAfectadosProvComarca(List<String> comarcasGipuzkoa, List<String> comarcasAraba,
            List<String> comarcasBizkaia, List<String> provinciasSinComarc, String provComarca, String provincia) {
        if (provComarca.split("-").length > 1 && Ac09jConstants.GIPUZKOA.equals(provincia)) {
            comarcasGipuzkoa.add(provComarca.split("-")[1]);
        } else if (provComarca.split("-").length > 1 && Ac09jConstants.ARABA.equals(provincia)) {
            comarcasAraba.add(provComarca.split("-")[1]);
        } else if (provComarca.split("-").length > 1 && Ac09jConstants.BIZKAIA.equals(provincia)) {
            comarcasBizkaia.add(provComarca.split("-")[1]);
        } else {
            provinciasSinComarc.add(provincia);
        }
    }

    /**
     * 
     * @param afectado
     * @return
     * @throws Ac09jException
     * 
     */
    public Long getIdTipoAviso(Afectado afectado) throws Ac09jException {
        if (Ac09jConstants.TEMP.equalsIgnoreCase(afectado.getIdMeteoro())
                && (Ac09jConstants.ALTAS_PERSISTENTES.equalsIgnoreCase(afectado.getZonaImpacto())
                        || Ac09jConstants.MUY_ALTAS.equalsIgnoreCase(afectado.getZonaImpacto()))) {
            return getMapAvisos().get(Ac09jConstants.TEMP_ALTAS);
        } else if (Ac09jConstants.TEMP.equalsIgnoreCase(afectado.getIdMeteoro())
                && Ac09jConstants.MINIMAS_HELADAS.equalsIgnoreCase(afectado.getZonaImpacto())) {
            return getMapAvisos().get(Ac09jConstants.TEMP_MIN);
        } else {
            return getMapAvisos().get(afectado.getIdMeteoro().toUpperCase());
        }
    }

    public Map<String, Long> getMapAvisos() {
        Map<String, Long> mapAvisos = new HashMap<String, Long>();
        mapAvisos.put(Ac09jConstants.MAR, (long) 1);
        mapAvisos.put(Ac09jConstants.TEMP_ALTAS, (long) 2);
        mapAvisos.put(Ac09jConstants.TEMP_MIN, (long) 3);
        mapAvisos.put(Ac09jConstants.NIEVE, (long) 4);
        mapAvisos.put(Ac09jConstants.LLUVIA, (long) 5);
        mapAvisos.put(Ac09jConstants.VIENTO, (long) 6);
        mapAvisos.put(Ac09jConstants.ESPECIAL, (long) 7);

        return mapAvisos;
    }

    /**
     * 
     * @param suceso
     * @param locale
     * @return
     */
    private String getFemAdverso(Suceso suceso, Locale locale) throws Ac09jException {
        String femAdverso = leerInputStream(
                Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_LINEA_FEM_ADVERSO));

        if (!Ac09jUtils.isEmptyOrNull(suceso.getIdMeteoro())) {
            femAdverso = femAdverso.replace(Ac09jConstants.SRC_IMAGEN_BLUE, getUrImages()
                    + suceso.getIdMeteoro().toLowerCase() + Ac09jConstants.BLUE + Ac09jConstants.EXTENSION_PNG);
        }
        if (!Ac09jUtils.isEmptyOrNull(suceso.getTituloEs()) && !Ac09jUtils.isEmptyOrNull(suceso.getDescripcionEs())
                && LANG_CASTELLANO.equals(locale.getLanguage())) {
            femAdverso = femAdverso.replace(Ac09jConstants.TITULO_FENOMENO, suceso.getTituloEs());
            femAdverso = femAdverso.replace(Ac09jConstants.DESCRIPCION_FENOMENO, suceso.getDescripcionEs());
            return femAdverso;
        } else if (!Ac09jUtils.isEmptyOrNull(suceso.getTituloEu())
                && !Ac09jUtils.isEmptyOrNull(suceso.getDescripcionEu()) && LANG_EUSKERA.equals(locale.getLanguage())) {
            femAdverso = femAdverso.replace(Ac09jConstants.TITULO_FENOMENO, suceso.getTituloEu());
            femAdverso = femAdverso.replace(Ac09jConstants.DESCRIPCION_FENOMENO, suceso.getDescripcionEu());
            return femAdverso;
        }

        this.errorControlado = true;
        erroresControladosEu.put(
                this.appMessageSource.getMessage(MENSAJE_ERROR_SUCESOS, null, new Locale(LANG_EUSKERA)),
                this.appMessageSource.getMessage(MENSAJE_ERROR_SUCESOS, null, new Locale(LANG_EUSKERA)));
        erroresControladosEs.put(
                this.appMessageSource.getMessage(MENSAJE_ERROR_SUCESOS, null, new Locale(LANG_CASTELLANO)),
                this.appMessageSource.getMessage(MENSAJE_ERROR_SUCESOS, null, new Locale(LANG_CASTELLANO)));
        femAdverso = femAdverso.replace(Ac09jConstants.TITULO_FENOMENO, Ac09jConstants.EMPTY_STRING);
        femAdverso = femAdverso.replace(Ac09jConstants.DESCRIPCION_FENOMENO, Ac09jConstants.EMPTY_STRING);
        return femAdverso;
    }

    /**
     * 
     * @param seguimiento
     * @param locale
     * @return
     */
    private String getPrediccion(Seguimiento seguimiento, Locale locale) throws Ac09jException {

        String prediccionPlantilla = leerInputStream(
                Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_PREDICCIONES));
        prediccionPlantilla = getTituloFechaPrediccion(seguimiento, locale, prediccionPlantilla);
        StringBuilder lineasPrediccion = new StringBuilder();
        for (Evolucion evolucion : seguimiento.getEvolucion()) {
            String evolucionCompleta = getEvolucion(evolucion, locale);
            lineasPrediccion.append(evolucionCompleta);
        }
        prediccionPlantilla = prediccionPlantilla.replace(Ac09jConstants.LINEAS_PREDICCION, lineasPrediccion);

        return prediccionPlantilla;
    }

    /**
     * 
     * @param seguimiento
     * @param locale
     * @param tituloFechaPrediccion
     * @return
     */
    private String getTituloFechaPrediccion(Seguimiento seguimiento, Locale locale, String tituloFechaPrediccion)
            throws Ac09jException {
        String tituloFechaPrediccion1;
        String tituloFechaPrediccion2;
        String tituloFechaPrediccion3 = Ac09jConstants.EMPTY_STRING;

        if (Ac09jUtils.isEmptyOrNull(seguimiento.getFecha()) || seguimiento.getFecha().length() < 10) {
            this.errorControlado = true;
            erroresControladosEu.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_PREDICCION, null, new Locale(LANG_EUSKERA)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_PREDICCION, null, new Locale(LANG_EUSKERA)));
            erroresControladosEs.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_PREDICCION, null, new Locale(LANG_CASTELLANO)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_PREDICCION, null,
                            new Locale(LANG_CASTELLANO)));
            tituloFechaPrediccion1 = tituloFechaPrediccion.replace(Ac09jConstants.DIA, Ac09jConstants.EMPTY_STRING);
            tituloFechaPrediccion2 = tituloFechaPrediccion1.replace(Ac09jConstants.DIA_SEMANA,
                    Ac09jConstants.EMPTY_STRING);
            tituloFechaPrediccion3 = tituloFechaPrediccion2.replace(Ac09jConstants.MES_ANYIO,
                    Ac09jConstants.EMPTY_STRING);
        }

        // Convert date representation to a date value

        Date dateFecha = null;
        try {
            tituloFechaPrediccion1 = tituloFechaPrediccion.replace(Ac09jConstants.DIA,
                    seguimiento.getFecha().substring(0, 2));
            dateFecha = sdf.parse(seguimiento.getFecha());
            Calendar cal = Calendar.getInstance();
            cal.setTime(dateFecha);
            if (LANG_CASTELLANO.equals(locale.getLanguage())) {
                tituloFechaPrediccion2 = tituloFechaPrediccion1.replace(Ac09jConstants.DIA_SEMANA,
                        Ac09jConstants.DIA_SEMANA_ES[cal.get(Calendar.DAY_OF_WEEK) - 1]);
                tituloFechaPrediccion3 = tituloFechaPrediccion2.replace(Ac09jConstants.MES_ANYIO,
                        Ac09jConstants.MES_ES[cal.get(Calendar.MONTH)] + " " + cal.get(Calendar.YEAR));
            } else if (LANG_EUSKERA.equals(locale.getLanguage())) {
                tituloFechaPrediccion2 = tituloFechaPrediccion1.replace(Ac09jConstants.DIA_SEMANA,
                        Ac09jConstants.DIA_SEMANA_EU[cal.get(Calendar.DAY_OF_WEEK) - 1]);
                tituloFechaPrediccion3 = tituloFechaPrediccion2.replace(Ac09jConstants.MES_ANYIO,
                        Ac09jConstants.MES_EU[cal.get(Calendar.MONTH)] + " " + cal.get(Calendar.YEAR));
            }
        } catch (ParseException e) {
            this.errorControlado = true;
            erroresControladosEs.put(e.getMessage(), e.getMessage());
            erroresControladosEu.put(e.getMessage(), e.getMessage());
            tituloFechaPrediccion1 = tituloFechaPrediccion.replace(Ac09jConstants.DIA, Ac09jConstants.EMPTY_STRING);
            tituloFechaPrediccion2 = tituloFechaPrediccion1.replace(Ac09jConstants.DIA_SEMANA,
                    Ac09jConstants.EMPTY_STRING);
            tituloFechaPrediccion3 = tituloFechaPrediccion2.replace(Ac09jConstants.MES_ANYIO,
                    Ac09jConstants.EMPTY_STRING);
        }

        return tituloFechaPrediccion3;
    }

    /**
     * 
     * @param evolucion
     * @param locale
     * @return
     */
    private String getEvolucion(Evolucion evolucion, Locale locale) throws Ac09jException {
        String alertaPrediccion = leerInputStream(
                Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_LINEA_PREDICCION));
        Object[] params = new Object[2];

        // Si el id meteoro es la temperatura, debemos mirar la zona impacto
        // para saber si es Temp. alta o mínima
        alertaPrediccion = getEvolucionTemperatura(evolucion, alertaPrediccion);
        alertaPrediccion = getEvolucionColor(evolucion, locale, alertaPrediccion, params);
        alertaPrediccion = getEvolucionMeteoroIdioma(evolucion, locale, alertaPrediccion, params);
        alertaPrediccion = getEvolucionTitulo(evolucion, locale, alertaPrediccion, params);

        StringBuilder datos = new StringBuilder();
        String zonaImpactoIdioma = Ac09jConstants.EMPTY_STRING;
        String desde = Ac09jConstants.EMPTY_STRING;
        String hasta = Ac09jConstants.EMPTY_STRING;
        if (!Ac09jUtils.isEmptyOrNull(getZonaImpactoIdioma(locale).get(evolucion.getZonaImpacto().toUpperCase()))) {
            zonaImpactoIdioma = getZonaImpactoIdioma(locale).get(evolucion.getZonaImpacto().toUpperCase());
            Object[] paramZonaImpacto = new Object[] { zonaImpactoIdioma };
            datos.append(Ac09jConstants.NEGRITA
                    + this.appMessageSource.getMessage("comun.datosMensaje", paramZonaImpacto, locale) + "."
                    + Ac09jConstants.NEGRITA_CIERRE + Ac09jConstants.ESPACIO_HTML);
        }

        if (!Ac09jUtils.isEmptyOrNull(evolucion.getDesde())) {
            desde = evolucion.getDesde();
            Object[] paramDesde = new Object[] { desde };
            datos.append(Ac09jConstants.NEGRITA
                    + this.appMessageSource.getMessage("comun.datosMensajeDesde", paramDesde, locale)
                    + Ac09jConstants.NEGRITA_CIERRE + Ac09jConstants.ESPACIO_HTML);
        }
        if (!Ac09jUtils.isEmptyOrNull(evolucion.getHasta())) {
            hasta = evolucion.getHasta();
            Object[] paramHasta = new Object[] { hasta };
            datos.append(Ac09jConstants.NEGRITA
                    + this.appMessageSource.getMessage("comun.datosMensajeHasta", paramHasta, locale)
                    + Ac09jConstants.NEGRITA_CIERRE + Ac09jConstants.ESPACIO_HTML);
        }
        if (!Ac09jUtils.isEmptyOrNull(evolucion.getDesde()) || !Ac09jUtils.isEmptyOrNull(evolucion.getHasta())) {
            datos.append(
                    Ac09jConstants.NEGRITA + this.appMessageSource.getMessage("comun.datosMensajeHora", null, locale)
                            + Ac09jConstants.NEGRITA_CIERRE + Ac09jConstants.ESPACIO_HTML);

        }
        if (!Ac09jUtils.isEmptyOrNull(evolucion.getDatosEs()) && !Ac09jUtils.isEmptyOrNull(evolucion.getDatosEu())) {
            if (locale.getLanguage().equals(LANG_EUSKERA)) {
                datos.append(evolucion.getDatosEu());
            } else {
                datos.append(evolucion.getDatosEs());
            }

        } else {
            this.errorControlado = true;
            erroresControladosEu.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_DATOS_PREDICCIONES, null, new Locale(LANG_EUSKERA)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_DATOS_PREDICCIONES, null, new Locale(LANG_EUSKERA)));
            erroresControladosEs.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_DATOS_PREDICCIONES, null,
                            new Locale(LANG_CASTELLANO)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_DATOS_PREDICCIONES, null,
                            new Locale(LANG_CASTELLANO)));

        }

        alertaPrediccion = alertaPrediccion.replace(Ac09jConstants.DATOS_ALERTA, datos.toString());
        return alertaPrediccion;
    }

    private String getEvolucionTitulo(Evolucion evolucion, Locale locale, String alertaPrediccion, Object[] params) {
        String tituloAlerta;
        String alertaPrediccionAux = alertaPrediccion;

        if (Ac09jUtils.isEmptyOrNull(evolucion.getColor())
                || Ac09jConstants.AMARILLO.equalsIgnoreCase(evolucion.getColor())) {
            tituloAlerta = this.appMessageSource.getMessage("comun.aviso", params, locale);
        } else if (Ac09jConstants.NARANJA.equalsIgnoreCase(evolucion.getColor())) {
            tituloAlerta = this.appMessageSource.getMessage("comun.alerta", params, locale);
        } else {
            tituloAlerta = this.appMessageSource.getMessage("comun.alarma", params, locale);
        }
        
        if(StringUtils.isNotEmpty(evolucion.getFin_aviso())
                && Ac09jConstants.FIN_AVISO_SI.equalsIgnoreCase(evolucion.getFin_aviso())) {
            tituloAlerta += "&nbsp;"+this.appMessageSource.getMessage("comun.finalizado", params, locale);
        }
                
        alertaPrediccionAux = alertaPrediccion.replace(Ac09jConstants.TITULO_ALERTA, tituloAlerta);
        return alertaPrediccionAux;
    }

    private String getEvolucionMeteoroIdioma(Evolucion evolucion, Locale locale, String alertaPrediccion,
            Object[] params) {
        String alertaPrediccionAux = alertaPrediccion;
        Object[] paramsErrorIdMeteoro;
        if (!Ac09jUtils.isEmptyOrNull(evolucion.getIdMeteoro())
                && !Ac09jUtils.isEmptyOrNull(getIdMeteoroIdioma(locale).get(evolucion.getIdMeteoro().toUpperCase()))) {
            params[1] = getIdMeteoroIdioma(locale).get(evolucion.getIdMeteoro().toUpperCase()).toUpperCase();
        } else {
            params[1] = evolucion.getIdMeteoro().toUpperCase();
            paramsErrorIdMeteoro = new Object[] { evolucion.getIdMeteoro() };
            alertaPrediccionAux = alertaPrediccion.replace(Ac09jConstants.TITULO_ALERTA, Ac09jConstants.EMPTY_STRING);
            errorControlado = true;
            erroresControladosEu.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_ID_METEORO, paramsErrorIdMeteoro,
                            new Locale(LANG_EUSKERA)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_ID_METEORO, paramsErrorIdMeteoro,
                            new Locale(LANG_EUSKERA)));
            erroresControladosEs.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_ID_METEORO, paramsErrorIdMeteoro,
                            new Locale(LANG_CASTELLANO)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_ID_METEORO, paramsErrorIdMeteoro,
                            new Locale(LANG_CASTELLANO)));

        }
        return alertaPrediccionAux;
    }

    private String getEvolucionColor(Evolucion evolucion, Locale locale, String alertaPrediccion, Object[] params) {
        Object[] paramsErrorColor;
        String alertaPrediccionAux = alertaPrediccion;

        final Boolean isFinAviso = StringUtils.isNotEmpty(evolucion.getFin_aviso())
                && Ac09jConstants.FIN_AVISO_SI.equalsIgnoreCase(evolucion.getFin_aviso());

        if (StringUtils.isNotEmpty(evolucion.getFin_aviso())
                && !Ac09jConstants.FIN_AVISO_SI.equalsIgnoreCase(evolucion.getFin_aviso())
                && !Ac09jConstants.FIN_AVISO_NO.equalsIgnoreCase(evolucion.getFin_aviso())) {
            paramsErrorColor = new Object[] { evolucion.getFin_aviso() };
            erroresControladosEs.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_FIN, paramsErrorColor,
                            new Locale(LANG_CASTELLANO)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_FIN, paramsErrorColor,
                            new Locale(LANG_CASTELLANO)));
            erroresControladosEu.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_FIN, paramsErrorColor,
                            new Locale(LANG_EUSKERA)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_FIN, paramsErrorColor,
                            new Locale(LANG_EUSKERA)));
        }

        if (!Ac09jUtils.isEmptyOrNull(evolucion.getColor())
                && !Ac09jUtils.isEmptyOrNull(getMapColorIdioma(locale).get(evolucion.getColor().toUpperCase()))) {
            params[0] = getMapColorIdioma(locale).get(evolucion.getColor().toUpperCase()).toUpperCase();
            alertaPrediccionAux = alertaPrediccion.replace(Ac09jConstants.COLOR,
                    isFinAviso ? Ac09jConstants.VERDE.toLowerCase() : evolucion.getColor());
        } else {
            params[0] = evolucion.getColor().toUpperCase();
            paramsErrorColor = new Object[] { evolucion.getColor() };
            alertaPrediccionAux = alertaPrediccion.replace(Ac09jConstants.COLOR, Ac09jConstants.EMPTY_STRING);
            errorControlado = true;
            erroresControladosEs.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_COLOR, paramsErrorColor,
                            new Locale(LANG_CASTELLANO)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_COLOR, paramsErrorColor,
                            new Locale(LANG_CASTELLANO)));
            erroresControladosEu.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_COLOR, paramsErrorColor,
                            new Locale(LANG_EUSKERA)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_LITERAL_COLOR, paramsErrorColor,
                            new Locale(LANG_EUSKERA)));
        }
        return alertaPrediccionAux;
    }

    private String getEvolucionTemperatura(Evolucion evolucion, String alertaPrediccion) {
        String alertaPrediccionAux = alertaPrediccion;

        if (Ac09jConstants.TEMP.equalsIgnoreCase(evolucion.getIdMeteoro())
                && (Ac09jConstants.ALTAS_PERSISTENTES.equalsIgnoreCase(evolucion.getZonaImpacto())
                        || Ac09jConstants.MUY_ALTAS.equalsIgnoreCase(evolucion.getZonaImpacto()))) {
            alertaPrediccionAux = alertaPrediccion.replace(Ac09jConstants.SRC_IMG_ALERTA, getUrImages()
                    + Ac09jConstants.TEMP_ALTAS.toLowerCase() + Ac09jConstants.WHITE + Ac09jConstants.EXTENSION_PNG);
        } else if (Ac09jConstants.TEMP.equalsIgnoreCase(evolucion.getIdMeteoro())
                && Ac09jConstants.MINIMAS_HELADAS.equalsIgnoreCase(evolucion.getZonaImpacto())) {
            alertaPrediccionAux = alertaPrediccion.replace(Ac09jConstants.SRC_IMG_ALERTA, getUrImages()
                    + Ac09jConstants.TEMP_MIN.toLowerCase() + Ac09jConstants.WHITE + Ac09jConstants.EXTENSION_PNG);
        } else {
            alertaPrediccionAux = alertaPrediccion.replace(Ac09jConstants.SRC_IMG_ALERTA, getUrImages()
                    + evolucion.getIdMeteoro().toLowerCase() + Ac09jConstants.WHITE + Ac09jConstants.EXTENSION_PNG);
        }
        return alertaPrediccionAux;
    }

    /**
     * 
     * @param seguimientos
     * @param locale
     * @return
     */
    private String getCausasAlertas(List<Afectado> afectados, Locale locale) throws Ac09jException {

        String causasAlertas = leerInputStream(
                Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_CAUSAS_ALERTAS));
        causasAlertas = causasAlertas.replace(Ac09jConstants.TITULO_CAUSAS_ALERTAS,
                this.appMessageSource.getMessage("mensaje.tituloCausasAlertas", null, locale));

        if (afectados == null) {
            this.errorControlado = true;
            erroresControladosEu.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_SEGUIMIENTOS, null, new Locale(LANG_EUSKERA)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_SEGUIMIENTOS, null, new Locale(LANG_EUSKERA)));
            erroresControladosEs.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_SEGUIMIENTOS, null, new Locale(LANG_CASTELLANO)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_SEGUIMIENTOS, null, new Locale(LANG_CASTELLANO)));
            return Ac09jConstants.EMPTY_STRING;
        }

        StringBuilder lineasCausasAlertas1 = new StringBuilder();
        StringBuilder lineasCausasAlertas2 = new StringBuilder();

        Map<String, String> lineasCausasAlertasMap = new HashMap<String, String>();

        for (int i = 0; i < afectados.size(); i++) {
            String linea = getDatosEvolucion(afectados.get(i), locale);
            // Ponemos las líneas en un mapa para evitar duplicados
            lineasCausasAlertasMap.put(linea, linea);
        }

        int lineasPrimeraColumna = lineasCausasAlertasMap.size() / 2 + lineasCausasAlertasMap.size() % 2;

        int j = 0;
        for (String linea : lineasCausasAlertasMap.keySet()) {
            if (j < lineasPrimeraColumna) {
                lineasCausasAlertas1.append(linea);
            } else if (j >= lineasPrimeraColumna) {
                lineasCausasAlertas2.append(linea);
            }
            j++;
        }

        causasAlertas = causasAlertas.replace(Ac09jConstants.LINEAS_CAUSAS_ALERTAS_1, lineasCausasAlertas1.toString());
        causasAlertas = causasAlertas.replace(Ac09jConstants.LINEAS_CAUSAS_ALERTAS_2, lineasCausasAlertas2.toString());

        return causasAlertas;

    }

    /**
     * 
     * @param afectado
     * @param locale
     * @return
     */
    private String getDatosEvolucion(Afectado afectado, Locale locale) throws Ac09jException {

        if (afectado != null) {
            String linea = leerInputStream(
                    Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_LINEA_CAUSAS_ALERTAS));
            // Si el id meteoro es la temperatura, debemos mirar la zona impacto
            // para saber si es Temp. alta o mínima
            if (!Ac09jUtils.isEmptyOrNull(afectado.getIdMeteoro())
                    && !Ac09jUtils.isEmptyOrNull(afectado.getZonaImpacto())) {
                return getStringEvolucion(afectado, linea, locale);
            } else {
                this.errorControlado = true;
                Object[] params = new Object[] { afectado.getIdMeteoro(), afectado.getZonaImpacto() };
                erroresControladosEu.put(
                        this.appMessageSource.getMessage(MENSAJE_ERROR_ID_ZONA, params, new Locale(LANG_EUSKERA)),
                        this.appMessageSource.getMessage(MENSAJE_ERROR_ID_ZONA, params, new Locale(LANG_EUSKERA)));
                erroresControladosEs.put(
                        this.appMessageSource.getMessage(MENSAJE_ERROR_ID_ZONA, params, new Locale(LANG_CASTELLANO)),
                        this.appMessageSource.getMessage(MENSAJE_ERROR_ID_ZONA, params, new Locale(LANG_CASTELLANO)));
                return Ac09jConstants.EMPTY_STRING;
            }
        }
        this.errorControlado = true;
        erroresControladosEu.put(
                this.appMessageSource.getMessage(MENSAJE_ERROR_EVOLUCION, null, new Locale(LANG_EUSKERA)),
                this.appMessageSource.getMessage(MENSAJE_ERROR_EVOLUCION, null, new Locale(LANG_EUSKERA)));
        erroresControladosEs.put(
                this.appMessageSource.getMessage(MENSAJE_ERROR_EVOLUCION, null, new Locale(LANG_CASTELLANO)),
                this.appMessageSource.getMessage(MENSAJE_ERROR_EVOLUCION, null, new Locale(LANG_CASTELLANO)));
        return Ac09jConstants.EMPTY_STRING;
    }

    private String getStringEvolucion(Afectado afectado, String linea, Locale locale) throws Ac09jException {

        String idMeteoroIdioma = getIdMeteoroIdioma(locale).get(afectado.getIdMeteoro().toUpperCase());
        String linea1 = linea.replace(Ac09jConstants.ID_METEORO, idMeteoroIdioma);
        String linea2;
        String linea3 = Ac09jConstants.EMPTY_STRING;

        String zonaImpactoIdioma = getZonaImpactoIdioma(locale).get(afectado.getZonaImpacto().toUpperCase());
        if (!Ac09jUtils.isEmptyOrNull(zonaImpactoIdioma)) {
            linea2 = linea1.replace(Ac09jConstants.ZONA_IMPACTO, zonaImpactoIdioma);
        } else {
            this.errorControlado = true;
            Object[] params = new Object[] { afectado.getIdMeteoro(), afectado.getZonaImpacto() };
            erroresControladosEu.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_ID_ZONA, params, new Locale(LANG_EUSKERA)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_ID_ZONA, params, new Locale(LANG_EUSKERA)));
            erroresControladosEs.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_ID_ZONA, params, new Locale(LANG_CASTELLANO)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_ID_ZONA, params, new Locale(LANG_CASTELLANO)));
            linea2 = linea1.replace(Ac09jConstants.ZONA_IMPACTO, Ac09jConstants.EMPTY_STRING);
        }
        if (Ac09jConstants.TEMP.equalsIgnoreCase(afectado.getIdMeteoro())
                && (Ac09jConstants.ALTAS_PERSISTENTES.equalsIgnoreCase(afectado.getZonaImpacto())
                        || Ac09jConstants.MUY_ALTAS.equalsIgnoreCase(afectado.getZonaImpacto()))) {
            linea3 = linea2.replace(Ac09jConstants.IMG_ALERTA, getUrImages() + Ac09jConstants.TEMP_ALTAS.toLowerCase()
                    + Ac09jConstants.BACK_BLUE + Ac09jConstants.EXTENSION_PNG);

        } else if (Ac09jConstants.TEMP.equalsIgnoreCase(afectado.getIdMeteoro())
                && Ac09jConstants.MINIMAS_HELADAS.equalsIgnoreCase(afectado.getZonaImpacto())) {
            linea3 = linea2.replace(Ac09jConstants.IMG_ALERTA, getUrImages() + Ac09jConstants.TEMP_MIN.toLowerCase()
                    + Ac09jConstants.BACK_BLUE + Ac09jConstants.EXTENSION_PNG);

        } else {
            linea3 = linea2.replace(Ac09jConstants.IMG_ALERTA, getUrImages() + afectado.getIdMeteoro().toLowerCase()
                    + Ac09jConstants.BACK_BLUE + Ac09jConstants.EXTENSION_PNG);

        }

        return linea3;
    }

    private String getLiteralFechaCreacion(String fechaCreacion, Locale locale) throws Ac09jException {

        if (fechaCreacion.length() < 15 || Integer.parseInt(fechaCreacion.substring(0, 2)) > Ac09jConstants.DIAS_MES
                || Integer.parseInt(fechaCreacion.substring(3, 5)) > Ac09jConstants.MESES_ANYO) {
            this.errorControlado = true;
            erroresControladosEs.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_CREACION, null, new Locale(LANG_CASTELLANO)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_CREACION, null, new Locale(LANG_CASTELLANO)));
            erroresControladosEu.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_CREACION, null, new Locale(LANG_EUSKERA)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_CREACION, null, new Locale(LANG_EUSKERA)));
            return Ac09jConstants.EMPTY_STRING;
        }
        String dia = fechaCreacion.substring(0, 2);
        String mes = fechaCreacion.substring(3, 5);
        String hora = fechaCreacion.substring(11, fechaCreacion.length());

        if (LANG_CASTELLANO.equals(locale.getLanguage())) {
            Object[] paramsEs = new Object[] { dia, Ac09jConstants.MES_ES[Integer.parseInt(mes) - 1], hora };
            return this.appMessageSource.getMessage("mensaje.fechaCreacion", paramsEs, new Locale(LANG_CASTELLANO));
        } else {
            Object[] paramsEu = new Object[] { dia, Ac09jConstants.MES_EU[Integer.parseInt(mes) - 1], hora };

            return this.appMessageSource.getMessage("mensaje.fechaCreacion", paramsEu, new Locale(LANG_EUSKERA));
        }

    }

    private String getLiteralFechaActualizacion(String fechaActualizacionProgramada, Locale locale)
            throws Ac09jException {

        if (fechaActualizacionProgramada.length() < 15
                || Integer.parseInt(fechaActualizacionProgramada.substring(0, 2)) > Ac09jConstants.DIAS_MES
                || Integer.parseInt(fechaActualizacionProgramada.substring(3, 5)) > Ac09jConstants.MESES_ANYO) {
            this.errorControlado = true;
            erroresControladosEs.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_ACTUALIZACION, null,
                            new Locale(LANG_CASTELLANO)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_ACTUALIZACION, null,
                            new Locale(LANG_CASTELLANO)));
            erroresControladosEu.put(
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_ACTUALIZACION, null, new Locale(LANG_EUSKERA)),
                    this.appMessageSource.getMessage(MENSAJE_ERROR_FECHA_ACTUALIZACION, null,
                            new Locale(LANG_EUSKERA)));
            return Ac09jConstants.EMPTY_STRING;
        }
        String fechaCompletaEs = fechaActualizacionProgramada.substring(0, 10);
        String hora = fechaActualizacionProgramada.substring(11, fechaActualizacionProgramada.length());

        if (LANG_CASTELLANO.equals(locale.getLanguage())) {
            return this.appMessageSource.getMessage("mensaje.fechaActualizacion",
                    new Object[] { hora, fechaCompletaEs }, new Locale(LANG_CASTELLANO));
        } else {
            String[] arrFecha = fechaCompletaEs.split("/");
            String fechaCompletaEu = arrFecha[2] + "/" + arrFecha[1] + "/" + arrFecha[0];

            return this.appMessageSource.getMessage("mensaje.fechaActualizacion",
                    new Object[] { hora, fechaCompletaEu }, new Locale(LANG_EUSKERA));
        }

    }

    private String getTitulo(String varTitulo) throws Ac09jException {
        String titulo = leerInputStream(Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_TITULO));
        titulo = titulo.replace(Ac09jConstants.TITULO_PRINCIPAL, varTitulo);
        return titulo;
    }

    /**
     * 
     * @return String codLogos
     */
    private String getLogos() throws Ac09jException {
        String logos = leerInputStream(Ac09jMensajeServiceImpl.class.getResourceAsStream(Ac09jConstants.RUTA_LOGOS_GV));
        logos = logos.replace(Ac09jConstants.SRC_LOGO_GV,
                getUrImages() + Ac09jConstants.LOGO_GV + Ac09jConstants.EXTENSION_JPJ);
        return logos;
    }

    private String leerInputStream(InputStream inputStream) throws Ac09jException {
        Charset utf8 = Charset.forName("UTF-8");
        BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inputStream, utf8));

        String str;
        StringBuilder htmlString = new StringBuilder();

        try {
            while ((str = bufferedReader.readLine()) != null) {
                htmlString.append(str);
            }
        } catch (IOException e) {
            Ac09jMensajeServiceImpl.LOGGER.error(e.getLocalizedMessage(), e);
            errorControlado = true;
            erroresControladosEs.put(e.getMessage(), e.getMessage());
            erroresControladosEu.put(e.getMessage(), e.getMessage());
            return Ac09jConstants.EMPTY_STRING;
        }

        return htmlString.toString();
    }

    /**
     * 
     * @param locale
     * @return
     */
    private Map<String, String> getMapColorIdioma(Locale locale) {
        Map<String, String> mapColores = new HashMap<String, String>();
        mapColores.put(Ac09jConstants.AMARILLO, this.appMessageSource.getMessage("comun.amarillo", null, locale));
        mapColores.put(Ac09jConstants.ROJO, this.appMessageSource.getMessage("comun.rojo", null, locale));
        mapColores.put(Ac09jConstants.NARANJA, this.appMessageSource.getMessage("comun.naranja", null, locale));
        return mapColores;
    }

    /**
     * 
     * @param locale
     * @return
     */
    private Map<String, String> getZonaImpactoIdioma(Locale locale) {
        Map<String, String> mapZonas = new HashMap<String, String>();

        mapZonas.put(Ac09jConstants.NAVEGACION, this.appMessageSource.getMessage("comun.navegacion", null, locale));
        mapZonas.put(Ac09jConstants.IMPACTO_COSTA,
                this.appMessageSource.getMessage("comun.impactoCosta", null, locale));
        mapZonas.put(Ac09jConstants.GALERNA, this.appMessageSource.getMessage("comun.galerna", null, locale));
        mapZonas.put(Ac09jConstants.ALTURA_OLA, this.appMessageSource.getMessage("comun.alturaOla", null, locale));
        mapZonas.put(Ac09jConstants.AGUA_SALPICADURAS,
                this.appMessageSource.getMessage("comun.aguaSalpicaduras", null, locale));
        mapZonas.put(Ac09jConstants.ZONAS_EXPUESTAS,
                this.appMessageSource.getMessage("comun.zonasExpuestas", null, locale));
        mapZonas.put(Ac09jConstants.ZONAS_NO_EXPUESTAS,
                this.appMessageSource.getMessage("comun.zonasNoExpuestas", null, locale));
        mapZonas.put(Ac09jConstants.INTENSAS, this.appMessageSource.getMessage("comun.intensas", null, locale));
        mapZonas.put(Ac09jConstants.PERSISTENTES, this.appMessageSource.getMessage("comun.persistentes", null, locale));
        mapZonas.put(Ac09jConstants.DEBAJO_MIL, this.appMessageSource.getMessage("comun.debajoMil", null, locale));
        mapZonas.put(Ac09jConstants.ALTAS_PERSISTENTES,
                this.appMessageSource.getMessage("comun.altasPersistentes", null, locale));
        mapZonas.put(Ac09jConstants.MUY_ALTAS, this.appMessageSource.getMessage("comun.muyAltas", null, locale));
        mapZonas.put(Ac09jConstants.MINIMAS_HELADAS,
                this.appMessageSource.getMessage("comun.minimasHeladas", null, locale));
        mapZonas.put(Ac09jConstants.INCENDIO_FORESTAL,
                this.appMessageSource.getMessage("comun.incendioForestal", null, locale));
        mapZonas.put(Ac09jConstants.OTROS, this.appMessageSource.getMessage("comun.otros", null, locale));
        return mapZonas;
    }

    /**
     * 
     * @param locale
     * @return
     */
    private Map<String, String> getIdMeteoroIdioma(Locale locale) {
        Map<String, String> mapAvisos = new HashMap<String, String>();
        mapAvisos.put(Ac09jConstants.MAR, this.appMessageSource.getMessage("comun.mar", null, locale));
        mapAvisos.put(Ac09jConstants.TEMP, this.appMessageSource.getMessage("comun.temperaturas", null, locale));
        mapAvisos.put(Ac09jConstants.NIEVE, this.appMessageSource.getMessage("comun.nieve", null, locale));
        mapAvisos.put(Ac09jConstants.LLUVIA, this.appMessageSource.getMessage("comun.lluvia", null, locale));
        mapAvisos.put(Ac09jConstants.VIENTO, this.appMessageSource.getMessage("comun.viento", null, locale));
        mapAvisos.put(Ac09jConstants.ESPECIAL, this.appMessageSource.getMessage("comun.especial", null, locale));

        return mapAvisos;
    }

}
