package com.ejie.y41b.control;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.servlet.ModelAndView;

import com.ejie.x38.control.exception.MethodFailureException;
import com.ejie.x38.control.exception.ResourceNotFoundException;
import com.ejie.x38.dto.JQGridJSONModel;
import com.ejie.x38.dto.Pagination;
import com.ejie.x38.util.ObjectConversionManager;
import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.model.CensoMonitor;
import com.ejie.y41b.model.Tribunal;
import com.ejie.y41b.service.TribunalService;
import com.ejie.y41b.utils.Y41bUIDGenerator;
import com.ejie.y41b.utils.exception.Y41bUDAErrorManager;
import com.ejie.y41b.utils.exception.Y41bUDAException;

/**
 * TribunalController  
 * 
 *  
 */

@Controller
@RequestMapping(value = "/tribunal")
public class TribunalController {

	private static final Logger logger = LoggerFactory
			.getLogger(TribunalController.class);

	@Autowired
	private TribunalService tribunalService;
	@Autowired
	private Properties appConfiguration;

	@Autowired
	private Y41bUDAErrorManager errorManager;

	/**
	 * Method 'udaExceptionHandle'.
	 * 
	 * @param e
	 *            Y41bUDAException
	 * @param request
	 *            HttpServletRequest
	 * @param response
	 *            HttpServletResponse
	 * @return
	 */
	@ExceptionHandler(Y41bUDAException.class)
	public @ResponseBody
	void udaExceptionHandle(Y41bUDAException e, HttpServletRequest request,
			HttpServletResponse response) {

		response.setStatus(HttpServletResponse.SC_CONFLICT);
		Writer w = null;

		try {
			w = response.getWriter();
			w.write(errorManager.process(e, request));
			w.flush();
		} catch (IOException e1) {
			throw new ResourceNotFoundException("No data Found.");
		}
	}

	/*
	 * --------------- Tratamiento de errores comun para los proyectos UDA.
	 * --------------------------------------------------------------------
	 */

	/**
	 * Method 'getCreateFormMantenimiento'.
	 * 
	 * @param model
	 *            Model
	 * @param request
	 *            HttpServletRequest
	 * @return String
	 */
	@RequestMapping(value = "tribunales", method = RequestMethod.GET)
	public ModelAndView getCreateFormMantenimiento(Model model,
			HttpServletRequest request) {
		model.addAttribute("defaultLanguage",
				appConfiguration.get("y41bVistaWar.default.language"));
		model.addAttribute("defaultLayout",
				appConfiguration.get("y41bVistaWar.default.layout"));

		HttpServletRequest httpRequest = (HttpServletRequest) request;
		HttpSession httpSession = httpRequest.getSession(false);
		CensoMonitor censoMonitor = (CensoMonitor) httpSession
				.getAttribute(Y41bConstantes.SESSIONCENSOMONITORCONNECTED);

		if (censoMonitor.getPerfilMaestros() == null
				|| censoMonitor.getPerfilMaestros().getIdPerfil()
						.equals(Y41bConstantes.PERFIL_SINACCESO_MANTENIMIENTO)) {
			return new ModelAndView("accessDeniedEscritorio", "model", model);
		} else {
			return new ModelAndView("tribunales", "model", model);
		}
	}

	/**
	 * Method 'getById'.
	 * 
	 * @param tbcodigo
	 *            String
	 * @return tribunal Tribunal
	 */
	@RequestMapping(value = "findTribunal/{tbcodigo}", method = RequestMethod.GET)
	public @ResponseBody
	Tribunal getById(@PathVariable String tbcodigo) {
		try {
			Tribunal tribunal = new Tribunal();
			tribunal.setTbcodigo(tbcodigo);
			tribunal = this.tribunalService.find(tribunal);
			if (tribunal == null) {
				throw new Exception(tbcodigo.toString());
			}

			return tribunal;
		} catch (Exception e) {
			throw new ResourceNotFoundException(tbcodigo.toString());
		}
	}

	/**
	 * Method 'getAll'.
	 * 
	 * @param tbcodigo_
	 *            String
	 * @param tbnombre_
	 *            String
	 * @param tbnombree_
	 *            String
	 * @param request
	 *            HttpServletRequest
	 * @return List
	 */
	@RequestMapping(value = "/like", method = RequestMethod.GET)
	public @ResponseBody
	Object getAll(
			@RequestParam(value = "tbcodigo", required = false) String tbcodigo_,
			@RequestParam(value = "tbnombre", required = false) String tbnombre_,
			@RequestParam(value = "tbnombree", required = false) String tbnombree_,
			HttpServletRequest request) {
		try {
			// control de los campos susceptibles de llegar a "" desde el filtro
			String tbcodigo = ("".equals(tbcodigo_)) ? null : tbcodigo_;
			String tbnombre = null;
			String tbnombree = null;

			Locale locale = LocaleContextHolder.getLocale();
			String idioma = locale.getLanguage();

			if (Y41bConstantes.CASTELLANO.equalsIgnoreCase(idioma)) {
				tbnombre = ("".equals(tbnombre_)) ? null : tbnombre_;
			} else {
				tbnombree = ("".equals(tbnombre_)) ? null : tbnombre_;
			}

			Tribunal filterTribunal = new Tribunal(tbcodigo, tbnombre,
					tbnombree);
			Pagination pagination = null;
			if (request.getHeader("JQGridModel") != null
					&& request.getHeader("JQGridModel").equals("true")) {
				pagination = new Pagination();
				pagination.setPage(Long.valueOf(request.getParameter("page")));
				pagination.setRows(Long.valueOf(request.getParameter("rows")));
				pagination.setSort(request.getParameter("sidx"));
				pagination.setAscDsc(request.getParameter("sord"));
				List<Tribunal> tribunales = this.tribunalService.findAllLike(
						filterTribunal, pagination, false);

				if (tribunales == null) {
					throw new Exception("No data Found.");
				}

				Long total = this.tribunalService.findAllCount(filterTribunal);
				JQGridJSONModel data = new JQGridJSONModel();
				data.setPage(request.getParameter("page"));
				data.setRecords(total.intValue());
				data.setTotal(total, pagination.getRows());
				data.setRows(tribunales);
				return data;
			} else {
				List<Tribunal> tribunales = this.tribunalService.findAllLike(
						filterTribunal, pagination, false);
				if (tribunales == null) {
					throw new Exception("No data Found.");
				}
				return tribunales;
			}
		} catch (Exception e) {
			throw new ResourceNotFoundException("No data Found.");
		}
	}

	/**
	 * Method 'getCreateFormAltaTribunal'.
	 * 
	 * @param model
	 *            Model
	 * @param request
	 *            HttpServletRequest
	 * @return String
	 */
	@RequestMapping(value = "altaTribunal", method = RequestMethod.GET)
	public ModelAndView getCreateFormAltaReclamacion(Model model,
			HttpServletRequest request) {
		model.addAttribute("defaultLanguage",
				appConfiguration.get("y41bVistaWar.default.language"));
		model.addAttribute("defaultLayout",
				appConfiguration.get("y41bVistaWar.default.layout"));

		model.addAttribute("codTribunal", Y41bUIDGenerator.getInstance()
				.generateId(Y41bConstantes.PK_SIZE));

		model.addAttribute("modoTribunal", Y41bConstantes.MODO_POPUP_ALTA);

		HttpServletRequest httpRequest = (HttpServletRequest) request;
		HttpSession httpSession = httpRequest.getSession(false);
		CensoMonitor censoMonitor = (CensoMonitor) httpSession
				.getAttribute(Y41bConstantes.SESSIONCENSOMONITORCONNECTED);

		if (censoMonitor.getPerfilMaestros() == null
				|| censoMonitor.getPerfilMaestros().getIdPerfil()
						.equals(Y41bConstantes.PERFIL_SINACCESO_MANTENIMIENTO)) {
			return new ModelAndView("accessDeniedModal", "model", model);
		} else {
			return new ModelAndView("detalleTribunal", "model", model);
		}
	}

	/**
	 * Method 'edit'.
	 * 
	 * @param tribunal
	 *            Tribunal
	 * @return Tribunal
	 */
	@RequestMapping(value = "/updateTribunal", method = RequestMethod.PUT)
	public @ResponseBody
	Tribunal edit(@RequestBody Tribunal tribunal) {
		try {
			Tribunal tribunalAux = this.tribunalService.update(tribunal);
			TribunalController.logger
					.info("[PUT] : Tribunal actualizado correctamente");
			return tribunalAux;
		} catch (Exception e) {
			throw new MethodFailureException("Method failed");
		}
	}

	/**
	 * Method 'add'.
	 * 
	 * @param tribunal
	 *            Tribunal
	 * @return Tribunal
	 */
	@RequestMapping(method = RequestMethod.POST)
	public @ResponseBody
	Tribunal add(@RequestBody Tribunal tribunal) {
		try {
			Tribunal tribunalAux = this.tribunalService.add(tribunal);
			TribunalController.logger
					.info("[POST] : Tribunal insertado correctamente");
			return tribunalAux;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e) {
			throw new MethodFailureException("Method failed");
		}
	}

	/**
	 * Method 'remove'.
	 * 
	 * @param tbcodigo
	 *            String
	 * @param response
	 *            HttpServletResponse
	 * @return
	 */
	@RequestMapping(value = "remove/{tbcodigo}", method = RequestMethod.DELETE)
	public void remove(@PathVariable String tbcodigo,
			HttpServletResponse response) {
		response.setContentType("text/javascript;charset=utf-8");
		response.setHeader("Pragma", "cache");
		response.setHeader("Expires", "0");
		response.setHeader("Cache-Control", "private");

		try {
			Tribunal tribunal = new Tribunal();
			tribunal.setTbcodigo(tbcodigo);
			this.tribunalService.remove(tribunal);
			response.setStatus(HttpServletResponse.SC_OK);
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e) {
			this.logger.error("Unable to delete " + tbcodigo);
			response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
			throw new MethodFailureException("Method failed");
		}

	}

	/**
	 * Method 'removeAll'.
	 * 
	 * @param tribunalIds
	 *            List
	 * @return tribunalList
	 */
	@RequestMapping(value = "/deleteAll", method = RequestMethod.POST)
	@ResponseStatus(value = HttpStatus.OK)
	public @ResponseBody
	List<List<String>> removeMultiple(
			@RequestBody List<List<String>> tribunalIds) {
		List<Tribunal> tribunalList = new ArrayList<Tribunal>();
		for (List<String> tribunalId : tribunalIds) {
			Iterator<String> iterator = tribunalId.iterator();
			Tribunal tribunal = new Tribunal(); // NOPMD - Objeto nuevo en la
												// lista (parametro del
												// servicio)
			tribunal.setTbcodigo(ObjectConversionManager.convert(
					iterator.next(), String.class));
			tribunalList.add(tribunal);
		}
		this.tribunalService.removeMultiple(tribunalList);
		TribunalController.logger
				.info("[POST - DELETE_ALL] : Tribunal borrados correctamente");
		return tribunalIds;
	}

	/**
	 * Method 'getAllJQGrid'.
	 * 
	 * @param filterTribunal
	 *            Tribunal
	 * @param pagination
	 *            Pagination
	 * @return JQGridJSONModel
	 */
	@RequestMapping(method = RequestMethod.GET, headers = { "JQGridModel=true" })
	public @ResponseBody
	JQGridJSONModel getAllJQGrid(@ModelAttribute Tribunal filterTribunal,
			@ModelAttribute Pagination pagination) {
		List<Tribunal> tribunals = this.tribunalService.findAll(filterTribunal,
				pagination);
		Long recordNum = this.tribunalService.findAllCount(filterTribunal);
		TribunalController.logger.info("[GET - jqGrid] : Obtener Tribunal");
		return new JQGridJSONModel(pagination, recordNum, tribunals);
	}

	/**
	 * Method 'getComboTribunal'.
	 * 
	 * @param request
	 *            HttpServletRequest
	 * @return List
	 */
	@RequestMapping(value = "/combo", method = RequestMethod.GET)
	public @ResponseBody
	List<Tribunal> getComboTribunal(HttpServletRequest request) {

		List<Tribunal> ltribunal = null;

		this.logger.info("[GET - find_ALL] : Obtener combo Tribunal");
		Tribunal filterTribunal = new Tribunal();
		/*
		 * desde la ficha de expedientes
		 */
		Locale locale = LocaleContextHolder.getLocale();
		String idioma = locale.getLanguage();
		Pagination pagination = new Pagination();
		if (Y41bConstantes.CASTELLANO.equals(idioma)) {
			pagination.setSort("tbnombre");
		} else {
			pagination.setSort("tbnombree");
		}
		pagination.setAscDsc("ASC");

		ltribunal = tribunalService.findAll(filterTribunal, null);

		return ltribunal;
	}

	/**
	 * Method 'getCreateFormModTribunal'.
	 * 
	 * @param codTribunal
	 *            String
	 * @param model
	 *            Model
	 * @param request
	 *            HttpServletRequest
	 * @return String
	 */
	@RequestMapping(value = "modTribunal/{codTribunal}", method = RequestMethod.GET)
	public ModelAndView getCreateFormModTribunal(
			@PathVariable String codTribunal, Model model,
			HttpServletRequest request) {
		model.addAttribute("defaultLanguage",
				appConfiguration.get("y41bVistaWar.default.language"));
		model.addAttribute("defaultLayout",
				appConfiguration.get("y41bVistaWar.default.layout"));

		model.addAttribute("codTribunal", codTribunal);

		model.addAttribute("modoTribunal",
				Y41bConstantes.MODO_POPUP_MODIFICACION);

		HttpServletRequest httpRequest = (HttpServletRequest) request;
		HttpSession httpSession = httpRequest.getSession(false);
		CensoMonitor censoMonitor = (CensoMonitor) httpSession
				.getAttribute(Y41bConstantes.SESSIONCENSOMONITORCONNECTED);

		Tribunal tribunal = new Tribunal();
		tribunal.setTbcodigo(codTribunal);

		if (censoMonitor.getPerfilMaestros() == null
				|| censoMonitor.getPerfilMaestros().getIdPerfil()
						.equals(Y41bConstantes.PERFIL_SINACCESO_MANTENIMIENTO)) {
			return new ModelAndView("accessDeniedModal", "model", model);
		} else {
			return new ModelAndView("detalleTribunal", "model", model);
		}
	}

}
