package com.ejie.y41b.control;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;

import com.ejie.x38.control.exception.ControlException;
import com.ejie.x38.control.exception.MethodFailureException;
import com.ejie.x38.control.exception.ResourceNotFoundException;
import com.ejie.x38.control.exception.ServiceUnavailableException;
import com.ejie.x38.dto.JQGridJSONModel;
import com.ejie.x38.dto.Pagination;
import com.ejie.x38.util.ObjectConversionManager;
import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.model.CensoMonitor;
import com.ejie.y41b.model.Inspector;
import com.ejie.y41b.service.InspectorService;

/**
 * * InspectorController  
 * 
 *  
 */

@Controller
@RequestMapping(value = "/inspector")
public class InspectorController {

	private static final Logger logger = LoggerFactory
			.getLogger(InspectorController.class);

	@Autowired
	private InspectorService inspectorService;

	@Autowired
	private Properties appConfiguration;

	/**
	 * Method 'getCreateForm'.
	 * 
	 * @param model
	 *            Model
	 * @return String
	 */
	@RequestMapping(value = "maint", method = RequestMethod.GET)
	public ModelAndView getCreateForm(Model model) {
		model.addAttribute("defaultLanguage",
				appConfiguration.get("y41bVistaWar.default.language"));
		model.addAttribute("defaultLayout",
				appConfiguration.get("y41bVistaWar.default.layout"));
		return new ModelAndView("inspector", "model", model);
	}

	/**
	 * Method 'getById'.
	 * 
	 * @param incodigo
	 *            String
	 * @return String
	 */
	@RequestMapping(value = "/{incodigo}", method = RequestMethod.GET)
	public @ResponseBody
	Inspector getById(@PathVariable String incodigo) {
		try {
			Inspector inspector = new Inspector();
			inspector.setIncodigo(incodigo);
			inspector = this.inspectorService.find(inspector);
			if (inspector == null) {
				throw new Exception(incodigo.toString());
			}
			return inspector;
		} catch (Exception e) {
			throw new ResourceNotFoundException(incodigo.toString());
		}
	}

	/**
	 * Method 'getAll'.
	 * 
	 * @param incodigo_
	 *            String
	 * @param innombre_
	 *            String
	 * @param incoor_
	 *            String
	 * @param request
	 *            HttpServletRequest
	 * @return String
	 */
	@RequestMapping(value = "inspectoresOrganismo", method = RequestMethod.GET)
	public @ResponseBody
	Object getAll(
			@RequestParam(value = "incodigo", required = false) String incodigo_,
			@RequestParam(value = "innombre", required = false) String innombre_,
			@RequestParam(value = "incoor", required = false) String incoor_,
			HttpServletRequest request) {
		try {
			CensoMonitor censoMonitor = (CensoMonitor) request.getSession()
					.getAttribute(Y41bConstantes.SESSIONCENSOMONITORCONNECTED);
			String incodigo = ("".equals(incodigo_)) ? null : incodigo_;
			String innombre = ("".equals(innombre_)) ? null : innombre_;
			String incoor = ("".equals(incoor_)) ? censoMonitor.getOrganismo()
					.getCodOrg() : incoor_;

			Inspector filterInspector = new Inspector(incodigo, innombre,
					incoor);
			Pagination pagination = null;
			if (request.getHeader("JQGridModel") != null
					&& request.getHeader("JQGridModel").equals("true")) {
				pagination = new Pagination();
				pagination.setPage(Long.valueOf(request.getParameter("page")));
				pagination.setRows(Long.valueOf(request.getParameter("rows")));
				pagination.setSort(request.getParameter("sidx"));
				pagination.setAscDsc(request.getParameter("sord"));
				List<Inspector> inspectors = this.inspectorService.findAll(
						filterInspector, pagination);

				if (inspectors == null) {
					throw new Exception("No data Found.");
				}

				Long total = getAllCount(filterInspector, request);
				JQGridJSONModel data = new JQGridJSONModel();
				data.setPage(request.getParameter("page"));
				data.setRecords(total.intValue());
				data.setTotal(total, pagination.getRows());
				data.setRows(inspectors);
				return data;
			} else {
				List<Inspector> inspectors = this.inspectorService.findAll(
						filterInspector, pagination);
				if (inspectors == null) {
					throw new Exception("No data Found.");
				}
				return inspectors;
			}
		} catch (Exception e) {
			throw new ResourceNotFoundException("No data Found.");
		}
	}

	/**
	 * Method 'getAllCount'.
	 * 
	 * @param filterInspector
	 *            Inspector
	 * @param request
	 *            HttpServletRequest
	 * @return Long
	 */
	@RequestMapping(value = "/count", method = RequestMethod.GET)
	public @ResponseBody
	Long getAllCount(
			@RequestParam(value = "inspector", required = false) Inspector filterInspector,
			HttpServletRequest request) {
		try {
			return inspectorService
					.findAllCount(filterInspector != null ? filterInspector
							: new Inspector());
		} catch (Exception e) {
			throw new ServiceUnavailableException(
					"Count Service is not responding.");
		}
	}

	/**
	 * Method 'edit'.
	 * 
	 * @param inspector
	 *            Inspector
	 * @param response
	 *            HttpServletResponse
	 * @return Inspector
	 */
	@RequestMapping(method = RequestMethod.PUT)
	public @ResponseBody
	Inspector edit(@RequestBody Inspector inspector,
			HttpServletResponse response) {
		try {
			Inspector inspectorAux = this.inspectorService.update(inspector);
			logger.info("Entity correctly inserted!");
			return inspectorAux;
		} catch (Exception e) {
			throw new MethodFailureException("Method failed");
		}
	}

	/**
	 * Method 'add'.
	 * 
	 * @param inspector
	 *            Inspector
	 * @return Inspector
	 */
	@RequestMapping(method = RequestMethod.POST)
	public @ResponseBody
	Inspector add(@RequestBody Inspector inspector) {
		try {
			Inspector inspectorAux = this.inspectorService.add(inspector);
			logger.info("Entity correctly inserted!");
			return inspectorAux;
		} catch (Exception e) {
			throw new MethodFailureException("Method failed");
		}
	}

	/**
	 * Method 'remove'.
	 * 
	 * @param incodigo
	 *            String
	 * @param response
	 *            HttpServletResponse
	 * @return
	 */
	@RequestMapping(value = "/{incodigo}", method = RequestMethod.DELETE)
	public void remove(@PathVariable String incodigo,
			HttpServletResponse response) {
		response.setContentType("text/javascript;charset=utf-8");
		response.setHeader("Pragma", "cache");
		response.setHeader("Expires", "0");
		response.setHeader("Cache-Control", "private");
		try {
			Inspector inspector = new Inspector();
			inspector.setIncodigo(incodigo);
			this.inspectorService.remove(inspector);
			response.setStatus(HttpServletResponse.SC_OK);
		} catch (Exception e) {
			logger.error("Unable to delete " + incodigo);
			response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
			throw new MethodFailureException("Method failed");
		}
	}

	/**
	 * Method 'removeAll'.
	 * 
	 * @param inspectorIds
	 *            ArrayList
	 * @param response
	 *            HttpServletResponse
	 * @return
	 */
	@RequestMapping(value = "/deleteAll", method = RequestMethod.POST)
	public void removeMultiple(
			@RequestBody ArrayList<ArrayList<String>> inspectorIds,
			HttpServletResponse response) {
		response.setContentType("text/javascript;charset=utf-8");
		response.setHeader("Pragma", "cache");
		response.setHeader("Expires", "0");
		response.setHeader("Cache-Control", "private");
		ArrayList<Inspector> inspectorList = new ArrayList<Inspector>();
		try {
			for (ArrayList<String> inspectorId : inspectorIds) {
				Iterator<String> iterator = inspectorId.iterator();
				Inspector inspector = new Inspector();
				inspector.setIncodigo(ObjectConversionManager.convert(
						iterator.next(), String.class));
				inspectorList.add(inspector);
			}
			this.inspectorService.removeMultiple(inspectorList);
			response.setStatus(HttpServletResponse.SC_OK);
		} catch (Exception e) {
			response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
			throw new MethodFailureException("Method failed");
		}
	}

	/**
	 * Method 'handle'.
	 * 
	 * @param e
	 *            ControlException
	 * @return String
	 */
	@ExceptionHandler
	public @ResponseBody
	String handle(ControlException e) {
		logger.warn(e.getMessage());
		return e.getMessage();
	}

	/**
	 * Method 'getInspectorService'.
	 * 
	 * @return InspectorService
	 */
	protected InspectorService getInspectorService() {
		return this.inspectorService;
	}

	/**
	 * Method 'setInspectorService'.
	 * 
	 * @param inspectorService
	 *            InspectorService
	 * @return
	 */
	public void setInspectorService(InspectorService inspectorService) {
		this.inspectorService = inspectorService;
	}

	/**
	 * Method 'getAppConfiguration'.
	 * 
	 * @return appConfiguration
	 */
	public Properties getAppConfiguration() {
		return appConfiguration;
	}

	/**
	 * Method 'setAppConfiguration'.
	 * 
	 * @param appConfiguration
	 *            Properties
	 * @return
	 */
	public void setAppConfiguration(Properties appConfiguration) {
		this.appConfiguration = appConfiguration;
	}
}
