package com.ejie.y41b.utils.ws;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.util.Iterator;
import java.util.Properties;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.Name;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPConnection;
import javax.xml.soap.SOAPConnectionFactory;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPEnvelope;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;
import javax.xml.soap.SOAPPart;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import sun.misc.BASE64Decoder;
import sun.misc.BASE64Encoder;

import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.nshf.EjgvDocument;
import com.ejie.y41b.nshf.EjgvDocumentType;
import com.ejie.y41b.utils.config.Y41bConfig;

/**
 * Y41bUtilsWS Clase que encapsula las invocaciones a WS
 * 
 *  
 * @version 1.0
 */
public class Y41bUtilsWS {

	/**
	 * Final static logger.
	 */
	private static final Logger logger = LoggerFactory
			.getLogger(Y41bUtilsWS.class);

	/**
	 * Constructor privado
	 */
	private Y41bUtilsWS() {
		super();
	}

	/**
	 * Metodo que realiza la invocacion a un WS
	 * 
	 * @param url
	 *            String URL de acceso al WS
	 * @param method
	 *            String metodo del WS al que se invoca
	 * @param paramNames
	 *            String[] con los nombres de los parametros de la funcion
	 * @param paramValues
	 *            String[] con los valores de los parametros de la funcion
	 * @return Object Objeto con el resultado
	 * @throws Exception
	 *             Cualquier excepcion durante la invocacion
	 */
	public static String invoke(String url, String method, String[] paramNames,
			String[] paramValues) throws Exception {

		logger.info("************************ SOAP Invoke ************************");
		logger.info("SOAP Invoke url:" + url);
		logger.info("SOAP Invoke method:" + method);
		for (int i = 0; i < paramNames.length; i++) {
			logger.info("SOAP Invoke paramNames:" + paramNames[i].toString()
					+ " ::: " + paramValues[i].toString());
		}

		SOAPMessage message = null;
		String rtdo = "";
		SOAPMessage response = null;
		String paramValor = "";

		MessageFactory mfactory = MessageFactory.newInstance();
		message = mfactory.createMessage();
		SOAPPart soapPart = message.getSOAPPart();
		SOAPEnvelope envelope = soapPart.getEnvelope();
		SOAPBody body = envelope.getBody();
		Name name = envelope.createName(method, "", "http://www.openuri.org/");

		/*
		 * SOAPElement identity = envelope.getHeader().addChildElement(
		 * envelope.createName(Y41bConstantes.IDENTITY));
		 * identity.addTextNode(Y41bConstantes.COD_APLICACION);
		 */

		SOAPBodyElement element = body.addBodyElement(name);

		Properties prop = Y41bConfig.loadProperties(Y41bConstantes.CONFIG_PATH);

		String local = (String) prop.getProperty("y41bVistaWar.entorno.local");

		for (int i = 0; i < paramNames.length && i < paramValues.length; i++) {
			SOAPElement param = element.addChildElement(envelope
					.createName(paramNames[i]));

			if (!paramNames[i].contains("oken")) {

				logger.info("#############################PARAM UTF-8:["
						+ new String(paramValues[i].getBytes("UTF-8"), "UTF-8")
						+ "]");
				logger.info("#############################PARAM ISO-8859-1:["
						+ new String(paramValues[i].getBytes("ISO-8859-1"),
								"ISO-8859-1") + "]");

				if (local.equals("false")) {
					byte[] btval = paramValues[i].getBytes("UTF-8");
					paramValor = new String(btval, "ISO-8859-1");// NOPMD
				} else {
					paramValor = paramValues[i];
				}
			} else {
				paramValor = paramValues[i];
			}
			param.addTextNode(paramValor);

			// "2017.04.12: Se quita este atributo ya que desde la invocacion a actualizar todas las remesas desde el WS pone el atributo xmlns vacio y da un error."
			// "            Esto no pasa si se envia a mensaje a la cola JMS y hace el tratamiento desde alli"
			// "            NOTA: en versiones posteriores el WS enviar mensajes JMS y por tanto, funcionar correctamente."
			// param.removeAttribute("xmlns");
		}

		SOAPConnectionFactory factory = SOAPConnectionFactory.newInstance();
		SOAPConnection con = factory.createConnection();
		logger.info("SOAP Invoke:" + Y41bUtilsWS.serializarSOAPMessage(message));
		response = con.call(message, url);
		logger.info("SOAP response:" + response);
		rtdo = Y41bUtilsWS.getResultado(response);
		logger.info("SOAP Resultado:" + rtdo);

		return rtdo;
	}

	/**
	 * Codifica en Base64 un array de bytes
	 * 
	 * @param decData
	 *            Datos a codificar
	 * @return String con los datos codificados en Base64
	 * @throws
	 */
	public static String encode(byte[] decData) {
		String encData;
		BASE64Encoder encoder;
		encoder = new BASE64Encoder();
		encData = encoder.encode(decData);
		return encData;
	}

	/**
	 * Codifica en Base64 una cadena de caracteres
	 * 
	 * @param decData
	 *            Datos a codificar
	 * @return String con los datos codificados en Base64
	 * @throws
	 */
	public static String encodeString(String decData) {
		return encode(decData.getBytes());
	}

	/**
	 * Codifica en Base64 una cadena de caracteres utilizando el encodig pasado
	 * como parametro
	 * 
	 * @param decData
	 *            Datos a codificar
	 * @param enc
	 *            encoding para codificar
	 * @return String con los datos codificados en Base64
	 * @throws
	 */
	public static String encodeString(String decData, String enc) {
		try {
			byte[] bytes = decData.getBytes(enc);
			return encode(bytes);
		} catch (UnsupportedEncodingException e) {
			logger.error("Y41bUtilsWS.encodeString - Error:", e);
			return null;
		}
	}

	/**
	 * Decodifica una cadena de caracteres que se encuentra codificada en
	 * formato Base64
	 * 
	 * @param encData
	 *            Datos a decodificar
	 * @return Bytes con los datos decodificados
	 * @throws W92jPlateaException
	 *             e
	 */
	public static byte[] decode(String encData) throws Exception {
		byte[] decData = null;
		BASE64Decoder decoder;
		decoder = new BASE64Decoder();
		try {
			decData = decoder.decodeBuffer(encData);
		} catch (IOException e) {
			logger.error("Y41bUtilsWS.decode - Error:", e);
			throw e;
		}
		return decData;
	}

	/**
	 * Decodifica una cadena de caracteres que se encuentra codificada en
	 * formato Base64
	 * 
	 * @param encData
	 *            Datos a decodificar
	 * @return String con los datos decodificados
	 * @throws Exception
	 *             e
	 */
	public static String decodeToString(String encData) throws Exception {
		String decoded = null;
		try {
			decoded = new String(decode(encData));
		} catch (Exception e) {
			logger.error("Y41bUtilsWS.decodeToString - Error:", e);
			throw new Exception("Y41bUtilsWS.decodeToString", e);
		}
		return decoded;
	}

	/**
	 * Extrae el resultado del mensaje SOAP
	 * 
	 * @param response
	 *            SOAPMessage a serializar
	 * 
	 * @return resultado
	 * @throws SOAPException
	 *             e
	 * @throws Exception
	 *             Esta excepcion se lanza cunado la invocacion devuleve un
	 *             SOAP:Fault
	 */

	private static String getResultado(SOAPMessage response)
			throws SOAPException, Exception {
		logger.debug("SOAP Resultado:"
				+ Y41bUtilsWS.serializarSOAPMessage(response));
		String rtdo = "";
		SOAPBodyElement soapBodyElement = Y41bUtilsWS
				.getSOAPBodyElement(response);
		String responseElementName = soapBodyElement.getElementName()
				.toString();
		String resultElementName = responseElementName.replaceAll("Response$",
				"Result");
		resultElementName = resultElementName.substring(
				resultElementName.indexOf(":") + 1, resultElementName.length());
		// Launch exception
		if (StringUtils.upperCase(responseElementName).endsWith("FAULT")) {
			throw new Exception(soapBodyElement.toString());
		}

		@SuppressWarnings("rawtypes")
		Iterator iterator = soapBodyElement.getChildElements();
		while (iterator.hasNext()) {
			Object objElement = iterator.next();
			if (objElement instanceof SOAPElement) {
				SOAPElement element = (SOAPElement) objElement;
				String elementName = element.getElementName().toString();
				elementName = elementName.substring(
						elementName.indexOf(":") + 1, elementName.length());
				if (elementName.equals(resultElementName)) {
					rtdo = element.getValue();
					break;
				}
			}
		}

		return rtdo;
	}

	/**
	 * Extrae el el nodo boy del mensaje SOAP
	 * 
	 * @param response
	 *            SOAPMessage a serializar
	 * @throws SOAPException
	 *             e
	 * @return SOAPBodyElement
	 */

	private static SOAPBodyElement getSOAPBodyElement(SOAPMessage response)
			throws SOAPException {
		SOAPBodyElement rtdo = null;
		@SuppressWarnings("rawtypes")
		Iterator iterator = response.getSOAPPart().getEnvelope().getBody()
				.getChildElements();
		while (iterator.hasNext()) {
			Object oUndefined = iterator.next();
			if (oUndefined instanceof SOAPBodyElement) {
				rtdo = (SOAPBodyElement) oUndefined;
				break;
			}
		}
		return rtdo;
	}

	/**
	 * Serializar un objeto SOAPMessage a un String y a un Fichero
	 * 
	 * @param response
	 *            Objeto SOAPMessage a serializar
	 * @return String con el contenido de un objeto SOAPMessage
	 */
	private static String serializarSOAPMessage(SOAPMessage response) {
		String retorno = "";
		try {
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			response.writeTo(baos);
			retorno = baos.toString();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (SOAPException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return retorno;
	}

	/**
	 * ejgvDocumentTypeToString
	 * 
	 * @param ejgvDocumentType
	 *            EjgvDocumentType
	 * @return String
	 * @throws Exception
	 *             e
	 */
	public static String ejgvDocumentTypeToString(
			EjgvDocumentType ejgvDocumentType) throws Exception {
		try {
			final JAXBContext jaxbContext = JAXBContext
					.newInstance(EjgvDocumentType.class);
			StringWriter writer = new StringWriter();
			Marshaller m = jaxbContext.createMarshaller();
			m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);

			jaxbContext.createMarshaller().marshal(ejgvDocumentType, writer);
			return writer.toString();

		} catch (JAXBException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;

	}

	/**
	 * ejgvDocumentToString
	 * 
	 * @param ejgvDocumentType
	 *            EjgvDocument
	 * @return String
	 * @throws Exception
	 *             e
	 */
	public static String ejgvDocumentToString(EjgvDocument ejgvDocumentType)
			throws Exception {

		final JAXBContext jaxbContext = JAXBContext
				.newInstance(EjgvDocument.class);
		StringWriter writer = new StringWriter();
		Marshaller m = jaxbContext.createMarshaller();
		m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);

		jaxbContext.createMarshaller().marshal(ejgvDocumentType, writer);
		return writer.toString();

	}

	/**
	 * stringToEjgvDocumentType
	 * 
	 * @param ejgvDocumentType
	 *            String
	 * @return EjgvDocumentType
	 * @throws Exception
	 *             e
	 */
	public static EjgvDocumentType stringToEjgvDocumentType(
			String ejgvDocumentType) throws Exception {

		final JAXBContext jaxbContext = JAXBContext
				.newInstance(EjgvDocumentType.class);
		return (EjgvDocumentType) jaxbContext.createUnmarshaller().unmarshal(
				new StringReader(ejgvDocumentType));
	}

	/**
	 * stringToEjgvDocument
	 * 
	 * @param ejgvDocument
	 *            String
	 * @return EjgvDocument
	 * @throws Exception
	 *             e
	 */
	public static EjgvDocument stringToEjgvDocument(String ejgvDocument)
			throws Exception {

		final JAXBContext jaxbContext = JAXBContext
				.newInstance(EjgvDocument.class);

		if (ejgvDocument
				.indexOf("<ns2:ejgvDocument xmlns:ns2=\"http://com.ejie.x43f/X43FNSHF/\">") == -1) {
			ejgvDocument = ejgvDocument
					.replaceFirst("<ejgvDocument>",
							"<ns2:ejgvDocument xmlns:ns2=\"http://com.ejie.x43f/X43FNSHF/\">");
			ejgvDocument = ejgvDocument.replace("</ejgvDocument>",
					"</ns2:ejgvDocument>");
		}
		Unmarshaller um = jaxbContext.createUnmarshaller();
		return (EjgvDocument) um.unmarshal(new StringReader(ejgvDocument));
	}

	/**
	 * unMarshallEjgvDocument
	 * 
	 * @param ejgvDocument
	 *            String
	 * @return EjgvDocument
	 * @throws Exception
	 *             e
	 */
	public static EjgvDocument unMarshallEjgvDocument(String ejgvDocument)
			throws Exception {
		final JAXBContext jaxbContext = JAXBContext
				.newInstance(EjgvDocument.class);

		if (ejgvDocument.indexOf("xmlns:ns2=\"http://com.ejie.x43f/X43FNSHF/") == -1) {
			ejgvDocument = ejgvDocument
					.replaceFirst("<ejgvDocument>",
							"<ns2:ejgvDocument xmlns:ns2=\"http://com.ejie.x43f/X43FNSHF/");
			ejgvDocument = ejgvDocument.replace("</ejgvDocument>",
					"</ns2:ejgvDocument>");
		}
		Unmarshaller um = jaxbContext.createUnmarshaller();
		return (EjgvDocument) um.unmarshal(new StringReader(ejgvDocument));
	}
}
