package com.ejie.y41b.utils.fichero;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.Calendar;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.utils.Y41bStringUtils;
import com.ejie.y41b.utils.config.Y41bConfig;

public class Y41bFicheroUtils {

	private static final int BUFFER_SIZE = 2048;

	/**
	 * Constructor privado
	 */
	private Y41bFicheroUtils() {

	}

	/**
	 * <p>
	 * Método que comprime un fichero
	 * <p>
	 * 
	 * @param file
	 *            File a comprimir
	 * @param zipName
	 *            nombre del fichero con la ruta completa
	 * 
	 * @throws ex
	 *             IOException
	 */
	public static void comprimirFichero(File file, String zipName)
			throws IOException {
		Properties prop = Y41bConfig.loadProperties(Y41bConstantes.CONFIG_PATH);

		if (file.isDirectory()) {
			if (file.list().length > 0) {
				String local = (String) prop
						.getProperty("y41bVistaWar.entorno.local");

				String folderStr = "";
				if (local.equals("true")) {
					folderStr = file.getPath() + "\\";
				} else {
					folderStr = file.getPath() + "/";
				}

				byte b[] = new byte[512];
				FileOutputStream out = new FileOutputStream(zipName);
				ZipOutputStream zout = new ZipOutputStream(out);

				File fileEntry = null;
				InputStream in = null;
				ZipEntry e = null;
				for (String fileName : file.list()) {
					fileEntry = new File(folderStr + fileName);
					in = new FileInputStream(fileEntry);
					// Hay que quitar los acentos y las tildes ya que el API de
					// Java para ZIP no lo interpreta correctamente
					e = new ZipEntry(Y41bStringUtils.convertNonAscii(fileName));
					zout.putNextEntry(e);
					int len = 0;
					while ((len = in.read(b)) != -1) {
						zout.write(b, 0, len);
					}
					in.close();
					zout.closeEntry();
				}
				zout.close();
			}
		} else {
			byte b[] = new byte[512];
			FileOutputStream out = new FileOutputStream(zipName);
			ZipOutputStream zout = new ZipOutputStream(out);
			InputStream in = new FileInputStream(file);
			ZipEntry e = new ZipEntry(file.getName());
			zout.putNextEntry(e);
			int len = 0;
			while ((len = in.read(b)) != -1) {
				zout.write(b, 0, len);
			}
			zout.closeEntry();
			zout.close();
		}
	}

	/**
	 * <p>
	 * Método que descomprime un ZIP en una ruta indicada
	 * <p>
	 * 
	 * @param origen
	 *            String
	 * @param destino
	 *            String
	 * 
	 * @throws ex
	 *             Exception
	 */
	public static void descomprimirZIP(String origen, String destino)
			throws Exception {
		BufferedOutputStream bos = null;
		FileInputStream fis = null;
		ZipInputStream zipis = null;
		FileOutputStream fos = null;
		try {
			fis = new FileInputStream(origen);
			zipis = new ZipInputStream(new BufferedInputStream(fis));
			ZipEntry entry;
			while ((entry = zipis.getNextEntry()) != null) {
				if (!entry.isDirectory()) {
					int len = 0;
					byte[] buffer = new byte[BUFFER_SIZE];
					String destFN = destino + File.separator + entry.getName();
					fos = new FileOutputStream(destFN);
					bos = new BufferedOutputStream(fos, BUFFER_SIZE);
					while ((len = zipis.read(buffer, 0, BUFFER_SIZE)) != -1)
						bos.write(buffer, 0, len);
					bos.flush();
				} else {
					Y41bFicheroUtils.crearDirectorio(entry.getName());
				}
			}
		} catch (Exception e) {
			throw e;
		} finally {
			bos.close();
			zipis.close();
			fos.close();
			fis.close();
		} // end try
	} // end descomprimirZIP

	public static void crearDirectorio(String directorio) throws IOException {
		File folder = new File(directorio);
		if (!folder.exists()) {
			folder.mkdir();
		}
	}

	/**
	 * <p>
	 * Método que borra el contenido de un directorio
	 * <p>
	 * 
	 * @param path
	 *            File
	 */
	public static void deleteDirectory(File path) {
		if (path.exists()) {
			File[] files = path.listFiles();
			for (int i = 0; i < files.length; i++) {
				if (files[i].isDirectory()) {
					Y41bFicheroUtils.deleteDirectory(files[i]);
				} else {
					files[i].delete();
				}
			}
		}
	}

	/**
	 * <p>
	 * Método que borra el directorio y el contenido de un directorio
	 * <p>
	 * 
	 * @param path
	 *            File
	 */
	public static void deleteDirectoryAndContents(File path) {
		if (path.exists()) {
			File[] files = path.listFiles();
			for (int i = 0; i < files.length; i++) {
				if (files[i].isDirectory()) {
					Y41bFicheroUtils.deleteDirectoryAndContents(files[i]);
				}
				files[i].delete();
			}
		}
	}

	/**
	 * <p>
	 * Método que borra un fichero
	 * <p>
	 * 
	 * @param file
	 *            File
	 */
	public static void deleteFile(File file) {
		if (file.exists()) {
			file.delete();
		}
	}

	/**
	 * <p>
	 * Funcion que dado un fichero, devuelve un String con su contenido
	 * <p>
	 * 
	 * @param file
	 *            File
	 * 
	 * @return str String
	 * @throws e
	 *             FileNotFoundException, ioe IOException
	 */
	public static String leerFichero(File file) throws FileNotFoundException,
			IOException {
		StringBuffer str = new StringBuffer();
		int ch;
		FileInputStream fin = null;
		try {
			fin = new FileInputStream(file);
			while ((ch = fin.read()) != -1) {
				str.append((char) ch);
			}

			fin.close();
		} catch (FileNotFoundException e) {
			throw e;
		} catch (IOException ioe) {
			throw ioe;
		}

		return str.toString();

	}

	/**
	 * Este método inserta el contenido de un String que se pasa por parámetro
	 * a un fichero de texto.
	 * 
	 * @param contenido
	 *            Contenido a insertar en el fichero
	 * @param fichero
	 *            File el fichero
	 * 
	 * @throws ex
	 *             IOException
	 */
	public static void escribirFichero(String contenido, File fichero)
			throws IOException {
		try {
			FileWriter fw = new FileWriter(fichero, true);
			BufferedWriter bw = new BufferedWriter(fw);
			PrintWriter salida = new PrintWriter(bw);
			salida.print(contenido);
			salida.close();
			bw.close();
			fw.close();
		} catch (IOException ex) {
			throw ex;
		}
	}

	/**
	 * Este método inserta el contenido de un InputStream que se pasa por
	 * parámetro a un fichero
	 * 
	 * @param entrada
	 *            InputStream
	 * @param fichero
	 *            File el fichero
	 * 
	 * @throws ex
	 *             IOException
	 */
	public static void escribirFichero(InputStream entrada, File fichero)
			throws IOException {
		OutputStream salida = null;
		try {
			salida = new FileOutputStream(fichero);
			byte[] buf = new byte[1024];
			int len;
			while ((len = entrada.read(buf)) > 0) {
				salida.write(buf, 0, len);
			}
		} catch (IOException ex) {
			throw ex;
		} finally {
			if (salida != null) {
				salida.close();
			}
		}
	}

	/**
	 * Método encargado de crear un nuevo fichero (objeto File) en caso de que
	 * no exista.
	 * 
	 * @param fichero
	 *            File el fichero
	 * 
	 * @throws ex
	 *             Exception
	 */
	public static void crearFichero(File fichero) throws IOException {
		try {
			if (!fichero.exists()) {
				fichero.createNewFile();
			}
		} catch (IOException ex) {
			throw ex;
		}
	}

	/**
	 * Método que elimina aquellos ficheros con una fecha creación con más
	 * que los milisegundos pasados por parametro
	 * 
	 * @param fichero
	 *            File el fichero
	 * @param fichero
	 *            File el fichero
	 */
	public static void deleteFilesFromDate(File directory, long timeMs) {
		if (directory.isDirectory() && directory.exists()) {
			File[] files = directory.listFiles();
			if (files != null && files.length > 0) {
				Long todayMs = Calendar.getInstance().getTimeInMillis();
				for (File file : files) {
					Long lastModifMs = file.lastModified();
					if ((todayMs - lastModifMs) > timeMs) {
						Y41bFicheroUtils.deleteFile(file);
					}
				}
			}
		}
	}

	/**
	 * Método que copia un fichero de un directorio a otro
	 * 
	 * @param fichero
	 *            File el fichero
	 * @param fichero
	 *            File el fichero
	 * @throws Exception
	 */
	public static void copiarFichero(String fromFile, String toFile)
			throws Exception {
		File origen = new File(fromFile);
		File destino = new File(toFile);
		InputStream in = new FileInputStream(origen);
		OutputStream out = new FileOutputStream(destino);
		// We use a buffer for the copy (Usamos un buffer para la copia).
		byte[] buf = new byte[1024];
		int len;
		while ((len = in.read(buf)) > 0) {
			out.write(buf, 0, len);
		}
		in.close();
		out.close();

	}

}