package com.ejie.y41b.utils.documentos;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.util.Date;
import java.util.StringTokenizer;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.velocity.anakia.Escape;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.utils.Y41bUtils;

/**
 *  
 * 
 */
public class Y41bGenerarDocumento {
	private static final long serialVersionUID = 1L;

	private static final Logger logger = LoggerFactory
			.getLogger(Y41bGenerarDocumento.class);

	private Y41bGenerarDocumento() {
	}

	/**
	 * Genera un documento
	 * 
	 * @param rutaPlantillaXML
	 *            ruta de la plantilla XML
	 * @param datosET
	 *            ET con los datos para generar el documento
	 * @param plantilla
	 *            String con la plantilla a rellenar
	 * @return String con el documento generado
	 * @throws Exception
	 */
	public static String generarDocumentoWord(String rutaPlantillaXML,
			Object datosET, String plantilla) throws Exception {
		StringBuffer datosPlantilla = Y41bGenerarDocumento
				.generarDatosPlantilla(rutaPlantillaXML, datosET);

		Y41bDocumento documento = new Y41bDocumento();

		String documentoGenerado = "";

		try {
			documentoGenerado = documento.generateDocument(
					datosPlantilla.toString(), plantilla);
		} catch (Exception e) {
			logger.error("Error al generar el documento");
		}

		return documentoGenerado;
	}

	/**
	 * Genera los datos con los que rellenar la plantilla
	 * 
	 * @param rutaPlantillaXML
	 *            Contiene la ruta de la plantilla xml
	 * @param datosET
	 *            ET con los datos para rellenar la plantilla
	 * @return ArrayList con los datos para rellenar la plantilla
	 * @throws Exception
	 */
	private static StringBuffer generarDatosPlantilla(String rutaPlantillaXML,
			Object datosET) throws Exception {
		Document xmlCamposPlantilla = null;

		try {
			xmlCamposPlantilla = Y41bGenerarDocumento
					.obtenerCamposPlantilla(rutaPlantillaXML);
		} catch (ParserConfigurationException e) {
			logger.error("Error al configurar el parser xml");
			throw new Exception("Error al configurar el parser xml");
		} catch (SAXException e) {
			logger.error("Error al parsear el documento xml");
			throw new Exception("Error al parsear el documento xml");
		} catch (IOException e) {
			logger.error("Error al localizar el documento xml");
			throw new Exception("Error al localizar el documento xml");
		}

		StringBuffer parametrosXml = new StringBuffer();
		parametrosXml.append("<TemplateDocument>");
		parametrosXml.append("<DocumentFields>");

		if (null != xmlCamposPlantilla) {
			NodeList listaCamposPlantilla = xmlCamposPlantilla
					.getElementsByTagName("campo");

			for (int c = 0; c < listaCamposPlantilla.getLength(); c++) {
				Node campo = listaCamposPlantilla.item(c);

				parametrosXml.append("<Field Name=\"#Campo");
				parametrosXml.append(campo.getChildNodes().item(0)
						.getChildNodes().item(0).getNodeValue());
				parametrosXml.append("#\">");

				String nombreCampo = campo.getChildNodes().item(1)
						.getChildNodes().item(0).getNodeValue();

				String valorCampo = Y41bGenerarDocumento.obtenerValorCampoET(
						nombreCampo, datosET);

				if (Y41bUtils.isFilled(valorCampo)) {
					// Escapar & y carateres especiales
					parametrosXml.append(Escape.getText(valorCampo));
				}

				parametrosXml.append("</Field>");
			}
		}

		parametrosXml.append("</DocumentFields>");
		parametrosXml.append("</TemplateDocument>");

		return parametrosXml;
	}

	/**
	 * Dado el nombre de un campo de una ET de tipo ToolkitET devuelve su valor
	 * 
	 * @param nombreCampo
	 *            para el cual obtener el valor
	 * @param datosET
	 *            ET con los valores
	 * @return String con el valor del campo en cuesti�n
	 */
	private static String obtenerValorCampoET(String nombreCampo, Object datosET) {
		String retorno = "";
		String prefijoMetodo = "get";
		StringBuffer nombreMetodo;

		StringTokenizer tokenizer = new StringTokenizer(nombreCampo, ".");

		Object et = datosET;

		while (tokenizer.hasMoreTokens()) {
			nombreMetodo = new StringBuffer(prefijoMetodo);

			String metodo = tokenizer.nextToken();
			boolean esListado = Y41bGenerarDocumento.campoEsListado(metodo);
			int indice = -1;

			if (esListado) {
				indice = Y41bGenerarDocumento.campoListadoObtenerIndice(metodo);
				metodo = Y41bGenerarDocumento.campoListadoObtenerMetodo(metodo);
			}

			nombreMetodo.append(metodo);

			try {
				if (!tokenizer.hasMoreTokens()) {

					if (et.getClass().getMethod(nombreMetodo.toString(), null)
							.invoke(et, null) instanceof String) {
						retorno = (String) et.getClass()
								.getMethod(nombreMetodo.toString(), null)
								.invoke(et, null);
					} else if (et.getClass()
							.getMethod(nombreMetodo.toString(), null)
							.invoke(et, null) instanceof Integer) {
						retorno = (String) Integer.toString((Integer) et
								.getClass()
								.getMethod(nombreMetodo.toString(), null)
								.invoke(et, null));
					} else if (et.getClass()
							.getMethod(nombreMetodo.toString(), null)
							.invoke(et, null) instanceof Date) {
						retorno = (String) Y41bUtils.parseDateToString(
								(Date) et
										.getClass()
										.getMethod(nombreMetodo.toString(),
												null).invoke(et, null),
								Y41bConstantes.CASTELLANO);
					} else if (et.getClass()
							.getMethod(nombreMetodo.toString(), null)
							.invoke(et, null) instanceof Long) {
						retorno = (String) (((Long) et.getClass()
								.getMethod(nombreMetodo.toString(), null)
								.invoke(et, null)).toString());
					} else if (et.getClass()
							.getMethod(nombreMetodo.toString(), null)
							.invoke(et, null) instanceof BigDecimal) {
						retorno = (String) ((BigDecimal) et.getClass()
								.getMethod(nombreMetodo.toString(), null)
								.invoke(et, null)).toString();
					}

				} else {
					et = et.getClass()
							.getMethod(nombreMetodo.toString(), new Class[0])
							.invoke(et, null);

					if (null == et) {
						break;
					}

					if (esListado) {
						et = ((Object[]) et)[indice];
					}
				}
			} catch (IllegalAccessException iae) {
				logger.error("Error al acceder al m�todo "
						+ nombreMetodo.toString());
			} catch (NoSuchMethodException iae) {
				logger.error("No existe el m�todo " + nombreMetodo.toString());
			} catch (InvocationTargetException iae) {
				logger.error("Error el invocar al m�todo "
						+ nombreMetodo.toString());
			}
		}
		// Escapa los dolares para evitar produzca un error al volcar cada
		// elemento en el documento.
		return escaparCadena(retorno);
	}

	/**
	 * Comprueba si un campo de una ET es un listado
	 * 
	 * @param nombreCampo
	 *            nombre del campo a comprobar
	 * @return True si es un listado, false en caso contrario
	 */
	private static boolean campoEsListado(String nombreCampo) {
		return (nombreCampo.indexOf("[") != -1) ? true : false;
	}

	/**
	 * Devuelve el indice de un campo que es un listado
	 * 
	 * @param nombreCampo
	 *            campo que es un listado
	 * @return indice
	 */
	private static int campoListadoObtenerIndice(String nombreCampo) {
		return Integer.parseInt(nombreCampo.substring(
				nombreCampo.indexOf("[") + 1, nombreCampo.indexOf("]")));
	}

	/**
	 * Devuelve el metodo de un campo que es un listado
	 * 
	 * @param nombreCampo
	 *            campo que es un listado
	 * @return metodo
	 */
	private static String campoListadoObtenerMetodo(String nombreCampo) {
		return nombreCampo.substring(0, nombreCampo.indexOf("["));
	}

	/**
	 * Lee el XML que contiene los campos de la plantilla a rellenar
	 * 
	 * @param rutaPlantillaXML
	 *            String
	 * @return XmlDocument con el docuento XML parseado
	 * @throws ParserConfigurationException
	 *             Excepci�n lanzada
	 * @throws SAXException
	 *             Excepci�n lanzada
	 * @throws IOException
	 *             Excepci�n lanzada
	 */
	private static Document obtenerCamposPlantilla(String rutaPlantillaXML)
			throws ParserConfigurationException, SAXException, IOException {
		Document documento = null;

		DocumentBuilderFactory documentFactory = DocumentBuilderFactory
				.newInstance();
		documentFactory.setIgnoringElementContentWhitespace(true);
		DocumentBuilder documentBuilder = documentFactory.newDocumentBuilder();
		documento = documentBuilder.parse(new File(rutaPlantillaXML));

		return documento;
	}

	/***
	 * 
	 * @param cadena
	 *            cadena
	 * @return devuelve la cadena con los dolares escapados
	 */
	private static String escaparCadena(String cadena) {
		// Reemplazar la cadena siempre que no sea en el caracter 0 o en el
		// último.
		if (cadena.length() > 0) {
			return cadena.replaceAll("\\$", "\\'24");
		}
		return cadena;
	}
}
