package com.ejie.y41b.utils;

import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;

/**
 * Esta clase provee la funcionalidad de convertir un numero representado en
 * digitos a una representacion en letras.
 * 
 *  
 */
public abstract class Y41bNumberToLetterConverter {

	private static final double MAX_NUMBER = 999999999;
	private static final int OCHO = 8;
	private static final int SIETE = 7;
	private static final int SEIS = 6;
	private static final int CINCO = 5;
	private static final int CUATRO = 4;
	private static final int TRES = 3;
	private static final int CUARENTA_Y_OCHO = 48;
	private static final int VEINTE = 20;
	private static final int TREINTA = 30;

	private static final String[] UNIDADES = { "", "UN ", "DOS ", "TRES ",
			"CUATRO ", "CINCO ", "SEIS ", "SIETE ", "OCHO ", "NUEVE ", "DIEZ ",
			"ONCE ", "DOCE ", "TRECE ", "CATORCE ", "QUINCE ", "DIECISEIS",
			"DIECISIETE", "DIECIOCHO", "DIECINUEVE", "VEINTE" };

	private static final String[] DECENAS = { "VEINTI", "TREINTA ",
			"CUARENTA ", "CINCUENTA ", "SESENTA ", "SETENTA ", "OCHENTA ",
			"NOVENTA ", "CIEN " };

	private static final String[] CENTENAS = { "CIENTO ", "DOSCIENTOS ",
			"TRESCIENTOS ", "CUATROCIENTOS ", "QUINIENTOS ", "SEISCIENTOS ",
			"SETECIENTOS ", "OCHOCIENTOS ", "NOVECIENTOS " };

	/**
	 * Convierte a letras un numero de la forma $123,456.32
	 * 
	 * @param number
	 *            Numero en representacion texto
	 * @throws NumberFormatException
	 *             Si valor del numero no es valido (fuera de rango o )
	 * @return Numero en letras
	 */
	public static String convertNumberToLetter(String number)
			throws NumberFormatException {
		return Y41bNumberToLetterConverter.convertNumberToLetter(Double
				.parseDouble(number));
	}

	/**
	 * Convierte un numero en representacion numerica a uno en representacion de
	 * texto. El numero es valido si esta entre 0 y 999'999.999
	 * 
	 * @param doubleNumberParam
	 *            Numero a convertir
	 * @return Numero convertido a texto
	 * @throws NumberFormatException
	 *             Si el numero esta fuera del rango
	 */
	public static String convertNumberToLetter(double doubleNumberParam)
			throws NumberFormatException {

		double doubleNumber = doubleNumberParam;

		StringBuilder converted = new StringBuilder();

		String patternThreeDecimalPoints = "#.###";

		DecimalFormatSymbols simbolo = new DecimalFormatSymbols();
		simbolo.setDecimalSeparator('.');
		simbolo.setGroupingSeparator(',');

		DecimalFormat format = new DecimalFormat(patternThreeDecimalPoints,
				simbolo);
		format.setRoundingMode(RoundingMode.DOWN);

		// formateamos el numero, para ajustarlo a el formato de tres puntos
		// decimales
		String formatedDouble = format.format(doubleNumber);
		doubleNumber = Double.parseDouble(formatedDouble);

		// Validamos que sea un numero legal
		if (doubleNumber > Y41bNumberToLetterConverter.MAX_NUMBER) {
			throw new NumberFormatException(
					"El numero es mayor de 999'999.999, "
							+ "no es posible convertirlo");
		}

		if (doubleNumber < 0) {
			throw new NumberFormatException("El numero debe ser positivo");
		}

		String splitNumber[] = formatedDouble.replace('.', '#').split("#");

		// Descompone el trio de millones
		int millon = Integer.parseInt(String
				.valueOf(Y41bNumberToLetterConverter.getDigitAt(splitNumber[0],
						Y41bNumberToLetterConverter.OCHO))
				+ String.valueOf(Y41bNumberToLetterConverter.getDigitAt(
						splitNumber[0], Y41bNumberToLetterConverter.SIETE))
				+ String.valueOf(Y41bNumberToLetterConverter.getDigitAt(
						splitNumber[0], Y41bNumberToLetterConverter.SEIS)));
		if (millon == 1) {
			converted.append("UN MILLON ");
		} else if (millon > 1) {
			converted.append(Y41bNumberToLetterConverter.convertNumber(String
					.valueOf(millon)) + "MILLONES ");
		}

		// Descompone el trio de miles
		int miles = Integer.parseInt(String.valueOf(Y41bNumberToLetterConverter
				.getDigitAt(splitNumber[0], Y41bNumberToLetterConverter.CINCO))
				+ String.valueOf(Y41bNumberToLetterConverter.getDigitAt(
						splitNumber[0], Y41bNumberToLetterConverter.CUATRO))
				+ String.valueOf(Y41bNumberToLetterConverter.getDigitAt(
						splitNumber[0], Y41bNumberToLetterConverter.TRES)));
		if (miles == 1) {
			converted.append("MIL ");
		} else if (miles > 1) {
			converted.append(Y41bNumberToLetterConverter.convertNumber(String
					.valueOf(miles)) + "MIL ");
		}

		// Descompone el ultimo trio de unidades
		int cientos = Integer.parseInt(String
				.valueOf(Y41bNumberToLetterConverter.getDigitAt(splitNumber[0],
						2))
				+ String.valueOf(Y41bNumberToLetterConverter.getDigitAt(
						splitNumber[0], 1))
				+ String.valueOf(Y41bNumberToLetterConverter.getDigitAt(
						splitNumber[0], 0)));
		if (cientos == 1) {
			converted.append("UN");
		}

		if (millon + miles + cientos == 0) {
			converted.append("CERO");
		}
		if (cientos > 1) {
			converted.append(Y41bNumberToLetterConverter.convertNumber(String
					.valueOf(cientos)));
		}

		// Descompone los centimos
		if (splitNumber.length > 1) {
			int centimos = Integer.parseInt(String
					.valueOf(Y41bNumberToLetterConverter.getDigitAt(
							splitNumber[1], 2))
					+ String.valueOf(Y41bNumberToLetterConverter.getDigitAt(
							splitNumber[1], 1))
					+ String.valueOf(Y41bNumberToLetterConverter.getDigitAt(
							splitNumber[1], 0)));
			if (centimos == 1) {
				converted.append(" CON UNO");
			} else if (centimos > 1) {
				converted.append(" CON "
						+ Y41bNumberToLetterConverter.convertNumber(String
								.valueOf(centimos)));
			}
		}
		return converted.toString().trim();
	}

	/**
	 * Convierte los trios de numeros que componen las unidades, las decenas y
	 * las centenas del numero.
	 * 
	 * @param number
	 *            Numero a convetir en digitos
	 * @return Numero convertido en letras
	 */
	private static String convertNumber(String number) {

		if (number.length() > Y41bNumberToLetterConverter.TRES) {
			throw new NumberFormatException(
					"La longitud maxima debe ser 3 digitos");
		}

		// Caso especial con el 100
		if ("100".equals(number)) {
			return "CIEN";
		}

		StringBuilder output = new StringBuilder();
		if (Y41bNumberToLetterConverter.getDigitAt(number, 2) != 0) {
			output.append(Y41bNumberToLetterConverter.CENTENAS[Y41bNumberToLetterConverter
					.getDigitAt(number, 2) - 1]);
		}

		int k = Integer.parseInt(String.valueOf(Y41bNumberToLetterConverter
				.getDigitAt(number, 1))
				+ String.valueOf(Y41bNumberToLetterConverter.getDigitAt(number,
						0)));

		if (k <= Y41bNumberToLetterConverter.VEINTE) {
			output.append(Y41bNumberToLetterConverter.UNIDADES[k]);
		} else if (k > Y41bNumberToLetterConverter.TREINTA
				&& Y41bNumberToLetterConverter.getDigitAt(number, 0) != 0) {
			output.append(Y41bNumberToLetterConverter.DECENAS[Y41bNumberToLetterConverter
					.getDigitAt(number, 1) - 2]
					+ "Y "
					+ Y41bNumberToLetterConverter.UNIDADES[Y41bNumberToLetterConverter
							.getDigitAt(number, 0)]);
		} else {
			output.append(Y41bNumberToLetterConverter.DECENAS[Y41bNumberToLetterConverter
					.getDigitAt(number, 1) - 2]
					+ Y41bNumberToLetterConverter.UNIDADES[Y41bNumberToLetterConverter
							.getDigitAt(number, 0)]);
		}

		return output.toString();
	}

	/**
	 * Retorna el digito numerico en la posicion indicada de derecha a izquierda
	 * 
	 * @param origin
	 *            Cadena en la cual se busca el digito
	 * @param position
	 *            Posicion de derecha a izquierda a retornar
	 * @return Digito ubicado en la posicion indicada
	 */
	private static int getDigitAt(String origin, int position) {
		if (origin.length() > position && position >= 0) {
			return origin.charAt(origin.length() - position - 1)
					- Y41bNumberToLetterConverter.CUARENTA_Y_OCHO;
		}
		return 0;
	}

}
