package com.ejie.y41b.service;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.x38.dto.Pagination;
import com.ejie.y41b.dao.ProductoDao;
import com.ejie.y41b.model.Campanha;
import com.ejie.y41b.model.CampanhaProd;
import com.ejie.y41b.model.Descriptor;
import com.ejie.y41b.model.Producto;
import com.ejie.y41b.utils.exception.Y41bUDAException;

/**
 * * ProductoServiceImpl  
 * 
 *  
 */

@Service(value = "productoService")
public class ProductoServiceImpl implements ProductoService {

	private static final Logger logger = LoggerFactory
			.getLogger(ProductoServiceImpl.class);
	@Autowired
	private ProductoDao productoDao;

	/**
	 * Inserts a single row in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @return Producto
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Producto add(Producto producto) {
		return this.productoDao.add(producto);
	}

	/**
	 * Inserts a single row in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @return Producto
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Producto addProducto(Producto producto) {
		try {
			producto.setPscodigo(producto.getPscodigo().toUpperCase());
			this.productoDao.add(producto);

			String idsDescriptores = producto.getIdsDescriptores();

			List<Descriptor> descriptores = new ArrayList<Descriptor>();

			// Generar el listado listado de etiquetas (comun)
			if (null != idsDescriptores && !"".equals(idsDescriptores)) {
				Descriptor descriptor = null;
				for (String id : idsDescriptores.split(",")) {
					descriptor = new Descriptor();
					descriptor.setIdDescriptor(id);
					descriptores.add(descriptor);
				}
			}

			producto.setDescriptores(descriptores);

			this.productoDao.addProductoDescriptor(producto);

			if (producto.getPscodigo().length() > 1) {
				if (producto.getPscodigo().length() == 2) {
					throw new Y41bUDAException(
							"error.producto.NoPadreException", true,
							new Exception());
				}

				String pscodigoPadre = null;

				if (producto.getPscodigo().length() == 3) {
					pscodigoPadre = producto.getPscodigo().substring(0,
							producto.getPscodigo().length() - 2);
				} else {
					pscodigoPadre = producto.getPscodigo().substring(0,
							producto.getPscodigo().length() - 1);
				}

				Producto productoAux = new Producto();
				productoAux.setPscodigo(pscodigoPadre);

				Long numProducto = this.productoDao.findAllCount(productoAux);

				if (numProducto == 0) {
					throw new Y41bUDAException(
							"error.producto.NoPadreException", true,
							new Exception());
				}
			}
			return producto;
		} catch (DuplicateKeyException e) {
			throw new Y41bUDAException("error.producto.DuplicateKeyException",
					true, e);
		}
	}

	/**
	 * Updates a single row in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @return Producto
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Producto update(Producto producto) {
		return this.productoDao.update(producto);
	}

	/**
	 * Updates a single row in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @return Producto
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Producto updateProducto(Producto producto) {
		this.productoDao.update(producto);

		String idsDescriptores = producto.getIdsDescriptores();

		List<Descriptor> descriptores = new ArrayList<Descriptor>();

		// Generar el listado listado de etiquetas (comun)
		if (null != idsDescriptores && !"".equals(idsDescriptores)) {
			Descriptor descriptor = null;
			for (String id : idsDescriptores.split(",")) {
				descriptor = new Descriptor();
				descriptor.setIdDescriptor(id);
				descriptores.add(descriptor);
			}
		}

		producto.setDescriptores(descriptores);

		// Descriptores
		this.productoDao.removeAllProductoDescriptor(producto);
		this.productoDao.addProductoDescriptor(producto);

		return producto;
	}

	/**
	 * Finds a single row in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @return Producto
	 */
	public Producto find(Producto producto) {
		return (Producto) this.productoDao.find(producto);
	}

	/**
	 * Finds a single row in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @return Producto
	 */
	public Producto findProducto(Producto producto) {
		producto = this.productoDao.find(producto);

		List<Descriptor> listDescriptor = this.productoDao
				.findAllProductoDescriptor(producto, null);

		StringBuffer strIdsDescriptores = new StringBuffer();
		StringBuffer strDescsDescriptores = new StringBuffer();

		strDescsDescriptores.append("[");

		boolean firstElement = true;

		// se recorre para obtener los ids de etiquetas
		for (Descriptor descriptor : listDescriptor) {
			if (firstElement) {
				strIdsDescriptores.append(descriptor.getIdDescriptor());
				strDescsDescriptores.append(descriptor.getNombre());
				firstElement = false;
			} else {
				strIdsDescriptores.append(",").append(
						descriptor.getIdDescriptor());
				strDescsDescriptores.append(",").append(descriptor.getNombre());
			}
		}
		strDescsDescriptores.append("]");

		producto.setIdsDescriptores(strIdsDescriptores.toString());
		producto.setDescsDescriptores(strDescsDescriptores.toString());

		return producto;
	}

	/**
	 * Finds a List of rows in the ProductoDescriptor table.
	 * 
	 * @param producto
	 *            Producto
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<Descriptor> findAllProductoDescriptor(Producto producto,
			Pagination pagination) {
		return (List<Descriptor>) this.productoDao.findAllProductoDescriptor(
				producto, pagination);
	}

	/**
	 * Finds a single row in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @return Producto
	 */
	public Producto findProductoSeleccionable(Producto producto) {
		return (Producto) this.productoDao.findProductoSeleccionable(producto);
	}

	/**
	 * Finds a single row in the Producto table.
	 * 
	 * @param campanhaProd
	 *            CampanhaProd
	 * @return Producto
	 */
	public Producto findProductoSeleccionableCampanha(CampanhaProd campanhaProd) {
		return (Producto) this.productoDao
				.findProductoSeleccionableCampanha(campanhaProd);
	}

	/**
	 * Finds a List of rows in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<Producto> findAll(Producto producto, Pagination pagination) {
		return (List<Producto>) this.productoDao.findAll(producto, pagination);
	}

	/**
	 * Counts rows in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @return Long
	 */
	public Long findAllCount(Producto producto) {
		return this.productoDao.findAllCount(producto);
	}

	/**
	 * Finds rows in the Producto table using like.
	 * 
	 * @param producto
	 *            Producto
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	public List<Producto> findAllLike(Producto producto, Pagination pagination,
			Boolean startsWith) {
		return (List<Producto>) this.productoDao.findAllLike(producto,
				pagination, startsWith);
	}

	/**
	 * Counts rows in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @param startsWith
	 *            Boolean
	 * @return Long
	 */
	public Long findAllLikeCount(Producto producto, Boolean startsWith) {
		return this.productoDao.findAllLikeCount(producto, startsWith);
	}

	/**
	 * Finds rows in the Producto table using like.
	 * 
	 * @param producto
	 *            Producto
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	public List<Producto> findAllLikeProducto(Producto producto,
			Pagination pagination, Boolean startsWith) {
		return (List<Producto>) this.productoDao.findAllLikeProducto(producto,
				pagination, startsWith);
	}

	/**
	 * Counts rows in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @param startsWith
	 *            Boolean
	 * @return Long
	 */
	public Long findAllLikeProductoCount(Producto producto, Boolean startsWith) {
		return this.productoDao.findAllLikeProductoCount(producto, startsWith);
	}

	/**
	 * Finds rows in the Producto table using like.
	 * 
	 * @param producto
	 *            Producto
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	public List<Producto> findAllLikeProductoSeleccion(Producto producto,
			Pagination pagination, Boolean startsWith) {
		return (List<Producto>) this.productoDao.findAllLikeProductoSeleccion(
				producto, pagination, startsWith);
	}

	/**
	 * Counts rows in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @param startsWith
	 *            Boolean
	 * @return Long
	 */
	public Long findAllLikeProductoSeleccionCount(Producto producto,
			Boolean startsWith) {
		return this.productoDao.findAllLikeProductoSeleccionCount(producto,
				startsWith);
	}

	/**
	 * Finds rows in the Producto table using like.
	 * 
	 * @param producto
	 *            Producto
	 * @param campanha
	 *            Campanha
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	public List<Producto> findAllLikeProductoSeleccionCampanha(
			Producto producto, Campanha campanha, Pagination pagination,
			Boolean startsWith) {
		return (List<Producto>) this.productoDao
				.findAllLikeProductoSeleccionCampanha(producto, campanha,
						pagination, startsWith);
	}

	/**
	 * Counts rows in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @param campanha
	 *            Campanha
	 * @param startsWith
	 *            Boolean
	 * @return Long
	 */
	public Long findAllLikeProductoSeleccionCampanhaCount(Producto producto,
			Campanha campanha, Boolean startsWith) {
		return this.productoDao.findAllLikeProductoSeleccionCampanhaCount(
				producto, campanha, startsWith);
	}

	/**
	 * Deletes a single row in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void remove(Producto producto) {
		this.productoDao.remove(producto);
	}

	/**
	 * Deletes a single row in the Producto table.
	 * 
	 * @param producto
	 *            Producto
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeProducto(Producto producto) {

		try {
			// Descriptores
			this.productoDao.removeAllProductoDescriptor(producto);

			this.productoDao.remove(producto);

			Producto productoAux = new Producto();
			productoAux.setPscodigo(producto.getPscodigo());

			Long numProducto = this.productoDao.findAllLikeCount(productoAux,
					true);

			if (numProducto > 0) {
				throw new Y41bUDAException(
						"error.producto.TieneHijosException", true,
						new Exception());
			}
		} catch (DataIntegrityViolationException e) {
			throw new Y41bUDAException(
					"error.producto.DataIntegrityViolationException", true, e);
		}
	}

	/**
	 * Deletes multiple rows in the Producto table.
	 * 
	 * @param productoList
	 *            ArrayList
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeMultiple(ArrayList<Producto> productoList) {
		for (Producto productoAux : productoList) {
			this.productoDao.remove(productoAux);
		}
	}

	/**
	 * Getter method for ProductoDao
	 * 
	 * @return ProductoDao
	 */
	public ProductoDao getProductoDao() {
		return this.productoDao;
	}

	/**
	 * Setter method for ProductoDao.
	 * 
	 * @param productoDao
	 *            ProductoDao
	 * @return
	 */
	public void setProductoDao(ProductoDao productoDao) {
		logger.info("Setting Dependency " + productoDao);
		this.productoDao = productoDao;
	}
}
