package com.ejie.y41b.service;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.x38.dto.Pagination;
import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.dao.PosibleTramiteDao;
import com.ejie.y41b.dao.TramiteDao;
import com.ejie.y41b.model.CensoMonitor;
import com.ejie.y41b.model.Decision;
import com.ejie.y41b.model.PosibleTramite;
import com.ejie.y41b.model.Tramite;
import com.ejie.y41b.utils.Y41bUIDGenerator;
import com.ejie.y41b.utils.exception.Y41bUDAException;

/**
 * PosibleTramiteServiceImpl  
 * 
 *  
 */

@Service(value = "posibleTramiteService")
public class PosibleTramiteServiceImpl implements PosibleTramiteService {

	@Autowired
	private PosibleTramiteDao posibleTramiteDao;

	@Autowired
	private TramiteDao tramiteDao;

	/**
	 * Inserts a single row in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @return PosibleTramite
	 */
	@Transactional(rollbackFor = Throwable.class)
	public PosibleTramite add(PosibleTramite posibleTramite) {
		return this.posibleTramiteDao.add(posibleTramite);
	}

	/**
	 * Updates a single row in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @return PosibleTramite
	 */
	@Transactional(rollbackFor = Throwable.class)
	public PosibleTramite update(PosibleTramite posibleTramite) {
		return this.posibleTramiteDao.update(posibleTramite);
	}

	/**
	 * Updates a single row in the Tramite table.
	 * 
	 * @param tramite
	 *            Tramite
	 * @return Tramite
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Tramite updateTramite(Tramite tramite) {
		this.tramiteDao.updateTramite(tramite);

		this.posibleTramiteDao.removeAllPosibleTramites(tramite);
		this.posibleTramiteDao.addAllPosibleTramites(tramite);
		this.posibleTramiteDao.removeAllPosiblesTramitesTemporal(tramite);

		return tramite;
	}

	/**
	 * Finds a single row in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @return PosibleTramite
	 */
	public PosibleTramite find(PosibleTramite posibleTramite) {
		return (PosibleTramite) this.posibleTramiteDao.find(posibleTramite);
	}

	/**
	 * Finds a single row in the Tramite table.
	 * 
	 * @param tramite
	 *            Tramite
	 * @return Tramite
	 */
	public Tramite findTramite(Tramite tramite) {
		tramite = (Tramite) this.tramiteDao.find(tramite);

		this.posibleTramiteDao.removeAllPosiblesTramitesTemporal(tramite);
		this.posibleTramiteDao.addPosiblesTramitesTemporal(tramite);

		return tramite;
	}

	/**
	 * Finds a List of rows in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<PosibleTramite> findAll(PosibleTramite posibleTramite, Pagination pagination) {
		return (List<PosibleTramite>) this.posibleTramiteDao.findAll(posibleTramite, pagination);
	}

	/**
	 * Counts rows in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @return Long
	 */
	public Long findAllCount(PosibleTramite posibleTramite) {
		return this.posibleTramiteDao.findAllCount(posibleTramite);
	}

	/**
	 * Finds rows in the PosibleTramite table using like.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	public List<PosibleTramite> findAllLike(PosibleTramite posibleTramite, Pagination pagination, Boolean startsWith) {
		return (List<PosibleTramite>) this.posibleTramiteDao.findAllLike(posibleTramite, pagination, startsWith);
	}

	/**
	 * Counts rows in the PosibleTramite table using like.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @param startsWith
	 *            Boolean
	 * @return Long
	 */
	public Long findAllLikeCount(PosibleTramite posibleTramite, Boolean startsWith) {
		return this.posibleTramiteDao.findAllLikeCount(posibleTramite, startsWith);
	}

	/**
	 * Deletes a single row in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void remove(PosibleTramite posibleTramite) {
		this.posibleTramiteDao.remove(posibleTramite);
	}

	/**
	 * Deletes multiple rows in the PosibleTramite table.
	 * 
	 * @param posibleTramiteList
	 *            List
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeMultiple(List<PosibleTramite> posibleTramiteList) {
		for (PosibleTramite posibleTramiteAux : posibleTramiteList) {
			this.posibleTramiteDao.remove(posibleTramiteAux);
		}
	}

	/**
	 * Finds a List of rows in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @param censoMonitor
	 *            CensoMonitor
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<Tramite> findAllPosibleTramite(PosibleTramite posibleTramite, CensoMonitor censoMonitor,
			Pagination pagination) {
		return (List<Tramite>) this.posibleTramiteDao.findAllPosibleTramite(posibleTramite, censoMonitor, pagination);
	}

	/**
	 * Finds a List of rows in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @param censoMonitor
	 *            CensoMonitor
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<Tramite> findAllPosibleTramiteConsumidor(PosibleTramite posibleTramite, CensoMonitor censoMonitor,
			Pagination pagination) {
		List<Tramite> listTramite = (List<Tramite>) this.posibleTramiteDao.findAllPosibleTramite(posibleTramite,
				censoMonitor, pagination);

		if (Y41bConstantes.TIPO_SOLICITUD_DENUNCIA.equals(posibleTramite.getTiposolicitud())) {
			Tramite tramiteAux = new Tramite();
			for (int i = 0; i < listTramite.size(); i++) {
				tramiteAux = listTramite.get(i);
				if (Y41bConstantes.TRAMITE_REQUERIMIENTO_ALEGACION.equals(tramiteAux.getTrcodigo())) {
					listTramite.remove(i);
					break;
				}
			}
		}
		return listTramite;
	}

	/**
	 * Finds a List of rows in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @param censoMonitor
	 *            CensoMonitor
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<Tramite> findAllPosibleTramiteInspecciones(PosibleTramite posibleTramite, CensoMonitor censoMonitor,
			Pagination pagination) {
		List<Tramite> listTramite = (List<Tramite>) this.posibleTramiteDao.findAllPosibleTramite(posibleTramite,
				censoMonitor, pagination);

		if (censoMonitor.getPerfilOrganizacion().getIdPerfil().equals(Y41bConstantes.PERFIL_ADMINISTRADOR_MANTENIMIENTO)
				|| censoMonitor.getPerfilInspecciones().getIdPerfil().equals(Y41bConstantes.PERFIL_COORDINADOR)) {
			// Si es control de mercado insertar en la lista el tramite orden de
			// servicio
			Tramite tramite = new Tramite();
			tramite.setTrcodigo(Y41bConstantes.TRAMITE_INSPECCIONES_ORDEN_SERVICIO);
			tramite = this.tramiteDao.find(tramite);
			listTramite.add(tramite);
		} else {
			// Si no es control de mercado eliminarlo de la lista
			for (Tramite tramiteAux : listTramite) {
				if (tramiteAux.getTrcodigo().equals(Y41bConstantes.TRAMITE_INSPECCIONES_ORDEN_SERVICIO)) {
					listTramite.remove(tramiteAux);
				}
			}
		}

		return listTramite;
	}

	/**
	 * Counts rows in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @param censoMonitor
	 *            CensoMonitor
	 * @return Long
	 */
	public Long findAllPosibleTramiteCount(PosibleTramite posibleTramite, CensoMonitor censoMonitor) {
		return this.posibleTramiteDao.findAllPosibleTramiteCount(posibleTramite, censoMonitor);
	}

	/**
	 * Finds a List of rows in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<Tramite> findAllPosibleTramiteTemporal(PosibleTramite posibleTramite, Pagination pagination) {
		if (pagination != null) {
			if (pagination.getSort() != null && pagination.getSort().equals("fase.fnombre")) {
				pagination.setSort("FNOMBRE");
			}
		}

		if (pagination != null) {
			if (pagination.getSort() != null && pagination.getSort().equals("fase.fnombree")) {
				pagination.setSort("FNOMBREE");
			}
		}

		return (List<Tramite>) this.posibleTramiteDao.findAllPosibleTramiteTemporal(posibleTramite, pagination);
	}

	/**
	 * Counts rows in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @return Long
	 */
	public Long findAllPosibleTramiteTemporalCount(PosibleTramite posibleTramite) {
		return this.posibleTramiteDao.findAllPosibleTramiteTemporalCount(posibleTramite);
	}

	/**
	 * Inserts a single row in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @return PosibleTramite
	 */
	@Transactional(rollbackFor = Throwable.class)
	public PosibleTramite addPosibleTramiteTemporal(PosibleTramite posibleTramite) {
		posibleTramite.setTrposiblecodigo(Y41bUIDGenerator.getInstance().generateId(Y41bConstantes.PK_SIZE));

		if (this.posibleTramiteDao.findAllExistePosibleTramiteTemporalCount(posibleTramite) > 0) {
			throw new Y41bUDAException("error.anyadirposibletramite.duplicateException", true, new Exception());
		}

		return this.posibleTramiteDao.addPosibleTramiteTemporal(posibleTramite);
	}

	/**
	 * Deletes a single row in the PosibleTramite table.
	 * 
	 * @param posibleTramite
	 *            PosibleTramite
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removePosibleTramiteTemporal(PosibleTramite posibleTramite) {
		this.posibleTramiteDao.removePosibleTramiteTemporal(posibleTramite);
	}
}
