package com.ejie.y41b.service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.x38.dto.Pagination;
import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.dao.ConsultaDao;
import com.ejie.y41b.dao.ConsultaFileDao;
import com.ejie.y41b.dao.NumConsultaDao;
import com.ejie.y41b.dao.OrganismoDao;
import com.ejie.y41b.dao.RespuestaDao;
import com.ejie.y41b.dao.RespuestaFileDao;
import com.ejie.y41b.model.AttachFile;
import com.ejie.y41b.model.Consulta;
import com.ejie.y41b.model.ConsultaBusqueda;
import com.ejie.y41b.model.ConsultaDetalle;
import com.ejie.y41b.model.ConsultaFile;
import com.ejie.y41b.model.DocumentoConsultaDetalle;
import com.ejie.y41b.model.EstadisticasConsultasResultado;
import com.ejie.y41b.model.EstadisticasConsultasTiempoAtencion;
import com.ejie.y41b.model.EstadisticasKontsumonetBusqueda;
import com.ejie.y41b.model.EstadisticasKontsumonetResultado;
import com.ejie.y41b.model.ListadoConsultaResultado;
import com.ejie.y41b.model.Organismo;
import com.ejie.y41b.model.Respuesta;
import com.ejie.y41b.model.RespuestaFile;
import com.ejie.y41b.model.TipoRespuesta;
import com.ejie.y41b.utils.Y41bUIDGenerator;
import com.ejie.y41b.utils.Y41bUtils;
import com.ejie.y41b.utils.config.Y41bConfig;
import com.ejie.y41b.utils.exception.Y41bUDAException;

/**
 * * ConsultaServiceImpl  
 * 
 *  
 */

@Service(value = "consultaService")
public class ConsultaServiceImpl implements ConsultaService {

	private static final Logger logger = LoggerFactory.getLogger(ConsultaServiceImpl.class);
	@Autowired
	private ConsultaDao consultaDao;
	@Autowired
	private RespuestaDao respuestaDao;
	@Autowired
	private RespuestaFileDao respuestaFileDao;
	@Autowired
	private ConsultaFileDao consultaFileDao;
	@Autowired
	private OrganismoDao organismoDao;
	@Autowired
	private NumConsultaDao numConsultaDao;

	@Resource
	ReloadableResourceBundleMessageSource appMessageSource;

	/**
	 * Inserts a single row in the Consulta table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @return Consulta
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Consulta add(Consulta consulta) {
		return this.consultaDao.add(consulta);
	}

	/**
	 * Inserts a single row in the Consulta table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param respuesta
	 *            Respuesta
	 * @return Consulta
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Consulta addConsulta(Consulta consulta, Respuesta respuesta) {
		consulta.setCocodigo(Y41bUIDGenerator.getInstance().generateId(Y41bConstantes.PK_SIZE));
		consulta.setCoinl37(Y41bConstantes.VALOR_NO);
		consulta.setCoinl21(Y41bConstantes.VALOR_SI);
		consulta.setCosituacion(Y41bConstantes.SITUACION_PENDIENTES);

		ConsultaDetalle consultaDetalle = new ConsultaDetalle();
		consultaDetalle.setCodOrg(consulta.getOrganismo().getCodOrg());
		consultaDetalle.setConranrg(consulta.getConranrg());
		consultaDetalle = this.numConsultaDao.findNumConsulta(consultaDetalle);
		consultaDetalle.setCodOrg(consulta.getOrganismo().getCodOrg());
		consultaDetalle.setConranrg(consulta.getConranrg());
		this.numConsultaDao.add(consultaDetalle);

		consulta.setConrreg(consultaDetalle.getConrreg());

		this.consultaDao.addConsulta(consulta);

		if (respuesta.getP9feresp() != null) {
			respuesta.setP9codigo(Y41bUIDGenerator.getInstance().generateId(Y41bConstantes.PK_SIZE));
			respuesta.setConsulta(consulta);

			this.respuestaDao.add(respuesta);

			consulta.setCoferes(respuesta.getP9feresp());
			consulta.setCosituacion(Y41bConstantes.SITUACION_RESUELTAS);

			this.consultaDao.updateRespuesta(consulta);
		}
		return consulta;
	}

	/**
	 * Updates a single row in the Consulta table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @return Consulta
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Consulta update(Consulta consulta) {
		return this.consultaDao.update(consulta);
	}

	/**
	 * Updates a single row in the Consulta table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param respuesta
	 *            Respuesta
	 * @return Consulta
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Consulta updateConsulta(Consulta consulta, Respuesta respuesta) {

		if (respuesta.getP9feresp() != null) {

			Respuesta busqueRespuestas = new Respuesta();
			Consulta cons = new Consulta();
			cons.setCocodigo(consulta.getCocodigo());
			busqueRespuestas.setConsulta(cons);

			List<Respuesta> listRespuesta = this.respuestaDao.findAll(busqueRespuestas, null);

			if (listRespuesta.size() == 1) {
				Respuesta respuestaAux = listRespuesta.get(0);
				respuesta.setP9codigo(respuestaAux.getP9codigo());
				respuesta.setConsulta(cons);
				this.respuestaDao.update(respuesta);
			} else {
				respuesta.setP9codigo(Y41bUIDGenerator.getInstance().generateId(Y41bConstantes.PK_SIZE));
				respuesta.setConsulta(consulta);

				this.respuestaDao.add(respuesta);
			}

			consulta.setCoferes(respuesta.getP9feresp());
			consulta.setCosituacion(Y41bConstantes.SITUACION_RESUELTAS);

			this.consultaDao.update(consulta);
		} else {
			consulta.setCosituacion(Y41bConstantes.SITUACION_PENDIENTES);

			this.consultaDao.update(consulta);

			this.respuestaDao.removeRespuestaConsulta(consulta);

		}
		return consulta;
	}

	/**
	 * Responde una consulta.
	 * 
	 * @param respuesta
	 *            Respuesta
	 * @return Consulta
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Respuesta responderConsulta(Respuesta respuesta) {

		Respuesta busqueRespuestas = new Respuesta();
		Consulta cons = new Consulta();
		cons.setCocodigo(respuesta.getConsulta().getCocodigo());
		busqueRespuestas.setConsulta(cons);

		List<Respuesta> listRespuesta = this.respuestaDao.findAll(busqueRespuestas, null);

		if (listRespuesta.size() == 1) {
			Respuesta respuestaAux = listRespuesta.get(0);
			respuesta.setP9codigo(respuestaAux.getP9codigo());
			this.respuestaDao.update(respuesta);
		} else {
			respuesta.setP9codigo(Y41bUIDGenerator.getInstance().generateId(Y41bConstantes.PK_SIZE));

			this.respuestaDao.add(respuesta);
		}

		Consulta consulta = new Consulta();
		consulta.setCocodigo(respuesta.getConsulta().getCocodigo());
		consulta.setCensoMonitor(respuesta.getConsulta().getCensoMonitor());
		consulta.setCoferes(respuesta.getP9feresp());
		consulta.setCosituacion(Y41bConstantes.SITUACION_RESUELTAS);

		this.consultaDao.updateRespuesta(consulta);

		return respuesta;
	}

	/**
	 * Responde por email a una consulta.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param respuesta
	 *            Respuesta
	 * @return Consulta
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Respuesta emailConsulta(Consulta consulta, Respuesta respuesta) {

		Respuesta busqueRespuestas = new Respuesta();
		Consulta cons = new Consulta();
		cons.setCocodigo(consulta.getCocodigo());
		busqueRespuestas.setConsulta(cons);

		List<Respuesta> listRespuesta = this.respuestaDao.findAll(busqueRespuestas, null);

		if (listRespuesta.size() == 1) {
			Respuesta respuestaAux = listRespuesta.get(0);
			respuesta.setP9codigo(respuestaAux.getP9codigo());
			respuesta.setP9inemlusu(Y41bConstantes.VALOR_SI);
			this.respuestaDao.update(respuesta);
		} else {
			respuesta.setP9codigo(Y41bUIDGenerator.getInstance().generateId(Y41bConstantes.PK_SIZE));
			respuesta.setP9inemlusu(Y41bConstantes.VALOR_SI);

			this.respuestaDao.add(respuesta);
		}

		String destinatariosMail = consulta.getCocoemco();

		// buscar organismo
		Organismo organismo = new Organismo();
		organismo.setCodOrg(consulta.getOrganismo().getCodOrg());
		organismo = this.organismoDao.find(organismo);

		String from = organismo.getEmail();

		Properties prop = Y41bConfig.loadProperties(Y41bConstantes.CONFIG_PATH);

		String smtpMail = (String) prop.getProperty("y41bVistaWar.mail.ejie.smtp");

		StringBuffer asunto = new StringBuffer();
		asunto.append(appMessageSource.getMessage("consulta.mail.asunto", null, new Locale(Y41bConstantes.CASTELLANO)));

		StringBuffer numeroConsulta = new StringBuffer();

		numeroConsulta.append(consulta.getOrganismo().getCodOrg()).append("-").append(consulta.getConrreg()).append("-")
				.append(consulta.getConranrg());

		asunto.append(" ").append(numeroConsulta).append(" / ");

		asunto.append(appMessageSource.getMessage("consulta.mail.asunto", null, new Locale(Y41bConstantes.EUSKERA)));
		asunto.append(" ").append(numeroConsulta);

		StringBuffer texto = new StringBuffer();
		texto.append("<B>").append(appMessageSource.getMessage("consulta.mail.texto.consulta", null,
				new Locale(Y41bConstantes.CASTELLANO)));
		texto.append(" / ").append(
				appMessageSource.getMessage("consulta.mail.texto.consulta", null, new Locale(Y41bConstantes.EUSKERA)))
				.append("</B><BR>");
		texto.append(consulta.getCotexto());

		texto.append("<BR><BR><B>").append(appMessageSource.getMessage("consulta.mail.texto.respuesta", null,
				new Locale(Y41bConstantes.CASTELLANO)));
		texto.append(" / ").append(
				appMessageSource.getMessage("consulta.mail.texto.respuesta", null, new Locale(Y41bConstantes.EUSKERA)))
				.append("</B><BR>");
		texto.append(respuesta.getP9teres());

		RespuestaFile respuestaFileAux = new RespuestaFile();
		Respuesta respuestaAux = new Respuesta();
		respuestaAux.setP9codigo(respuesta.getP9codigo());
		respuestaFileAux.setQ3enviadomail(Y41bConstantes.VALOR_NO);

		List<RespuestaFile> listRespuestaFile = this.respuestaFileDao.findAll(respuestaFileAux, null);
		ArrayList attachments = new ArrayList();

		for (RespuestaFile respFile : listRespuestaFile) {
			AttachFile fileattach = new AttachFile();

			fileattach.setContentType(respFile.getQ3contenttypefile());
			fileattach.setNombre(respFile.getQ3nofile());

			respFile.setQ3datafile(respFile.getQ3datafile());

			fileattach.setAdjunto(respFile.getQ3datafile());

			respFile.setQ3enviadomail(Y41bConstantes.VALOR_SI);

			this.respuestaFileDao.update(respFile);

			attachments.add(fileattach);
		}

		// Envio mail
		boolean envio = Y41bUtils.sendMail(smtpMail, asunto.toString(), texto.toString(), from, destinatariosMail, "",
				attachments);

		if (envio) {

			consulta.setCocodigo(respuesta.getConsulta().getCocodigo());
			consulta.setCensoMonitor(respuesta.getConsulta().getCensoMonitor());
			consulta.setCoferes(respuesta.getP9feresp());
			consulta.setCosituacion(Y41bConstantes.SITUACION_RESUELTAS);

			this.consultaDao.updateRespuestaMail(consulta);
		} else {
			throw new Y41bUDAException("error.consulta.respuestaemail", true, new Exception());
		}

		return respuesta;
	}

	/**
	 * Finds a single row in the Consulta table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @return Consulta
	 */
	public Consulta find(Consulta consulta) {
		return (Consulta) this.consultaDao.find(consulta);
	}

	/**
	 * Finds a single row in the Consulta table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @return ConsultaDetalle
	 */
	public ConsultaDetalle findConsulta(Consulta consulta) {
		Consulta consul = (Consulta) this.consultaDao.find(consulta);

		Organismo organismo = new Organismo();
		organismo.setCodOrg(consul.getOrganismo().getCodOrg());
		organismo = this.organismoDao.find(organismo);

		ConsultaDetalle consultaDetalle = new ConsultaDetalle(consul.getCocodigo(), consul.getMotivo().getTircodigo(),
				consul.getMotivo().getTirnombre(), consul.getProducto().getPscodigo(),
				consul.getProducto().getPsnombre(), consul.getProducto().getPsnombree(),
				consul.getRangoEdad().getRancod(), consul.getRangoEdad().getRandesc(),
				consul.getFormaPresentacion().getTfpcodigo(), consul.getFormaPresentacion().getTfpnombre(),
				consul.getFormaPresentacion().getTfpnombree(), consul.getOrganismo().getCodOrg(),
				organismo.getNombreEs(), organismo.getNombreEu(), organismo.getTelefono(), organismo.getFax(),
				organismo.getEmail(), organismo.getUrl(), organismo.getNoraPaisId(), organismo.getNombrePais(),
				organismo.getNoraProvinciaId(), organismo.getNombreProvincia(), organismo.getNoraMunicipioId(),
				organismo.getNombreMunicipio(), organismo.getNoraLocalidadId(), organismo.getNombreLocalidad(),
				organismo.getNoraCalleId(), organismo.getNombreCalle(), organismo.getNoraCpId(),
				organismo.getNombreCp(), organismo.getNoraPortalId(), organismo.getNombrePortal(),
				consul.getCensoMonitor().getMocodigo(), consul.getCensoMonitor().getMonombre(), consul.getCofecha(),
				consul.getCocosx(), consul.getConran(), consul.getCocose(), consul.getCotexto(), consul.getConrreg(),
				consul.getConoco(), consul.getConodirco(), consul.getConrcpco(), consul.getCocoloco(),
				consul.getConoloco(), consul.getConomuco(), consul.getCocopvco(), consul.getCocopaco(),
				consul.getConrtelco(), consul.getConrfaxco(), consul.getCocoemco(), consul.getCoferes(),
				consul.getCoinreenv(), consul.getCoteresp(), consul.getCocotfpr(), consul.getCoinl37(),
				consul.getConranrg(), consul.getCocopro(), consul.getCohdcodigo(), consul.getCohora(),
				consul.getCofechaconsumidor(), null, consul.getNoraPaisId(), consul.getNombrePais(),
				consul.getNoraProvinciaId(), consul.getNombreProvincia(), consul.getNoraMunicipioId(),
				consul.getNombreMunicipio(), consul.getNoraLocalidadId(), consul.getNombreLocalidad(),
				consul.getNoraCalleId(), consul.getNombreCalle(), consul.getNoraCpId(), consul.getNombreCp(),
				consul.getNoraPortalId(), consul.getNombrePortal(), consul.getCoinl21(), consul.getCosituacion(),
				consul.getNombrePiso(), consul.getDireccion_nocapv(), null, null, null, null, null, null, null, null,
				null, null, null, null, null, null, consul.getCoextranjero(), consul.getCoconsultaaaccempresas(),
				consul.getCocif(), consul.getCorazonsocial(), consul.getConombrecomercial());

		Respuesta respuesta = new Respuesta();
		respuesta.setConsulta(consul);

		try {
			respuesta = (Respuesta) this.respuestaDao.findRespuesta(respuesta);

			consultaDetalle.setP9codigo(respuesta.getP9codigo());
			consultaDetalle.setP9feanot(respuesta.getP9feanot());
			consultaDetalle.setP9inemlusu(respuesta.getP9inemlusu());
			consultaDetalle.setP9feresp(respuesta.getP9feresp());

			consultaDetalle.setP9nufiladj(respuesta.getP9nufiladj());
			consultaDetalle.setP8codigo(respuesta.getTipoRespuesta().getP8codigo());
			consultaDetalle.setP8nombre(respuesta.getTipoRespuesta().getP8nombre());
			consultaDetalle.setP8nombree(respuesta.getTipoRespuesta().getP8nombree());

			consultaDetalle.setP9teres(respuesta.getP9teres());

		} catch (EmptyResultDataAccessException e) {

		}

		String cocopvco = (consul.getCocopvco() == null) ? "" : consul.getCocopvco();
		String conomuco = (consul.getConomuco() == null) ? "" : consul.getConomuco();
		String conoloco = (consul.getConoloco() == null) ? "" : consul.getConoloco();
		String conodirco = (consul.getConodirco() == null) ? "" : consul.getConodirco();
		String conrcpco = (consul.getConrcpco() == null) ? "" : consul.getConrcpco();

		StringBuffer direccionAntiguaCompleta = new StringBuffer();
		direccionAntiguaCompleta.append(cocopvco).append(" ").append(conomuco).append(" ").append(conoloco).append(" ")
				.append(conodirco).append(" ").append(conrcpco);

		consultaDetalle.setDireccionAntiguaCompleta(direccionAntiguaCompleta.toString());

		return consultaDetalle;
	}

	/**
	 * Finds a List of rows in the Consulta table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<Consulta> findAll(Consulta consulta, Pagination pagination) {
		return (List<Consulta>) this.consultaDao.findAll(consulta, pagination);
	}

	/**
	 * Counts rows in the Consulta table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @return Long
	 */
	public Long findAllCount(Consulta consulta) {
		return this.consultaDao.findAllCount(consulta);
	}

	/**
	 * Finds rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	public List<Consulta> findAllLike(Consulta consulta, Pagination pagination, Boolean startsWith) {
		return (List<Consulta>) this.consultaDao.findAllLike(consulta, pagination, startsWith);
	}

	/**
	 * Deletes a single row in the Consulta table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void remove(Consulta consulta) {
		this.consultaDao.remove(consulta);
	}

	/**
	 * Deletes a single row in the Consulta table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeConsulta(Consulta consulta) {
		// Si la consulta est respondida no deja eliminar
		Consulta consul = (Consulta) this.consultaDao.find(consulta);

		if (consul != null && consul.getCosituacion() != null
				&& consul.getCosituacion().equals(Y41bConstantes.SITUACION_RESUELTAS)) {
			throw new Y41bUDAException("error.consulta.eliminarrespondida", true, new Exception());
		}
		// eliminar los ficheros de la respuesta
		this.respuestaFileDao.removeRespuestaFileConsulta(consulta);
		// eliminar la respuesta
		this.respuestaDao.removeRespuestaConsulta(consulta);
		// eliminar los ficheros de la consulta
		this.consultaFileDao.removeConsultaFileConsulta(consulta);
		// eliminar la consulta
		this.consultaDao.remove(consulta);
	}

	/**
	 * Deletes multiple rows in the Consulta table.
	 * 
	 * @param consultaList
	 *            ArrayList
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeMultiple(ArrayList<Consulta> consultaList) {
		for (Consulta consultaAux : consultaList) {
			this.consultaDao.remove(consultaAux);
		}
	}

	/**
	 * Finds rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @param fechaDesde
	 *            Date
	 * @param fechaHasta
	 *            Date
	 * @return List
	 */
	public List<ConsultaBusqueda> findAllLikeConsulta(Consulta consulta, Pagination pagination, Boolean startsWith,
			Date fechaDesde, Date fechaHasta) {

		// Control para poder paginar por un campo calculado - numeroConsulta
		if (pagination != null) {
			if (pagination.getSort() != null && pagination.getAscDsc().equals("asc")
					&& pagination.getSort().equals("numeroConsulta")) {

				pagination.setSort("cocoor ASC, conranrg ASC, LPAD(conrreg, 12) ");
			}
		}

		if (pagination != null) {
			if (pagination.getSort() != null && pagination.getAscDsc().equals("desc")
					&& pagination.getSort().equals("numeroConsulta")) {

				pagination.setSort("cocoor DESC, conranrg DESC, LPAD(conrreg,12) ");
			}
		}

		List<ConsultaBusqueda> listConsultaBusqueda = new ArrayList<ConsultaBusqueda>();
		List<Consulta> listConsulta = this.consultaDao.findAllLikeConsulta(consulta, pagination, startsWith, fechaDesde,
				fechaHasta);

		String documentos = null;
		for (Consulta cons : listConsulta) {
			StringBuffer numeroConsulta = new StringBuffer();

			numeroConsulta.append(cons.getOrganismo().getCodOrg()).append("-").append(cons.getConrreg()).append("-")
					.append(cons.getConranrg());

			ConsultaFile consultafile = new ConsultaFile();
			consultafile.setQ4origen(Y41bConstantes.ORIGEN_APORTADO_CIUDADANO);
			Consulta consultaAux = new Consulta();
			consultaAux.setCocodigo(cons.getCocodigo());
			consultafile.setConsulta(consultaAux);

			Long numDocumentosConsumidor = this.consultaFileDao.findAllCount(consultafile);

			if (numDocumentosConsumidor > 0) {
				documentos = Y41bConstantes.VALOR_SI;
			} else {
				documentos = null;
			}

			listConsultaBusqueda.add(new ConsultaBusqueda(cons.getCocodigo(), numeroConsulta.toString(),
					cons.getCofecha(), cons.getCensoMonitor().getMonombre(), cons.getNoraMunicipioId(),
					cons.getNombreMunicipio(), cons.getProducto().getPsnombre(), cons.getProducto().getPsnombree(),
					cons.getCoinl37(), cons.getCosituacion(), cons.getFormaPresentacion().getTfpnombre(),
					cons.getFormaPresentacion().getTfpnombree(), documentos, cons.getConombrecomercial()));
		}

		return listConsultaBusqueda;
	}

	/**
	 * Counts rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param startsWith
	 *            Boolean
	 * @param fechaDesde
	 *            Date
	 * @param fechaHasta
	 *            Date
	 * @return Long
	 */
	public Long findAllLikeConsultaCount(Consulta consulta, Boolean startsWith, Date fechaDesde, Date fechaHasta) {
		return this.consultaDao.findAllLikeConsultaCount(consulta, startsWith, fechaDesde, fechaHasta);
	}

	/**
	 * Finds all files of the Consulta File table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<DocumentoConsultaDetalle> findAllFileConsulta(Consulta consulta, Pagination pagination) {

		return (List<DocumentoConsultaDetalle>) this.consultaDao.findAllDocumentosConsulta(consulta, pagination);
	}

	/**
	 * Finds rows in the Consulta table.
	 * 
	 * @param estadisticasKontsumonetBusqueda
	 *            EstadisticasKontsumonetBusqueda
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<EstadisticasKontsumonetResultado> findAllConsultaEstadisticasKontsumonet(
			EstadisticasKontsumonetBusqueda estadisticasKontsumonetBusqueda, Pagination pagination) {

		return (List<EstadisticasKontsumonetResultado>) this.consultaDao
				.findAllConsultaEstadisticasKontsumonet(estadisticasKontsumonetBusqueda, pagination);
	}

	/**
	 * Counts rows in the Consulta File table.
	 * 
	 * @param consulta
	 *            Consulta
	 * @return Long
	 */
	public Long findAllFileConsultaCount(Consulta consulta) {
		return this.consultaDao.findAllDocumentosConsultaCount(consulta);
	}

	/**
	 * Finds a single row in the File Consulta table.
	 * 
	 * @param documentoConsultaDetalle
	 *            DocumentoConsultaDetalle
	 * @return DocumentoConsultaDetalle
	 */
	public DocumentoConsultaDetalle findFileConsulta(DocumentoConsultaDetalle documentoConsultaDetalle) {
		if (documentoConsultaDetalle != null && documentoConsultaDetalle.getTipofile() != null
				&& documentoConsultaDetalle.getTipofile().equals(Y41bConstantes.TIPO_DOCUMENTO_CONSULTA_CONSULTA)) {

			ConsultaFile consultaFile = new ConsultaFile();
			consultaFile.setQ4codocumento(documentoConsultaDetalle.getCofile());

			// obtiene el documento de la consulta
			consultaFile = this.consultaFileDao.find(consultaFile);

			consultaFile.setQ4datafile(consultaFile.getQ4datafile());

			documentoConsultaDetalle = new DocumentoConsultaDetalle(consultaFile.getQ4codocumento(),
					consultaFile.getConsulta().getCocodigo(), null, Y41bConstantes.TIPO_DOCUMENTO_CONSULTA_CONSULTA,
					consultaFile.getQ4fecha(), consultaFile.getQ4nofile(), consultaFile.getQ4datafile(),
					consultaFile.getQ4sizefile(), consultaFile.getQ4tedesfile(), consultaFile.getQ4contenttypefile(),
					consultaFile.getQ4origen(), null);
		} else if (documentoConsultaDetalle != null && documentoConsultaDetalle.getTipofile() != null
				&& documentoConsultaDetalle.getTipofile().equals(Y41bConstantes.TIPO_DOCUMENTO_CONSULTA_RESPUESTA)) {

			RespuestaFile respuestaFile = new RespuestaFile();
			respuestaFile.setQ3cofile(documentoConsultaDetalle.getCofile());

			// obtiene el documento de la respuesta
			respuestaFile = this.respuestaFileDao.find(respuestaFile);

			respuestaFile.setQ3datafile(respuestaFile.getQ3datafile());

			documentoConsultaDetalle = new DocumentoConsultaDetalle(respuestaFile.getQ3cofile(),
					respuestaFile.getRespuesta().getConsulta().getCocodigo(),
					respuestaFile.getRespuesta().getP9codigo(), Y41bConstantes.TIPO_DOCUMENTO_CONSULTA_RESPUESTA,
					respuestaFile.getQ3fecha(), respuestaFile.getQ3nofile(), respuestaFile.getQ3datafile(),
					respuestaFile.getQ3sizefile(), respuestaFile.getQ3tedesfile(), respuestaFile.getQ3contenttypefile(),
					Y41bConstantes.ORIGEN_APORTADO_ADMINISTRACION, respuestaFile.getRespuesta().getP9inemlusu());
		}

		return documentoConsultaDetalle;
	}

	/**
	 * Deletes a document of the Consulta.
	 * 
	 * @param documentoConsultaDetalle
	 *            DocumentoConsultaDetalle
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeLobTablaTemporal(DocumentoConsultaDetalle documentoConsultaDetalle) {
		if (documentoConsultaDetalle != null && documentoConsultaDetalle.getTipofile() != null
				&& documentoConsultaDetalle.getTipofile().equals(Y41bConstantes.TIPO_DOCUMENTO_CONSULTA_CONSULTA)) {

			ConsultaFile consultaFile = new ConsultaFile();
			consultaFile.setQ4codocumento(documentoConsultaDetalle.getCofile());

		} else if (documentoConsultaDetalle != null && documentoConsultaDetalle.getTipofile() != null
				&& documentoConsultaDetalle.getTipofile().equals(Y41bConstantes.TIPO_DOCUMENTO_CONSULTA_RESPUESTA)) {

			RespuestaFile respuestaFile = new RespuestaFile();
			respuestaFile.setQ3cofile(documentoConsultaDetalle.getCofile());
		}
	}

	/**
	 * Deletes a document of the Consulta.
	 * 
	 * @param documentoConsultaDetalle
	 *            DocumentoConsultaDetalle
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeFileConsulta(DocumentoConsultaDetalle documentoConsultaDetalle) {
		if (documentoConsultaDetalle != null && documentoConsultaDetalle.getTipofile() != null
				&& documentoConsultaDetalle.getTipofile().equals(Y41bConstantes.TIPO_DOCUMENTO_CONSULTA_CONSULTA)) {

			ConsultaFile consultaFile = new ConsultaFile();
			consultaFile.setQ4codocumento(documentoConsultaDetalle.getCofile());

			// elimina el documento de la consulta
			this.consultaFileDao.remove(consultaFile);

		} else if (documentoConsultaDetalle != null && documentoConsultaDetalle.getTipofile() != null
				&& documentoConsultaDetalle.getTipofile().equals(Y41bConstantes.TIPO_DOCUMENTO_CONSULTA_RESPUESTA)) {

			RespuestaFile respuestaFile = new RespuestaFile();
			respuestaFile.setQ3cofile(documentoConsultaDetalle.getCofile());

			// elimina el documento de la respuesta
			this.respuestaFileDao.remove(respuestaFile);
		}
	}

	/**
	 * Inserts a single row in the Consulta File table.
	 * 
	 * @param consultaFile
	 *            ConsultaFile
	 * @return ConsultaFile
	 */
	@Transactional(rollbackFor = Throwable.class)
	public ConsultaFile addFileConsulta(ConsultaFile consultaFile) {

		consultaFile.setQ4codocumento(this.consultaFileDao.getNewPK());

		this.consultaFileDao.add(consultaFile);

		return consultaFile;
	}

	/**
	 * Inserts a single row in the Respuesta File table.
	 * 
	 * @param respuestaFile
	 *            RespuestaFile
	 * @return RespuestaFile
	 */
	@Transactional(rollbackFor = Throwable.class)
	public RespuestaFile addFileRespuesta(RespuestaFile respuestaFile) {

		respuestaFile.setQ3cofile(this.respuestaFileDao.getNewPK());

		Respuesta busqueRespuestas = new Respuesta();
		Consulta cons = new Consulta();
		cons.setCocodigo(respuestaFile.getRespuesta().getConsulta().getCocodigo());
		busqueRespuestas.setConsulta(cons);

		List<Respuesta> listRespuesta = this.respuestaDao.findAll(busqueRespuestas, null);

		if (listRespuesta.size() == 1) {
			Respuesta respuestaAux = listRespuesta.get(0);
			respuestaFile.setRespuesta(respuestaAux);
		} else {
			Respuesta respuestaAux = new Respuesta();
			respuestaAux.setP9codigo(Y41bUIDGenerator.getInstance().generateId(Y41bConstantes.PK_SIZE));
			respuestaAux.setConsulta(cons);
			respuestaAux.setP9feanot(new Date());
			respuestaAux.setP9inemlusu(Y41bConstantes.VALOR_NO);

			TipoRespuesta tipoRespuesta = new TipoRespuesta();
			tipoRespuesta.setP8codigo(Y41bConstantes.TIPO_RESPUESTA_CONSULTA_EMAIL);

			respuestaAux.setTipoRespuesta(tipoRespuesta);

			respuestaAux.setP9nufiladj(Y41bConstantes.NUM_CERO);

			this.respuestaDao.add(respuestaAux);

			respuestaFile.setRespuesta(respuestaAux);
		}

		respuestaFile.setQ3enviadomail(Y41bConstantes.VALOR_NO);
		respuestaFile.setQ3fecha(new Date());

		this.respuestaFileDao.add(respuestaFile);

		return respuestaFile;
	}

	/**
	 * Deletes a document of the Consulta.
	 * 
	 * @param documentoConsultaDetalle
	 *            DocumentoConsultaDetalle
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeFileRespuesta(DocumentoConsultaDetalle documentoConsultaDetalle) {
		Respuesta busqueRespuestas = new Respuesta();
		Consulta cons = new Consulta();
		cons.setCocodigo(documentoConsultaDetalle.getCocodigo());
		busqueRespuestas.setConsulta(cons);

		List<Respuesta> listRespuesta = this.respuestaDao.findAll(busqueRespuestas, null);

		if (listRespuesta.size() == 1) {
			Respuesta respuestaAux = listRespuesta.get(0);
			RespuestaFile respuestaFile = new RespuestaFile();
			Respuesta respuesta = new Respuesta();
			respuesta.setP9codigo(respuestaAux.getP9codigo());
			respuestaFile.setRespuesta(respuesta);

			List<RespuestaFile> listRespuestaFile = this.respuestaFileDao.findAll(respuestaFile, null);

			for (RespuestaFile respuestaFileAux : listRespuestaFile) {
				// elimina el documento de la respuesta
				this.respuestaFileDao.remove(respuestaFileAux);
			}
		}
	}

	/**
	 * Getter method for ConsultaDao
	 * 
	 * @return ConsultaDao
	 */
	public ConsultaDao getConsultaDao() {
		return this.consultaDao;
	}

	/**
	 * Setter method for ConsultaDao.
	 * 
	 * @param consultaDao
	 *            ConsultaDao
	 * @return
	 */
	public void setConsultaDao(ConsultaDao consultaDao) {
		logger.info("Setting Dependency " + consultaDao);
		this.consultaDao = consultaDao;
	}

	/**
	 * Finds rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @param fechaDesde
	 *            Date
	 * @param fechaHasta
	 *            Date
	 * @return List
	 */
	public List<EstadisticasConsultasResultado> findAllConsultaEstadisticaFormaPresentacion(Consulta consulta,
			Pagination pagination, Boolean startsWith, Date fechaDesde, Date fechaHasta) {
		return this.consultaDao.findAllConsultaEstadisticaFormaPresentacion(consulta, pagination, startsWith,
				fechaDesde, fechaHasta);
	}

	/**
	 * Finds rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @param fechaDesde
	 *            Date
	 * @param fechaHasta
	 *            Date
	 * @return List
	 */
	public List<EstadisticasConsultasResultado> findAllConsultaEstadisticaMotivosProductos(Consulta consulta,
			Pagination pagination, Boolean startsWith, Date fechaDesde, Date fechaHasta) {
		return this.consultaDao.findAllConsultaEstadisticaMotivosProductos(consulta, pagination, startsWith, fechaDesde,
				fechaHasta);
	}

	/**
	 * Finds rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @param fechaDesde
	 *            Date
	 * @param fechaHasta
	 *            Date
	 * @return List
	 */
	public List<EstadisticasConsultasResultado> findAllConsultaEstadisticaProductoServicio(Consulta consulta,
			Pagination pagination, Boolean startsWith, Date fechaDesde, Date fechaHasta) {
		return this.consultaDao.findAllConsultaEstadisticaProductoServicio(consulta, pagination, startsWith, fechaDesde,
				fechaHasta);
	}

	/**
	 * Finds rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @param fechaDesde
	 *            Date
	 * @param fechaHasta
	 *            Date
	 * @return List
	 */
	public List<EstadisticasConsultasResultado> findAllConsultaEstadisticaSexoEdad(Consulta consulta,
			Pagination pagination, Boolean startsWith, Date fechaDesde, Date fechaHasta) {
		return this.consultaDao.findAllConsultaEstadisticaSexoEdad(consulta, pagination, startsWith, fechaDesde,
				fechaHasta);
	}

	/**
	 * Finds rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @param fechaDesde
	 *            Date
	 * @param fechaHasta
	 *            Date
	 * @return List
	 */
	public List<EstadisticasConsultasResultado> findAllConsultaEstadisticaPromedioMedioResponderConsulta(
			Consulta consulta, Pagination pagination, Boolean startsWith, Date fechaDesde, Date fechaHasta) {
		return this.consultaDao.findAllConsultaEstadisticaPromedioMedioResponderConsulta(consulta, pagination,
				startsWith, fechaDesde, fechaHasta);
	}

	/**
	 * Finds rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @param fechaDesde
	 *            Date
	 * @param fechaHasta
	 *            Date
	 * @return List
	 */
	public List<EstadisticasConsultasTiempoAtencion> findAllConsultaEstadisticaConsultasPresencialesTiempoAtencion(
			Consulta consulta, Pagination pagination, Boolean startsWith, Date fechaDesde, Date fechaHasta) {
		return this.consultaDao.findAllConsultaEstadisticaConsultasPresencialesTiempoAtencion(consulta, pagination,
				startsWith, fechaDesde, fechaHasta);
	}

	/**
	 * Finds rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @param fechaDesde
	 *            Date
	 * @param fechaHasta
	 *            Date
	 * @return List
	 */
	public List<EstadisticasConsultasTiempoAtencion> findAllConsultaEstadisticaConsultasWebFaxTiempoRespuesta(
			Consulta consulta, Pagination pagination, Boolean startsWith, Date fechaDesde, Date fechaHasta) {
		return this.consultaDao.findAllConsultaEstadisticaConsultasWebFaxTiempoRespuesta(consulta, pagination,
				startsWith, fechaDesde, fechaHasta);
	}

	/**
	 * Finds rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @param fechaDesde
	 *            Date
	 * @param fechaHasta
	 *            Date
	 * @return List
	 */
	public List<ListadoConsultaResultado> findAllConsultaListadoRelacionConsultas(Consulta consulta,
			Pagination pagination, Boolean startsWith, Date fechaDesde, Date fechaHasta) {
		return this.consultaDao.findAllConsultaListadoRelacionConsultas(consulta, pagination, startsWith, fechaDesde,
				fechaHasta);
	}

	/**
	 * Finds rows in the Consulta table using like.
	 * 
	 * @param consulta
	 *            Consulta
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @param fechaDesde
	 *            Date
	 * @param fechaHasta
	 *            Date
	 * @return List
	 */
	public List<ListadoConsultaResultado> findAllConsultaListadoRelacionConsultasAbreviada(Consulta consulta,
			Pagination pagination, Boolean startsWith, Date fechaDesde, Date fechaHasta) {
		return this.consultaDao.findAllConsultaListadoRelacionConsultasAbreviada(consulta, pagination, startsWith,
				fechaDesde, fechaHasta);
	}
}
