package com.ejie.y41b.jms;

import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.y41b.adapter.Y41bDokusiAdapterService;
import com.ejie.y41b.adapter.Y41bPlateaAdapterService;
import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.dao.AdjuntardocsDao;
import com.ejie.y41b.dao.BandejaFirmaArbitrajeDao;
import com.ejie.y41b.dao.BandejaFirmaDao;
import com.ejie.y41b.dao.DocumentoHechoDao;
import com.ejie.y41b.dao.PlantillaTramiteDao;
import com.ejie.y41b.model.BandejaFirma;
import com.ejie.y41b.model.BandejaFirmaArbitraje;
import com.ejie.y41b.model.DocumentoHecho;
import com.ejie.y41b.model.PlantillaTramite;
import com.ejie.y41b.model.dokusi.Documento;
import com.ejie.y41b.model.dokusi.MensajeIncorporarLocalizador;
import com.ejie.y41b.model.dokusi.Metadato;
import com.ejie.y41b.model.dokusi.User;
import com.ejie.y41b.model.platea.Procedure;
import com.ejie.y41b.utils.Y41bUtils;
import com.ejie.y41b.utils.exception.Y41bUDAException;
import com.ejie.y41b.utils.ws.Y41bUtilsWS;

/**
 * Y41bTratamientoDokusiJMSServiceImpl Clase encargada del tratamiento de los
 * mensajes recibidos en la cola JMS para DOKUSI
 * 
 * Los metodos deberian ser TRANSACCIONALES del tipo "propagation =
 * Propagation.REQUIRES_NEW" para que no participe de ninguna transaccion global
 * y no de problemas de reenvio del mensaje a la cola.
 * 
 *  
 * 
 */
@Service(value = "y41bTratamientoDokusiJMSService")
public class Y41bTratamientoDokusiJMSServiceImpl implements Y41bTratamientoDokusiJMSService {
	private static final Logger logger = LoggerFactory.getLogger(Y41bTratamientoDokusiJMSServiceImpl.class);

	@Autowired
	private Y41bPlateaAdapterService y41bPlateaAdapterService;
	@Autowired
	private Y41bDokusiAdapterService y41bDokusiAdapterService;

	@Autowired
	private DocumentoHechoDao documentoHechoDao;
	@Autowired
	private AdjuntardocsDao adjuntardocsDao;
	@Autowired
	private BandejaFirmaDao bandejaFirmaDao;
	@Autowired
	private BandejaFirmaArbitrajeDao bandejaFirmaArbitrajeDao;
	@Autowired
	private PlantillaTramiteDao plantillaTramiteDao;

	/**
	 * Constructor privado
	 */
	private Y41bTratamientoDokusiJMSServiceImpl() {
		super();
	}

	/**
	 * tratamientoLocalizador
	 * 
	 * @param stPfToken
	 *            token
	 * @param mapaMensaje
	 *            mensaje
	 * @throws Exception
	 *             Cualquier excepcion
	 */
	@Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = Throwable.class)
	public void tratamientoLocalizador(String stPfToken, Map mapaMensaje) throws Exception {
		Y41bTratamientoDokusiJMSServiceImpl.logger
				.info("Y41bTratamientoDokusiJMSServiceImpl.tratamientoLocalizador - Inicio");
		try {
			// 1. Recuperamos informacion necesaria
			String eventTipology = (String) mapaMensaje.get("EventTipology");

			String details = (String) mapaMensaje.get("details");
			String documentName = (String) mapaMensaje.get("documentName");
			String documentType = (String) mapaMensaje.get("documentType");
			String documentId = (String) mapaMensaje.get("documentId");
			String requestID = (String) mapaMensaje.get("requestID");
			String succesful = (String) mapaMensaje.get("succesful");
			String documentVersion = (String) mapaMensaje.get("documentVersion");

			// 2. Creamos el objeto con la informacion necesaria
			MensajeIncorporarLocalizador mensajeIncorporarLocalizador = new MensajeIncorporarLocalizador();
			mensajeIncorporarLocalizador.setDetalles(details);
			mensajeIncorporarLocalizador.setNombre(documentName);
			mensajeIncorporarLocalizador.setTipoDocumental(documentType);
			mensajeIncorporarLocalizador.setOidDokusi(documentId);
			mensajeIncorporarLocalizador.setIdPeticion(requestID);
			mensajeIncorporarLocalizador.setSuccesful(succesful);
			mensajeIncorporarLocalizador.setVersion(documentVersion);

			// 3. Logica de negocio para tratamiento de aportacion
			// documentacion
			Y41bTratamientoDokusiJMSServiceImpl.logger.debug(
					"Y41bTratamientoDokusiJMSServiceImpl.tratamientoLocalizador - eventTipology:" + eventTipology);
			Y41bTratamientoDokusiJMSServiceImpl.logger
					.debug("Y41bTratamientoDokusiJMSServiceImpl.tratamientoLocalizador - mensaje localizador:"
							+ mensajeIncorporarLocalizador.toString());

			DocumentoHecho documentohecho = new DocumentoHecho();
			documentohecho.setQ5iddocumentum(documentId);
			Y41bTratamientoDokusiJMSServiceImpl.logger
					.debug("Y41bTratamientoDokusiJMSServiceImpl.tratamientoLocalizador - documentId:" + documentId);
			this.actualizarDocumentoLocalizadorBBDD(documentohecho);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Throwable e_tratamientoLocalizador) {
			logger.error("Y41bTratamientoDokusiJMSServiceImpl.tratamientoLocalizador", e_tratamientoLocalizador);
			throw new Y41bUDAException("error.platea.invocation", true, e_tratamientoLocalizador);
		} finally {
			logger.info("Y41bTratamientoDokusiJMSServiceImpl.tratamientoLocalizador - Final");
		}

	}

	/**
	 * Actualiza el documento en BBDD despues de insertar el localizador.
	 * 
	 * @param documentohecho
	 *            DocumentoHecho
	 * @return Long
	 * @throws Exception
	 *             Exception
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Long actualizarDocumentoLocalizadorBBDD(DocumentoHecho documentohecho) throws Exception {

		try {
			int numRowsUpdated = 0;
			Documento docDokusi = null;

			Documento documento = new Documento();
			User user = new User();
			user.setNombre(Y41bConstantes.COD_APLICACION);
			user.setDni(Y41bConstantes.COD_APLICACION);
			documento.setUser(user);
			Procedure procedure = new Procedure();
			procedure.setIdProcedimiento(Y41bConstantes.ID_PROCEDIMIENTO_SOLICITUDES);
			documento.setProcedure(procedure);
			documento.setOidDokusi(documentohecho.getQ5iddocumentum());

			docDokusi = y41bDokusiAdapterService.descargarDocumentoDokusi(documento);

			if (this.isLocalizado(docDokusi) || this.isTransformado(docDokusi)
					|| this.isNoLocalizable(docDokusi, documentohecho)) {
				DocumentoHecho documentohechoAux = new DocumentoHecho();
				documentohechoAux.setQ5nofile(docDokusi.getNombre());
				documentohechoAux.setQ5iddocumentum(documentohecho.getQ5iddocumentum());
				documentohechoAux.setQ5sizefile((int) docDokusi.getLength());
				// Blob blob = new
				// SerialBlob(Y41bUtilsWS.decode(docDokusi.getBytes()));
				// documentohechoAux.setQ5datafile(blob);
				documentohechoAux.setQ5contenttypefile(docDokusi.getContentType());

				numRowsUpdated = this.documentoHechoDao.updateDocumentoHecho(documentohechoAux);

				this.adjuntardocsDao.updateDocumentoHecho(documentohechoAux);

				logger.debug(
						"###################################DenunciaServiceImpl.actualizarDocumentoLocalizadorBBDD numRowsUpdated ["
								+ numRowsUpdated + "]");

				if (numRowsUpdated == 1) {
					BandejaFirma bandejafirma = new BandejaFirma();
					bandejafirma.setNofile(docDokusi.getNombre());
					bandejafirma.setIddocumentum(documentohecho.getQ5iddocumentum());

					this.bandejaFirmaDao.updateDocumentoBandejaFirma(bandejafirma);

					BandejaFirmaArbitraje bandejaFirmaArbitraje = new BandejaFirmaArbitraje();
					bandejaFirmaArbitraje.setNofile(docDokusi.getNombre());
					bandejaFirmaArbitraje.setIddocumentum(documentohecho.getQ5iddocumentum());

					this.bandejaFirmaArbitrajeDao.updateDocumentoBandejaFirma(bandejaFirmaArbitraje);

					try {
						documento = new Documento();
						user = new User();
						user.setNombre(Y41bConstantes.COD_APLICACION);
						user.setDni(Y41bConstantes.COD_APLICACION);
						documento.setUser(user);
						procedure = new Procedure();
						procedure.setIdProcedimiento(Y41bConstantes.ID_PROCEDIMIENTO_SOLICITUDES);
						documento.setProcedure(procedure);
						documento.setOidDokusi(documentohecho.getQ5iddocumentum());

						documento.setBytes(new String(Y41bUtilsWS.decode(docDokusi.getBytes())));

						String hashDocumento = y41bPlateaAdapterService.generarHashDocumento(documento);
						bandejafirma.setHashfile(hashDocumento);

						this.bandejaFirmaDao.updateHashFileBandejaFirma(bandejafirma);

						bandejaFirmaArbitraje.setHashfile(hashDocumento);
						this.bandejaFirmaArbitrajeDao.updateHashFileBandejaFirma(bandejaFirmaArbitraje);

						documentohechoAux.setQ5hashfile(hashDocumento);
						this.documentoHechoDao.updateHashByOIDDokusi(documentohechoAux);

						this.adjuntardocsDao.updateHashByOIDDokusi(documentohechoAux);
					} catch (Throwable e) {
						bandejafirma.setHashfile(null);

						this.bandejaFirmaDao.updateHashFileBandejaFirma(bandejafirma);

						bandejaFirmaArbitraje.setHashfile(null);
						this.bandejaFirmaArbitrajeDao.updateHashFileBandejaFirma(bandejaFirmaArbitraje);

						documentohechoAux.setQ5hashfile(null);
						this.documentoHechoDao.updateHashByOIDDokusi(documentohechoAux);

						this.adjuntardocsDao.updateHashByOIDDokusi(documentohechoAux);

					}
					return new Long(1);
				} else {
					return new Long(0);
				}
			} else {
				return new Long(0);
			}
		} catch (Exception e) {
			return new Long(0);
		}
	}

	/**
	 * isLocalizado
	 * 
	 * @param documento
	 *            Documento
	 * @return boolean
	 */
	private boolean isLocalizado(Documento documento) {
		Y41bTratamientoDokusiJMSServiceImpl.logger.debug("Y41bTratamientoDokusiJMSServiceImpl.isLocalizado - INICIO ");
		boolean resultado = false;
		if (documento != null) {
			List<Metadato> listaMetadatos = documento.getMetadatos();
			if (listaMetadatos != null && listaMetadatos.size() > 0) {
				for (Metadato metadato : listaMetadatos) {
					if (Y41bConstantes.METADATO_CODIGO_VERIFICACION.equalsIgnoreCase(metadato.getNombre())) {
						resultado = !Y41bUtils.esNullOCadenaVacia(metadato.getValor());
						break;
					}
				}
			}
		}
		Y41bTratamientoDokusiJMSServiceImpl.logger
				.debug("Y41bTratamientoDokusiJMSServiceImpl.isLocalizado - RESULTADO: " + resultado);
		Y41bTratamientoDokusiJMSServiceImpl.logger.debug("Y41bTratamientoDokusiJMSServiceImpl.isLocalizado - FIN ");
		return resultado;
	}

	/**
	 * isOrigenCiudadania
	 * 
	 * @param documento
	 *            Documento
	 * @return boolean
	 */
	private boolean isOrigenCiudadania(Documento documento) {
		Y41bTratamientoDokusiJMSServiceImpl.logger
				.debug("Y41bTratamientoDokusiJMSServiceImpl.isOrigenCiudadania - INICIO ");
		boolean resultado = false;
		if (documento != null) {
			List<Metadato> listaMetadatos = documento.getMetadatos();
			if (listaMetadatos != null && listaMetadatos.size() > 0) {
				for (Metadato metadato : listaMetadatos) {
					if (Y41bConstantes.METADATO_ORIGEN.equalsIgnoreCase(metadato.getNombre())) {
						resultado = Y41bConstantes.ORIGEN_CIUDADANIA.equalsIgnoreCase(metadato.getValor());
						break;
					}
				}
			}
		}
		Y41bTratamientoDokusiJMSServiceImpl.logger
				.debug("Y41bTratamientoDokusiJMSServiceImpl.isOrigenCiudadania - RESULTADO: " + resultado);
		Y41bTratamientoDokusiJMSServiceImpl.logger
				.debug("Y41bTratamientoDokusiJMSServiceImpl.isOrigenCiudadania - FIN ");
		return resultado;
	}

	/**
	 * isTransformado
	 * 
	 * @param documento
	 *            Documento
	 * @return boolean
	 */
	private boolean isTransformado(Documento documento) {
		Y41bTratamientoDokusiJMSServiceImpl.logger
				.debug("Y41bTratamientoDokusiJMSServiceImpl.isTransformado - INICIO ");
		boolean resultado = false;
		if (documento != null && documento.getExtension() != null
				&& documento.getExtension().equalsIgnoreCase(Y41bConstantes.PDF)
				&& this.isOrigenCiudadania(documento)) {
			resultado = true;
		}
		Y41bTratamientoDokusiJMSServiceImpl.logger
				.debug("Y41bTratamientoDokusiJMSServiceImpl.isTransformado - RESULTADO: " + resultado);
		Y41bTratamientoDokusiJMSServiceImpl.logger.debug("Y41bTratamientoDokusiJMSServiceImpl.isTransformado - FIN ");
		return resultado;
	}

	/**
	 * isNoLocalizable
	 * 
	 * @param documento
	 *            Documento
	 * @param documentohecho
	 *            DocumentoHecho
	 * @return boolean
	 */
	private boolean isNoLocalizable(Documento documento, DocumentoHecho documentohecho) {
		Y41bTratamientoDokusiJMSServiceImpl.logger
				.debug("Y41bTratamientoDokusiJMSServiceImpl.isNoLocalizable - INICIO ");
		boolean resultado = false;

		documentohecho = this.documentoHechoDao.findDocumentoHechoByOIDDokusi(documentohecho);

		if (documentohecho != null && Y41bUtils.isFilled(documentohecho.getQ5iddocumentum())) {

			String nolocalizable = null;
			if (documentohecho != null && documentohecho.getQ5ptcodigo() != null
					&& !documentohecho.getQ5ptcodigo().equals("")) {
				PlantillaTramite plantillaTramite = new PlantillaTramite();
				plantillaTramite.setPtcodigo(documentohecho.getQ5ptcodigo());
				// Obtener la plantilla del tramite
				plantillaTramite = this.plantillaTramiteDao.find(plantillaTramite);

				nolocalizable = plantillaTramite.getNolocalizable();
			}

			if (documento != null && documento.getExtension() != null
					&& documento.getExtension().equalsIgnoreCase(Y41bConstantes.PDF)
					&& (nolocalizable != null && nolocalizable.equals(Y41bConstantes.VALOR_SI))) {
				resultado = true;
			}
		}

		Y41bTratamientoDokusiJMSServiceImpl.logger
				.debug("Y41bTratamientoDokusiJMSServiceImpl.isNoLocalizable - RESULTADO: " + resultado);
		Y41bTratamientoDokusiJMSServiceImpl.logger.debug("Y41bTratamientoDokusiJMSServiceImpl.isNoLocalizable - FIN ");
		return resultado;
	}
}
