package com.ejie.y41b.jms;

import java.util.Hashtable;
import java.util.Map;
import java.util.Properties;

import javax.jms.DeliveryMode;
import javax.jms.JMSException;
import javax.jms.MapMessage;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.TextMessage;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.utils.config.Y41bConfig;

/**
 * Y41bRemesasEmisionJMSServiceImpl Clase encargada del tratamiento de los
 * mensajes recibidos en la cola JMS para la emision de remesas
 * 
 * Los metodos deberian ser TRANSACCIONALES del tipo
 * "propagation = Propagation.REQUIRES_NEW" para que no participe de ninguna
 * transaccion global y no de problemas de reenvio del mensaje a la cola.
 * 
 *  
 * 
 */
@Service(value = "y41bRemesasClientJMSService")
public class Y41bRemesasClientJMSServiceImpl implements
		Y41bRemesasClientJMSService {
	private static final Logger logger = LoggerFactory
			.getLogger(Y41bRemesasClientJMSServiceImpl.class);

	/**
	 * Constructor privado
	 */
	private Y41bRemesasClientJMSServiceImpl() {
		super();
	}

	public void sendMessage(String messageText) {
		InitialContext ctx = null;
		QueueConnectionFactory qcf = null;
		QueueConnection qc = null;
		QueueSession qsess = null;
		Queue q = null;
		QueueSender qsndr = null;
		TextMessage message = null;
		// NOTE: The next two lines set the name of the Queue Connection Factory
		// and the Queue that we want to use.

		// TODO Crear carpeta con el id de remesa
		Properties prop = Y41bConfig.loadProperties(Y41bConstantes.CONFIG_PATH);
		// SE especifica la ruta temporal de las remesas

		String QCF_NAME = prop.getProperty("jndi.jsm.remesasFactory");
		String QUEUE_NAME = prop.getProperty("jndi.jsm.remesasQueue");

		// create InitialContext
		Hashtable properties = new Hashtable();
		properties.put(Context.INITIAL_CONTEXT_FACTORY,
				"weblogic.jndi.WLInitialContextFactory");
		try {
			ctx = new InitialContext(properties);
		} catch (NamingException ne) {
			ne.printStackTrace(System.err);
		}
		logger.debug("Got InitialContext " + ctx.toString());
		// create QueueConnectionFactory
		try {
			qcf = (QueueConnectionFactory) ctx.lookup(QCF_NAME);
		} catch (NamingException ne) {
			logger.error("ERROR JMS: " + ne.getMessage());
		}
		logger.debug("Got QueueConnectionFactory " + qcf.toString());
		// create QueueConnection
		try {
			qc = qcf.createQueueConnection();
		} catch (JMSException jmse) {
			logger.error("ERROR JMS: " + jmse.getMessage());
		}
		logger.debug("Got QueueConnection " + qc.toString());
		// create QueueSession
		try {
			qsess = qc.createQueueSession(false, 0);
		} catch (JMSException jmse) {
			logger.error("ERROR JMS: " + jmse.getMessage());
		}
		logger.debug("Got QueueSession " + qsess.toString());
		// lookup Queue
		try {
			q = (Queue) ctx.lookup(QUEUE_NAME);
		} catch (NamingException ne) {
			logger.error("ERROR JMS: " + ne.getMessage());
		}
		logger.debug("Got Queue " + q.toString());
		// create QueueSender
		try {
			qsndr = qsess.createSender(q);
		} catch (JMSException jmse) {
			logger.error("ERROR JMS: " + jmse.getMessage());
		}
		logger.debug("Got QueueSender " + qsndr.toString());
		// create TextMessage
		try {
			message = qsess.createTextMessage();
			message.setJMSExpiration(100000);
			message.setJMSDeliveryMode(DeliveryMode.NON_PERSISTENT);
		} catch (JMSException jmse) {
			logger.error("ERROR JMS: " + jmse.getMessage());
		}
		logger.debug("Got TextMessage " + message.toString());
		// set message text in TextMessage
		try {
			message.setText(messageText);
		} catch (JMSException jmse) {
			logger.error("ERROR JMS: " + jmse.getMessage());
		}
		logger.debug("Set text in TextMessage " + message.toString());
		// send message
		try {
			qsndr.send(message);
		} catch (JMSException jmse) {
			logger.error("ERROR JMS: " + jmse.getMessage());
		}
		logger.debug("Sent message ");
		// clean up
		try {
			message = null;
			qsndr.close();
			qsndr = null;
			q = null;
			qsess.close();
			qsess = null;
			qc.close();
			qc = null;
			qcf = null;
			ctx = null;
		} catch (JMSException jmse) {
			logger.error("ERROR JMS: " + jmse.getMessage());
		}
		logger.debug("Cleaned up and done.");
	}

	/**
	 * Manda el mensaje a la cola de monitorizacion
	 * 
	 * @param messageMap
	 *            Map<String, String>
	 */
	public void sendMessage(Map<String, String> messageMap) {
		InitialContext ctx = null;
		QueueConnectionFactory qcf = null;
		QueueConnection qc = null;
		QueueSession qsess = null;
		Queue q = null;
		QueueSender qsndr = null;
		MapMessage message = null;

		Properties prop = Y41bConfig.loadProperties(Y41bConstantes.CONFIG_PATH);

		String QCF_NAME = prop.getProperty("jndi.jsm.remesasFactory");
		String QUEUE_NAME = prop.getProperty("jndi.jsm.remesasQueue");

		Hashtable<String, String> properties = new Hashtable<String, String>();
		properties.put(Context.INITIAL_CONTEXT_FACTORY,
				"weblogic.jndi.WLInitialContextFactory");

		try {
			ctx = new InitialContext(properties);
		} catch (NamingException ne) {
			logger.error("Error JMS:" + ne.getMessage());
		}
		try {
			qcf = (QueueConnectionFactory) ctx.lookup(QCF_NAME);
		} catch (NamingException ne) {
			logger.error("Error JMS:" + ne.getMessage());
		}
		try {
			qc = qcf.createQueueConnection();
		} catch (JMSException jmse) {
			logger.error("Error JMS:" + jmse.getMessage());
		}
		try {
			qsess = qc.createQueueSession(true, 0);
		} catch (JMSException jmse) {
			logger.error("Error JMS:" + jmse.getMessage());
		}
		try {
			q = (Queue) ctx.lookup(QUEUE_NAME);
		} catch (NamingException ne) {
			logger.error("Error JMS:" + ne.getMessage());
		}
		try {
			qsndr = qsess.createSender(q);
		} catch (JMSException jmse) {
			logger.error("Error JMS:" + jmse.getMessage());
		}
		try {
			message = createMessage(qsess, messageMap);
		} catch (JMSException jmse) {
			logger.error("Error JMS:" + jmse.getMessage());
		}
		try {
			qsndr.send(message);
		} catch (JMSException jmse) {
			logger.error("Error JMS:" + jmse.getMessage());
		}
		try {
			message = null;
			qsndr.close();
			qsndr = null;
			q = null;
			qsess.commit();
			qsess.close();
			qsess = null;
			qc.close();
			qc = null;
			qcf = null;
			ctx = null;
		} catch (JMSException jmse) {
			logger.error("Error JMS:" + jmse.getMessage());
		}
	}

	/**
	 * Crea un mensaje JMS MapMessage, a partir del mapa recibido por parametro
	 * 
	 * @param qsess
	 *            QueueSession
	 * @param messageMap
	 *            Map<String, String>
	 * @return MapMessage
	 * @throws JMSException
	 */
	private static MapMessage createMessage(QueueSession qsess,
			Map<String, String> messageMap) throws JMSException {
		MapMessage message = qsess.createMapMessage();
		message.setJMSExpiration(100000); // en milisegundos (100 segundos)
		message.setJMSDeliveryMode(DeliveryMode.NON_PERSISTENT);
		if (messageMap != null && messageMap.size() > 0) {
			for (Map.Entry<String, String> entry : messageMap.entrySet()) {
				message.setString(entry.getKey(), entry.getValue());
			}
		}
		return message;
	}

}
