package com.ejie.y41b.dao;

import com.ejie.x38.dto.Pagination;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.annotation.Resource;
import javax.sql.DataSource;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.y41b.model.Tribunal;

/**
 * TribunalDaoImpl  
 * 
 *  
 */

@Repository
@Transactional
public class TribunalDaoImpl implements TribunalDao {
	private JdbcTemplate jdbcTemplate;
	private RowMapper<Tribunal> rwMap = new RowMapper<Tribunal>() {
		public Tribunal mapRow(ResultSet resultSet, int rowNum)
				throws SQLException {
			return new Tribunal(resultSet.getString("TBCODIGO"),
					resultSet.getString("TBNOMBRE"),
					resultSet.getString("TBNOMBREE"));
		}
	};

	/**
	 * Method use to set the datasource.
	 * 
	 * @param dataSource
	 *            DataSource
	 * @return
	 */
	@Resource
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Inserts a single row in the Tribunal table.
	 * 
	 * @param tribunal
	 *            Pagination
	 * @return Tribunal
	 */
	public Tribunal add(Tribunal tribunal) {
		String query = "INSERT INTO TRIBUNAL (TBCODIGO, TBNOMBRE, TBNOMBREE) VALUES (?,?,?)";
		this.jdbcTemplate.update(query, tribunal.getTbcodigo(),
				tribunal.getTbnombre(), tribunal.getTbnombree());
		return tribunal;
	}

	/**
	 * Updates a single row in the Tribunal table.
	 * 
	 * @param tribunal
	 *            Pagination
	 * @return Tribunal
	 */
	public Tribunal update(Tribunal tribunal) {
		String query = "UPDATE TRIBUNAL SET TBNOMBRE=?,TBNOMBREE=? WHERE TBCODIGO=?";
		this.jdbcTemplate.update(query, tribunal.getTbnombre(),
				tribunal.getTbnombree(), tribunal.getTbcodigo());
		return tribunal;
	}

	/**
	 * Finds a single row in the Tribunal table.
	 * 
	 * @param tribunal
	 *            Pagination
	 * @return Tribunal
	 */
	@Transactional(readOnly = true)
	public Tribunal find(Tribunal tribunal) {
		String query = "SELECT t1.TBCODIGO TBCODIGO, t1.TBNOMBRE TBNOMBRE, t1.TBNOMBREE TBNOMBREE FROM TRIBUNAL t1  WHERE t1.TBCODIGO = ?  ";

		List<Tribunal> tribunalList = this.jdbcTemplate.query(query,
				this.rwMap, tribunal.getTbcodigo());
		return (Tribunal) DataAccessUtils.uniqueResult(tribunalList);
	}

	/**
	 * Removes a single row in the Tribunal table.
	 * 
	 * @param tribunal
	 *            Pagination
	 * @return
	 */
	public void remove(Tribunal tribunal) {
		String query = "DELETE FROM TRIBUNAL WHERE TBCODIGO=?";
		this.jdbcTemplate.update(query, tribunal.getTbcodigo());
	}

	/**
	 * Finds a List of rows in the Tribunal table.
	 * 
	 * @param tribunal
	 *            Tribunal
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	@Transactional(readOnly = true)
	public List<Tribunal> findAll(Tribunal tribunal, Pagination pagination) {
		StringBuilder query = new StringBuilder(
				"SELECT  t1.TBCODIGO TBCODIGO,t1.TBNOMBRE TBNOMBRE, NVL(t1.TBNOMBREE,t1.TBNOMBRE) TBNOMBREE ");
		query.append("FROM TRIBUNAL t1 ");

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereMap(tribunal);
		StringBuilder where = new StringBuilder(" WHERE 1=1 ");
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> params = (List<?>) mapaWhere.get("params");

		if (pagination != null) {
			query = pagination.getPaginationQuery(query);
		}

		return (List<Tribunal>) this.jdbcTemplate.query(query.toString(),
				this.rwMap, params.toArray());
	}

	/**
	 * Counts rows in the Tribunal table.
	 * 
	 * @param tribunal
	 *            Tribunal
	 * @return Long
	 */
	@Transactional(readOnly = true)
	public Long findAllCount(Tribunal tribunal) {
		StringBuilder query = new StringBuilder(
				"SELECT COUNT(1) FROM TRIBUNAL t1 ");

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereMap(tribunal);
		StringBuilder where = new StringBuilder(" WHERE 1=1 ");
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> params = (List<?>) mapaWhere.get("params");

		return this.jdbcTemplate.queryForLong(query.toString(),
				params.toArray());
	}

	/**
	 * Finds rows in the Tribunal table using like.
	 * 
	 * @param tribunal
	 *            Tribunal
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	@Transactional(readOnly = true)
	public List<Tribunal> findAllLike(Tribunal tribunal, Pagination pagination,
			Boolean startsWith) {
		StringBuilder query = new StringBuilder(
				"SELECT  t1.TBCODIGO TBCODIGO,t1.TBNOMBRE TBNOMBRE, NVL(t1.TBNOMBREE,t1.TBNOMBRE) TBNOMBREE ");
		query.append("FROM TRIBUNAL t1 ");

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereLikeMap(tribunal, startsWith);
		StringBuilder where = new StringBuilder(" WHERE 1=1 ");
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> params = (List<?>) mapaWhere.get("params");

		if (pagination != null) {
			query = pagination.getPaginationQuery(query);
		}

		return (List<Tribunal>) this.jdbcTemplate.query(query.toString(),
				this.rwMap, params.toArray());
	}

	/**
	 * Counts rows in the Tribunal table using like.
	 * 
	 * @param tribunal
	 *            Tribunal
	 * @param startsWith
	 *            Boolean
	 * @return Long
	 */
	@Transactional(readOnly = true)
	public Long findAllLikeCount(Tribunal tribunal, Boolean startsWith) {
		StringBuilder query = new StringBuilder(
				"SELECT COUNT(1) FROM TRIBUNAL t1 ");

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereLikeMap(tribunal, startsWith);
		StringBuilder where = new StringBuilder(" WHERE 1=1 ");
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> params = (List<?>) mapaWhere.get("params");

		return this.jdbcTemplate.queryForLong(query.toString(),
				params.toArray());
	}

	/**
	 * Returns a map with the needed value to create the conditions to filter by
	 * the Tribunal entity
	 * 
	 * @param tribunal
	 *            Tribunal Bean with the criteria values to filter by.
	 * @return Map created with two keys key query stores the sql query syntax
	 *         key params stores the parameter values to be used in the
	 *         condition sentence.
	 */
	// CHECKSTYLE:OFF CyclomaticComplexity - Generación de código de UDA
	private Map<String, ?> getWhereMap(Tribunal tribunal) {

		StringBuilder where = new StringBuilder(
				TribunalDaoImpl.STRING_BUILDER_INIT);
		List<Object> params = new ArrayList<Object>();

		if (tribunal != null && tribunal.getTbcodigo() != null) {
			where.append(" AND t1.TBCODIGO = ?");
			params.add(tribunal.getTbcodigo());
		}
		if (tribunal != null && tribunal.getTbnombre() != null) {
			where.append(" AND t1.TBNOMBRE = ?");
			params.add(tribunal.getTbnombre());
		}
		if (tribunal != null && tribunal.getTbnombree() != null) {
			where.append(" AND t1.TBNOMBREE = ?");
			params.add(tribunal.getTbnombree());
		}

		Map<String, Object> mapWhere = new HashMap<String, Object>();
		mapWhere.put("query", where);
		mapWhere.put("params", params);

		return mapWhere;
	}

	// CHECKSTYLE:ON CyclomaticComplexity - Generación de código de UDA

	/**
	 * Returns a map with the needed value to create the conditions to filter by
	 * the Tribunal entity
	 * 
	 * @param tribunal
	 *            Tribunal Bean with the criteria values to filter by.
	 * @param startsWith
	 *            Boolean
	 * @return Map created with two keys key query stores the sql query syntax
	 *         key params stores the parameter values to be used in the
	 *         condition sentence.
	 */
	// CHECKSTYLE:OFF CyclomaticComplexity - Generación de código de UDA
	private Map<String, ?> getWhereLikeMap(Tribunal tribunal, Boolean startsWith) {

		StringBuilder where = new StringBuilder(
				TribunalDaoImpl.STRING_BUILDER_INIT);
		List<Object> params = new ArrayList<Object>();

		if (tribunal != null && tribunal.getTbcodigo() != null) {
			where.append(" AND UPPER(t1.TBCODIGO) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(tribunal.getTbcodigo().toUpperCase() + "%");
			} else {
				params.add("%" + tribunal.getTbcodigo().toUpperCase() + "%");
			}
			where.append(" AND t1.TBCODIGO IS NOT NULL");
		}
		if (tribunal != null && tribunal.getTbnombre() != null) {
			where.append(" AND TRANSLATE(UPPER(t1.TBNOMBRE),'','AEIOU') like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(tribunal.getTbnombre().toUpperCase() + "%");
			} else {
				params.add("%" + tribunal.getTbnombre().toUpperCase() + "%");
			}
			where.append(" AND t1.TBNOMBRE IS NOT NULL");
		}
		if (tribunal != null && tribunal.getTbnombree() != null) {
			where.append(" AND (TRANSLATE(UPPER(t1.TBNOMBREE),'','AEIOU') like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(tribunal.getTbnombree().toUpperCase() + "%");
			} else {
				params.add("%" + tribunal.getTbnombree().toUpperCase() + "%");
			}
			where.append(" AND t1.TBNOMBREE IS NOT NULL) ");

			where.append(" OR (TRANSLATE(UPPER(t1.TBNOMBRE),'','AEIOU') like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(tribunal.getTbnombree().toUpperCase() + "%");
			} else {
				params.add("%" + tribunal.getTbnombree().toUpperCase() + "%");
			}
			where.append(" AND t1.TBNOMBRE IS NOT NULL AND t1.TBNOMBREE IS NULL)");
		}

		Map<String, Object> mapWhere = new HashMap<String, Object>();
		mapWhere.put("query", where);
		mapWhere.put("params", params);

		return mapWhere;
	}

	// CHECKSTYLE:ON CyclomaticComplexity - Generación de código de UDA

	/**
	 * StringBuilder initilization value
	 */
	public static final int STRING_BUILDER_INIT = 4096;
}
