package com.ejie.y41b.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.model.PlantillaInformeCierreCampanha;

/**
 * Extraccion de datos para la plantilla Informe de Cierre de Campanha.
 * 
 *  
 */
@Repository()
@Transactional()
public class PlantillaInformeCierreCampanhaDaoImpl implements
		PlantillaInformeCierreCampanhaDao {
	private JdbcTemplate jdbcTemplate;

	/**
	 * Method use to set the datasource.
	 * 
	 * @param dataSource
	 *            DataSource
	 * @return
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * 01. Cabecera de la plantilla de Informe de cierre de informe.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findCabecera(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT CMNOMBRE, TO_CHAR(CMFECHACIERRE, 'YYYY') CMANIO FROM CAMPANHA WHERE CMCODIGO = ?";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setCmNombre(resultSet
												.getString("CMNOMBRE"));
								plantillaInformeCierreCampanha
										.setCmAnio(resultSet.getInt("CMANIO"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 02. Actuaciones por protocolo y toma de muestras.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findActuacionesProtocoloMuestras(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT ADCOOR, ATISTOMAMUESTRAS, ATISCONTROLESTANDAR, COUNT(1) TOTAL "
				+ "FROM ACTUACION, ACTDEC "
				+ "WHERE ATCODIGO = ADCOAT AND ATCOCM = ? AND (ATISTOMAMUESTRAS = 'S' OR ATISCONTROLESTANDAR = 'S') "
				+ "GROUP BY ADCOOR, ATISTOMAMUESTRAS, ATISCONTROLESTANDAR";

		final class TotalesSSTT {
			public String adcoor;
			public String atistomamuestras;
			public String atiscontrolestandar;
			public int total;
		}

		List<TotalesSSTT> listaTotalesSSTT = (List<TotalesSSTT>) this.jdbcTemplate
				.query(query, new RowMapper<TotalesSSTT>() {
					public TotalesSSTT mapRow(ResultSet resultSet, int rowNum)
							throws SQLException {
						TotalesSSTT totalesSSTT = new TotalesSSTT();

						totalesSSTT.adcoor = resultSet.getString("ADCOOR");
						totalesSSTT.atistomamuestras = resultSet
								.getString("ATISTOMAMUESTRAS");
						totalesSSTT.atiscontrolestandar = resultSet
								.getString("ATISCONTROLESTANDAR");
						totalesSSTT.total = resultSet.getInt("TOTAL");

						return totalesSSTT;
					}
				}, plantillaInformeCierreCampanha.getCmCodigo());

		plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

		int sumaProtocolos = 0;
		int sumaMuestras = 0;
		for (TotalesSSTT totalesSSTT : listaTotalesSSTT) {
			if (Y41bConstantes.CODIGO_ORGANISMO_ARABA
					.equals(totalesSSTT.adcoor)) {
				if (Y41bConstantes.VALOR_SI
						.equals(totalesSSTT.atiscontrolestandar)) {
					plantillaInformeCierreCampanha
							.setProtocolosAraba(totalesSSTT.total);
					sumaProtocolos += totalesSSTT.total;
				}
				if (Y41bConstantes.VALOR_SI
						.equals(totalesSSTT.atistomamuestras)) {
					plantillaInformeCierreCampanha
							.setMuestrasAraba(totalesSSTT.total);
					sumaMuestras += totalesSSTT.total;
				}
			} else if (Y41bConstantes.CODIGO_ORGANISMO_GIPUZKOA
					.equals(totalesSSTT.adcoor)) {
				if (Y41bConstantes.VALOR_SI
						.equals(totalesSSTT.atiscontrolestandar)) {
					plantillaInformeCierreCampanha
							.setProtocolosGipuzkoa(totalesSSTT.total);
					sumaProtocolos += totalesSSTT.total;
				}
				if (Y41bConstantes.VALOR_SI
						.equals(totalesSSTT.atistomamuestras)) {
					plantillaInformeCierreCampanha
							.setMuestrasGipuzkoa(totalesSSTT.total);
					sumaMuestras += totalesSSTT.total;
				}
			} else if (Y41bConstantes.CODIGO_ORGANISMO_BIZKAIA
					.equals(totalesSSTT.adcoor)) {
				if (Y41bConstantes.VALOR_SI
						.equals(totalesSSTT.atiscontrolestandar)) {
					plantillaInformeCierreCampanha
							.setProtocolosBizkaia(totalesSSTT.total);
					sumaProtocolos += totalesSSTT.total;
				}
				if (Y41bConstantes.VALOR_SI
						.equals(totalesSSTT.atistomamuestras)) {
					plantillaInformeCierreCampanha
							.setMuestrasBizkaia(totalesSSTT.total);
					sumaMuestras += totalesSSTT.total;
				}
			}
		}
		plantillaInformeCierreCampanha.setProtocolosTotal(sumaProtocolos);
		plantillaInformeCierreCampanha.setMuestrasTotal(sumaMuestras);

		plantillaInformeCierreCampanha.setSumaProtocolosMuestras(sumaProtocolos
				+ sumaMuestras);

		return plantillaInformeCierreCampanha;
	}

	/**
	 * 03. Productos de la campanha.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findProductosCampanha(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "WITH TABLADATOS AS ("
				+ "SELECT DISTINCT (ATCOPSNOM) DATOS "
				+ "FROM ACTUACION WHERE ATCOCM = ? AND ATISTOMAMUESTRAS = 'S' "
				+ ") "
				+ "SELECT (SELECT LISTAGG(DATOS, '; ') WITHIN GROUP (ORDER BY NULL) FROM TABLADATOS) PRODUCTOS "
				+ "FROM DUAL";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setProductosCampanha(resultSet
												.getString("PRODUCTOS"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 04. Origen de las tomas de muestras.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findOrigenTomas(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "WITH TABLADATOS AS ( "
				+ "SELECT DISTINCT (CASE ADCOOR "
				+ "WHEN '01A001' THEN 'Araba' "
				+ "WHEN '20A001' THEN 'Gipuzkoa' "
				+ "WHEN '48A001' THEN 'Bizkaia' "
				+ "ELSE '' END) DATOS "
				+ "FROM ACTUACION, ACTDEC "
				+ "WHERE ATCODIGO = ADCOAT AND ATCOCM = ? AND ATISTOMAMUESTRAS = 'S' "
				+ ") "
				+ "SELECT (SELECT LISTAGG(DATOS, '/') WITHIN GROUP (ORDER BY NULL) FROM TABLADATOS) PROVINCIAS "
				+ "FROM DUAL";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setProvincias(resultSet
												.getString("PROVINCIAS"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 05. Suma de muestras indicativas y reglamentarias realizadas.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findSumaMuestrasIndicativasReglamentarias(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT ANCODCARACTERMU, COUNT(1) TOTAL "
				+ "FROM ACTUACION t1, ACTA_NUEVA t2 "
				+ "WHERE t1.ATCODIGO = t2.ANCODINS AND t1.ATCOCM = ? AND ANTIPO LIKE '%T%' "
				+ "GROUP BY ANCODCARACTERMU";

		final class IndsYRegs {
			public String tipo;
			public int total;
		}

		List<IndsYRegs> listaIndsYRegs = (List<IndsYRegs>) this.jdbcTemplate
				.query(query, new RowMapper<IndsYRegs>() {
					public IndsYRegs mapRow(ResultSet resultSet, int rowNum)
							throws SQLException {
						IndsYRegs indsYRegs = new IndsYRegs();

						indsYRegs.tipo = resultSet.getString("ANCODCARACTERMU");
						indsYRegs.total = resultSet.getInt("TOTAL");

						return indsYRegs;
					}
				}, plantillaInformeCierreCampanha.getCmCodigo());

		plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

		for (IndsYRegs indsYRegs : listaIndsYRegs) {
			if ("I".equals(indsYRegs.tipo)) {
				plantillaInformeCierreCampanha
						.setSumaMuestrasIndicativas(indsYRegs.total);
			} else if ("R".equals(indsYRegs.tipo)) {
				plantillaInformeCierreCampanha
						.setSumaMuestrasReglamentarias(indsYRegs.total);
			}
		}

		return plantillaInformeCierreCampanha;
	}

	/**
	 * 06. Suma de protocolos estandar.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findSumaProtocolosEstandar(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT COUNT(1) TOTAL FROM ACTUACION "
				+ "WHERE ATCOCM = ? AND ATISCONTROLESTANDAR = 'S'";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setSumaProtocolosEstandar(resultSet
												.getInt("TOTAL"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 07. Suma de protocolos estandar con infraccion.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findSumaProtocolosEstandarInfraccion(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT COUNT(DISTINCT PTCOAT) TOTAL "
				+ "FROM CAMPANHA_TIPOPROTOCOLO CTP, TIP_PROTOCOLO TP, TIP_PREG_PROTOC TPP, PROTOCOLO P, ACTUACION A "
				+ "WHERE CTP.CMCODIGO = ? AND CTP.TPCODIGO = TP.TPCODIGO AND TP.TPCODIGO = TPP.TRCOTP AND TPP.TRCODIGO = P.PTCOTR AND P.PTCOAT = A.ATCODIGO AND A.ATCOCM = ? AND TP.TPESTANDAR = 'S' AND TPP.TRINTI = 'N' AND TPP.TRINRE = 'S' AND TRCOTTI IS NOT NULL AND TPP.TRREES <> P.PTCORETP AND P.PTCORETP <> '-' ";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(
						query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setSumaProtocolosEstandarInfraccion(resultSet
												.getInt("TOTAL"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo(),
						plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 08. Total de protocolos estandar.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findTotalProtocolosEstandar(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT COUNT(DISTINCT TP.TPCODIGO) TOTAL "
				+ "FROM CAMPANHA_TIPOPROTOCOLO CTP, TIP_PROTOCOLO TP, TIP_PREG_PROTOC TPP "
				+ "WHERE CTP.CMCODIGO = ? AND CTP.TPCODIGO = TP.TPCODIGO AND TP.TPCODIGO = TPP.TRCOTP AND TP.TPESTANDAR = 'S'";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setTotalProtocolos(resultSet
												.getInt("TOTAL"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 08.5. Total de infracciones.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findTotalInfracciones(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT COUNT(TPP.TRCODIGO) TOTAL "
				+ "FROM CAMPANHA_TIPOPROTOCOLO CTP, TIP_PROTOCOLO TP, TIP_PREG_PROTOC TPP, PROTOCOLO P, TIP_REQ_NORMATIVO TRN, ACTUACION A "
				+ "WHERE CTP.CMCODIGO = ? AND CTP.TPCODIGO = TP.TPCODIGO AND TP.TPCODIGO = TPP.TRCOTP AND TPP.TRCODIGO = P.PTCOTR AND TP.TPESTANDAR = 'S' AND TPP.TRINTI = 'N' AND TPP.TRINRE = 'S' AND TRCOTTI IS NOT NULL AND TPP.TRREES <> P.PTCORETP AND TPP.TRCOTTI = TRN.TTICODIGO AND P.PTCOAT = A.ATCODIGO AND A.ATCOCM = ? AND P.PTCORETP <> '-' ";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(
						query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setTotalInfracciones(resultSet
												.getInt("TOTAL"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo(),
						plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 09. Preguntas mas frecuentemente infringidas.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findPreguntasFrecuentementeInfringidas(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT ROWNUMBER, TRCODIGO, TRNRTIPP, TRCOTIPP, TRDETIPP, TOTAL, TTINOMBRE FROM ( "
				+ "SELECT ROW_NUMBER() OVER(ORDER BY COUNT(TPP.TRCODIGO) DESC) AS ROWNUMBER, TPP.TRCODIGO, TPP.TRNRTIPP, TPP.TRCOTIPP, TPP.TRDETIPP, COUNT(TPP.TRCODIGO) TOTAL, TRN.TTINOMBRE "
				+ "FROM CAMPANHA_TIPOPROTOCOLO CTP, TIP_PROTOCOLO TP, TIP_PREG_PROTOC TPP, PROTOCOLO P, TIP_REQ_NORMATIVO TRN, ACTUACION A "
				+ "WHERE CTP.CMCODIGO = ? AND CTP.TPCODIGO = TP.TPCODIGO AND TP.TPCODIGO = TPP.TRCOTP AND TPP.TRCODIGO = P.PTCOTR AND TP.TPESTANDAR = 'S' AND TPP.TRINTI = 'N' AND TPP.TRINRE = 'S' AND TRCOTTI IS NOT NULL AND TPP.TRREES <> P.PTCORETP AND TPP.TRCOTTI = TRN.TTICODIGO AND P.PTCOAT = A.ATCODIGO AND A.ATCOCM = ? AND P.PTCORETP <> '-' "
				+ "GROUP BY TPP.TRCODIGO, TPP.TRNRTIPP, TPP.TRDETIPP, TPP.TRCOTIPP, TRN.TTINOMBRE "
				+ "ORDER BY TOTAL DESC ) WHERE ROWNUM <= 3";

		final class Preguntas {
			public int rownumber;
			public String trcodigo;
			private Long trnrtipp;
			public String trcotipp;
			public String trdetipp;
			public int total;
			public String ttinombre;
		}

		List<Preguntas> listaPreguntas = (List<Preguntas>) this.jdbcTemplate
				.query(query, new RowMapper<Preguntas>() {
					public Preguntas mapRow(ResultSet resultSet, int rowNum)
							throws SQLException {
						Preguntas preguntas = new Preguntas();

						preguntas.rownumber = resultSet.getInt("ROWNUMBER");

						preguntas.trcodigo = resultSet.getString("TRCODIGO");
						preguntas.trnrtipp = resultSet.getLong("TRNRTIPP");
						preguntas.trcotipp = resultSet.getString("TRCOTIPP");
						preguntas.trdetipp = resultSet.getString("TRDETIPP");
						preguntas.total = resultSet.getInt("TOTAL");
						preguntas.ttinombre = resultSet.getString("TTINOMBRE");

						return preguntas;
					}
				}, plantillaInformeCierreCampanha.getCmCodigo(),
						plantillaInformeCierreCampanha.getCmCodigo());

		plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

		for (Preguntas preguntas : listaPreguntas) {
			switch (preguntas.rownumber) {
			case Y41bConstantes.NUM_1:
				plantillaInformeCierreCampanha.setTrcodigo1(preguntas.trcodigo);
				plantillaInformeCierreCampanha.setTrnrtipp1(preguntas.trnrtipp);
				plantillaInformeCierreCampanha
						.setProtocolosPregunta1(preguntas.trcotipp + " - "
								+ preguntas.trdetipp);
				plantillaInformeCierreCampanha
						.setProtocolosInfraccion1(preguntas.ttinombre);
				plantillaInformeCierreCampanha
						.setProtocolosTotal1(preguntas.total);
				break;
			case Y41bConstantes.NUM_2:
				plantillaInformeCierreCampanha.setTrcodigo2(preguntas.trcodigo);
				plantillaInformeCierreCampanha.setTrnrtipp2(preguntas.trnrtipp);
				plantillaInformeCierreCampanha
						.setProtocolosPregunta2(preguntas.trcotipp + " - "
								+ preguntas.trdetipp);
				plantillaInformeCierreCampanha
						.setProtocolosInfraccion2(preguntas.ttinombre);
				plantillaInformeCierreCampanha
						.setProtocolosTotal2(preguntas.total);
				break;
			case Y41bConstantes.NUM_3:
				plantillaInformeCierreCampanha.setTrcodigo3(preguntas.trcodigo);
				plantillaInformeCierreCampanha.setTrnrtipp3(preguntas.trnrtipp);
				plantillaInformeCierreCampanha
						.setProtocolosPregunta3(preguntas.trcotipp + " - "
								+ preguntas.trdetipp);
				plantillaInformeCierreCampanha
						.setProtocolosInfraccion3(preguntas.ttinombre);
				plantillaInformeCierreCampanha
						.setProtocolosTotal3(preguntas.total);
				break;
			}
		}

		return plantillaInformeCierreCampanha;
	}

	/**
	 * 09.4. Titulo pregunta1.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findTituloPregunta1(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {

		try {

			String query = "SELECT TRDETIPP "
					+ "FROM  CAMPANHA_TIPOPROTOCOLO CTP, TIP_PROTOCOLO TP, TIP_PREG_PROTOC TPP, "
					+ " (SELECT MAX(TRNRTIPP) TRNRTIPP "
					+ " FROM  CAMPANHA_TIPOPROTOCOLO CTP1, TIP_PROTOCOLO TP1, TIP_PREG_PROTOC TPP1 "
					+ " WHERE CTP1.CMCODIGO = ? AND CTP1.TPCODIGO = TP1.TPCODIGO AND TP1.TPCODIGO = TPP1.TRCOTP "
					+ " AND TP1.TPESTANDAR = 'S' "
					+ " AND TPP1.TRINTI = 'S' "
					+ " AND TPP1.TRNRTIPP < ?) AUX "
					+ " WHERE CTP.CMCODIGO = ? AND CTP.TPCODIGO = TP.TPCODIGO AND TP.TPCODIGO = TPP.TRCOTP "
					+ " AND TP.TPESTANDAR = 'S' "
					+ " AND TPP.TRNRTIPP = AUX.TRNRTIPP ";

			return (PlantillaInformeCierreCampanha) this.jdbcTemplate
					.queryForObject(
							query,
							new RowMapper<PlantillaInformeCierreCampanha>() {
								public PlantillaInformeCierreCampanha mapRow(
										ResultSet resultSet, int rowNum)
										throws SQLException {
									PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

									if (resultSet.getString("TRDETIPP") != null) {
										plantillaInformeCierreCampanha
												.setTitulo1(resultSet
														.getString("TRDETIPP"));
									} else {
										plantillaInformeCierreCampanha
												.setTitulo1("");
									}

									return plantillaInformeCierreCampanha;
								}
							}, plantillaInformeCierreCampanha.getCmCodigo(),
							plantillaInformeCierreCampanha.getTrnrtipp1(),
							plantillaInformeCierreCampanha.getCmCodigo());

		} catch (EmptyResultDataAccessException e) {
			return new PlantillaInformeCierreCampanha();
		}
	}

	/**
	 * 09.5. Titulo pregunta2.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findTituloPregunta2(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		try {
			String query = "SELECT TRDETIPP "
					+ "FROM  CAMPANHA_TIPOPROTOCOLO CTP, TIP_PROTOCOLO TP, TIP_PREG_PROTOC TPP, "
					+ " (SELECT MAX(TRNRTIPP) TRNRTIPP "
					+ " FROM  CAMPANHA_TIPOPROTOCOLO CTP1, TIP_PROTOCOLO TP1, TIP_PREG_PROTOC TPP1 "
					+ " WHERE CTP1.CMCODIGO = ? AND CTP1.TPCODIGO = TP1.TPCODIGO AND TP1.TPCODIGO = TPP1.TRCOTP "
					+ " AND TP1.TPESTANDAR = 'S' "
					+ " AND TPP1.TRINTI = 'S' "
					+ " AND TPP1.TRNRTIPP < ?) AUX "
					+ " WHERE CTP.CMCODIGO = ? AND CTP.TPCODIGO = TP.TPCODIGO AND TP.TPCODIGO = TPP.TRCOTP "
					+ " AND TP.TPESTANDAR = 'S' "
					+ " AND TPP.TRNRTIPP = AUX.TRNRTIPP ";

			return (PlantillaInformeCierreCampanha) this.jdbcTemplate
					.queryForObject(
							query,
							new RowMapper<PlantillaInformeCierreCampanha>() {
								public PlantillaInformeCierreCampanha mapRow(
										ResultSet resultSet, int rowNum)
										throws SQLException {
									PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

									if (resultSet.getString("TRDETIPP") != null) {
										plantillaInformeCierreCampanha
												.setTitulo2(resultSet
														.getString("TRDETIPP"));
									} else {
										plantillaInformeCierreCampanha
												.setTitulo2("");
									}

									return plantillaInformeCierreCampanha;
								}
							}, plantillaInformeCierreCampanha.getCmCodigo(),
							plantillaInformeCierreCampanha.getTrnrtipp2(),
							plantillaInformeCierreCampanha.getCmCodigo());

		} catch (EmptyResultDataAccessException e) {
			return new PlantillaInformeCierreCampanha();
		}

	}

	/**
	 * 09.6. Titulo pregunta3.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findTituloPregunta3(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		try {
			String query = "SELECT TRDETIPP "
					+ "FROM  CAMPANHA_TIPOPROTOCOLO CTP, TIP_PROTOCOLO TP, TIP_PREG_PROTOC TPP, "
					+ " (SELECT MAX(TRNRTIPP) TRNRTIPP "
					+ " FROM  CAMPANHA_TIPOPROTOCOLO CTP1, TIP_PROTOCOLO TP1, TIP_PREG_PROTOC TPP1 "
					+ " WHERE CTP1.CMCODIGO = ? AND CTP1.TPCODIGO = TP1.TPCODIGO AND TP1.TPCODIGO = TPP1.TRCOTP "
					+ " AND TP1.TPESTANDAR = 'S' "
					+ " AND TPP1.TRINTI = 'S' "
					+ " AND TPP1.TRNRTIPP < ?) AUX "
					+ " WHERE CTP.CMCODIGO = ? AND CTP.TPCODIGO = TP.TPCODIGO AND TP.TPCODIGO = TPP.TRCOTP "
					+ " AND TP.TPESTANDAR = 'S' "
					+ " AND TPP.TRNRTIPP = AUX.TRNRTIPP ";

			return (PlantillaInformeCierreCampanha) this.jdbcTemplate
					.queryForObject(
							query,
							new RowMapper<PlantillaInformeCierreCampanha>() {
								public PlantillaInformeCierreCampanha mapRow(
										ResultSet resultSet, int rowNum)
										throws SQLException {
									PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

									if (resultSet.getString("TRDETIPP") != null) {
										plantillaInformeCierreCampanha
												.setTitulo3(resultSet
														.getString("TRDETIPP"));
									} else {
										plantillaInformeCierreCampanha
												.setTitulo3("");
									}

									return plantillaInformeCierreCampanha;
								}
							}, plantillaInformeCierreCampanha.getCmCodigo(),
							plantillaInformeCierreCampanha.getTrnrtipp3(),
							plantillaInformeCierreCampanha.getCmCodigo());

		} catch (EmptyResultDataAccessException e) {
			return new PlantillaInformeCierreCampanha();
		}
	}

	/**
	 * 10. Actas con incumplimiento.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findActasIncumplimiento(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT LISTAGG(AN.ANCODAC, '; ') WITHIN GROUP(ORDER BY NULL) ANCODAC "
				+ "FROM ACTA_NUEVA AN, ACTUACION A "
				+ "WHERE AN.ANCODINS = A.ATCODIGO AND A.ATCOCM = ? AND AN.ANTIPO LIKE '%I%'";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setProtocolosActas(resultSet
												.getString("ANCODAC"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 11. Suma de actuaciones con toma de muestras.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findSumaMuestras(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT COUNT(1) TOTAL FROM ACTUACION "
				+ "WHERE ATISTOMAMUESTRAS = 'S' AND ATCOCM = ?";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setSumaMuestras(resultSet
												.getInt("TOTAL"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 12. Suma de tomas de muestra con infraccion.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findSumaMuestrasInfraccion(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT COUNT(DISTINCT ATCODIGO) TOTAL FROM ACTUACION A, TRAMITE_INSPECCION TI "
				+ "WHERE A.ATCODIGO = TI.TIATCODIGO(+) AND A.ATCOCM = ? AND TI.TICOTEE = 'IN12' AND TI.TIESTADOTRAMITE = 'T' AND TI.TICONSTATACION = 'S'";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setSumaMuestrasInfraccion(resultSet
												.getInt("TOTAL"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 13. Suma de incumplimientos detectados.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findSumaIncumplimientosDetectados(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT COUNT(DISTINCT AI.AICOTTI) TOTAL "
				+ "FROM ACTUACION A, TRAMITE_INSPECCION TI, ACTA_INFRACCION AI "
				+ "WHERE A.ATCODIGO = TI.TIATCODIGO(+) AND A.ATCOCM = ? AND TI.TICOTEE = 'IN12' AND TI.TIESTADOTRAMITE = 'T' AND TI.TICONSTATACION = 'S' AND TI.TICODIGO = AI.AICOAT(+)";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setSumaInfraccionesMuestras(resultSet
												.getInt("TOTAL"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 14. Texto de incumplimientos detectados.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findTextoIncumplimientosDetectados(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "WITH TABLADATOS AS ( "
				+ "SELECT DISTINCT TRN.TTINOMBRE DATOS "
				+ "FROM ACTUACION A, TRAMITE_INSPECCION TI, ACTA_INFRACCION AI, TIP_REQ_NORMATIVO TRN "
				+ "WHERE A.ATCODIGO = TI.TIATCODIGO(+) AND A.ATCOCM = ? AND TI.TICOTEE = 'IN12' AND TI.TIESTADOTRAMITE = 'T' AND TI.TICONSTATACION = 'S' AND TI.TICODIGO = AI.AICOAT(+) AND AI.AICOTTI = TRN.TTICODIGO"
				+ ") SELECT (SELECT LISTAGG(DATOS, '; ') WITHIN GROUP (ORDER BY NULL) FROM TABLADATOS) INFRACCIONES FROM DUAL";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setMuestrasTextoInfraccion(resultSet
												.getString("INFRACCIONES"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 15. Actas de toma de muestras relacionadas con incumplimientos.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findActasMuestrasIncumplimientos(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT LISTAGG(AN.ANCODAC, '; ') WITHIN GROUP (ORDER BY NULL) ACTAS "
				+ "FROM ACTUACION A, TRAMITE_INSPECCION TI, ACTA_NUEVA AN "
				+ "WHERE A.ATCODIGO = TI.TIATCODIGO(+) AND A.ATCOCM = ? AND TI.TICOTEE = 'IN12' AND TI.TIESTADOTRAMITE = 'T' AND TI.TICONSTATACION = 'S' AND TI.TIACCODIGO = AN.ANCODIGO";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setMuestrasActas(resultSet
												.getString("ACTAS"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 16. Actuaciones propuestas por SSCC.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findActuacionesPropuestasSSCC(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "SELECT TYPROPUESTASSCC FROM CAMPANHA_CIERRE WHERE TYCMCODIGO = ?";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setPropuestasSSCC(resultSet
												.getString("TYPROPUESTASSCC"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}

	/**
	 * 17. Establecimientos visitados durante las actuaciones.
	 * 
	 * @param plantillaInformeCierreCampanha
	 *            PlantillaInformeCierreCampanha
	 * @return PlantillaInformeCierreCampanha
	 */
	@Transactional(readOnly = true)
	public PlantillaInformeCierreCampanha findEstablecimientosVisitados(
			PlantillaInformeCierreCampanha plantillaInformeCierreCampanha) {
		String query = "WITH TABLADATOS AS ("
				+ "SELECT DISTINCT I.SENOMBRE DATOS "
				+ "FROM  ACTUACION A, ACTA AC, INTERVINIENTES I "
				+ "WHERE A.ATCOAC = AC.ACCODIGO AND AC.ACCOES IS NOT NULL AND AC.ACCOES = I.ESCODIGO(+) AND A.ATCODIGO = I.HDCODIGO AND I.TIPO_INTERVINIENTE = 'V' AND A.ATCOCM = ?"
				+ ") SELECT (SELECT LISTAGG(DATOS, '; ') WITHIN GROUP (ORDER BY NULL) FROM TABLADATOS) ESTABLECIMIENTOS FROM DUAL";

		return (PlantillaInformeCierreCampanha) this.jdbcTemplate
				.queryForObject(query,
						new RowMapper<PlantillaInformeCierreCampanha>() {
							public PlantillaInformeCierreCampanha mapRow(
									ResultSet resultSet, int rowNum)
									throws SQLException {
								PlantillaInformeCierreCampanha plantillaInformeCierreCampanha = new PlantillaInformeCierreCampanha();

								plantillaInformeCierreCampanha
										.setEstablecimientosVisitados(resultSet
												.getString("ESTABLECIMIENTOS"));

								return plantillaInformeCierreCampanha;
							}
						}, plantillaInformeCierreCampanha.getCmCodigo());
	}
}