package com.ejie.y41b.adapter.transformer;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import noNamespace.DocumentDocument;
import noNamespace.DocumentIDDocument;
import noNamespace.FormatListDocument;
import noNamespace.SignatureListDocument;
import noNamespace.T65BAttributeListType;
import noNamespace.T65BAttributeType;
import noNamespace.T65BDocumentIDType;
import noNamespace.T65BDocumentType;
import noNamespace.T65BFormatListType;
import noNamespace.T65BFormatType;
import noNamespace.T65BSignatureListType;
import noNamespace.T65BSignatureType;

import org.apache.commons.lang.StringEscapeUtils;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.bea.xml.XmlException;
import com.ejie.y41b.model.dokusi.Documento;
import com.ejie.y41b.model.dokusi.Firma;
import com.ejie.y41b.model.dokusi.Formato;
import com.ejie.y41b.model.dokusi.Metadato;
import com.ejie.y41b.model.dokusi.RespuestaEliminarDocumento;
import com.ejie.y41b.model.dokusi.RespuestaLocalizadorDocumento;
import com.ejie.y41b.model.dokusi.RespuestaTransformarDocumento;
import com.ejie.y41b.model.sir.NotificationBody;
import com.ejie.y41b.model.sir.schemaSIRDocumentList.DocumentList;
import com.ejie.y41b.model.sir.schemaSIRDocumentList.DocumentList.Document;
import com.ejie.y41b.model.sir.schemaSIRDocumentList.DocumentList.Document.AttributeList;

/**
 * <p>
 * Titulo: Y41bDokusiTransformer
 * </p>
 * <p>
 * Descripcion: Contiene metodos para devolver el resultado en las llamadas a
 * los WebServices de Dokusi
 * </p>
 * <p>
 * Copyright: Copyright (c) 2013
 * </p>
 * <p>
 * Empresa: Bilbomatica
 * </p>
 * 
 *  
 * @version 1.0
 */
public final class Y41bDokusiTransformer {

	/**
	 * Constructor
	 */
	private Y41bDokusiTransformer() {
		super();
	}

	/**
	 * Transforma un xml resultado de la llamada al metodo de busqueda de
	 * documentos en dokusi en un objeto List<Documento> de nuestro modelo de
	 * negocio.
	 * 
	 * @param xml
	 *            ListaIdsDocumentosDocumentum
	 * @return List<Documento>
	 * @throws XmlException
	 *             XmlException
	 * @throws IOException
	 *             IOException
	 * @throws SAXException
	 *             SAXException
	 * @throws ParserConfigurationException
	 *             ParserConfigurationException
	 */
	public static List<Documento> transformBuscarDocumentosDokusi(String xml)
			throws XmlException, SAXException, IOException, ParserConfigurationException {
		List<Documento> listaDocumentos = new ArrayList<Documento>();
		List<Metadato> documentoMetadatos = new ArrayList<Metadato>();
		Metadato documentoMetadato = null;
		Documento documento = null;
		DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
		DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
		InputStream input = new ByteArrayInputStream(xml.getBytes("UTF-8"));

		org.w3c.dom.Document doc = docBuilder.parse(input);
		doc.getDocumentElement().normalize();
		NodeList lista = doc.getElementsByTagName("object");
		for (int i = 0; i < lista.getLength(); i++) {
			documentoMetadatos = new ArrayList<Metadato>();
			Node objeto = lista.item(i);
			NodeList metadatos = objeto.getChildNodes();
			documento = new Documento();
			documento.setOidDokusi(metadatos.item(0).getParentNode().getAttributes().getNamedItem("ID").getNodeValue());
			for (int x = 0; x < metadatos.getLength(); x++) {
				documentoMetadato = new Metadato();
				documentoMetadato.setNombre(metadatos.item(x).getNodeName());
				documentoMetadato.setValor((metadatos.item(x).getFirstChild() != null)
						? metadatos.item(x).getFirstChild().getNodeValue() : "");
				if (!"#text".equals(documentoMetadato.getNombre())) {
					documentoMetadatos.add(documentoMetadato);
				}
			}
			documento.setMetadatos(documentoMetadatos);
			listaDocumentos.add(documento);
		}
		return listaDocumentos;
	}

	/**
	 * Transforma un xml resultado de la llamada al metodo de registrar
	 * documento en dokusi en un objeto Documento de nuestro modelo de negocio.
	 * 
	 * @param xml
	 *            ListaIdsDocumentosDocumentum
	 * @return String con el OID de DOKUSI
	 * 
	 * @throws XmlException
	 *             XmlException
	 */
	public static String transformRegistrarDocumentoDokusi(String xml) throws XmlException {

		DocumentIDDocument documentID = DocumentIDDocument.Factory.parse(xml);

		return documentID.getDocumentID().getId();
	}

	/**
	 * Transforma un xml resultado de la llamada al metodo de
	 * shareDocumentContents en dokusi en un objeto Documento de nuestro modelo
	 * de negocio.
	 * 
	 * @param xml
	 *            ListaIdsDocumentosDocumentum
	 * @return String con el OID de DOKUSI
	 * 
	 * @throws XmlException
	 *             XmlException
	 */
	public static String transformShareDocumentContents(String xml) throws XmlException {

		DocumentIDDocument documentID = DocumentIDDocument.Factory.parse(xml);

		return documentID.getDocumentID().getId();
	}

	/**
	 * Transforma un xml resultado de la llamada al metodo de modificar
	 * documento en dokusi en un objeto Documento de nuestro modelo de negocio.
	 * 
	 * @param xml
	 *            ListaIdsDocumentosDocumentum
	 * @return String con el OID de DOKUSI
	 * 
	 * @throws XmlException
	 *             XmlException
	 */
	public static String transformModificarDocumentoDokusi(String xml) throws XmlException {

		DocumentIDDocument documentID = DocumentIDDocument.Factory.parse(xml);

		return documentID.getDocumentID().getId();
	}

	/**
	 * Transforma un xml resultado de la llamada al metodo de descargar
	 * documento de dokusi en un objeto Documento de nuestro modelo de negocio.
	 * 
	 * @param xml
	 *            ListaIdsDocumentosDocumentum
	 * @return Documento documento
	 * 
	 * @throws XmlException
	 *             XmlException
	 */
	public static Documento transformDescargarDocumentoDokusi(String xml) throws XmlException {

		DocumentDocument documentDocument = DocumentDocument.Factory.parse(xml);
		T65BDocumentType t65BDocumentType = documentDocument.getDocument();

		Documento documento = new Documento();

		documento.setPifId(t65BDocumentType.getContent().getPifId());

		documento.setExtension(t65BDocumentType.getContent().getFormat().getExtension());

		documento.setContentType(t65BDocumentType.getContent().getFormat().getMimeType());
		documento.setLength(t65BDocumentType.getContent().getLenght());

		StringBuffer nombreCompleto = new StringBuffer();

		T65BAttributeListType t65BAttributeListType = t65BDocumentType.getAttributeList();
		List<Metadato> metadatos = new ArrayList<Metadato>();
		if (t65BAttributeListType != null && t65BAttributeListType.getAttributeArray().length > 0) {
			for (T65BAttributeType t65BAttributeType : t65BAttributeListType.getAttributeArray()) {
				Metadato metadato = new Metadato();

				metadato.setNombre(t65BAttributeType.getKey());
				metadato.setValor(t65BAttributeType.getValue());

				if (t65BAttributeType.getKey() != null) {
					if (t65BAttributeType.getKey().equals("object_name")) {
						nombreCompleto.append(t65BAttributeType.getValue());
					} else if (t65BAttributeType.getKey().equals("r_object_type")) {
						documento.setTipoDocumental(t65BAttributeType.getValue());
					}
				}
				metadatos.add(metadato);
			}
		}
		documento.setMetadatos(metadatos);
		nombreCompleto.append(".").append(documento.getExtension());
		documento.setNombre(nombreCompleto.toString());
		return documento;
	}

	/**
	 * Transforma un xml resultado de la llamada al metodo de descargar
	 * documento de SIR en un objeto Documento de nuestro modelo de negocio.
	 * 
	 * @param xml
	 *            ListaIdsDocumentosDocumentum
	 * @return Documento documento
	 * 
	 * @throws JAXBException
	 */
	public static Documento transformDescargarDocumentoSIR(String xml) throws JAXBException {

		// Parseo del XML mediante JAXB
		JAXBContext jaxbContext;
		DocumentList documentList = null;

		jaxbContext = JAXBContext.newInstance(DocumentList.class);
		Unmarshaller jaxbUnmarshaller = jaxbContext.createUnmarshaller();

		StringReader reader = new StringReader(StringEscapeUtils.unescapeHtml(xml).replaceAll("xsi:nil=\"true\"", ""));
		documentList = (DocumentList) jaxbUnmarshaller.unmarshal(reader);

		Document document = documentList.getDocument();

		Documento documento = new Documento();

		documento.setPifId(document.getContent().getPifId());

		documento.setExtension(document.getContent().getFormat().getExtension());

		documento.setContentType(document.getContent().getFormat().getMimeType());
		documento.setLength(Long.parseLong(document.getContent().getLenght()));

		StringBuffer nombreCompleto = new StringBuffer();

		AttributeList attributeList = document.getAttributeList();
		List<Metadato> metadatos = new ArrayList<Metadato>();
		if (attributeList != null && attributeList.getAttribute() != null && attributeList.getAttribute().size() > 0) {
			for (DocumentList.Document.AttributeList.Attribute attribute : attributeList.getAttribute()) {
				Metadato metadato = new Metadato();

				metadato.setNombre(attribute.getKey());
				metadato.setValor(attribute.getValue());

				if (attribute.getKey() != null) {
					if (attribute.getKey().equals("object_name")) {
						nombreCompleto.append(attribute.getValue());
					} else if (attribute.getKey().equals("r_object_type")) {
						documento.setTipoDocumental(attribute.getValue());
					}
				}
				metadatos.add(metadato);
			}
		}
		documento.setMetadatos(metadatos);
		nombreCompleto.append(".").append(documento.getExtension());
		documento.setNombre(nombreCompleto.toString());
		return documento;
	}

	/**
	 * Transforma un xml resultado de la llamada al metodo de eliminar documento
	 * en dokusi en un objeto Documento de nuestro modelo de negocio.
	 * 
	 * @param xml
	 *            xml de respuesta eliminar
	 * @return RespuestaEliminarDocumento RespuestaEliminarDocumento
	 * 
	 * @throws XmlException
	 *             XmlException
	 */
	public static RespuestaEliminarDocumento transformEliminarDocumentoDokusi(String xml) throws XmlException {

		DocumentIDDocument documentID = DocumentIDDocument.Factory.parse(xml);
		T65BDocumentIDType t65BDocumentIDType = documentID.getDocumentID();

		RespuestaEliminarDocumento respuestaEliminarDocumento = new RespuestaEliminarDocumento();
		respuestaEliminarDocumento.setOidDokusi(t65BDocumentIDType.getId());
		respuestaEliminarDocumento.setVersion(t65BDocumentIDType.getVersion());

		return respuestaEliminarDocumento;
	}

	/**
	 * Transforma un xml resultado de la llamada al metodo de recuperar firmas
	 * de un documento en dokusi en un objeto Documento de nuestro modelo de
	 * negocio.
	 * 
	 * @param xml
	 *            xml de respuesta eliminar
	 * @return Documento Documento
	 * 
	 * @throws XmlException
	 *             XmlException
	 */
	public static Documento transformRecuperarFirmarDocumentosDokusi(String xml) throws XmlException {

		SignatureListDocument signatureListDocument = SignatureListDocument.Factory.parse(xml);
		T65BSignatureListType t65BSignatureListType = signatureListDocument.getSignatureList();

		Documento documento = new Documento();
		List<Firma> listaFirmas = new ArrayList<Firma>();

		if (t65BSignatureListType != null && t65BSignatureListType.getSignatureArray().length > 0) {
			for (T65BSignatureType t65BSignatureType : t65BSignatureListType.getSignatureArray()) {
				Firma firma = new Firma();

				firma.setFecha(t65BSignatureType.getDate());
				firma.setFirma(t65BSignatureType.getSignature());
				listaFirmas.add(firma);
			}
		}

		documento.setFirmas(listaFirmas);

		return documento;
	}

	/**
	 * Transforma un String resultado de la llamada al metodo de incorporar
	 * localizador en dokusi en un objeto RespuestaLocalizadorDocumento de
	 * nuestro modelo de negocio.
	 * 
	 * @param oidPeticion
	 *            String
	 * @return RespuestaLocalizadorDocumento RespuestaLocalizadorDocumento
	 * 
	 * @throws XmlException
	 *             XmlException
	 */
	public static RespuestaLocalizadorDocumento transformIncorporarLocalizadorDokusi(String oidPeticion)
			throws XmlException {
		RespuestaLocalizadorDocumento respuestaLocalizadorDocumento = new RespuestaLocalizadorDocumento();
		respuestaLocalizadorDocumento.setOidPeticion(oidPeticion);
		return respuestaLocalizadorDocumento;
	}

	/**
	 * Transforma un String resultado de la llamada al metodo de incorporar
	 * transformar en dokusi en un objeto RespuestaTransformarDocumento de
	 * nuestro modelo de negocio.
	 * 
	 * @param oidPeticion
	 *            String
	 * @return RespuestaTransformarDocumento RespuestaTransformarDocumento
	 * 
	 * @throws XmlException
	 *             XmlException
	 */
	public static RespuestaTransformarDocumento transformTransformarDocumentoDokusi(String oidPeticion)
			throws XmlException {
		RespuestaTransformarDocumento respuestaTransformarDocumento = new RespuestaTransformarDocumento();
		respuestaTransformarDocumento.setOidPeticion(oidPeticion);
		return respuestaTransformarDocumento;
	}

	/**
	 * Transforma un xml resultado de la llamada al metodo de recuperar formatos
	 * de un documento en dokusi en un objeto Documento de nuestro modelo de
	 * negocio.
	 * 
	 * @param xml
	 *            xml de respuesta eliminar
	 * @return Formato Formato
	 * 
	 * @throws XmlException
	 *             XmlException
	 */
	public static Formato transformRecuperarFormatosDocumentosDokusi(String xml) throws XmlException {

		FormatListDocument formatListDocument = FormatListDocument.Factory.parse(xml);
		T65BFormatListType t65BFormatListType = formatListDocument.getFormatList();

		Formato formato = new Formato();

		if (t65BFormatListType != null && t65BFormatListType.getFormatArray().length > 0) {
			for (T65BFormatType t65BFormatType : t65BFormatListType.getFormatArray()) {
				formato = new Formato();

				formato.setDescription(t65BFormatType.getDescription());
				formato.setExtension(t65BFormatType.getExtension());
				formato.setMimeType(t65BFormatType.getMimeType());
				formato.setName(t65BFormatType.getName());
				formato.setFullText(t65BFormatType.getFullText());
			}
		}

		return formato;
	}
}
