package com.ejie.y41b.adapter.services;

import java.util.Properties;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.stereotype.Service;
import org.springframework.context.i18n.LocaleContextHolder;

import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.utils.exception.Y41bUDAException;
import com.ejie.y41b.utils.ws.Y41bUtilsWS;

/**
 * <p>
 * Titulo: Y41bPlateaServiceImpl
 * </p>
 * <p>
 * Descipcion: Realiza invocaciones a los webServices de Platea
 * </p>
 * <p>
 * Copyright: Copyright (c) 2013
 * </p>
 * <p>
 * Empresa: Bilbomatica
 * </p>
 * 
 *  
 * @version 1.0
 */
@Service(value = "Y41bPlateaServiceImpl")
public class Y41bPlateaServiceImpl implements Y41bPlateaService {

	/**
	 * Final static logger.
	 */
	private static final Logger logger = LoggerFactory.getLogger(Y41bPlateaServiceImpl.class);

	@Autowired
	private Properties appConfiguration;

	@Resource()
	private ReloadableResourceBundleMessageSource appMessageSource;

	/**
	 * Clase que hara de facade para invocar los servicios de Platea
	 */
	private Y41bPlateaServiceImpl() {
		super();
	}

	/**
	 * Método que invoca al webService sendEjgvDocumentPresencial de la
	 * administración para crear la solicitud presencial
	 * 
	 * @param tokenSesion
	 *            String: Token de sesion
	 * @param contextSubmission
	 *            String: Xml que lleva toda los datos necesarios para crear la
	 *            Solicitud presencial
	 * @return String: Xml que representa la respuesta del envío de la
	 *         solicitud.
	 * @throws Exception
	 * 
	 */
	public String crearSolicitudPresencial(String tokenSesion, String contextSubmission) throws Exception {
		logger.info("Y41bPlateaServiceImpl.crearSolicitudPresencial - Inicio");
		try {
			String respuesta;

			Q99MPPSFacadeWSServiceImpl q99MPPSFacadeWSServiceImpl = new Q99MPPSFacadeWSServiceImpl();
			respuesta = q99MPPSFacadeWSServiceImpl.crearSolicitudPresencial(tokenSesion, contextSubmission);

			logger.debug("Y41bPlateaServiceImpl.crearSolicitudPresencial - Return: " + respuesta);
			respuesta = respuesta.replace("<RequestReturn>", "<RequestReturn xmlns=\"com/ejie/folderManagement/xml\">");
			return respuesta;

		} catch (Exception e_crearSolicitudPresencial) {
			logger.error("Y41bPlateaServiceImpl.crearSolicitudPresencial", e_crearSolicitudPresencial);
			throw e_crearSolicitudPresencial;
		} finally {
			logger.info("Y41bPlateaServiceImpl.crearSolicitudPresencial - Final");
		}
	}

	/**
	 * Metodo que invoca al webService saveOrUpdateProceedingsNT de la
	 * administracin para actualizar mis gestiones
	 * 
	 * @param tokenSesion
	 *            String: Token de sesion
	 * @param sXMLExpedient
	 *            String: Xml que lleva toda los datos necesarios para atualizar
	 *            mis gestiones
	 * @return String: Xml que representa la respuesta
	 * @throws Exception
	 * 
	 */
	public String actualizarMisGestiones(String tokenSesion, String sXMLExpedient) throws Exception {
		logger.info("Y41bPlateaServiceImpl.actualizarMisGestiones - Inicio");
		try {

			String respuesta = "";

			Q99USendMessageSCPFacadeWSServiceImpl q99USendMessageSCPFacadeWSServiceImpl = new Q99USendMessageSCPFacadeWSServiceImpl();
			q99USendMessageSCPFacadeWSServiceImpl.actualizarMisGestiones(tokenSesion, sXMLExpedient);

			logger.debug("Y41bPlateaServiceImpl.actualizarMisGestiones - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_actualizarMisGestiones) {
			logger.error("Y41bPlateaServiceImpl.actualizarMisGestiones", e_actualizarMisGestiones);
			throw e_actualizarMisGestiones;
		} finally {
			logger.info("Y41bPlateaServiceImpl.actualizarMisGestiones - Final");
		}
	}

	/**
	 * Metodo que invoca al webService deleteProceedingsNT de la administracin
	 * para eliminar de mis gestiones
	 * 
	 * @param tokenSesion
	 *            String: Token de sesion
	 * @param sXMLExpedient
	 *            String: Xml que lleva toda los datos necesarios para atualizar
	 *            mis gestiones
	 * @return String: Xml que representa la respuesta
	 * @throws Exception
	 * 
	 */
	public String eliminarMisGestiones(String tokenSesion, String sXMLExpedient) throws Exception {
		logger.info("Y41bPlateaServiceImpl.eliminarMisGestiones - Inicio");
		try {
			String respuesta = "";

			Q99USendMessageSCPFacadeWSServiceImpl q99USendMessageSCPFacadeWSServiceImpl = new Q99USendMessageSCPFacadeWSServiceImpl();
			q99USendMessageSCPFacadeWSServiceImpl.eliminarMisGestiones(tokenSesion, sXMLExpedient);

			logger.debug("Y41bPlateaServiceImpl.eliminarMisGestiones - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_actualizarMisGestiones) {
			logger.error("Y41bPlateaServiceImpl.eliminarMisGestiones", e_actualizarMisGestiones);
			throw e_actualizarMisGestiones;
		} finally {
			logger.info("Y41bPlateaServiceImpl.eliminarMisGestiones - Final");
		}
	}

	/**
	 * Metodo que invoca al webService saveOrUpdateExpedientNT de la
	 * administracin para actualizar mis gestiones
	 * 
	 * @param tokenSesion
	 *            String: Token de sesion
	 * @param sXMLExpedient
	 *            String: Xml que lleva toda los datos necesarios para atualizar
	 *            mis gestiones
	 * @return String: Xml que representa la respuesta
	 * @throws Exception
	 * 
	 */
	public String actualizarExpediente(String tokenSesion, String sXMLExpedient) throws Exception {
		logger.info("Y41bPlateaServiceImpl.actualizarExpediente - Inicio");
		try {
			String respuesta = "";

			Q99USendMessageSCPFacadeWSServiceImpl q99USendMessageSCPFacadeWSServiceImpl = new Q99USendMessageSCPFacadeWSServiceImpl();
			q99USendMessageSCPFacadeWSServiceImpl.actualizarExpediente(tokenSesion, sXMLExpedient);

			logger.info("Y41bPlateaServiceImpl.actualizarExpediente - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_actualizarMisGestiones) {
			logger.error("Y41bPlateaServiceImpl.actualizarMisGestiones", e_actualizarMisGestiones);
			throw e_actualizarMisGestiones;
		} finally {
			logger.info("Y41bPlateaServiceImpl.actualizarMisGestiones - Final");
		}
	}

	/**
	 * Método que invoca al webService getProcedureDataAsXml de la
	 * administración para obtener los datos del catálogo de Procedimientos
	 * 
	 * @param tokenSesion
	 *            String: Token de sesion
	 * @param procedureId
	 *            String: id del procedimiento en el catálogo de procedimientos
	 * @return String: XML con los datos del procedimiento
	 * @throws Exception
	 * 
	 */
	public String obtenerDatosCatalogoProcedimientos(String tokenSesion, String procedureId) throws Exception {
		logger.info("Y41bPlateaServiceImpl.obtenerDatosCatalogoProcedimientos - Inicio");
		try {
			String respuesta;

			Q99RCPFacadeWSServiceImpl q99RCPFacadeWSServiceImpl = new Q99RCPFacadeWSServiceImpl();
			respuesta = q99RCPFacadeWSServiceImpl.obtenerDatosCatalogoProcedimientos(tokenSesion, procedureId);

			logger.debug("Y41bPlateaServiceImpl.obtenerDatosCatalogoProcedimientos - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_obtenerDatosCatalogoProcedimientos) {
			logger.error("Y41bPlateaServiceImpl.obtenerDatosCatalogoProcedimientos",
					e_obtenerDatosCatalogoProcedimientos);
			throw e_obtenerDatosCatalogoProcedimientos;
		} finally {
			logger.info("Y41bPlateaServiceImpl.obtenerDatosCatalogoProcedimientos - Final");
		}
	}

	/**
	 * Método que realiza la apertura de un expediente
	 * 
	 * @param tokenSesion
	 *            String
	 * @param sXMLExpedient
	 *            String
	 * @return String
	 * @throws Exception
	 */
	public String abrirExpediente(String tokenSesion, String sXMLExpedient) throws Exception {
		logger.info("Y41bPlateaServiceImpl.abrirExpediente - Inicio");
		try {
			String respuesta = "";

			Q99USendMessageSCPFacadeWSServiceImpl q99USendMessageSCPFacadeWSServiceImpl = new Q99USendMessageSCPFacadeWSServiceImpl();
			q99USendMessageSCPFacadeWSServiceImpl.abrirExpediente(tokenSesion, sXMLExpedient);

			logger.info("Y41bPlateaServiceImpl.abrirExpediente - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_abrirExpediente) {
			logger.error("Y41bPlateaServiceImpl.abrirExpediente", e_abrirExpediente);
			throw e_abrirExpediente;
		} finally {
			logger.info("Y41bPlateaServiceImpl.abrirExpediente - Final");
		}
	}

	/**
	 * Método que realiza la apertura de un expediente de oficio
	 * 
	 * @param tokenSesion
	 *            String
	 * @param sXMLExpedient
	 *            String
	 * @return String
	 * @throws Exception
	 *             ex
	 */
	public String abrirExpedienteOficio(String tokenSesion, String sXMLExpedient) throws Exception {
		logger.info("Y41bPlateaServiceImpl.abrirExpedienteOficio - Inicio");
		try {
			String respuesta = "";

			Q99OFacadeWSServiceImpl q99OFacadeWSServiceImpl = new Q99OFacadeWSServiceImpl();
			respuesta = q99OFacadeWSServiceImpl.abrirExpedienteOficio(tokenSesion, sXMLExpedient);

			logger.debug("Y41bPlateaServiceImpl.abrirExpedienteOficio - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_abrirExpedienteOficio) {
			logger.error("Y41bPlateaServiceImpl.abrirExpedienteOficio", e_abrirExpedienteOficio);
			throw e_abrirExpedienteOficio;
		} finally {
			logger.info("Y41bPlateaServiceImpl.abrirExpedienteOficio - Final");
		}
	}

	/**
	 * Método que realiza el cierre de un expediente en Mis Gestiones
	 * 
	 * @param tokenSesion
	 *            String
	 * @param xmlCierre
	 *            String
	 * @return String
	 * @throws Exception
	 */
	public String cerrarExpediente(String tokenSesion, String xmlCierre) throws Exception {
		logger.info("Y41bPlateaServiceImpl.cerrarExpediente - Inicio");
		try {
			String respuesta = "";

			Q99USendMessageSCPFacadeWSServiceImpl q99USendMessageSCPFacadeWSServiceImpl = new Q99USendMessageSCPFacadeWSServiceImpl();
			q99USendMessageSCPFacadeWSServiceImpl.cerrarExpediente(tokenSesion, xmlCierre);

			logger.debug("Y41bPlateaServiceImpl.cerrarExpediente - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_cerrarExpediente) {
			logger.error("Y41bPlateaServiceImpl.cerrarExpediente", e_cerrarExpediente);
			throw e_cerrarExpediente;
		} finally {
			logger.info("Y41bPlateaServiceImpl.cerrarExpediente - Final");
		}
	}

	/**
	 * Método que crea un registro de entrada
	 * 
	 * @param tokenSesion
	 *            String
	 * @param sXmlContextRegistry
	 *            String
	 * @return String
	 * @throws Exception
	 */
	public String obtenerRegistroEntrada(String tokenSesion, String sXmlContextRegistry) throws Exception {
		logger.info("Y41bPlateaServiceImpl.obtenerRegistroEntrada - Inicio");
		try {

			String respuesta = "";

			Q99DSRTFacadeWSServiceImpl q99DSRTFacadeWSServiceImpl = new Q99DSRTFacadeWSServiceImpl();
			respuesta = q99DSRTFacadeWSServiceImpl.obtenerRegistroEntrada(tokenSesion, sXmlContextRegistry);

			logger.debug("Y41bPlateaServiceImpl.obtenerRegistroEntrada - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_obtenerRegistroEntrada) {
			logger.error("Y41bPlateaServiceImpl.obtenerRegistroEntrada", e_obtenerRegistroEntrada);
			throw e_obtenerRegistroEntrada;
		} finally {
			logger.info("Y41bPlateaServiceImpl.obtenerRegistroEntrada - Final");
		}
	}

	/**
	 * Método que crea un registro de salida
	 * 
	 * @param tokenSesion
	 *            String
	 * @param sXmlExitRegistryCreationInfo
	 *            String
	 * @return String
	 * @throws Exception
	 */
	public String obtenerRegistroSalida(String tokenSesion, String sXmlExitRegistryCreationInfo) throws Exception {
		logger.info("Y41bPlateaServiceImpl.obtenerRegistroSalida - Inicio");
		try {

			String respuesta = "";

			Q99DSRTFacadeWSServiceImpl q99DSRTFacadeWSServiceImpl = new Q99DSRTFacadeWSServiceImpl();
			respuesta = q99DSRTFacadeWSServiceImpl.obtenerRegistroSalida(tokenSesion, sXmlExitRegistryCreationInfo);

			logger.debug("Y41bPlateaServiceImpl.obtenerRegistroSalida - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_obtenerRegistroSalida) {
			logger.error("Y41bPlateaServiceImpl.obtenerRegistroSalida", e_obtenerRegistroSalida);
			throw e_obtenerRegistroSalida;
		} finally {
			logger.info("Y41bPlateaServiceImpl.obtenerRegistroSalida - Final");
		}
	}

	/**
	 * Obtiene un xml con la información contenida en el certificado
	 * 
	 * @param tokenSesion
	 *            el token
	 * @param firmaSimple
	 *            la firma simple
	 * @return String datos del certificado
	 * @throws Exception
	 */
	public String obtenerDatosCertificadoFirma(String tokenSesion, String firmaSimple) throws Exception {
		logger.info("Y41bPlateaServiceImpl.obtenerDatosCertificadoFirma - Inicio");
		try {

			String paramValues[] = new String[Y41bConstantes.NUM_2];
			paramValues[0] = tokenSesion;// Session token de XL-Nets
			paramValues[1] = firmaSimple;

			String[] paramNames = new String[Y41bConstantes.NUM_2];
			paramNames[0] = "sSessionToken";
			paramNames[1] = "baSimpleSign";

			String respuesta;

			respuesta = Y41bUtilsWS.invoke(appConfiguration.getProperty("URL_SHF"),
					"getSignatureCertificateInfoGeneric", paramNames, paramValues);

			logger.debug("Y41bPlateaServiceImpl.obtenerDatosCertificadoFirma - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_obtenerDatosCertificadoFirma) {
			logger.error("Y41bPlateaServiceImpl.obtenerDatosCertificadoFirma", e_obtenerDatosCertificadoFirma);
			throw e_obtenerDatosCertificadoFirma;
		} finally {
			logger.info("Y41bPlateaServiceImpl.obtenerDatosCertificadoFirma - Final");
		}
	}

	/**
	 * Obtiene el ejgvDocument de la solicitud o aportación.
	 * 
	 * @param tokenSesion
	 *            Token de sesión en XLNets
	 * @param xmlDocument
	 *            Documento XML con los datos computables de la solicitud
	 * @param xslDocument
	 *            Documento XSL con la presentación de la solicitud
	 * @param textoPlano
	 *            Texto a firmar por el ciudadano.
	 * @return String Documento de la solicitud, representa el html del
	 *         justificante en base64
	 * @throws Exception
	 */
	public String crearEJGVDocumentSolicitud(String tokenSesion, String xmlDocument, String xslDocument,
			String textoPlano) throws Exception {
		logger.info("Y41bPlateaServiceImpl.crearEJGVDocumentSolicitud - Inicio");
		try {
			// crear parametros
			String paramValues[] = new String[Y41bConstantes.NUM_4];
			paramValues[0] = tokenSesion;// Session token de XL-Nets
			paramValues[1] = xmlDocument;// Xml de solicitud
			paramValues[2] = xslDocument;// Xsl de solicitud.
			paramValues[Y41bConstantes.NUM_3] = textoPlano;

			String[] paramNames = new String[Y41bConstantes.NUM_4];
			paramNames[0] = "sessionToken";
			paramNames[1] = "xmlDocument";
			paramNames[2] = "xslDocument";
			paramNames[Y41bConstantes.NUM_3] = "textoPlano";

			String respuesta;

			respuesta = Y41bUtilsWS.invoke(appConfiguration.getProperty("URL_PFS"), "createEjgvDocument", paramNames,
					paramValues);

			logger.debug("Y41bPlateaServiceImpl.crearEJGVDocumentSolicitud - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_crearEJGVDocumentSolicitud) {
			logger.error("Y41bPlateaServiceImpl.crearEJGVDocumentSolicitud", e_crearEJGVDocumentSolicitud);
			throw e_crearEJGVDocumentSolicitud;
		} finally {
			logger.info("Y41bPlateaServiceImpl.crearEJGVDocumentSolicitud - Final");
		}
	}

	/**
	 * Obtiene el ejgvDocument de notificacion.
	 * 
	 * @param tokenSesion
	 *            Token de sesión en XLNets
	 * @param xmlDocumentNotif
	 *            Documento XML con los datos de la notificacion
	 * 
	 * @return String Documento de la solicitud, representa el html del
	 *         justificante en base64
	 * @throws Exception
	 */
	public String crearEJGVDocumentNotificacion(String tokenSesion, String xmlDocumentNotif) throws Exception {
		logger.info("Y41bPlateaServiceImpl.crearEJGVDocumentNotificacion - Inicio");
		try {
			String respuesta;

			Q99MServiceIfzFacadeWSServiceImpl q99MServiceIfzFacadeWSServiceImpl = new Q99MServiceIfzFacadeWSServiceImpl();
			respuesta = q99MServiceIfzFacadeWSServiceImpl.crearEJGVDocumentNotificacion(tokenSesion, xmlDocumentNotif);

			logger.debug("Y41bPlateaServiceImpl.crearEJGVDocumentNotificacion - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_crearEJGVDocumentNotificacion) {

			if (e_crearEJGVDocumentNotificacion.getMessage().contains("R02G-10004080")) {
				logger.error("Y41bPlateaServiceImpl.crearEJGVDocumentNotificacion", e_crearEJGVDocumentNotificacion);

				int strInicio = xmlDocumentNotif.lastIndexOf("<documentName>");
				int strFinal = xmlDocumentNotif.lastIndexOf("</documentName>");

				String[] args = new String[1];
				args[0] = xmlDocumentNotif.substring(strInicio, strFinal);

				String errormsg = this.appMessageSource.getMessage("error.dokusi.extension", args,
						LocaleContextHolder.getLocale());

				throw new Y41bUDAException(errormsg, true, e_crearEJGVDocumentNotificacion);
			} else {
				// fin - control extension fichero
				// e_registrarDocumentoDokusi.printStackTrace();
				logger.error("Y41bPlateaServiceImpl.crearEJGVDocumentNotificacion", e_crearEJGVDocumentNotificacion);
				throw new Y41bUDAException("error.dokusi.invocation", true, e_crearEJGVDocumentNotificacion);
			}

			// logger.error("Y41bPlateaServiceImpl.crearEJGVDocumentNotificacion",
			// e_crearEJGVDocumentNotificacion);
			// throw e_crearEJGVDocumentNotificacion;
		} finally {
			logger.info("Y41bPlateaServiceImpl.crearEJGVDocumentNotificacion - Final");
		}
	}

	/**
	 * Obtiene el HTML de la Notificacion a partir del Identificador de la
	 * Notificacion
	 * 
	 * @param tokenSesion
	 *            Token de sesión en XLNets
	 * @param sNotifID
	 *            Id de la notificacion
	 * @param sAuditUser
	 *            Usuario
	 * @return String HTML de la Notificacion
	 * @throws Exception
	 */
	public String getNotificacionById(String tokenSesion, String sNotifID, String sAuditUser) throws Exception {
		logger.info("Y41bPlateaServiceImpl.crearEJGVDocumentNotificacion - Inicio");
		try {
			String respuesta;

			Q99SSHNFacadeWSServiceImpl q99SSHNFacadeWSServiceImpl = new Q99SSHNFacadeWSServiceImpl();
			respuesta = q99SSHNFacadeWSServiceImpl.getNotificacionPubReceiptById(tokenSesion, sNotifID);

			logger.debug("Y41bPlateaServiceImpl.getNotificacionById - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_getNotificacionById) {
			logger.error("Y41bPlateaServiceImpl.getNotificacionById", e_getNotificacionById);
			throw e_getNotificacionById;
		} finally {
			logger.info("Y41bPlateaServiceImpl.getNotificacionById - Final");
		}
	}

	/**
	 * Obtiene el HTML del acuse a partir del Identificador de la Notificacion
	 * 
	 * @param tokenSesion
	 *            Token de sesión en XLNets
	 * @param sNotifID
	 *            Id de la notificacion
	 * @param sAuditUser
	 *            Usuario
	 * @return String HTML de la Notificacion
	 * @throws Exception
	 */
	public String getAcuseById(String tokenSesion, String sNotifID, String sAuditUser) throws Exception {
		logger.info("Y41bPlateaServiceImpl.crearEJGVDocumentNotificacion - Inicio");
		try {
			String respuesta;

			Q99SSHNFacadeWSServiceImpl q99SSHNFacadeWSServiceImpl = new Q99SSHNFacadeWSServiceImpl();
			respuesta = q99SSHNFacadeWSServiceImpl.getNotificacionReceiptById(tokenSesion, sNotifID);

			logger.debug("Y41bPlateaServiceImpl.getNotificacionById - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_getNotificacionById) {
			logger.error("Y41bPlateaServiceImpl.getNotificacionById", e_getNotificacionById);
			throw e_getNotificacionById;
		} finally {
			logger.info("Y41bPlateaServiceImpl.getNotificacionById - Final");
		}
	}

	/**
	 * Añade en el pie de un EJGVDocument los datos de la firma del mismo
	 * 
	 * @param tokenSesion
	 *            String
	 * @param ejgvDocument
	 *            String
	 * @param firma
	 *            String
	 * @return String con el EJGVdocument y la firma
	 * @throws Exception
	 */
	public String anyadirFirmaEjgvDocumentSolicitud(String tokenSesion, String ejgvDocument, String firma)
			throws Exception {
		logger.info("Y41bPlateaServiceImpl.anyadirFirmaEjgvDocumentSolicitud - Inicio");
		try {
			String paramValues[] = new String[Y41bConstantes.NUM_3];
			paramValues[0] = tokenSesion;// Session token de XL-Nets
			paramValues[1] = firma;
			paramValues[2] = ejgvDocument;

			String[] paramNames = new String[Y41bConstantes.NUM_3];
			paramNames[0] = "sSessionToken";
			paramNames[1] = "abSignature";
			paramNames[2] = "sEjgvDocument";

			String respuesta;

			respuesta = Y41bUtilsWS.invoke(appConfiguration.getProperty("URL_SHF"), "addSignatureFooterFAC", paramNames,
					paramValues);

			logger.debug("Y41bPlateaServiceImpl.anyadirFirmaEjgvDocumentSolicitud - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_anyadirFirmaEjgvDocumentSolicitud) {
			logger.error("Y41bPlateaServiceImpl.anyadirFirmaEjgvDocumentSolicitud",
					e_anyadirFirmaEjgvDocumentSolicitud);
			throw e_anyadirFirmaEjgvDocumentSolicitud;
		} finally {
			logger.info("Y41bPlateaServiceImpl.anyadirFirmaEjgvDocumentSolicitud - Final");
		}
	}

	/**
	 * Realiza el envío telemático de una solicitud(EJGVDocument)
	 * 
	 * @param tokenSesion
	 *            String
	 * @param contextSubmission
	 *            String: Xml que lleva toda los datos necesarios para crear la
	 *            Solicitud
	 * @param baEjgvDocument
	 *            String
	 * @return String con el EJGVdocument y la firma
	 * @throws Exception
	 */
	public String envioSolicitudes(String tokenSesion, String contextSubmission, String baEjgvDocument)
			throws Exception {
		logger.info("Y41bPlateaServiceImpl.envioSolicitudes - Inicio");
		try {
			// Llamada a la función para enviar la solicitud
			String paramValues[] = new String[Y41bConstantes.NUM_3];
			paramValues[0] = tokenSesion;// Session token de XL-Nets
			paramValues[1] = contextSubmission;
			paramValues[2] = baEjgvDocument;// Sin codificar porque ya llega
											// codificado
			// HTML(ejgvDocument)

			String paramNames[] = new String[Y41bConstantes.NUM_3];
			paramNames[0] = "sSessionToken";// Session token de XL-Nets
			paramNames[1] = "sContextSubmission";// Documento HTML(ejgvDocument)
			paramNames[2] = "baEjgvDocument";// byte[] b64 del ejgvDocument con
			// la firma al pie

			String respuesta;

			respuesta = Y41bUtilsWS.invoke(appConfiguration.getProperty("URL_PFS"), "sendEjgvDocumentST", paramNames,
					paramValues);

			logger.debug("Y41bPlateaServiceImpl.envioSolicitudes - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_envioSolicitudes) {
			logger.error("Y41bPlateaServiceImpl.envioSolicitudes", e_envioSolicitudes);
			throw e_envioSolicitudes;
		} finally {
			logger.info("Y41bPlateaServiceImpl.envioSolicitudes - Final");
		}
	}

	/**
	 * Realiza el envío de una notificacion
	 * 
	 * @param tokenSesion
	 *            String
	 * @param sUnsignedNotification
	 *            String
	 * @param bytesNotification
	 *            String
	 * @return String con el Numero de Registro
	 * @throws Exception
	 */
	public String envioNotificaciones(String tokenSesion, String sUnsignedNotification, String bytesNotification)
			throws Exception {
		logger.info("Y41bPlateaServiceImpl.envioNotificaciones - Inicio");
		try {
			String respuesta = "";

			Q99MServiceIfzFacadeWSServiceImpl q99MServiceIfzFacadeWSServiceImpl = new Q99MServiceIfzFacadeWSServiceImpl();
			respuesta = q99MServiceIfzFacadeWSServiceImpl.envioNotificaciones(tokenSesion, sUnsignedNotification,
					bytesNotification);

			logger.debug("Y41bPlateaServiceImpl.envioNotificaciones - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_envioNotificaciones) {
			logger.error("Y41bPlateaServiceImpl.envioNotificaciones", e_envioNotificaciones);
			throw e_envioNotificaciones;
		} finally {
			logger.info("Y41bPlateaServiceImpl.envioNotificaciones - Final");
		}
	}

	/**
	 * Método que sirve para procesar las esperas
	 * 
	 * @param tokenSesion
	 *            token xlnets
	 * @param xmlEspera
	 *            String
	 * @param tipoEspera
	 *            String
	 * @return String
	 * @throws Exception
	 *             ex
	 */
	public String crearMatarEspera(String tokenSesion, String xmlEspera, String tipoEspera) throws Exception {
		logger.info("Y41bPlateaServiceImpl.procesarEsperas - Inicio");
		try {
			String respuesta = "";

			Q99USendMessageSCPFacadeWSServiceImpl q99USendMessageSCPFacadeWSServiceImpl = new Q99USendMessageSCPFacadeWSServiceImpl();
			q99USendMessageSCPFacadeWSServiceImpl.crearMatarEspera(tokenSesion, xmlEspera, tipoEspera);

			logger.debug("Y41bPlateaServiceImpl.procesarEsperas - Return: " + respuesta);
			return respuesta;
		} catch (Exception e_procesarEsperas) {
			logger.error("Y41bPlateaServiceImpl.procesarEsperas", e_procesarEsperas);
			throw e_procesarEsperas;
		} finally {
			logger.info("Y41bPlateaServiceImpl.procesarEsperas - Final");
		}
	}

	/**
	 * Método que sirve para registrar en mis gestiones datos del histórico
	 * 
	 * @param tokenSesion
	 *            token xlnets
	 * @param xmlHistorico
	 *            String
	 * @param tipoHistorico
	 *            String
	 * @return String
	 * @throws Exception
	 *             ex
	 */
	public String registrarHistorico(String tokenSesion, String xmlHistorico, String tipoHistorico) throws Exception {
		logger.info("Y41bPlateaServiceImpl.registrarHistorico - Inicio");
		try {
			String respuesta = "";

			Q99USendMessageSCPFacadeWSServiceImpl q99USendMessageSCPFacadeWSServiceImpl = new Q99USendMessageSCPFacadeWSServiceImpl();
			q99USendMessageSCPFacadeWSServiceImpl.registrarHistorico(tokenSesion, xmlHistorico, tipoHistorico);

			logger.debug("Y41bPlateaServiceImpl.registrarHistorico - Return: " + respuesta);
			return respuesta;
		} catch (Exception e_registrarHistorico) {
			logger.error("Y41bPlateaServiceImpl.registrarHistorico", e_registrarHistorico);
			throw e_registrarHistorico;
		} finally {
			logger.info("Y41bPlateaServiceImpl.registrarHistorico - Final");
		}
	}

	/**
	 * Método que sirve para eliminar en mis gestiones datos del histórico
	 * 
	 * @param tokenSesion
	 *            token xlnets
	 * @param xmlHistorico
	 *            String
	 * @param tipoHistorico
	 *            String
	 * @return String
	 * @throws Exception
	 *             ex
	 */
	public String eliminarHistorico(String tokenSesion, String xmlHistorico, String tipoHistorico) throws Exception {
		logger.info("Y41bPlateaServiceImpl.eliminarHistorico - Inicio");
		try {
			String respuesta = "";

			Q99USendMessageSCPFacadeWSServiceImpl q99USendMessageSCPFacadeWSServiceImpl = new Q99USendMessageSCPFacadeWSServiceImpl();
			q99USendMessageSCPFacadeWSServiceImpl.eliminarHistorico(tokenSesion, xmlHistorico, tipoHistorico);

			logger.debug("Y41bPlateaServiceImpl.eliminarHistorico - Return: " + respuesta);
			return respuesta;
		} catch (Exception e_registrarHistorico) {
			logger.error("Y41bPlateaServiceImpl.eliminarHistorico", e_registrarHistorico);
			throw e_registrarHistorico;
		} finally {
			logger.info("Y41bPlateaServiceImpl.eliminarHistorico - Final");
		}
	}

	/**
	 * Metodo que invoca al webService saveOrUpdateExpedientNT de la
	 * administracin para actualizar mis gestiones
	 * 
	 * @param tokenSesion
	 *            String: Token de sesion
	 * @param sXMLExpedient
	 *            String: Xml que lleva toda los datos necesarios para atualizar
	 *            mis gestiones
	 * @return String: Xml que representa la respuesta
	 * @throws Exception
	 * 
	 */
	public String actualizarParticipantExpediente(String tokenSesion, String sXMLExpedient) throws Exception {
		logger.info("Y41bPlateaServiceImpl.actualizarParticipantExpediente - Inicio");
		try {
			String respuesta = "";

			Q99USendMessageSCPFacadeWSServiceImpl q99USendMessageSCPFacadeWSServiceImpl = new Q99USendMessageSCPFacadeWSServiceImpl();
			q99USendMessageSCPFacadeWSServiceImpl.actualizarParticipacion(tokenSesion, sXMLExpedient);

			logger.info("Y41bPlateaServiceImpl.actualizarParticipantExpediente - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_actualizarMisGestiones) {
			logger.error("Y41bPlateaServiceImpl.actualizarMisGestiones", e_actualizarMisGestiones);
			throw e_actualizarMisGestiones;
		} finally {
			logger.info("Y41bPlateaServiceImpl.actualizarMisGestiones - Final");
		}
	}

	/**
	 * Metodo que invoca al webService saveOrUpdateExpedientNT de la
	 * administracin para actualizar mis gestiones
	 * 
	 * @param tokenSesion
	 *            String: Token de sesion
	 * @param sXMLExpedient
	 *            String: Xml que lleva toda los datos necesarios para atualizar
	 *            mis gestiones
	 * @return String: Xml que representa la respuesta
	 * @throws Exception
	 * 
	 */
	public String cancelarCierreExpediente(String tokenSesion, String sXMLExpedient) throws Exception {
		logger.info("Y41bPlateaServiceImpl.cancelarCierreExpediente - Inicio");
		try {
			String paramValues[] = new String[Y41bConstantes.NUM_3];
			paramValues[0] = tokenSesion;// Session token de XL-Nets
			paramValues[1] = sXMLExpedient;// Número de registro
			paramValues[2] = "cancelCloseExpedientNT";

			String paramNames[] = new String[Y41bConstantes.NUM_3];
			paramNames[0] = "sSessionToken";// Session token de XL-Nets
			paramNames[1] = "sXMLExpedient";
			paramNames[2] = "sServiceType";

			String respuesta;

			respuesta = Y41bUtilsWS.invoke(appConfiguration.getProperty("URL_NOTIF"), "sendMessageSC", paramNames,
					paramValues);

			logger.info("Y41bPlateaServiceImpl.cancelarCierreExpediente - Return: " + respuesta);

			return respuesta;
		} catch (Exception e_actualizarMisGestiones) {
			logger.error("Y41bPlateaServiceImpl.cancelarCierreExpediente", e_actualizarMisGestiones);
			throw e_actualizarMisGestiones;
		} finally {
			logger.info("Y41bPlateaServiceImpl.cancelarCierreExpediente - Final");
		}
	}
}