package com.ejie.y41b.adapter;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Date;
import java.util.List;
import java.util.Properties;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.ejie.cp.xml.ProcedureDocument;
import com.ejie.x43s.v2.mgs.xml.TProceedingList;
import com.ejie.x43s.v2.ttn.xml.AExtensionPoint;
import com.ejie.y31.vo.Y31AttachmentBean;
import com.ejie.y41b.adapter.helper.Y41bPlateaHelper;
import com.ejie.y41b.adapter.services.NSHFServiceImpl;
import com.ejie.y41b.adapter.services.X43CuentasInteresadosServiceImpl;
import com.ejie.y41b.adapter.services.X43NotificacionesServiceImpl;
import com.ejie.y41b.adapter.services.X43kServiceImpl;
import com.ejie.y41b.adapter.services.X43sServiceImpl;
import com.ejie.y41b.adapter.services.Y41bPlateaService;
import com.ejie.y41b.adapter.transformer.Y41bPlateaTransformer;
import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.model.dokusi.Documento;
import com.ejie.y41b.model.dokusi.Formato;
import com.ejie.y41b.model.platea.ActualizarExpediente;
import com.ejie.y41b.model.platea.ActualizarMisGestiones;
import com.ejie.y41b.model.platea.CierreExpediente;
import com.ejie.y41b.model.platea.CrearEJGV;
import com.ejie.y41b.model.platea.CrearNotifEJGV;
import com.ejie.y41b.model.platea.DatosCertificado;
import com.ejie.y41b.model.platea.DatosPersona;
import com.ejie.y41b.model.platea.EnvioEJGV;
import com.ejie.y41b.model.platea.EnvioNotifEJGV;
import com.ejie.y41b.model.platea.Espera;
import com.ejie.y41b.model.platea.ExpedienteOficio;
import com.ejie.y41b.model.platea.Historico;
import com.ejie.y41b.model.platea.Procedure;
import com.ejie.y41b.model.platea.RegistroEntrada;
import com.ejie.y41b.model.platea.RegistroSalida;
import com.ejie.y41b.model.platea.RespuestaCrearEJGVDocumentNotificacion;
import com.ejie.y41b.model.platea.RespuestaCrearEJGVDocumentSolicitud;
import com.ejie.y41b.model.platea.RespuestaNotificacion;
import com.ejie.y41b.model.platea.RespuestaProcesarEspera;
import com.ejie.y41b.model.platea.RespuestaRegistrarHistorico;
import com.ejie.y41b.model.platea.RespuestaSolicitudTelematica;
import com.ejie.y41b.model.platea.SolicitudOficio;
import com.ejie.y41b.model.platea.SolicitudPresencial;
import com.ejie.y41b.model.platea.SolicitudTelematica;
import com.ejie.y41b.nshf.SignCertificateType;
import com.ejie.y41b.nshf.SignCertificatesType;
import com.ejie.y41b.nshf.VerificationResultType;
import com.ejie.y41b.utils.exception.Y41bUDAException;
import com.ejie.y41b.utils.ws.Y41bUtilsWS;
import com.ejie.y41b.utils.xlnets.Y41bUtils;
import com.ejie.y41b.ws.cliente.x43CuentasInteresados.TCanalNotificacion;
import com.ejie.y41b.ws.cliente.x43CuentasInteresados.TContacto;
import com.ejie.y41b.ws.cliente.x43CuentasInteresados.TCuenta;
import com.ejie.y41b.ws.cliente.x43CuentasInteresados.TInteresado;
import com.ejie.y41b.ws.cliente.x43CuentasInteresados.TInterestedSearch;
import com.ejie.y41b.ws.cliente.x43notificaciones.TDocField;
import com.ejie.y41b.ws.cliente.x43notificaciones.TDocFieldList;
import com.ejie.y41b.ws.cliente.x43notificaciones.TDocumentFilter;
import com.ejie.y41b.ws.cliente.x43notificaciones.TDocumentList;
import com.ejie.y41b.ws.cliente.x43notificaciones.TNotification;
import com.ejie.y41b.ws.cliente.x43notificaciones.TNotificationCreateNew;
import com.ejie.y41b.ws.cliente.x43notificaciones.TNotificationFilter;
import com.ejie.y41b.ws.cliente.x43notificaciones.TNotificationList;
import com.ejie.y41b.ws.cliente.x43notificaciones.TStringList;

/**
 * <p>
 * Titulo: Y41bPlateaAdapterServiceImpl
 * </p>
 * <p>
 * Descipcion: Adapta nuestro sistema con Platea
 * </p>
 * <p>
 * Copyright: Copyright (c) 2013
 * </p>
 * <p>
 * Empresa: Bilbomatica
 * </p>
 * 
 *  
 * @version 1.0
 */
@Service(value = "Y41bPlateaAdapterServiceImpl")
public class Y41bPlateaAdapterServiceImpl implements Y41bPlateaAdapterService {

	/**
	 * Final static logger.
	 */
	private static final Logger logger = LoggerFactory.getLogger(Y41bPlateaAdapterServiceImpl.class);

	@Autowired
	private Y41bPlateaService plateaService;

	@Autowired
	private Y41bDokusiAdapterService dokusiAdapterService;

	@Autowired
	private Properties appConfiguration;

	@Autowired
	private X43sServiceImpl x43sServiceImpl;

	@Autowired
	private X43kServiceImpl x43kServiceImpl;

	@Autowired
	private X43NotificacionesServiceImpl x43NotificacionesServiceImpl;

	@Autowired
	private X43CuentasInteresadosServiceImpl x43CuentasInteresadosServiceImpl;

	@Autowired()
	private Y41bPifAdapterService y41bPifAdapterService;

	/**
	 * Clase que adaptara los servicios de Platea a nuestro sistema
	 */
	private Y41bPlateaAdapterServiceImpl() {
		super();
	}

	/**
	 * Método que abre un expediente presencial
	 * 
	 * @param solicitudPresencial
	 *            Objeto con los datos de la solicitud presencial
	 * @return SolicitudPresencial
	 * 
	 */
	public SolicitudPresencial crearSolicitudPresencial(SolicitudPresencial solicitudPresencial) {
		logger.info("Y41bPlateaAdapterServiceImpl.crearSolicitudPresencial - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			Procedure procedure = solicitudPresencial.getProcedure();

			if (null != solicitudPresencial.getDocumentosAdjuntos()
					&& solicitudPresencial.getDocumentosAdjuntos().size() > 0) {
				for (Documento documento : solicitudPresencial.getDocumentosAdjuntos()) {
					if (StringUtils.isBlank(documento.getOidDokusi())) {
						if (null != documento.getBytes() && documento.getBytes().length() > 0) {

							documento.setProcedure(procedure);
							documento.setNumeroExpediente(solicitudPresencial.getNumeroExpediente());

							documento = dokusiAdapterService.registrarDocumentoDokusi(documento);

						}
					}
				}
			}

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					solicitudPresencial.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLInicioExpedientePresencial(solicitudPresencial,
					procedureDocument.getProcedure());

			plateaService.abrirExpediente(tokenSesion, sXMLExpedient);

			return solicitudPresencial;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_crearSolicitudPresencial) {
			logger.error("Y41bPlateaAdapterServiceImpl.crearSolicitudPresencial", e_crearSolicitudPresencial);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearSolicitudPresencial);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.crearSolicitudPresencial - Final");
		}
	}

	/**
	 * Método que abre un expediente presencial.
	 * 
	 * @param solicitudOficio
	 *            the solicitud oficio
	 */
	public void crearSolicitudOficio(SolicitudOficio solicitudOficio) {
		logger.info("Y41bPlateaAdapterServiceImpl.crearSolicitudOficio - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			Procedure procedure = solicitudOficio.getProcedure();
			logger.info("Y41bPlateaAdapterServiceImpl.crearSolicitudOficio - Paso 1/4: Registrar Documento en Dokusi");
			if (null != solicitudOficio.getDocumentosAdjuntos() && solicitudOficio.getDocumentosAdjuntos().size() > 0) {
				for (Documento documento : solicitudOficio.getDocumentosAdjuntos()) {
					if (StringUtils.isBlank(documento.getOidDokusi())) {
						if (null != documento.getBytes() && documento.getBytes().length() > 0) {

							documento.setProcedure(procedure);
							documento.setNumeroExpediente(solicitudOficio.getNumeroExpediente());
							/*
							 * PARA LA APERTURA NO LOCALIZO DOCUMENTOS PORQUE
							 * LAS SANCIONES NO LLEVAN DOCUMENTOS ADJUNTOS, ESTA
							 * FUNCION ESTA ASI POR SI EN UN FUTURO SE USA PARA
							 * OTROS PROCEDIMIENTOS EN ESE CASO HABRA QUE PROBAR
							 * QUE LA SIGUIENTE SENTENCIA FUNCIONA CORRECTAMENTE
							 */
							documento = dokusiAdapterService.registrarDocumentoDokusi(documento);

						}
					}
				}
			}
			logger.info("Y41bPlateaAdapterServiceImpl.ExpedienController ObtenerDatosCatalogoProcedimientos");
			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					solicitudOficio.getProcedure().getIdProcedimiento());

			logger.info("Y41bPlateaAdapterServiceImpl.crearSolicitudOficio - Paso 3/4: crearXMLInicioExpedienteOficio"
					+ respuesta.toString() + ".");

			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			logger.info(
					"Y41bPlateaAdapterServiceImpl.crearSolicitudOficio - Paso 3a/4: procedureDocument.getProcedure():"
							+ procedureDocument.getProcedure() + ".");

			String sXMLExpedient = Y41bPlateaHelper.crearXMLInicioExpedienteOficio(solicitudOficio,
					procedureDocument.getProcedure());
			logger.info("Y41bPlateaAdapterServiceImpl.crearSolicitudOficio - Paso 4/4: abrirExpediente");

			String respuesta2 = plateaService.abrirExpediente(tokenSesion, sXMLExpedient);
			logger.info("Y41bPlateaAdapterServiceImpl.crearSolicitudOficio - Fin" + respuesta2);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_crearSolicitudPresencial) {
			logger.error("Y41bPlateaAdapterServiceImpl.crearSolicitudOficio", e_crearSolicitudPresencial);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearSolicitudPresencial);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.crearSolicitudOficio - Final");
		}
	}

	/**
	 * Método que abre el expediente de oficio
	 * 
	 * @param expedienteOficio
	 *            ExpedienteOficio
	 * @return ExpedienteOficio
	 */
	public ExpedienteOficio abrirExpedienteOficio(ExpedienteOficio expedienteOficio) {
		logger.info("Y41bPlateaAdapterServiceImpl.abrirExpedienteOficio - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			Procedure procedure = expedienteOficio.getProcedure();

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					procedure.getIdProcedimiento());

			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLInicioExpedienteOficio(expedienteOficio,
					procedureDocument.getProcedure());

			String resultado = plateaService.abrirExpedienteOficio(tokenSesion, sXMLExpedient);

			return Y41bPlateaTransformer.transformAbrirExpedienteOficio(resultado, expedienteOficio);
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_abrirExpedienteOficio) {
			logger.error("Y41bPlateaAdapterServiceImpl.abrirExpedienteOficio", e_abrirExpedienteOficio);
			throw new Y41bUDAException("error.platea.invocation", true, e_abrirExpedienteOficio);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.abrirExpedienteOficio - Final");
		}
	}

	/**
	 * Método que abre un expediente telematica
	 * 
	 * @param solicitudTelematica
	 *            Objeto con los datos de la solicitud telematica
	 * 
	 */
	public void crearSolicitudTelematica(SolicitudTelematica solicitudTelematica) {
		logger.info("Y41bPlateaAdapterServiceImpl.crearSolicitudTelematica - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					solicitudTelematica.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLInicioExpedienteTelematico(solicitudTelematica,
					procedureDocument.getProcedure());

			plateaService.abrirExpediente(tokenSesion, sXMLExpedient);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_crearSolicitudTelematica) {
			logger.error("Y41bPlateaAdapterServiceImpl.crearSolicitudTelematica", e_crearSolicitudTelematica);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearSolicitudTelematica);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.crearSolicitudTelematica - Final");
		}
	}

	/**
	 * Metodo que actualiza Mis Gestiones
	 * 
	 * @param actualizarMisGestiones
	 *            Objeto con los datos de mis gestiones
	 * 
	 */
	public void actualizarMisGestiones(ActualizarMisGestiones actualizarMisGestiones) {
		logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones - Inicio");
		try {

			actualizarMisGestiones = this.eliminarBlancosDNI(actualizarMisGestiones);

			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					actualizarMisGestiones.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLActualizacionMisGestiones(actualizarMisGestiones,
					procedureDocument.getProcedure());

			plateaService.actualizarMisGestiones(tokenSesion, sXMLExpedient);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_actualizarMisGestiones) {
			logger.error("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones", e_actualizarMisGestiones);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestiones);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones - Final");
		}
	}

	/**
	 * Metodo que actualiza Mis Gestiones para el acta
	 * 
	 * @param actualizarMisGestiones
	 *            Objeto con los datos de mis gestiones
	 * 
	 */
	public void actualizarMisGestionesActa(ActualizarMisGestiones actualizarMisGestiones) {
		logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones - Inicio");
		try {

			actualizarMisGestiones = this.eliminarBlancosDNI(actualizarMisGestiones);

			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					actualizarMisGestiones.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLActualizacionMisGestionesActa(actualizarMisGestiones,
					procedureDocument.getProcedure());

			plateaService.actualizarMisGestiones(tokenSesion, sXMLExpedient);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_actualizarMisGestiones) {
			logger.error("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones", e_actualizarMisGestiones);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestiones);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones - Final");
		}
	}

	/**
	 * Metodo que actualiza Mis Gestiones
	 * 
	 * @param actualizarMisGestiones
	 *            Objeto con los datos de mis gestiones
	 * 
	 */
	public void actualizarMisGestionesX43(ActualizarMisGestiones actualizarMisGestiones) {
		logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones - Inicio");
		try {

			actualizarMisGestiones = this.eliminarBlancosDNI(actualizarMisGestiones);

			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					actualizarMisGestiones.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			TProceedingList proceedingList = Y41bPlateaHelper
					.crearXMLActualizacionMisGestionesX43(actualizarMisGestiones, procedureDocument.getProcedure());

			x43sServiceImpl.reportProceedingsUpdate(actualizarMisGestiones.getNumeroExpediente(),
					actualizarMisGestiones.getEjercicio(),
					actualizarMisGestiones.getProcedure().getIdProcedimiento(), com.ejie.y41b.utils.Y41bUtils
							.formatearDateToString(new Date(), Y41bConstantes.FORMATO_FECHAHORA_TOOLKIT),
					proceedingList, new AExtensionPoint());

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Throwable e_actualizarMisGestiones) {
			logger.error("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones", e_actualizarMisGestiones);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestiones);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones - Final");
		}
	}

	/**
	 * Metodo que recupera el EJGVDocument
	 * 
	 * @param idSolicitud
	 *            String
	 * @param folderId
	 *            String
	 * @return String
	 */
	public String getRequestInformation(String idSolicitud, String folderId) {
		logger.info("Y41bPlateaAdapterServiceImpl.getRequestInformation - Inicio");
		try {
			return x43kServiceImpl.getRequestInformation(idSolicitud, folderId);
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Throwable e_actualizarMisGestiones) {
			logger.error("Y41bPlateaAdapterServiceImpl.getRequestInformation", e_actualizarMisGestiones);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestiones);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.getRequestInformation - Final");
		}
	}

	/**
	 * Metodo que devuelve la lista de todas las notificaciones existentes segn
	 * los parmetros de entrada. Al menos tiene que tener un parmetro de
	 * filtro.
	 * 
	 * @param filter
	 *            TDocumentFilter
	 * @param ep
	 *            AExtensionPoint
	 * @return TDocumentList
	 */
	public TDocumentList searchDocument(TDocumentFilter filter,
			com.ejie.y41b.ws.cliente.x43notificaciones.AExtensionPoint ep) {
		logger.info("Y41bPlateaAdapterServiceImpl.searchDocument - Inicio");
		try {
			return x43NotificacionesServiceImpl.searchDocument(filter, ep);
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Throwable e_actualizarMisGestiones) {
			logger.error("Y41bPlateaAdapterServiceImpl.searchDocument", e_actualizarMisGestiones);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestiones);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.searchDocument - Final");
		}
	}

	/**
	 * Metodo que devuelve la lista de todas las notificaciones existentes segn
	 * los parmetros de entrada. Al menos tiene que tener un parmetro de
	 * filtro.
	 * 
	 * @param filter
	 *            TNotificationFilter
	 * @param ep
	 *            AExtensionPoint
	 * @return TNotificationList
	 */
	public TNotificationList searchNotification(TNotificationFilter filter,
			com.ejie.y41b.ws.cliente.x43notificaciones.AExtensionPoint ep) {
		logger.info("Y41bPlateaAdapterServiceImpl.searchNotification - Inicio");
		try {
			return x43NotificacionesServiceImpl.searchNotification(filter, ep);
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Throwable e_actualizarMisGestiones) {
			logger.error("Y41bPlateaAdapterServiceImpl.searchNotification", e_actualizarMisGestiones);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestiones);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.searchNotification - Final");
		}
	}

	/**
	 * Metodo que actualiza Mis Gestiones
	 * 
	 * @param actualizarMisGestiones
	 *            Objeto con los datos de mis gestiones
	 * 
	 */
	public void actualizarMisGestionesComunicacion(ActualizarMisGestiones actualizarMisGestiones) {
		logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestionesComunicacion - Inicio");
		try {

			actualizarMisGestiones = this.eliminarBlancosDNI(actualizarMisGestiones);

			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					actualizarMisGestiones.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLActualizacionMisGestionesComunicacion(
					actualizarMisGestiones, procedureDocument.getProcedure());

			plateaService.actualizarMisGestiones(tokenSesion, sXMLExpedient);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_actualizarMisGestionesComunicacion) {
			logger.error("Y41bPlateaAdapterServiceImpl.actualizarMisGestionesComunicacion",
					e_actualizarMisGestionesComunicacion);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestionesComunicacion);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestionesComunicacion - Final");
		}
	}

	/**
	 * Metodo que actualiza Mis Gestiones
	 * 
	 * @param actualizarMisGestiones
	 *            Objeto con los datos de mis gestiones
	 * 
	 */
	public void actualizarMisGestionesNotificacion(ActualizarMisGestiones actualizarMisGestiones) {
		logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestionesNotificacion - Inicio");
		try {

			actualizarMisGestiones = this.eliminarBlancosDNI(actualizarMisGestiones);

			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					actualizarMisGestiones.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLActualizacionMisGestionesNotificacion(
					actualizarMisGestiones, procedureDocument.getProcedure());

			plateaService.actualizarMisGestiones(tokenSesion, sXMLExpedient);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_actualizarMisGestionesNotificacion) {
			logger.error("Y41bPlateaAdapterServiceImpl.actualizarMisGestionesNotificacion",
					e_actualizarMisGestionesNotificacion);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestionesNotificacion);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestionesNotificacion - Final");
		}
	}

	/**
	 * Metodo que actualiza Mis Gestiones
	 * 
	 * @param actualizarMisGestiones
	 *            Objeto con los datos de mis gestiones
	 * 
	 */
	public void actualizarMisGestionesDecision(ActualizarMisGestiones actualizarMisGestiones) {
		logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestionesDecision - Inicio");
		try {

			actualizarMisGestiones = this.eliminarBlancosDNI(actualizarMisGestiones);

			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					actualizarMisGestiones.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLActualizacionMisGestionesDecision(actualizarMisGestiones,
					procedureDocument.getProcedure());

			plateaService.actualizarMisGestiones(tokenSesion, sXMLExpedient);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_actualizarMisGestionesNotificacion) {
			logger.error("Y41bPlateaAdapterServiceImpl.actualizarMisGestionesDecision",
					e_actualizarMisGestionesNotificacion);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestionesNotificacion);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestionesDecision - Final");
		}
	}

	/**
	 * Metodo que elimina de Mis Gestiones
	 * 
	 * @param actualizarMisGestiones
	 *            Objeto con los datos de mis gestiones
	 * 
	 */
	public void eliminarMisGestiones(ActualizarMisGestiones actualizarMisGestiones) {
		logger.info("Y41bPlateaAdapterServiceImpl.eliminarMisGestiones - Inicio");
		try {

			actualizarMisGestiones = this.eliminarBlancosDNI(actualizarMisGestiones);

			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					actualizarMisGestiones.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLEliminarMisGestiones(actualizarMisGestiones,
					procedureDocument.getProcedure());

			plateaService.eliminarMisGestiones(tokenSesion, sXMLExpedient);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_eliminarMisGestiones) {
			logger.error("Y41bPlateaAdapterServiceImpl.eliminarMisGestiones", e_eliminarMisGestiones);
			throw new Y41bUDAException("error.platea.invocation", true, e_eliminarMisGestiones);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.eliminarMisGestiones - Final");
		}
	}

	/**
	 * Metodo que elimina de Mis Gestiones
	 * 
	 * @param actualizarMisGestiones
	 *            Objeto con los datos de mis gestiones
	 * 
	 */
	public void eliminarMisGestionesX43(ActualizarMisGestiones actualizarMisGestiones) {
		logger.info("Y41bPlateaAdapterServiceImpl.eliminarMisGestiones - Inicio");
		try {

			actualizarMisGestiones = this.eliminarBlancosDNI(actualizarMisGestiones);

			x43sServiceImpl.reportProceedingsDelete(actualizarMisGestiones.getNumeroExpediente(),
					actualizarMisGestiones.getEjercicio(), actualizarMisGestiones.getProcedure().getIdProcedimiento(),
					actualizarMisGestiones.getProceedingsID(), actualizarMisGestiones.getSolicitante().getDni().trim(),
					new AExtensionPoint());

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Throwable e_eliminarMisGestiones) {
			logger.error("Y41bPlateaAdapterServiceImpl.eliminarMisGestiones", e_eliminarMisGestiones);
			throw new Y41bUDAException("error.platea.invocation", true, e_eliminarMisGestiones);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.eliminarMisGestiones - Final");
		}
	}

	/**
	 * Metodo que actualiza Mis Gestiones
	 * 
	 * @param actualizarExpediente
	 *            Objeto con los datos de mis gestiones
	 * 
	 */
	public void actualizarExpediente(ActualizarExpediente actualizarExpediente) {
		logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					actualizarExpediente.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLActualizacionExpediente(actualizarExpediente,
					procedureDocument.getProcedure());

			plateaService.actualizarExpediente(tokenSesion, sXMLExpedient);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_actualizarMisGestiones) {
			logger.error("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones", e_actualizarMisGestiones);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestiones);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones - Final");
		}
	}

	/**
	 * Cerrará los expedientes telemáticos que se encuentren abiertos
	 * 
	 * @param cierreExpediente
	 *            CierreExpediente
	 */
	public void cerrarExpediente(CierreExpediente cierreExpediente) {
		logger.info("Y41bPlateaAdapterServiceImpl.cerrarExpediente - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			logger.info("Y41bPlateaAdapterServiceImpl.cerrarExpediente - Paso 1/3: obtenerDatosCatalogoProcedimientos");
			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					cierreExpediente.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());
			logger.info("Y41bPlateaAdapterServiceImpl.cerrarExpediente - Paso 2/3: crearXMLCerrarExpediente");
			String xmlCierre = Y41bPlateaHelper.crearXMLCerrarExpediente(cierreExpediente,
					procedureDocument.getProcedure());
			logger.info("Y41bPlateaAdapterServiceImpl.cerrarExpediente - Paso 3/3: cerrarExpediente");
			plateaService.cerrarExpediente(tokenSesion, xmlCierre);
			logger.info("Y41bPlateaAdapterServiceImpl.cerrarExpediente - Fin");
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_crearSolicitudPresencial) {
			logger.error("Y41bPlateaAdapterServiceImpl.cerrarExpediente", e_crearSolicitudPresencial);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearSolicitudPresencial);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.cerrarExpediente - Final");
		}
	}

	/**
	 * Método que recupera los datos del catálogo de Procedimientos
	 * 
	 * @param procedure
	 *            Procedure: Procedimiento de platea
	 * 
	 * @return ProcedureDocument: objeto con los datos del procedimiento
	 * 
	 */
	public ProcedureDocument obtenerDatosCatalogoProcedimientos(Procedure procedure) {
		logger.info("Y41bPlateaAdapterServiceImpl.obtenerDatosCatalogoProcedimientos - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String idProcedimiento = procedure.getIdProcedimiento();

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion, idProcedimiento);
			logger.debug("Y41bPlateaAdapterServiceImpl.obtenerDatosCatalogoProcedimientos - Return: " + respuesta);
			return Y41bPlateaTransformer.transformObtenerDatosCatalogoProcedimientos(respuesta);
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_obtenerDatosCatalogoProcedimientos) {
			logger.error("Y41bPlateaAdapterServiceImpl.obtenerDatosCatalogoProcedimientos",
					e_obtenerDatosCatalogoProcedimientos);
			throw new Y41bUDAException("error.platea.invocation", true, e_obtenerDatosCatalogoProcedimientos);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.obtenerDatosCatalogoProcedimientos - Final");
		}
	}

	/**
	 * Método que crea un registro de entrada
	 * 
	 * @param registroEntrada
	 *            RegistroEntrada
	 * 
	 * @return RegistroEntrada
	 * 
	 */
	public RegistroEntrada obtenerRegistroEntrada(RegistroEntrada registroEntrada) {

		try {
			logger.info("Y41bPlateaAdapterServiceImpl.obtenerRegistroEntrada - Inicio");
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			logger.info(
					"Y41bPlateaAdapterServiceImpl.obtenerRegistroEntrada - Paso 1/4: ObtenerDatosCatalogoProcedimientos");
			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					registroEntrada.getProcedure().getIdProcedimiento());

			logger.info("Y41bPlateaAdapterServiceImpl.obtenerRegistroEntrada - Paso 2/4: crearXMLContextRegistry");
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());
			String sXmlContextRegistry = Y41bPlateaHelper.crearXMLContextRegistry(registroEntrada,
					procedureDocument.getProcedure());
			logger.info("Y41bPlateaAdapterServiceImpl.obtenerRegistroEntrada - Paso 3/4: obtenerRegistroEntrada");
			respuesta = plateaService.obtenerRegistroEntrada(tokenSesion, sXmlContextRegistry);
			logger.info(
					"Y41bPlateaAdapterServiceImpl.obtenerRegistroEntrada - Paso 4/4: transformObtenerRegistroEntrada: "
							+ respuesta);
			RegistroEntrada registro = Y41bPlateaTransformer.transformObtenerRegistroEntrada(respuesta);
			logger.info("Y41bPlateaAdapterServiceImpl.obtenerRegistroEntrada - Fin");
			return registro;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_obtenerRegistroEntrada) {
			logger.error("Y41bPlateaAdapterServiceImpl.obtenerRegistroEntrada", e_obtenerRegistroEntrada);
			throw new Y41bUDAException("error.platea.invocation", true, e_obtenerRegistroEntrada);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.obtenerRegistroEntrada - Final");
		}
	}

	/**
	 * Método que crea un registro de salida
	 * 
	 * @param registroSalida
	 *            RegistroSalida
	 * 
	 * @return RegistroSalida
	 * 
	 */
	public RegistroSalida obtenerRegistroSalida(RegistroSalida registroSalida) {
		logger.info("Y41bPlateaAdapterServiceImpl.obtenerRegistroSalida - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			logger.info(
					"Y41bPlateaAdapterServiceImpl.obtenerRegistroSalida - Paso 1/4: obtenerDatosCatalogoProcedimientos ");
			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					registroSalida.getProcedure().getIdProcedimiento());
			logger.info(
					"Y41bPlateaAdapterServiceImpl.obtenerRegistroSalida - Paso 2/4: crearXMLExitRegistryCreationInfo ");
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXmlContextRegistry = Y41bPlateaHelper.crearXMLExitRegistryCreationInfo(registroSalida,
					procedureDocument.getProcedure());
			logger.info("Y41bPlateaAdapterServiceImpl.obtenerRegistroSalida - Paso 3/4: obtenerRegistroSalida ");
			respuesta = plateaService.obtenerRegistroSalida(tokenSesion, sXmlContextRegistry);
			logger.info(
					"Y41bPlateaAdapterServiceImpl.obtenerRegistroSalida - Paso 4/4: transformObtenerRegistroSalida: "
							+ respuesta);
			RegistroSalida registro = Y41bPlateaTransformer.transformObtenerRegistroSalida(respuesta);
			logger.info("Y41bPlateaAdapterServiceImpl.obtenerRegistroSalida - Fin");
			return registro;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_obtenerRegistroSalida) {
			logger.error("Y41bPlateaAdapterServiceImpl.obtenerRegistroSalida", e_obtenerRegistroSalida);
			throw new Y41bUDAException("error.platea.invocation", true, e_obtenerRegistroSalida);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.obtenerRegistroSalida - Final");
		}
	}

	/**
	 * Método que obtiene los datos contenidos en el certificado de la firma
	 * 
	 * @param firmaSimple
	 *            la firma simple
	 * 
	 * @return DatosCertificado
	 * 
	 */
	public DatosCertificado obtenerDatosCertificadoFirma(String firmaSimple) {
		logger.info("Y41bPlateaAdapterServiceImpl.obtenerDatosCertificadoFirma - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCertificadoFirma(tokenSesion, firmaSimple);

			logger.debug("Y41bPlateaAdapterServiceImpl.obtenerDatosCertificadoFirma - Return: " + respuesta);
			return Y41bPlateaTransformer.transformObtenerDatosCertificadoFirma(respuesta);
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_obtenerDatosCertificadoFirma) {
			logger.error("Y41bPlateaAdapterServiceImpl.obtenerDatosCertificadoFirma", e_obtenerDatosCertificadoFirma);
			throw new Y41bUDAException("error.platea.invocation", true, e_obtenerDatosCertificadoFirma);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.obtenerDatosCertificadoFirma - Final");
		}
	}

	/**
	 * Este metodo sirve para procesar el crear el ejgv
	 * 
	 * @param crearEJGV
	 *            CrearEJGV
	 * 
	 * @return RespuestaCrearEJGVDocumentSolicitud
	 * 
	 */
	public RespuestaCrearEJGVDocumentSolicitud crearEJGVDocumentSolicitud(CrearEJGV crearEJGV) {
		logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentSolicitud - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			Procedure procedure = crearEJGV.getProcedure();

			if (null != crearEJGV.getDocumentosAdjuntos() && crearEJGV.getDocumentosAdjuntos().size() > 0) {
				for (Documento documento : crearEJGV.getDocumentosAdjuntos()) {
					if (StringUtils.isBlank(documento.getOidDokusi())) {
						if (null != documento.getBytes() && documento.getBytes().length() > 0) {

							documento.setProcedure(procedure);
							documento.setNumeroExpediente(crearEJGV.getNumeroExpediente());
							documento = dokusiAdapterService.registrarDocumentoDokusi(documento);
						}
					}
				}
			}

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					crearEJGV.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String xmlDocument = Y41bPlateaHelper.crearXMLSolicitud(crearEJGV, procedureDocument.getProcedure());
			String xslDocument = Y41bPlateaHelper.crearXSLSolicitud(crearEJGV);
			String textoPlano = Y41bPlateaHelper.crearTextoPlanoSolicitud(crearEJGV);

			// Preparamos el texto plano segun guia de servicio horizontal firma
			// - Inicio
			String textoPlanoPreparado = "\r\n" + textoPlano.replaceAll("\n", "\r\n").trim() + "\r\n";
			textoPlanoPreparado = textoPlanoPreparado.replaceAll("\r\n", "\n");
			textoPlanoPreparado = textoPlanoPreparado.replaceAll("\n", "\r\n");
			textoPlanoPreparado = Y41bUtilsWS.encodeString(textoPlanoPreparado);
			// Preparamos el texto plano segun guia de servicio horizontal firma
			// - Fin

			respuesta = plateaService.crearEJGVDocumentSolicitud(tokenSesion, xmlDocument, xslDocument,
					textoPlanoPreparado);

			logger.debug("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentSolicitud - Return: "
					+ Y41bUtilsWS.decodeToString(respuesta));
			RespuestaCrearEJGVDocumentSolicitud respuestaEjgv = Y41bPlateaTransformer
					.transformCrearEJGVDocumentSolicitud(Y41bUtilsWS.decodeToString(respuesta),
							crearEJGV.getDocumentosAdjuntos());
			return respuestaEjgv;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_crearEJGVDocumentSolicitud) {
			logger.error("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentSolicitud", e_crearEJGVDocumentSolicitud);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearEJGVDocumentSolicitud);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentSolicitud - Final");
		}
	}

	/**
	 * Método que sirve para procesar el crear el ejgv de la Notificacion
	 * 
	 * @param crearNotifEJGV
	 *            CrearNotifEJGV
	 * 
	 * @return RespuestaCrearEJGVDocumentNotificacion
	 * 
	 */
	public RespuestaCrearEJGVDocumentNotificacion crearEJGVDocumentNotificacion(CrearNotifEJGV crearNotifEJGV) {
		logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentNotificacion - Inicio");
		try {
			if (!com.ejie.y41b.utils.Y41bUtils.isFilled(crearNotifEJGV.getDescTipoNotificacion())) {
				crearNotifEJGV.setDescTipoNotificacion(Y41bConstantes.NOTIFICACION_DESC_COMUNICACION);
			}
			logger.info(
					"Y41bPlateaAdapterServiceImpl.crearEJGVDocumentNotificacion - Paso 1/1: crearEJGVDocumentNotificacionGenerico");
			RespuestaCrearEJGVDocumentNotificacion respuestaEjgv = this
					.crearEJGVDocumentNotificacionGenerico(crearNotifEJGV);
			logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentNotificacion - Fin");
			return respuestaEjgv;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_crearEJGVDocumentNotificacion) {
			logger.error("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentNotificacion", e_crearEJGVDocumentNotificacion);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearEJGVDocumentNotificacion);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentNotificacion - Final");
		}
	}

	/**
	 * Método que sirve para procesar el crear el ejgv de la Comunicacion
	 * 
	 * @param crearNotifEJGV
	 *            CrearNotifEJGV
	 * 
	 * @return RespuestaCrearEJGVDocumentNotificacion
	 * 
	 */
	public RespuestaCrearEJGVDocumentNotificacion crearEJGVDocumentComunicacion(CrearNotifEJGV crearNotifEJGV) {
		logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentComunicacion - Inicio");
		try {
			// Completamos
			crearNotifEJGV.setIdTipoNotificacion(Y41bConstantes.NOTIFICACION_COMUNICACION);
			crearNotifEJGV.setDescTipoNotificacion(Y41bConstantes.NOTIFICACION_DESC_COMUNICACION);
			logger.info(
					"Y41bPlateaAdapterServiceImpl.crearEJGVDocumentComunicacion - Paso 1/1 : crearEJGVDocumentNotificacionGenerico");
			RespuestaCrearEJGVDocumentNotificacion respuestaEjgv = this
					.crearEJGVDocumentNotificacionGenerico(crearNotifEJGV);
			logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentComunicacion - Fin");
			return respuestaEjgv;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_crearEJGVDocumentComunicacion) {
			logger.error("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentComunicacion", e_crearEJGVDocumentComunicacion);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearEJGVDocumentComunicacion);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentComunicacion - Final");
		}
	}

	/**
	 * Método que sirve para recuperar el Registro de Salida de una
	 * Notificacion/Comunicacion Electronica
	 * 
	 * @param idNotificacion
	 *            String
	 * 
	 * @return RegistroSalida
	 * 
	 */
	public RegistroSalida getRegistroSalidaFromIdNotificacion(String idNotificacion) {
		logger.info("Y41bPlateaAdapterServiceImpl.getRegistroSalidaFromIdNotificacion - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String sNotifID = idNotificacion;
			String sAuditUser = "";

			String respuesta = plateaService.getNotificacionById(tokenSesion, sNotifID, sAuditUser);

			logger.debug("Y41bPlateaAdapterServiceImpl.getRegistroSalidaFromIdNotificacion - Return: "
					+ Y41bUtilsWS.decodeToString(respuesta));

			RegistroSalida registroSalida = Y41bPlateaTransformer
					.transformGetRegistroSalidaFromIdNotificacion(Y41bUtilsWS.decodeToString(respuesta));

			logger.info("Y41bPlateaAdapterServiceImpl.getRegistroSalidaFromIdNotificacion - Fin");
			return registroSalida;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_crearEJGVDocumentNotificacion) {
			logger.error("Y41bPlateaAdapterServiceImpl.getRegistroSalidaFromIdNotificacion",
					e_crearEJGVDocumentNotificacion);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearEJGVDocumentNotificacion);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.getRegistroSalidaFromIdNotificacion - Final");
		}
	}

	/**
	 * Este metodo sirve para procesar el envio de la notificacion
	 * 
	 * @param EnvioNotifEJGV
	 *            envioNotifEJGV
	 * 
	 * @return RespuestaNotificacion numero de registro de la notificacion
	 * 
	 */
	public RespuestaNotificacion procesarEnviarNotificacion(EnvioNotifEJGV envioNotifEJGV) {
		logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacion - Inicio");
		try {
			logger.info(
					"Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacion - Paso 1/1 : procesarEnviarNotificacionGenerico");
			RespuestaNotificacion respuestaNotificacion = this.procesarEnviarNotificacionGenerico(envioNotifEJGV);
			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacion - Fin");
			return respuestaNotificacion;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_procesarEnviarNotificacion) {
			logger.error("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacion", e_procesarEnviarNotificacion);
			throw new Y41bUDAException("error.platea.invocation", true, e_procesarEnviarNotificacion);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacion - Final");
		}
	}

	/**
	 * Este metodo sirve para procesar el envio de la notificacion mediante
	 * X43Notificacion
	 * 
	 * @param crearNotifEJGV
	 *            CrearNotifEJGV
	 * 
	 * @return RespuestaNotificacion numero de registro de la notificacion
	 * 
	 */
	public RespuestaNotificacion procesarEnviarNotificacionX43(CrearNotifEJGV crearNotifEJGV) {
		logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacion - Inicio");
		try {
			if (!com.ejie.y41b.utils.Y41bUtils.isFilled(crearNotifEJGV.getDescTipoNotificacion())) {
				crearNotifEJGV.setDescTipoNotificacion(Y41bConstantes.NOTIFICACION_DESC_COMUNICACION);
			}

			logger.info(
					"Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacion - Paso 1/1 : procesarEnviarNotificacionGenerico");
			RespuestaNotificacion respuestaNotificacion = this.procesarEnviarNotificacionGenericoX43(crearNotifEJGV);
			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacion - Fin");
			return respuestaNotificacion;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_procesarEnviarNotificacion) {
			logger.error("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacion", e_procesarEnviarNotificacion);
			throw new Y41bUDAException("error.platea.invocation", true, e_procesarEnviarNotificacion);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacion - Final");
		}
	}

	/**
	 * Este metodo sirve para procesar el envio de la comunicacion
	 * 
	 * @param EnvioNotifEJGV
	 *            envioNotifEJGV
	 * 
	 * @return RespuestaNotificacion numero de registro de la notificacion
	 * 
	 */
	public RespuestaNotificacion procesarEnviarComunicacion(EnvioNotifEJGV envioNotifEJGV) {
		logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarComunicacion - Inicio");
		try {
			RespuestaNotificacion respuestaComunicacion = this.procesarEnviarNotificacionGenerico(envioNotifEJGV);
			return respuestaComunicacion;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_procesarEnviarComunicacionn) {
			logger.error("Y41bPlateaAdapterServiceImpl.procesarEnviarComunicacion", e_procesarEnviarComunicacionn);
			throw new Y41bUDAException("error.platea.invocation", true, e_procesarEnviarComunicacionn);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarComunicacion - Final");
		}
	}

	/**
	 * Método que centraliza la parte comun a la creacion del EJGV de cualquier
	 * notificacion
	 * 
	 * @param crearNotifEJGV
	 *            CrearNotifEJGV
	 * 
	 * @return RespuestaCrearEJGVDocumentNotificacion
	 * 
	 */
	private RespuestaCrearEJGVDocumentNotificacion crearEJGVDocumentNotificacionGenerico(
			CrearNotifEJGV crearNotifEJGV) {
		logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentNotificacionGenerico - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			if (null != crearNotifEJGV.getDocumentosAdjuntos() && crearNotifEJGV.getDocumentosAdjuntos().size() > 0) {

				logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentNotificacionGenerico - Numero documentos: "
						+ crearNotifEJGV.getDocumentosAdjuntos().size() + ".");

				for (Documento documento : crearNotifEJGV.getDocumentosAdjuntos()) {
					if (com.ejie.y41b.utils.Y41bUtils.isDocumentoLocalizable(documento)) {

						if (!StringUtils.isBlank(documento.getOidDokusi())) {

							Formato formato = dokusiAdapterService.recuperarFormatosDocumentosDokusi(documento);

							if (formato != null && formato.getName().equals(Y41bConstantes.PDFA)) {
								if (documento != null && documento.getNombre() != null && !com.ejie.y41b.utils.Y41bUtils
										.getExtensionFile(documento.getNombre()).equalsIgnoreCase(Y41bConstantes.PDF)) {
									throw new Y41bUDAException("error.documentos.sinlocalizador", true,
											new Exception());
								}

								documento.setTieneLocalizador(Y41bConstantes.SI);

							} else {
								throw new Y41bUDAException("error.documentos.sinlocalizador", true, new Exception());
							}
						} else {
							throw new Y41bUDAException("error.documentos.sinlocalizador", true, new Exception());
						}
					} else {
						if (!StringUtils.isBlank(documento.getOidDokusi())) {

							Formato formato = dokusiAdapterService.recuperarFormatosDocumentosDokusi(documento);

							if (formato != null && formato.getName().equals(Y41bConstantes.PDFA)) {
								documento.setTieneLocalizador(Y41bConstantes.SI);
							}
						}
					}

					// 86178: [SOLICITUDES] - Nuevos tramites - Solicitud de...
					if (Y41bConstantes.TIPO_DOC_RESP_SOL_COLABORACION.equals(documento.getTipoDocumental())) {
						documento.setTipoDocumental(Y41bConstantes.TIPO_DOC_ARCHIVO_GENERICO);
					} else if (Y41bConstantes.TIPO_DOC_SOL_ARBITRAJE.equals(documento.getTipoDocumental())) {
						documento.setTipoDocumental(Y41bConstantes.TIPO_DOC_SOLICITUD);
					} else if (Y41bConstantes.TIPO_DOC_INS_INFORME_INSPECCION.equals(documento.getTipoDocumental())) {
						documento.setTipoDocumental(Y41bConstantes.TIPO_DOC_INFORME);
					} else if (Y41bConstantes.TIPO_DOC_RESPUESTA_MEDIACION.equals(documento.getTipoDocumental())) {
						documento.setTipoDocumental(Y41bConstantes.TIPO_DOC_ALEGACION);
					} else if (Y41bConstantes.TIPO_DOC_PROTOCOLO.equals(documento.getTipoDocumental())) {
						documento.setTipoDocumental(Y41bConstantes.TIPO_DOC_INFORME);
					}
				}
			}

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					crearNotifEJGV.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String xmlDocumentNotif = Y41bPlateaHelper.crearXMLNotificacion(crearNotifEJGV,
					procedureDocument.getProcedure());

			respuesta = plateaService.crearEJGVDocumentNotificacion(tokenSesion, xmlDocumentNotif);

			logger.debug("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentNotificacionGenerico - Return: "
					+ Y41bUtilsWS.decodeToString(respuesta));
			RespuestaCrearEJGVDocumentNotificacion respuestaEjgv = Y41bPlateaTransformer
					.transformCrearEJGVDocumentNotificacion(Y41bUtilsWS.decodeToString(respuesta),
							crearNotifEJGV.getDocumentosAdjuntos());
			return respuestaEjgv;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_crearEJGVDocumentNotificacionGenerico) {
			logger.error("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentNotificacionGenerico",
					e_crearEJGVDocumentNotificacionGenerico);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearEJGVDocumentNotificacionGenerico);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentNotificacionGenerico - Final");
		}
	}

	/**
	 * Este metodo sirve para procesar el envio de la notificacion
	 * 
	 * @param envioNotifEJGV
	 *            EnvioNotifEJGV
	 * 
	 * @return RespuestaNotificacion numero de registro de la solicitud
	 * 
	 */
	private RespuestaNotificacion procesarEnviarNotificacionGenerico(EnvioNotifEJGV envioNotifEJGV) {
		logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico - Inicio");
		try {
			RespuestaNotificacion respuestaNotificacion = new RespuestaNotificacion();
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String ejgvDocument = envioNotifEJGV.getEjgvDocument();
			String ejgvDocumentBase64 = Y41bUtilsWS.encodeString(ejgvDocument);
			String firma = envioNotifEJGV.getFirma();
			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico - Inicio envioNotificaciones");
			String respuestaEnvioNotificacion = plateaService.envioNotificaciones(tokenSesion, ejgvDocumentBase64,
					firma);
			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico - Fin envioNotificaciones");

			respuestaNotificacion.setNumeroRegistro(respuestaEnvioNotificacion);
			logger.debug("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico - Return: "
					+ respuestaNotificacion);
			return respuestaNotificacion;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_procesarEnviarNotificacionGenerico) {
			logger.error("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico",
					e_procesarEnviarNotificacionGenerico);
			throw new Y41bUDAException("error.platea.invocation", true, e_procesarEnviarNotificacionGenerico);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico - Final");
		}
	}

	/**
	 * Este metodo sirve para procesar el envio de la notificacion mediante
	 * X43Notificacion
	 * 
	 * @param crearNotifEJGV
	 *            CrearNotifEJGV
	 * 
	 * @return RespuestaNotificacion numero de registro de la solicitud
	 * 
	 */
	private RespuestaNotificacion procesarEnviarNotificacionGenericoX43(CrearNotifEJGV crearNotifEJGV) {
		logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico - Inicio");
		try {
			RespuestaNotificacion respuestaNotificacion = new RespuestaNotificacion();

			if (null != crearNotifEJGV.getDocumentosAdjuntos() && crearNotifEJGV.getDocumentosAdjuntos().size() > 0) {

				logger.info("Y41bPlateaAdapterServiceImpl.crearEJGVDocumentNotificacionGenerico - Numero documentos: "
						+ crearNotifEJGV.getDocumentosAdjuntos().size() + ".");

				for (Documento documento : crearNotifEJGV.getDocumentosAdjuntos()) {
					if (com.ejie.y41b.utils.Y41bUtils.isDocumentoLocalizable(documento)) {

						if (!StringUtils.isBlank(documento.getOidDokusi())) {

							Formato formato = dokusiAdapterService.recuperarFormatosDocumentosDokusi(documento);

							if (formato != null && formato.getName().equals(Y41bConstantes.PDFA)) {
								if (documento != null && documento.getNombre() != null && !com.ejie.y41b.utils.Y41bUtils
										.getExtensionFile(documento.getNombre()).equalsIgnoreCase(Y41bConstantes.PDF)) {
									throw new Y41bUDAException("error.documentos.sinlocalizador", true,
											new Exception());
								}

								documento.setTieneLocalizador(Y41bConstantes.SI);

							} else {
								throw new Y41bUDAException("error.documentos.sinlocalizador", true, new Exception());
							}
						} else {
							throw new Y41bUDAException("error.documentos.sinlocalizador", true, new Exception());
						}
					} else {
						if (!StringUtils.isBlank(documento.getOidDokusi())) {

							Formato formato = dokusiAdapterService.recuperarFormatosDocumentosDokusi(documento);

							if (formato != null && formato.getName().equals(Y41bConstantes.PDFA)) {
								documento.setTieneLocalizador(Y41bConstantes.SI);
							}
						}
					}

					// 86178: [SOLICITUDES] - Nuevos tramites - Solicitud de...
					if (Y41bConstantes.TIPO_DOC_RESP_SOL_COLABORACION.equals(documento.getTipoDocumental())) {
						documento.setTipoDocumental(Y41bConstantes.TIPO_DOC_ARCHIVO_GENERICO);
					} else if (Y41bConstantes.TIPO_DOC_SOL_ARBITRAJE.equals(documento.getTipoDocumental())) {
						documento.setTipoDocumental(Y41bConstantes.TIPO_DOC_SOLICITUD);
					} else if (Y41bConstantes.TIPO_DOC_INS_INFORME_INSPECCION.equals(documento.getTipoDocumental())) {
						documento.setTipoDocumental(Y41bConstantes.TIPO_DOC_INFORME);
					} else if (Y41bConstantes.TIPO_DOC_RESPUESTA_MEDIACION.equals(documento.getTipoDocumental())) {
						documento.setTipoDocumental(Y41bConstantes.TIPO_DOC_ALEGACION);
					} else if (Y41bConstantes.TIPO_DOC_PROTOCOLO.equals(documento.getTipoDocumental())) {
						documento.setTipoDocumental(Y41bConstantes.TIPO_DOC_INFORME);
					}
				}
			}

			if (crearNotifEJGV.getSolicitante() != null && (StringUtils
					.isBlank(crearNotifEJGV.getSolicitante().getDni())
					|| (!com.ejie.y41b.utils.Y41bUtils.isValidCIF(crearNotifEJGV.getSolicitante().getDni())
							&& !com.ejie.y41b.utils.Y41bUtils.isValidNIF(crearNotifEJGV.getSolicitante().getDni())
							&& !com.ejie.y41b.utils.Y41bUtils.isValidNIE(crearNotifEJGV.getSolicitante().getDni())))) {
				throw new Y41bUDAException("error.notificacionelectronica.nifCifNoValido", true, new Exception());
			}

			TNotificationCreateNew notification = Y41bPlateaHelper.crearXMLTNotificationCreateNew(crearNotifEJGV);

			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico - Inicio envioNotificaciones");
			String respuestaEnvioNotificacion = x43NotificacionesServiceImpl.notificationPublish(notification,
					new com.ejie.y41b.ws.cliente.x43notificaciones.AExtensionPoint());
			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico - Fin envioNotificaciones");

			respuestaNotificacion.setNumeroRegistro(respuestaEnvioNotificacion);
			logger.debug("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico - Return: "
					+ respuestaNotificacion);
			return respuestaNotificacion;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Throwable e_procesarEnviarNotificacionGenerico) {
			logger.error("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico",
					e_procesarEnviarNotificacionGenerico);
			throw new Y41bUDAException("error.platea.invocation", true, e_procesarEnviarNotificacionGenerico);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarNotificacionGenerico - Final");
		}
	}

	/**
	 * Este metodo sirve para procesar el envio de la solicitud
	 * 
	 * @param envioEJGV
	 *            EnvioEJGV
	 * 
	 * @return RespuestaSolicitudTelematica numero de registro de la solicitud +
	 *         Ejgv Firmado
	 * 
	 */
	public RespuestaSolicitudTelematica procesarEnviarSolicitud(EnvioEJGV envioEJGV) {
		logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarSolicitud - Inicio");
		try {
			RespuestaSolicitudTelematica respuestaSolicitudTelematica = new RespuestaSolicitudTelematica();
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String ejgvDocument = envioEJGV.getEjgvDocument();
			String firma = envioEJGV.getFirma();

			// Inicio Bloque para "anyadirFirmaEjgvDocumentSolicitud"
			String strEjgvFirma = Y41bPlateaHelper.fncCrearXMLFirmaSimple(firma, true);
			String strEjgvDocumentFirma = ejgvDocument;

			strEjgvDocumentFirma = strEjgvDocumentFirma.replaceAll("<!-- ejgvSignatureejgvSignature -->",
					"<!-- ejgvSignature" + strEjgvFirma + "ejgvSignature -->");

			String strEjgvdocumentFirmaBase64 = Y41bUtilsWS.encodeString(strEjgvDocumentFirma);

			String ejgvDocumentFirmado = plateaService.anyadirFirmaEjgvDocumentSolicitud(tokenSesion,
					strEjgvdocumentFirmaBase64, firma);

			respuestaSolicitudTelematica.setEjgvDocumentFirmado(Y41bUtilsWS.decodeToString(ejgvDocumentFirmado));
			// Fin Bloque para "anyadirFirmaEjgvDocumentSolicitud"

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					envioEJGV.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			respuesta = plateaService.obtenerDatosCertificadoFirma(tokenSesion, firma);

			DatosCertificado datosCertificado = Y41bPlateaTransformer.transformObtenerDatosCertificadoFirma(respuesta);

			String contextSubmission = Y41bPlateaHelper.crearXMLContentSubmisionEnvio(envioEJGV, datosCertificado,
					procedureDocument.getProcedure());

			respuesta = plateaService.envioSolicitudes(tokenSesion, contextSubmission, ejgvDocumentFirmado);

			respuestaSolicitudTelematica.setNumeroRegistro(respuesta);
			logger.debug("Y41bPlateaAdapterServiceImpl.procesarEnviarSolicitud - Return: " + respuesta);
			return respuestaSolicitudTelematica;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_procesarEnviarSolicitud) {
			logger.error("Y41bPlateaAdapterServiceImpl.procesarEnviarSolicitud", e_procesarEnviarSolicitud);
			throw new Y41bUDAException("error.platea.invocation", true, e_procesarEnviarSolicitud);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.procesarEnviarSolicitud - Final");
		}
	}

	/**
	 * Metodo que genera el HASH de un documento
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para generar el HASH en
	 *            Dokusi
	 * 
	 * @return String Hash Generado.
	 */
	public String generarHashDocumento(Documento documento) {
		try {

			Documento documentoRdo = dokusiAdapterService.descargarPifDocumentoDokusi(documento);

			Y31AttachmentBean y31AttachmentBean = this.y41bPifAdapterService.moveToPLATEA(documentoRdo.getPifId(),
					true);

			NSHFServiceImpl nshfService = new NSHFServiceImpl();
			String resultado = nshfService.createHashLocation(y31AttachmentBean.getFilePath());

			logger.debug("Y41bPlateaAdapterServiceImpl.generarHashDocumento - Return: " + resultado);

			return resultado;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_generarHashDocumento) {
			logger.error("Y41bPlateaAdapterServiceImpl.generarHashDocumento", e_generarHashDocumento);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_generarHashDocumento);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.generarHashDocumento - Final");
		}
	}

	/**
	 * Método que sirve para crear una espera
	 * 
	 * @param espera
	 *            Espera
	 * 
	 * @return RespuestaProcesarEspera
	 * 
	 */
	public RespuestaProcesarEspera crearEspera(Espera espera) {
		logger.info("Y41bPlateaAdapterServiceImpl.crearEspera - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			logger.debug("Y41bPlateaAdapterServiceImpl.crearEspera - Paso 1/3: obtenerDatosCatalogoProcedimientos");

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					espera.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			// Inicio Bloque para crear los datos de la espera
			// Generamos el identificador de la espera
			String idEspera = "";
			if (!com.ejie.y41b.utils.Y41bUtils.isFilled(espera.getOidEspera())) {
				idEspera = (Long.toString((new java.util.Date()).getTime()));
				espera.setOidEspera(idEspera);
			} else {
				idEspera = espera.getOidEspera();
			}

			logger.debug("Y41bPlateaAdapterServiceImpl.crearEspera - Paso 2/3: crearXMLEspera");

			String xmlEspera = Y41bPlateaHelper.crearXMLEspera(espera, procedureDocument.getProcedure(),
					Y41bConstantes.TIPO_TAREA_CREAR_ESPERA);
			String tipoEspera = Y41bConstantes.TIPO_ESPERA_SAVE_OR_UPDATE_NT;
			// Fin Bloque para crear los datos de la espera

			logger.debug("Y41bPlateaAdapterServiceImpl.crearEspera - Paso 3/3: crearMatarEspera");

			respuesta = plateaService.crearMatarEspera(tokenSesion, xmlEspera, tipoEspera);
			logger.debug("Y41bPlateaAdapterServiceImpl.crearEspera - Return: " + respuesta);
			RespuestaProcesarEspera respuestaProcesarEsperas = new RespuestaProcesarEspera();
			respuestaProcesarEsperas.setOidEspera(idEspera);
			logger.debug("Y41bPlateaAdapterServiceImpl.crearEspera - Fin ");
			return respuestaProcesarEsperas;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_crearEspera) {
			logger.error("Y41bPlateaAdapterServiceImpl.crearEspera", e_crearEspera);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearEspera);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.crearEspera - Final");
		}
	}

	/**
	 * Método que sirve para matar una espera
	 * 
	 * @param espera
	 *            Espera
	 * 
	 * @return RespuestaProcesarEspera
	 * 
	 */
	public RespuestaProcesarEspera matarEspera(Espera espera) {
		logger.info("Y41bPlateaAdapterServiceImpl.matarEspera - Inicio");
		try {
			RespuestaProcesarEspera respuestaProcesarEsperas = new RespuestaProcesarEspera();
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					espera.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			// Inicio Bloque para crear los datos de la espera
			String xmlEspera = Y41bPlateaHelper.crearXMLEspera(espera, procedureDocument.getProcedure(),
					Y41bConstantes.TIPO_TAREA_MATAR_ESPERA);
			String tipoEspera = Y41bConstantes.TIPO_ESPERA_SAVE_OR_UPDATE_NT;

			respuesta = plateaService.crearMatarEspera(tokenSesion, xmlEspera, tipoEspera);

			respuestaProcesarEsperas.setOidEspera(espera.getOidEspera());
			logger.debug("Y41bPlateaAdapterServiceImpl.matarEspera - Return: " + respuesta);
			return respuestaProcesarEsperas;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_matarEspera) {
			logger.error("Y41bPlateaAdapterServiceImpl.matarEspera", e_matarEspera);
			throw new Y41bUDAException("error.platea.invocation", true, e_matarEspera);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.matarEspera - Final");
		}
	}

	/**
	 * Método que sirve para registrar información en el historico del
	 * expediente
	 * 
	 * @param historico
	 *            Historico
	 * @return RespuestaRegistrarHistorico
	 */
	public RespuestaRegistrarHistorico registrarHistorico(Historico historico) {
		logger.info("Y41bPlateaAdapterServiceImpl.registrarHistorico - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					historico.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			// Inicio Bloque para crear los datos del historico
			// Generamos el identificador del historico
			String idHistorico = "";
			if (!com.ejie.y41b.utils.Y41bUtils.isFilled(historico.getIdHistorico())) {
				idHistorico = (Long.toString((new java.util.Date()).getTime()));
				historico.setIdHistorico(idHistorico);
			} else {
				idHistorico = historico.getIdHistorico();
			}
			String xmlHistorico = Y41bPlateaHelper.crearXMLRegistrarHistorico(historico,
					procedureDocument.getProcedure());
			String tipoHistorico = Y41bConstantes.TIPO_HISTORICO_SAVE_OR_UPDATE_NT;

			respuesta = plateaService.registrarHistorico(tokenSesion, xmlHistorico, tipoHistorico);

			RespuestaRegistrarHistorico respuestaRegistrarHistorico = new RespuestaRegistrarHistorico();
			respuestaRegistrarHistorico.setIdHistorico(idHistorico);

			logger.debug("Y41bPlateaAdapterServiceImpl.registrarHistorico - Return: " + respuesta);
			return respuestaRegistrarHistorico;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_registrarHistorico) {
			logger.error("Y41bPlateaAdapterServiceImpl.registrarHistorico", e_registrarHistorico);
			throw new Y41bUDAException("error.platea.invocation", true, e_registrarHistorico);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.registrarHistorico - Final");
		}
	}

	/**
	 * Método que sirve para eliminar información en el historico del
	 * expediente
	 * 
	 * @param historico
	 *            Historico
	 * 
	 */
	public void eliminarHistorico(Historico historico) {
		logger.info("Y41bPlateaAdapterServiceImpl.eliminarHistorico - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					historico.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			// Inicio Bloque para crear los datos del historico
			String xmlHistorico = Y41bPlateaHelper.crearXMLEliminarHistorico(historico,
					procedureDocument.getProcedure());
			String tipoHistorico = Y41bConstantes.TIPO_HISTORICO_DELETE_NT;

			respuesta = plateaService.eliminarHistorico(tokenSesion, xmlHistorico, tipoHistorico);

			logger.debug("Y41bPlateaAdapterServiceImpl.eliminarHistorico - Return: " + respuesta);
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_eliminarHistorico) {
			logger.error("Y41bPlateaAdapterServiceImpl.eliminarHistorico", e_eliminarHistorico);
			throw new Y41bUDAException("error.platea.invocation", true, e_eliminarHistorico);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.eliminarHistorico - Final");
		}
	}

	/**
	 * Este metodo sirve para verificar el certificado que ha firmado
	 * 
	 * @param certificateB64
	 *            byte[]
	 * 
	 * @return SignCertificateType
	 * 
	 */
	public SignCertificateType verifyCertificate(byte[] certificateB64) {
		logger.info("Y41bPlateaAdapterServiceImpl.verifyCertificate - Inicio");
		try {

			Date fechaHoy = new Date();
			String fechaActual = com.ejie.y41b.utils.Y41bUtils.formatearDateToString(fechaHoy, "yyyyMMddHHmmss");

			StringBuffer nombreCertificado = new StringBuffer().append("verifyCertificateLocation").append(fechaActual)
					.append(".xml");

			InputStream myInputStream = new ByteArrayInputStream(certificateB64);
			String idPif = this.y41bPifAdapterService.putFileToPLATEA(myInputStream, nombreCertificado.toString());

			NSHFServiceImpl nSHFServiceImpl = new NSHFServiceImpl();
			VerificationResultType verificationResultType = nSHFServiceImpl.verifyCertificateLocation(idPif);

			SignCertificatesType signCertificatesType = verificationResultType.getVerificationResult()
					.getSignCertificates();
			List<SignCertificateType> listCertificateType = signCertificatesType.getSignCertificate();

			SignCertificateType certificadoType = new SignCertificateType();

			if (listCertificateType != null && listCertificateType.size() > 0) {
				certificadoType = listCertificateType.get(0);
			}

			return certificadoType;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_crearEJGVDocumentSolicitud) {
			logger.error("Y41bPlateaAdapterServiceImpl.verifyCertificate", e_crearEJGVDocumentSolicitud);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearEJGVDocumentSolicitud);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.verifyCertificate - Final");
		}
	}

	/**
	 * Elimina blancos del DNI al actualizar MIS GESTIONES
	 * 
	 * @param actualizarMisGestiones
	 *            ActualizarMisGestiones
	 * @return ActualizarMisGestiones
	 */
	private ActualizarMisGestiones eliminarBlancosDNI(ActualizarMisGestiones actualizarMisGestiones) {

		// eliminar blacos del DNI
		DatosPersona datosPersona = new DatosPersona();
		datosPersona = actualizarMisGestiones.getSolicitante();

		if (datosPersona.getDni() != null) {
			datosPersona.setDni(datosPersona.getDni().trim());
		}
		actualizarMisGestiones.setSolicitante(datosPersona);
		return actualizarMisGestiones;
	}

	/**
	 * Metodo que actualiza Mis Gestiones
	 * 
	 * @param actualizarExpediente
	 *            Objeto con los datos de mis gestiones
	 * 
	 */
	public void actualizarParticipantExpediente(ActualizarExpediente actualizarExpediente) {
		logger.info("Y41bPlateaAdapterServiceImpl.actualizarParticipantExpediente - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					actualizarExpediente.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLActualizacionParticipantExpediente(actualizarExpediente,
					procedureDocument.getProcedure());

			plateaService.actualizarParticipantExpediente(tokenSesion, sXMLExpedient);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_actualizarMisGestiones) {
			logger.error("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones", e_actualizarMisGestiones);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestiones);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.actualizarMisGestiones - Final");
		}
	}

	/**
	 * Metodo que actualiza Mis Gestiones
	 * 
	 * @param actualizarExpediente
	 *            Objeto con los datos de mis gestiones
	 * 
	 */
	public void cancelarCierreExpediente(ActualizarExpediente actualizarExpediente) {
		logger.info("Y41bPlateaAdapterServiceImpl.cancelarCierreExpediente - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					actualizarExpediente.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String sXMLExpedient = Y41bPlateaHelper.crearXMLCancelCloseExpediente(actualizarExpediente,
					procedureDocument.getProcedure());

			// logger.info("Y41bPlateaAdapterServiceImpl.cancelarCierreExpediente
			// - Inicio"
			// + 1 / 0);
			plateaService.cancelarCierreExpediente(tokenSesion, sXMLExpedient);

		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_actualizarMisGestiones) {
			logger.error("Y41bPlateaAdapterServiceImpl.cancelarCierreExpediente", e_actualizarMisGestiones);
			throw new Y41bUDAException("error.platea.invocation", true, e_actualizarMisGestiones);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.cancelarCierreExpediente - Final");
		}
	}

	/**
	 * Método que sirve para recuperar el justificante de la notificacin
	 * 
	 * @param idNotificacion
	 *            String
	 * 
	 * @return TNotification
	 * 
	 */
	public TNotification getJustificanteNotificacion(String idNotificacion) {
		logger.info("Y41bPlateaAdapterServiceImpl.getJustificanteNotificacion - Inicio");
		try {
			TNotificationFilter filter = new TNotificationFilter();

			TStringList tStringList = new TStringList();
			List<String> listString = tStringList.getItem();
			listString.add(idNotificacion);

			filter.setNotificationIDList(tStringList);

			TDocFieldList value = new TDocFieldList();
			List<TDocField> listTDocField = value.getItem();

			TDocField e = TDocField.ROLE_RECEIPT;
			listTDocField.add(e);
			e = TDocField.DOC_INFO;
			listTDocField.add(e);
			filter.setDocFieldList(value);

			TNotificationList tNotificationList = x43NotificacionesServiceImpl.searchNotification(filter,
					new com.ejie.y41b.ws.cliente.x43notificaciones.AExtensionPoint());

			if (tNotificationList != null && tNotificationList.getNotification() != null
					&& tNotificationList.getNotification().size() > 0) {
				return tNotificationList.getNotification().get(0);
			} else {
				return null;
			}
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Throwable e_crearEJGVDocumentNotificacion) {
			logger.error("Y41bPlateaAdapterServiceImpl.getJustificanteNotificacion", e_crearEJGVDocumentNotificacion);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearEJGVDocumentNotificacion);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.getJustificanteNotificacion - Final");
		}
	}

	/**
	 * Método que sirve para recuperar el justificante del acuse de la
	 * notificacin
	 * 
	 * @param idNotificacion
	 *            String
	 * 
	 * @return TNotification
	 * 
	 */
	public TNotification getJustificanteAcuseNotificacion(String idNotificacion) {
		logger.info("Y41bPlateaAdapterServiceImpl.getJustificanteAcuseNotificacion - Inicio");
		try {
			TNotificationFilter filter = new TNotificationFilter();

			TStringList tStringList = new TStringList();
			List<String> listString = tStringList.getItem();
			listString.add(idNotificacion);

			filter.setNotificationIDList(tStringList);

			TDocFieldList value = new TDocFieldList();
			List<TDocField> listTDocField = value.getItem();

			TDocField e = TDocField.ROLE_ACCUSE;
			listTDocField.add(e);
			e = TDocField.DOC_INFO;
			listTDocField.add(e);
			filter.setDocFieldList(value);

			TNotificationList tNotificationList = x43NotificacionesServiceImpl.searchNotification(filter,
					new com.ejie.y41b.ws.cliente.x43notificaciones.AExtensionPoint());

			if (tNotificationList != null && tNotificationList.getNotification() != null
					&& tNotificationList.getNotification().size() > 0) {
				return tNotificationList.getNotification().get(0);
			} else {
				return null;
			}
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Throwable e_crearEJGVDocumentNotificacion) {
			logger.error("Y41bPlateaAdapterServiceImpl.getJustificanteAcuseNotificacion",
					e_crearEJGVDocumentNotificacion);
			throw new Y41bUDAException("error.platea.invocation", true, e_crearEJGVDocumentNotificacion);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.getJustificanteAcuseNotificacion - Final");
		}
	}

	/**
	 * Método que sirve para recuperar la cuenta del interesado
	 * 
	 * @param numIdentificacion
	 *            String
	 * 
	 * @return String
	 * 
	 */
	public String getCuentaInteresado(String numIdentificacion) {
		logger.info("Y41bPlateaAdapterServiceImpl.getCuentaInteresado - Inicio");
		try {
			TInterestedSearch interestedSearch = new TInterestedSearch();

			com.ejie.y41b.ws.cliente.x43CuentasInteresados.TStringList tStringList = new com.ejie.y41b.ws.cliente.x43CuentasInteresados.TStringList();
			List<String> listString = tStringList.getValor();
			listString.add(numIdentificacion);

			interestedSearch.setNumerosIdentificacion(tStringList);

			List<TInteresado> listTInteresado = x43CuentasInteresadosServiceImpl.getPersonsConsent(interestedSearch);

			if (listTInteresado != null && listTInteresado.size() > 0) {
				TInteresado tInteresado = listTInteresado.get(0);

				if (tInteresado != null && tInteresado.getCuentaList() != null
						&& tInteresado.getCuentaList().getCuenta() != null
						&& tInteresado.getCuentaList().getCuenta().size() > 0) {
					TCuenta tCuenta = tInteresado.getCuentaList().getCuenta().get(0);
					TCanalNotificacion tCanalNotificacion = tCuenta.getCanalNotificacion();

					return tCanalNotificacion.name();
				} else {
					return null;
				}
			} else {
				return null;
			}
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Throwable e_getCuentaInteresado) {
			logger.error("Y41bPlateaAdapterServiceImpl.getCuentaInteresado", e_getCuentaInteresado);
			throw new Y41bUDAException("error.platea.invocation", true, e_getCuentaInteresado);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.getCuentaInteresado - Final");
		}
	}

	/**
	 * getCuentaInteresado
	 * 
	 * @param numIdentificacion
	 *            String
	 * 
	 * @return boolean
	 * 
	 */
	public boolean tieneEmailCuentaInteresado(String numIdentificacion) {
		logger.info("Y41bPlateaAdapterServiceImpl.tieneEmailCuentaInteresado - Inicio");
		boolean resultado = false;
		try {

			TInterestedSearch interestedSearch = new TInterestedSearch();

			com.ejie.y41b.ws.cliente.x43CuentasInteresados.TStringList tStringList = new com.ejie.y41b.ws.cliente.x43CuentasInteresados.TStringList();
			List<String> listString = tStringList.getValor();
			listString.add(numIdentificacion);

			interestedSearch.setNumerosIdentificacion(tStringList);

			List<TInteresado> listTInteresado = x43CuentasInteresadosServiceImpl.getPersonsConsent(interestedSearch);

			if (listTInteresado != null && !listTInteresado.isEmpty()) {
				TInteresado tInteresado = listTInteresado.get(0);

				if (tInteresado != null && tInteresado.getCuentaList() != null
						&& tInteresado.getCuentaList().getCuenta() != null
						&& !tInteresado.getCuentaList().getCuenta().isEmpty()) {
					TCuenta tCuenta = tInteresado.getCuentaList().getCuenta().get(0);

					List<TContacto> listaTContactos = tCuenta.getContactoList().getContacto();
					for (TContacto tContacto : listaTContactos) {
						if (Y41bConstantes.CONTACT_TOOLKIT.EMAIL.equals(tContacto.getTipoContacto().toString())) {
							resultado = true;
							break;
						}
					}
				}

			}
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Throwable exception) {
			logger.error("Y41bPlateaAdapterServiceImpl.tieneEmailCuentaInteresado", exception);
			throw new Y41bUDAException("error.platea.invocation", true, exception);
		} finally {
			logger.info("Y41bPlateaAdapterServiceImpl.tieneEmailCuentaInteresado - Final");
		}

		return resultado;

	}
}
