package com.ejie.y41b.adapter;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.sql.Timestamp;
import java.util.List;
import java.util.Properties;

import javax.annotation.Resource;

import org.apache.commons.lang.StringEscapeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.stereotype.Service;

import com.ejie.cp.xml.ProcedureDocument;
import com.ejie.y41b.adapter.helper.Y41bDokusiHelper;
import com.ejie.y41b.adapter.services.SirServiceImpl;
import com.ejie.y41b.adapter.services.Y41bDokusiService;
import com.ejie.y41b.adapter.services.Y41bPlateaService;
import com.ejie.y41b.adapter.transformer.Y41bDokusiTransformer;
import com.ejie.y41b.constantes.Y41bConstantes;
import com.ejie.y41b.model.dokusi.Documento;
import com.ejie.y41b.model.dokusi.DocumentoBusqueda;
import com.ejie.y41b.model.dokusi.Formato;
import com.ejie.y41b.model.dokusi.LocalizadorDocumento;
import com.ejie.y41b.model.dokusi.Metadato;
import com.ejie.y41b.model.dokusi.RespuestaEliminarDocumento;
import com.ejie.y41b.model.dokusi.RespuestaLocalizadorDocumento;
import com.ejie.y41b.model.dokusi.RespuestaTransformarDocumento;
import com.ejie.y41b.model.dokusi.TransformarDocumento;
import com.ejie.y41b.utils.documentos.Y41bDocumento;
import com.ejie.y41b.utils.dokusi.Y41bUtilsDokusi;
import com.ejie.y41b.utils.exception.Y41bUDAException;
import com.ejie.y41b.utils.ws.Y41bUtilsWS;
import com.ejie.y41b.utils.xlnets.Y41bUtils;
import com.ejie.y41b.utils.xml.Y41bXMLUtils;

import noNamespace.T65BAttributeListListType;
import noNamespace.T65BAttributeListType;
import noNamespace.T65BAttributeType;

/**
 * <p>
 * Titulo: Y41bDokusiAdapterServiceImpl
 * </p>
 * <p>
 * Descipcion: Adapta nuestro sistema con Dokusi
 * </p>
 * <p>
 * Copyright: Copyright (c) 2013
 * </p>
 * <p>
 * Empresa: Bilbomatica
 * </p>
 * 
 *  
 * @version 1.0
 */
@Service(value = "Y41bDokusiAdapterServiceImpl")
public class Y41bDokusiAdapterServiceImpl implements Y41bDokusiAdapterService {

	/**
	 * Final static logger.
	 */
	private static final Logger logger = LoggerFactory.getLogger(Y41bDokusiAdapterServiceImpl.class);

	@Autowired()
	private Y41bDokusiService dokusiService;

	@Autowired()
	private Y41bPlateaService plateaService;

	@Autowired()
	private Y41bPifAdapterService y41bPifAdapterService;

	@Autowired()
	private Properties appConfiguration;

	@Resource()
	private ReloadableResourceBundleMessageSource appMessageSource;

	/**
	 * Clase que adaptara los servicios de Dokusi a nuestro sistema
	 */
	private Y41bDokusiAdapterServiceImpl() {
		super();
	}

	/**
	 * Metodo que obtiene los ids de documentos obtenidos filtrando por una
	 * lista de metadatos
	 * 
	 * @param documentoBusqueda
	 *            Objeto con los atributos necesarios para el filtro de
	 *            documentos
	 * @return List<Documento>.
	 */
	public List<Documento> buscarDocumentosDokusi(DocumentoBusqueda documentoBusqueda) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.buscarDocumentosDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(documentoBusqueda.getProcedure().getIdProcedimiento(),
					documentoBusqueda.getUser().getDni(), documentoBusqueda.getUser().getNombre());
			String documentalType = documentoBusqueda.getTipoDocumental();
			String returnedAttributeKeysList = Y41bDokusiHelper
					.crearXMLReturnedAttributeKeysList(documentoBusqueda.getMetadatosResultado());
			String searchConditionsList = Y41bDokusiHelper
					.crearXMLSearchConditionsList(documentoBusqueda.getMetadatosFiltro());
			boolean boolAllVersions = documentoBusqueda.isAllVersions();
			int intMaxRows = documentoBusqueda.getMaxRows();

			String resultado = this.dokusiService.buscarDocumentosDokusi(tokenSesion, auditUser, documentalType,
					returnedAttributeKeysList, searchConditionsList, boolAllVersions, intMaxRows);
			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.buscarDocumentosDokusi - Return: " + resultado);
			return Y41bDokusiTransformer.transformBuscarDocumentosDokusi(resultado);
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_buscarDocumentosDokusi) {
			Y41bDokusiAdapterServiceImpl.logger.error("Y41bDokusiAdapterServiceImpl.buscarDocumentosDokusi",
					e_buscarDocumentosDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_buscarDocumentosDokusi);
		} finally {
			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.buscarDocumentosMetadatosDokusi - Final");
		}
	}

	/**
	 * Metodo que registra el documento en DOKUSI
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para registrar el
	 *            documento en dokusi
	 * 
	 * @return Documento.
	 */
	public Documento registrarDocumentoDokusi(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(documento.getProcedure().getIdProcedimiento(),
					documento.getUser().getDni(), documento.getUser().getNombre());

			String documentalType = documento.getTipoDocumental();

			byte[] bytes = null;
			if (null != documento.getBytes()) {
				bytes = Y41bUtilsDokusi.decode(documento.getBytes());
			}

			InputStream myInputStream = new ByteArrayInputStream(bytes);
			String idPif = this.y41bPifAdapterService.putFileToDokusi(myInputStream, documento.getNombre());
			documento.setPifId(idPif);

			String content = Y41bDokusiHelper.crearXMLContentPif(documento.getPifId(), documento.getNombre(),
					documento.getExtension());

			String respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
					documento.getProcedure().getIdProcedimiento());
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String attributeList = Y41bDokusiHelper.crearXMLAttributeList(documento, procedureDocument);
			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusi - Inicio llamada Servicio");
			String resultado = this.dokusiService.registrarDocumentoDokusi(tokenSesion, auditUser, content,
					documentalType, attributeList);
			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusi - Fin llamada Servicio");

			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusi - Return: " + resultado);

			documento.setOidDokusi(Y41bDokusiTransformer.transformRegistrarDocumentoDokusi(resultado));

			if (com.ejie.y41b.utils.Y41bUtils.isDocumentoLocalizable(documento) && (documento.getNolocalizable() == null
					|| !documento.getNolocalizable().equals(Y41bConstantes.VALOR_SI))) {

				LocalizadorDocumento localizadorDocumento = new LocalizadorDocumento();
				localizadorDocumento.setProcedure(documento.getProcedure());
				localizadorDocumento.setUser(documento.getUser());
				localizadorDocumento.setOidDocumento(documento.getOidDokusi());
				localizadorDocumento.setBoolReducir(true);
				localizadorDocumento.setBoolVersion(false);
				localizadorDocumento.setSede("");

				this.incorporarLocalizadorDokusi(localizadorDocumento);
			} else if (com.ejie.y41b.utils.Y41bUtils.isDocumentoTransformable(documento)) {
				TransformarDocumento transformarDocumento = new TransformarDocumento();
				transformarDocumento.setProcedure(documento.getProcedure());
				transformarDocumento.setUser(documento.getUser());
				transformarDocumento.setOidDocumento(documento.getOidDokusi());

				this.transformarDocumentoDokusi(transformarDocumento);
			}

			return documento;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_registrarDocumentoDokusi) {
			// inicio - control extension fichero
			// R02G-10004080

			if (e_registrarDocumentoDokusi.getMessage().contains("R02G-10004080")) {
				Y41bDokusiAdapterServiceImpl.logger.error("Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusi",
						e_registrarDocumentoDokusi);

				String[] args = new String[1];
				args[0] = documento.getNombre();

				String errormsg = this.appMessageSource.getMessage("error.dokusi.extension", args,
						LocaleContextHolder.getLocale());

				throw new Y41bUDAException(errormsg, true, e_registrarDocumentoDokusi);
			} else {
				// fin - control extension fichero
				// e_registrarDocumentoDokusi.printStackTrace();
				Y41bDokusiAdapterServiceImpl.logger.error("Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusi",
						e_registrarDocumentoDokusi);
				throw new Y41bUDAException("error.dokusi.invocation", true, e_registrarDocumentoDokusi);
			}
		} finally {
			Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusi - Final");
		}
	}

	/**
	 * Metodo que registra el documento en DOKUSI Sin localizar Ni transformar
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para registrar el
	 *            documento en dokusi
	 * 
	 * @return Documento.
	 */
	public Documento registrarDocumentoDokusiSinLocalizar(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger
				.info("Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusiSinLocalizar - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(documento.getProcedure().getIdProcedimiento(),
					documento.getUser().getDni(), documento.getUser().getNombre());

			String documentalType = documento.getTipoDocumental();

			byte[] bytes = null;
			if (null != documento.getBytes()) {
				bytes = Y41bUtilsDokusi.decode(documento.getBytes());
			}

			InputStream myInputStream = new ByteArrayInputStream(bytes);
			String idPif = this.y41bPifAdapterService.putFileToDokusi(myInputStream, documento.getNombre());
			documento.setPifId(idPif);

			String content = Y41bDokusiHelper.crearXMLContentPif(documento.getPifId(), documento.getNombre(),
					documento.getExtension());
			String respuesta;
			if (Y41bConstantes.ID_PROCEDIMIENTO_ESTABLECIMIENTOS
					.equalsIgnoreCase(documento.getProcedure().getIdProcedimiento())) {
				respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
						Y41bConstantes.ID_PROCEDIMIENTO_ADHALTA);
			} else {
				respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
						documento.getProcedure().getIdProcedimiento());
			}
			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String attributeList = Y41bDokusiHelper.crearXMLAttributeList(documento, procedureDocument);
			Y41bDokusiAdapterServiceImpl.logger.info(
					"Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusiSinLocalizar - Inicio llamada Servicio");
			String resultado = this.dokusiService.registrarDocumentoDokusi(tokenSesion, auditUser, content,
					documentalType, attributeList);
			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusiSinLocalizar - Fin llamada Servicio");

			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusiSinLocalizar - Return: " + resultado);

			documento.setOidDokusi(Y41bDokusiTransformer.transformRegistrarDocumentoDokusi(resultado));

			return documento;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_registrarDocumentoDokusi) {
			// inicio - control extension fichero
			// R02G-10004080

			if (e_registrarDocumentoDokusi.getMessage().contains("R02G-10004080")) {
				Y41bDokusiAdapterServiceImpl.logger.error(
						"Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusiSinLocalizar",
						e_registrarDocumentoDokusi);

				String[] args = new String[1];
				args[0] = documento.getNombre();

				String errormsg = this.appMessageSource.getMessage("error.dokusi.extension", args,
						LocaleContextHolder.getLocale());

				throw new Y41bUDAException(errormsg, true, e_registrarDocumentoDokusi);
			} else {
				// fin - control extension fichero
				// e_registrarDocumentoDokusi.printStackTrace();
				Y41bDokusiAdapterServiceImpl.logger.error(
						"Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusiSinLocalizar",
						e_registrarDocumentoDokusi);
				throw new Y41bUDAException("error.dokusi.invocation", true, e_registrarDocumentoDokusi);
			}
		} finally {
			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.registrarDocumentoDokusiSinLocalizar - Final");
		}
	}

	/**
	 * Metodo que retramita el documento en DOKUSI
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para retramitar el
	 *            documento en dokusi
	 * 
	 * @return String.
	 */
	public String retramitarDocumentoDokusi(Documento documento) {
		logger.info("Y41bDokusiAdapterServiceImpl.retramitarDocumentoDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(documento.getProcedure().getIdProcedimiento(),
					documento.getUser().getDni(), documento.getUser().getNombre());

			String oidDocumentoDokusi = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			String documentalType = documento.getTipoDocumental();

			T65BAttributeListListType t65BAttributeListListType = T65BAttributeListListType.Factory.newInstance();

			T65BAttributeListType listaAtributos = t65BAttributeListListType.addNewAttributeList();

			T65BAttributeType atributo = null;

			if (com.ejie.y41b.utils.Y41bUtils.isFilled(documento.getTipoDocumental())
					&& documento.getTipoDocumental().equals(Y41bConstantes.TIPO_DOC_BOLETIN)) {
				atributo = listaAtributos.addNewAttribute();
				atributo.setKey("ejgv_tipo_boletin");
				atributo.setValue(Y41bConstantes.EJGV_TIPO_BOLETIN_BOPV);
			}

			if (com.ejie.y41b.utils.Y41bUtils.isFilled(documento.getTipoDocumental())
					&& documento.getTipoDocumental().equals(Y41bConstantes.TIPO_DOC_SENTENCIA)) {
				if (com.ejie.y41b.utils.Y41bUtils.isFilled(documento.getNumeroExpediente())) {
					atributo = listaAtributos.addNewAttribute();
					atributo.setKey("ejgv_num_sentencia");
					atributo.setValue(documento.getNumeroExpediente());
				}

				atributo = listaAtributos.addNewAttribute();
				atributo.setKey("ejgv_fecha_sentencia");
				java.util.Date date = new java.util.Date();
				atributo.setValue(
						com.ejie.y41b.utils.Y41bUtils.fncParsearFechaLargaAlternativa(new Timestamp(date.getTime())));
			}

			if (com.ejie.y41b.utils.Y41bUtils.isFilled(documento.getTipoDocumental())
					&& documento.getTipoDocumental().equals(Y41bConstantes.TIPO_DOC_FACTURA)) {
				if (com.ejie.y41b.utils.Y41bUtils.isFilled(documento.getNombre())) {
					atributo = listaAtributos.addNewAttribute();
					atributo.setKey("ejgv_num_factura");
					atributo.setValue(com.ejie.y41b.utils.Y41bUtils
							.removeAcentosCaracteresEspeciales(StringEscapeUtils.unescapeHtml(documento.getNombre())));
				}
			}

			String attributeList = Y41bXMLUtils.quitarNameSpacesString(t65BAttributeListListType);

			logger.info("Y41bDokusiAdapterServiceImpl.retramitarDocumentoDokusi - Inicio llamada Servicio");
			String resultado = dokusiService.retramitarDocumentoDokusi(tokenSesion, auditUser, oidDocumentoDokusi,
					documentalType, attributeList);
			logger.info("Y41bDokusiAdapterServiceImpl.retramitarDocumentoDokusi - Fin llamada Servicio");

			logger.debug("Y41bDokusiAdapterServiceImpl.retramitarDocumentoDokusi - Return: " + resultado);

			return resultado;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_retramitarDocumentoDokusi) {
			logger.error("Y41bDokusiAdapterServiceImpl.retramitarDocumentoDokusi", e_retramitarDocumentoDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_retramitarDocumentoDokusi);
		} finally {
			logger.info("Y41bDokusiAdapterServiceImpl.retramitarDocumentoDokusi - Final");
		}
	}

	/**
	 * Método que permitirá cambiar un documento de contexto de Registro a
	 * contexto de Tramitación y modifica el acl o la serie documental de un
	 * documento de DOKUSI
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para para modificar el acl
	 *            o la serie documental del documento en dokusi
	 * 
	 * @return boolean.
	 */
	public boolean tramitarDocumentoDokusi(Documento documento) {
		logger.info("tramitarDocumentoDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(documento.getProcedure().getIdProcedimiento(),
					documento.getUser().getDni(), documento.getUser().getNombre());

			String documentalType = "";
			if (documento.getTipoDocumental() != null)
				documentalType = documento.getTipoDocumental();

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			T65BAttributeListListType t65BAttributeListListType = T65BAttributeListListType.Factory.newInstance();

			T65BAttributeListType listaAtributos = t65BAttributeListListType.addNewAttributeList();

			T65BAttributeType atributo = null;

			if (documento.getMetadatos() != null && documento.getMetadatos().size() > 0) {

				for (Metadato metadato : documento.getMetadatos()) {

					atributo = listaAtributos.addNewAttribute();
					atributo.setKey(metadato.getNombre());
					atributo.setValue(metadato.getValor());
				}
			}

			/* mod */

			if (com.ejie.y41b.utils.Y41bUtils.isFilled(documento.getTipoDocumental())
					&& documento.getTipoDocumental().equals(Y41bConstantes.TIPO_DOC_BOLETIN)) {
				atributo = listaAtributos.addNewAttribute();
				atributo.setKey("ejgv_tipo_boletin");
				atributo.setValue(Y41bConstantes.EJGV_TIPO_BOLETIN_BOPV);
			}

			if (com.ejie.y41b.utils.Y41bUtils.isFilled(documento.getTipoDocumental())
					&& documento.getTipoDocumental().equals(Y41bConstantes.TIPO_DOC_SENTENCIA)) {
				if (com.ejie.y41b.utils.Y41bUtils.isFilled(documento.getNumeroExpediente())) {
					atributo = listaAtributos.addNewAttribute();
					atributo.setKey("ejgv_num_sentencia");
					atributo.setValue(documento.getNumeroExpediente());
				}

				atributo = listaAtributos.addNewAttribute();
				atributo.setKey("ejgv_fecha_sentencia");
				java.util.Date date = new java.util.Date();
				atributo.setValue(
						com.ejie.y41b.utils.Y41bUtils.fncParsearFechaLargaAlternativa(new Timestamp(date.getTime())));
			}

			if (com.ejie.y41b.utils.Y41bUtils.isFilled(documento.getTipoDocumental())
					&& documento.getTipoDocumental().equals(Y41bConstantes.TIPO_DOC_FACTURA)) {
				if (com.ejie.y41b.utils.Y41bUtils.isFilled(documento.getNombre())) {
					atributo = listaAtributos.addNewAttribute();
					atributo.setKey("ejgv_num_factura");
					atributo.setValue(com.ejie.y41b.utils.Y41bUtils
							.removeAcentosCaracteresEspeciales(StringEscapeUtils.unescapeHtml(documento.getNombre())));
				}
			}
			/* mod */

			String attributeList = Y41bXMLUtils.quitarNameSpacesString(t65BAttributeListListType);

			boolean bolResultado = dokusiService.tramitarDocumentoDokusi(tokenSesion, auditUser, documentID,
					documentalType, attributeList);

			return bolResultado;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_tramitarDocumentoDokusi) {
			logger.error("Y41bDokusiAdapterServiceImpl.tramitarDocumentoDokusi", e_tramitarDocumentoDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_tramitarDocumentoDokusi);
		} finally {
			logger.info("Y41bDokusiAdapterServiceImpl.tramitarDocumentoDokusi - Final");
		}
	}

	/**
	 * Metodo que modifica el documento en DOKUSI
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para registrar el
	 *            documento en dokusi
	 * 
	 * @return Documento.
	 */
	public Documento modificarDocumentoDokusi(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.modificarDocumentoDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(documento.getProcedure().getIdProcedimiento(),
					documento.getUser().getDni(), documento.getUser().getNombre());

			String oidDocumentoDokusi = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());
			String saveAsVersion = documento.getSaveAsVersion();
			boolean boolSetCurrent = Boolean.valueOf(documento.getBoolSetCurrent());

			byte[] bytes = null;
			if (null != documento.getBytes()) {
				bytes = Y41bUtilsDokusi.decode(documento.getBytes());

				InputStream myInputStream = new ByteArrayInputStream(bytes);
				String idPif = this.y41bPifAdapterService.putFileToDokusi(myInputStream, documento.getNombre());
				documento.setPifId(idPif);
			}

			String content = Y41bDokusiHelper.crearXMLContentPif(documento.getPifId(), documento.getNombre(),
					documento.getExtension());

			String resultado = "";
			if ((null != documento.getBytes() && !documento.getBytes().equalsIgnoreCase(""))
					&& null != documento.getMetadatos() && documento.getMetadatos().size() > 0) {
				String attributeList = Y41bDokusiHelper.crearXMLAttributeListModificacion(documento);
				resultado = this.dokusiService.modificarDocumentoDokusi(tokenSesion, auditUser, oidDocumentoDokusi,
						saveAsVersion, boolSetCurrent, attributeList, content);
			} else if (null != documento.getBytes() && !documento.getBytes().equalsIgnoreCase("")) {
				resultado = this.dokusiService.modificarDocumentoDokusiContent(tokenSesion, auditUser,
						oidDocumentoDokusi, saveAsVersion, boolSetCurrent, content);
			} else if (null != documento.getMetadatos() && documento.getMetadatos().size() > 0) {
				String attributeList = Y41bDokusiHelper.crearXMLAttributeListModificacion(documento);
				resultado = this.dokusiService.modificarDocumentoDokusiAtrib(tokenSesion, auditUser, oidDocumentoDokusi,
						saveAsVersion, boolSetCurrent, attributeList);
			} else {
				Y41bDokusiAdapterServiceImpl.logger.error(
						"Y41bDokusiAdapterServiceImpl.modificarDocumentoDokusi - Faltan Parametros para modificar un documento");
				throw new Y41bUDAException("error.dokusi.invocation", true, null);
			}

			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.modificarDocumentoDokusi - Return: " + resultado);

			documento.setOidDokusi(Y41bDokusiTransformer.transformModificarDocumentoDokusi(resultado));
			return documento;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_modificarDocumentoDokusi) {
			Y41bDokusiAdapterServiceImpl.logger.error("Y41bDokusiAdapterServiceImpl.modificarDocumentoDokusi",
					e_modificarDocumentoDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_modificarDocumentoDokusi);
		} finally {
			Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.modificarDocumentoDokusi - Final");
		}
	}

	/**
	 * Metodo que descarga el documento de DOKUSI
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para registrar el
	 *            documento en dokusi
	 * 
	 * @return Documento.
	 */
	public Documento descargarDocumentoDokusi(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.descargarDocumentoDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String procedureId = null;
			if (documento.getProcedure() != null) {
				procedureId = documento.getProcedure().getIdProcedimiento();
			}

			String auditUser = Y41bDokusiHelper.getAuditUser(procedureId, documento.getUser().getDni(),
					documento.getUser().getNombre());

			List<Metadato> metadatos = documento.getMetadatos();

			Metadato metadato = new Metadato();
			metadato.setNombre("object_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("title");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_serie");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre(Y41bConstantes.METADATO_ORIGEN);
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("i_chronicle_id");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_version_label");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("subject");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("keywords");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("authors");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_creation_date");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_modify_date");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_object_type");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("acl_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("owner_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_entrada_salida");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_num_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_formato_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_libro_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_registrado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_procedimiento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_expediente");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_entidad");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_departamento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_proc_infraestructura");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_tipo_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_num_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_ap1_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_ap2_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_archivado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_subsistema_archivo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_archivo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_cod_familia");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_idioma");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_documento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_naturaleza");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_digitalizacion");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_estado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_vigencia");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_anyo_ejercicio");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_firmacontenida");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_id_firmantes");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_nom_firmantes");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_infraestructura");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_editable");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_apoyo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre(Y41bConstantes.METADATO_CODIGO_VERIFICACION);
			metadatos.add(metadato);

			documento.setMetadatos(metadatos);

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			Formato formato = this.recuperarFormatosDocumentosDokusi(documento);

			String formatoName = formato.getName();
			String keysList = Y41bDokusiHelper.crearXMLReturnedAttributeKeysList(documento.getMetadatos());

			String pathTo = "/" + Y41bConstantes.COD_APLICACION.toLowerCase() + "/"
					+ this.appConfiguration.getProperty("PIF.CARPETA.DOKUSI");
			String content = Y41bDokusiHelper.crearXMLContentName(formatoName, pathTo);

			String resultado = this.dokusiService.descargarDocumentoDokusi(tokenSesion, auditUser, documentID, keysList,
					content);
			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.descargarDocumentoDokusi - Return: " + resultado);

			Documento documentoDescargado = Y41bDokusiTransformer.transformDescargarDocumentoDokusi(resultado);

			// DESCARGA EL DOCUMENTO DEL PIF
			BufferedInputStream bIS = this.y41bPifAdapterService.getFile(documentoDescargado.getPifId());

			// pasa a byte[]
			documentoDescargado.setBytes(Y41bUtilsWS.encode(Y41bDocumento.readDocumentContent(bIS)));

			// ELIMINA EL DOCUMENTO DEL PIF
			this.y41bPifAdapterService.deleteFile(documentoDescargado.getPifId());

			return documentoDescargado;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_descargarDocumentoDokusi) {
			this.logger.error("Y41bDokusiAdapterServiceImpl.descargarDocumentoDokusi", e_descargarDocumentoDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_descargarDocumentoDokusi);
		} finally {
			this.logger.info("Y41bDokusiAdapterServiceImpl.descargarDocumentoDokusi - Final");
		}
	}

	/**
	 * Metodo que descarga el documento de SIR
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para registrar el
	 *            documento en dokusi
	 * 
	 * @return Documento.
	 */
	public Documento descargarDocumentoSIR(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.descargarDocumentoSIR - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String procedureId = null;
			if (documento.getProcedure() != null) {
				procedureId = documento.getProcedure().getIdProcedimiento();
			}

			String auditUser = Y41bDokusiHelper.getAuditUser(procedureId, documento.getUser().getDni(),
					documento.getUser().getNombre());

			List<Metadato> metadatos = documento.getMetadatos();

			Metadato metadato = new Metadato();
			metadato.setNombre("object_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("title");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_serie");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre(Y41bConstantes.METADATO_ORIGEN);
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("i_chronicle_id");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_version_label");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("subject");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("keywords");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("authors");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_creation_date");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_modify_date");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_object_type");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("acl_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("owner_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_entrada_salida");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_formato_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_libro_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_registrado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_expediente");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_entidad");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_departamento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_proc_infraestructura");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_tipo_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_num_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_ap1_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_ap2_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_archivado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_subsistema_archivo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_archivo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_cod_familia");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_idioma");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_documento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_naturaleza");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_digitalizacion");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_estado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_vigencia");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_anyo_ejercicio");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_firmacontenida");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_id_firmantes");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_nom_firmantes");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_infraestructura");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_editable");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_apoyo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre(Y41bConstantes.METADATO_CODIGO_VERIFICACION);
			metadatos.add(metadato);

			documento.setMetadatos(metadatos);

			String documentID = Y41bDokusiHelper.crearXMLdocumentIDSIR(documento.getOidDokusi());

			String keysList = Y41bDokusiHelper.crearXMLReturnedAttributeKeysListSIR(documento.getMetadatos());

			String pathTo = "/" + Y41bConstantes.COD_APLICACION.toLowerCase() + "/"
					+ this.appConfiguration.getProperty("PIF.CARPETA.DOKUSI");

			String content = Y41bDokusiHelper.crearXMLContentNameSIR(pathTo);

			SirServiceImpl sirServiceImpl = new SirServiceImpl();
			String resultado = sirServiceImpl.retrieveRegistryDocuments(tokenSesion, auditUser, documentID, keysList,
					content);
			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.descargarDocumentoSIR - Return: " + resultado);

			Documento documentoDescargado = Y41bDokusiTransformer.transformDescargarDocumentoSIR(resultado);

			// DESCARGA EL DOCUMENTO DEL PIF
			BufferedInputStream bIS = this.y41bPifAdapterService.getFile(documentoDescargado.getPifId());

			// pasa a byte[]
			documentoDescargado.setBytes(Y41bUtilsWS.encode(Y41bDocumento.readDocumentContent(bIS)));

			// ELIMINA EL DOCUMENTO DEL PIF
			this.y41bPifAdapterService.deleteFile(documentoDescargado.getPifId());

			return documentoDescargado;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_descargarDocumentoDokusi) {
			this.logger.error("Y41bDokusiAdapterServiceImpl.descargarDocumentoSIR", e_descargarDocumentoDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_descargarDocumentoDokusi);
		} finally {
			this.logger.info("Y41bDokusiAdapterServiceImpl.descargarDocumentoSIR - Final");
		}
	}

	/**
	 * Metodo que descarga el documento de DOKUSI
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para registrar el
	 *            documento en dokusi
	 * 
	 * @return Documento.
	 */
	public Documento descargarDocumentoDokusiSinDocumento(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.descargarDocumentoDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String procedureId = null;
			if (documento.getProcedure() != null) {
				procedureId = documento.getProcedure().getIdProcedimiento();
			}

			String auditUser = Y41bDokusiHelper.getAuditUser(procedureId, documento.getUser().getDni(),
					documento.getUser().getNombre());

			List<Metadato> metadatos = documento.getMetadatos();

			Metadato metadato = new Metadato();
			metadato.setNombre("object_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("title");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_serie");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre(Y41bConstantes.METADATO_ORIGEN);
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("i_chronicle_id");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_version_label");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("subject");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("keywords");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("authors");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_creation_date");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_modify_date");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_object_type");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("acl_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("owner_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_entrada_salida");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_num_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_formato_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_libro_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_registrado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_procedimiento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_expediente");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_entidad");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_departamento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_proc_infraestructura");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_tipo_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_num_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_ap1_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_ap2_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_archivado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_subsistema_archivo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_archivo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_cod_familia");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_idioma");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_documento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_naturaleza");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_digitalizacion");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_estado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_vigencia");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_anyo_ejercicio");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_firmacontenida");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_id_firmantes");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_nom_firmantes");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_infraestructura");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_editable");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_apoyo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre(Y41bConstantes.METADATO_CODIGO_VERIFICACION);
			metadatos.add(metadato);

			documento.setMetadatos(metadatos);

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			Formato formato = this.recuperarFormatosDocumentosDokusi(documento);

			String formatoName = formato.getName();
			String keysList = Y41bDokusiHelper.crearXMLReturnedAttributeKeysList(documento.getMetadatos());

			String pathTo = "/" + Y41bConstantes.COD_APLICACION.toLowerCase() + "/"
					+ this.appConfiguration.getProperty("PIF.CARPETA.DOKUSI");
			String content = Y41bDokusiHelper.crearXMLContentName(formatoName, pathTo);

			String resultado = this.dokusiService.descargarDocumentoDokusi(tokenSesion, auditUser, documentID, keysList,
					content);
			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.descargarDocumentoDokusi - Return: " + resultado);

			Documento documentoDescargado = Y41bDokusiTransformer.transformDescargarDocumentoDokusi(resultado);

			return documentoDescargado;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_descargarDocumentoDokusi) {
			this.logger.error("Y41bDokusiAdapterServiceImpl.descargarDocumentoDokusi", e_descargarDocumentoDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_descargarDocumentoDokusi);
		} finally {
			this.logger.info("Y41bDokusiAdapterServiceImpl.descargarDocumentoDokusi - Final");
		}
	}

	/**
	 * Metodo que descarga el documento de DOKUSI y lo deja en PIF
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para registrar el
	 *            documento en dokusi
	 * 
	 * @return Documento.
	 */
	public Documento descargarPifDocumentoDokusi(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.descargarPifDocumentoDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String procedureId = null;
			if (documento.getProcedure() != null) {
				procedureId = documento.getProcedure().getIdProcedimiento();
			}

			String auditUser = Y41bDokusiHelper.getAuditUser(procedureId, documento.getUser().getDni(),
					documento.getUser().getNombre());

			List<Metadato> metadatos = documento.getMetadatos();

			Metadato metadato = new Metadato();
			metadato.setNombre("object_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("title");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_serie");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre(Y41bConstantes.METADATO_ORIGEN);
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("i_chronicle_id");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_version_label");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("subject");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("keywords");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("authors");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_creation_date");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_modify_date");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_object_type");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("acl_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("owner_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_entrada_salida");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_num_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_formato_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_libro_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_registrado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_procedimiento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_expediente");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_entidad");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_departamento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_proc_infraestructura");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_tipo_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_num_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_ap1_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_ap2_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_archivado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_subsistema_archivo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_archivo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_cod_familia");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_idioma");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_documento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_naturaleza");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_digitalizacion");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_estado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_vigencia");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_anyo_ejercicio");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_firmacontenida");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_id_firmantes");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_nom_firmantes");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_infraestructura");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_editable");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_apoyo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre(Y41bConstantes.METADATO_CODIGO_VERIFICACION);
			metadatos.add(metadato);

			documento.setMetadatos(metadatos);

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			Formato formato = this.recuperarFormatosDocumentosDokusi(documento);

			String formatoName = formato.getName();
			String keysList = Y41bDokusiHelper.crearXMLReturnedAttributeKeysList(documento.getMetadatos());

			String pathTo = "/" + Y41bConstantes.COD_APLICACION.toLowerCase() + "/"
					+ this.appConfiguration.getProperty("PIF.CARPETA.DOKUSI");
			String content = Y41bDokusiHelper.crearXMLContentName(formatoName, pathTo);

			String resultado = this.dokusiService.descargarDocumentoDokusi(tokenSesion, auditUser, documentID, keysList,
					content);
			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.descargarDocumentoDokusi - Return: " + resultado);

			Documento documentoDescargado = Y41bDokusiTransformer.transformDescargarDocumentoDokusi(resultado);

			return documentoDescargado;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_descargarDocumentoDokusi) {
			this.logger.error("Y41bDokusiAdapterServiceImpl.descargarPifDocumentoDokusi", e_descargarDocumentoDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_descargarDocumentoDokusi);
		} finally {
			this.logger.info("Y41bDokusiAdapterServiceImpl.descargarPifDocumentoDokusi - Final");
		}
	}

	/**
	 * Metodo que descarga el documento de DOKUSI y lo deja en PIF
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para registrar el
	 *            documento en dokusi
	 * @param pathTo
	 *            Ruta donde dejar el documento
	 * @return Documento.
	 */
	public Documento descargarPifDocumentoDokusiRuta(Documento documento, String pathTo) {
		Y41bDokusiAdapterServiceImpl.logger
				.info("Y41bDokusiAdapterServiceImpl.descargarPifDocumentoDokusiRuta - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String procedureId = null;
			if (documento.getProcedure() != null) {
				procedureId = documento.getProcedure().getIdProcedimiento();
			}

			String auditUser = Y41bDokusiHelper.getAuditUser(procedureId, documento.getUser().getDni(),
					documento.getUser().getNombre());

			List<Metadato> metadatos = documento.getMetadatos();

			Metadato metadato = new Metadato();
			metadato.setNombre("object_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("title");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_serie");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre(Y41bConstantes.METADATO_ORIGEN);
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("i_chronicle_id");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_version_label");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("subject");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("keywords");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("authors");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_creation_date");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_modify_date");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("r_object_type");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("acl_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("owner_name");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_entrada_salida");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_num_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_formato_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_libro_registro");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_registrado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_procedimiento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_expediente");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_entidad");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_departamento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_proc_infraestructura");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_tipo_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_num_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_ap1_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_ap2_titularoaport");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_archivado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_subsistema_archivo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_archivo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_cod_familia");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_idioma");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_documento");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_naturaleza");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_digitalizacion");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_estado");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_fecha_vigencia");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_anyo_ejercicio");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_firmacontenida");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_id_firmantes");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_nom_firmantes");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_infraestructura");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_editable");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre("ejgv_es_apoyo");
			metadatos.add(metadato);

			metadato = new Metadato();
			metadato.setNombre(Y41bConstantes.METADATO_CODIGO_VERIFICACION);
			metadatos.add(metadato);

			documento.setMetadatos(metadatos);

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			Formato formato = this.recuperarFormatosDocumentosDokusi(documento);

			String formatoName = formato.getName();
			String keysList = Y41bDokusiHelper.crearXMLReturnedAttributeKeysList(documento.getMetadatos());

			String content = Y41bDokusiHelper.crearXMLContentName(formatoName, pathTo);

			String resultado = this.dokusiService.descargarDocumentoDokusi(tokenSesion, auditUser, documentID, keysList,
					content);
			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.descargarPifDocumentoDokusiRuta - Return: " + resultado);

			Documento documentoDescargado = Y41bDokusiTransformer.transformDescargarDocumentoDokusi(resultado);

			return documentoDescargado;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_descargarPifDocumentoDokusiRuta) {
			this.logger.error("Y41bDokusiAdapterServiceImpl.descargarPifDocumentoDokusiRuta",
					e_descargarPifDocumentoDokusiRuta);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_descargarPifDocumentoDokusiRuta);
		} finally {
			this.logger.info("Y41bDokusiAdapterServiceImpl.descargarPifDocumentoDokusiRuta - Final");
		}
	}

	/**
	 * Metodo que elimina el documento de DOKUSI
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para eliminar el documento
	 *            en dokusi
	 * 
	 * @return RespuestaEliminarDocumento.
	 */
	public RespuestaEliminarDocumento eliminarDocumentoDokusi(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.eliminarDocumentoDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(documento.getProcedure().getIdProcedimiento(),
					documento.getUser().getDni(), documento.getUser().getNombre());

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			boolean boolAllVersions = Boolean.valueOf(documento.getBoolAllVersions());

			String resultado = this.dokusiService.eliminarDocumentoDokusi(tokenSesion, auditUser, documentID,
					boolAllVersions);
			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.eliminarDocumentoDokusi - Return: " + resultado);

			return Y41bDokusiTransformer.transformEliminarDocumentoDokusi(resultado);
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_eliminarDocumentoDokusi) {
			Y41bDokusiAdapterServiceImpl.logger.error("Y41bDokusiAdapterServiceImpl.eliminarDocumentoDokusi",
					e_eliminarDocumentoDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_eliminarDocumentoDokusi);
		} finally {
			Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.eliminarDocumentoDokusi - Final");
		}
	}

	/**
	 * Metodo que recupera las firmas de un documento de DOKUSI
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para eliminar el documento
	 *            en dokusi
	 * 
	 * @return Documento.
	 */
	public Documento recuperarFirmarDocumentosDokusi(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger
				.info("Y41bDokusiAdapterServiceImpl.recuperarFirmarDocumentosDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(documento.getProcedure().getIdProcedimiento(),
					documento.getUser().getDni(), documento.getUser().getNombre());

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			String format = Y41bDokusiHelper.crearXMLformat(documento);

			String resultado = this.dokusiService.recuperarFirmarDocumentosDokusi(tokenSesion, auditUser, documentID,
					format);

			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.recuperarFirmarDocumentosDokusi - Return: " + resultado);

			Documento documentoResultado = Y41bDokusiTransformer.transformRecuperarFirmarDocumentosDokusi(resultado);

			documentoResultado.setOidDokusi(documento.getOidDokusi());

			return documentoResultado;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_recuperarFirmarDocumentosDokusi) {
			Y41bDokusiAdapterServiceImpl.logger.error("Y41bDokusiAdapterServiceImpl.recuperarFirmarDocumentosDokusi",
					e_recuperarFirmarDocumentosDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_recuperarFirmarDocumentosDokusi);
		} finally {
			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.recuperarFirmarDocumentosDokusi - Final");
		}
	}

	/**
	 * Metodo que recupera las firmas de un documento de DOKUSI
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para eliminar el documento
	 *            en dokusi
	 * 
	 * @return String.
	 */
	public String recuperarFirmarDocumentosDokusiString(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger
				.info("Y41bDokusiAdapterServiceImpl.recuperarFirmarDocumentosDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(documento.getProcedure().getIdProcedimiento(),
					documento.getUser().getDni(), documento.getUser().getNombre());

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			String format = Y41bDokusiHelper.crearXMLformat(documento);

			String resultado = this.dokusiService.recuperarFirmarDocumentosDokusi(tokenSesion, auditUser, documentID,
					format);

			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.recuperarFirmarDocumentosDokusi - Return: " + resultado);

			return resultado;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_recuperarFirmarDocumentosDokusi) {
			Y41bDokusiAdapterServiceImpl.logger.error("Y41bDokusiAdapterServiceImpl.recuperarFirmarDocumentosDokusi",
					e_recuperarFirmarDocumentosDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_recuperarFirmarDocumentosDokusi);
		} finally {
			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.recuperarFirmarDocumentosDokusi - Final");
		}
	}

	/**
	 * Método que permite añadir firmas a un documento almacenado en DOKUSI
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para eliminar el documento
	 *            en dokusi
	 * 
	 * @return String.
	 */
	public String firmarDocumentosDokusi(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.firmarDocumentosDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(documento.getProcedure().getIdProcedimiento(),
					documento.getUser().getDni(), documento.getUser().getNombre());

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			Formato formato = this.recuperarFormatosDocumentosDokusi(documento);

			Documento documentoRdo = this.descargarPifDocumentoDokusi(documento);

			documentoRdo.setFirmas(documento.getFirmas());

			documentoRdo.setTsaSerialNumber(documento.getTsaSerialNumber());

			documentoRdo.setOidDokusi(documento.getOidDokusi());

			String signatures = Y41bDokusiHelper.crearXMLsignaturesNSHF(documentoRdo, formato);

			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.firmarDocumentosDokusi - signatures" + signatures);

			String resultado = this.dokusiService.firmarDocumentosDokusi(tokenSesion, auditUser, documentID,
					signatures);

			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.firmarDocumentosDokusi - Return: " + resultado);

			return resultado;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_firmarDocumentosDokusi) {
			Y41bDokusiAdapterServiceImpl.logger.error("Y41bDokusiAdapterServiceImpl.firmarDocumentosDokusi",
					e_firmarDocumentosDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_firmarDocumentosDokusi);
		} finally {
			Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.firmarDocumentosDokusi - Final");
		}
	}

	/**
	 * Método que incorpora un localizador a un documento de DOKUSI
	 * 
	 * @param localizadorDocumento
	 *            LocalizadorDocumento
	 * 
	 * @return RespuestaLocalizadorDocumento.
	 */
	public RespuestaLocalizadorDocumento incorporarLocalizadorDokusi(LocalizadorDocumento localizadorDocumento) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.incorporarLocalizadorDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(localizadorDocumento.getProcedure().getIdProcedimiento(),
					localizadorDocumento.getUser().getDni(), localizadorDocumento.getUser().getNombre());

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(localizadorDocumento.getOidDocumento());

			boolean boolVersion = localizadorDocumento.isBoolVersion();
			String boolReducir = localizadorDocumento.isBoolReducir() + "";
			String sede = localizadorDocumento.getSede();

			String resultado = this.dokusiService.incorporarLocalizadorDocumentoDokusi(tokenSesion, auditUser,
					documentID, boolVersion, boolReducir, sede);
			RespuestaLocalizadorDocumento respuestaLocalizadorDocumento = Y41bDokusiTransformer
					.transformIncorporarLocalizadorDokusi(resultado);
			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.incorporarLocalizadorDokusi - Return: " + resultado);
			return respuestaLocalizadorDocumento;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_incorporarLocalizadorDokusi) {
			String error_msge = "error.dokusi.invocation";
			if (e_incorporarLocalizadorDokusi.getMessage().contains("t65bFSDWar-addLocatorToDocument-10506")) {
				error_msge = "error.dokusi.t65bFSDWar.addLocatorToDocument.10506";
			} else if (e_incorporarLocalizadorDokusi.getMessage().contains("t65bFSDWar-addLocatorToDocument-10504")) {
				error_msge = "error.dokusi.t65bFSDWar.addLocatorToDocument.10504";
			}
			throw new Y41bUDAException(error_msge, true, e_incorporarLocalizadorDokusi);
		} finally {
			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.incorporarLocalizadorDokusi - Final");
		}
	}

	/**
	 * Metodo que transforma un documento de DOKUSI
	 * 
	 * @param transformarDocumento
	 *            TransformarDocumento
	 * 
	 * @return RespuestaTransformarDocumento.
	 */
	public RespuestaTransformarDocumento transformarDocumentoDokusi(TransformarDocumento transformarDocumento) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.transformarDocumentoDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(transformarDocumento.getProcedure().getIdProcedimiento(),
					transformarDocumento.getUser().getDni(), transformarDocumento.getUser().getNombre());

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(transformarDocumento.getOidDocumento());

			String format = Y41bDokusiHelper.crearXMLformatTransformacion();

			String resultado = this.dokusiService.transformarDocumentoDokusi(tokenSesion, auditUser, documentID,
					format);
			RespuestaTransformarDocumento respuestaTransformarDocumento = Y41bDokusiTransformer
					.transformTransformarDocumentoDokusi(resultado);
			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.transformarDocumentoDokusi - Return: " + resultado);
			return respuestaTransformarDocumento;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_incorporarLocalizadorDokusi) {
			String error_msge = "error.dokusi.invocation";

			throw new Y41bUDAException(error_msge, true, e_incorporarLocalizadorDokusi);
		} finally {
			Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.transformarDocumentoDokusi - Final");
		}
	}

	/**
	 * Metodo que recupera los formatos de un documento de DOKUSI
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para recuperar el formato
	 *            del documento en dokusi
	 * 
	 * @return Formato.
	 */
	public Formato recuperarFormatosDocumentosDokusi(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger
				.info("Y41bDokusiAdapterServiceImpl.recuperarFormatosDocumentosDokusi - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));

			String procedureId = null;
			if (documento.getProcedure() != null) {
				procedureId = documento.getProcedure().getIdProcedimiento();
			}

			String auditUser = Y41bDokusiHelper.getAuditUser(procedureId, documento.getUser().getDni(),
					documento.getUser().getNombre());

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			String resultado = this.dokusiService.recuperarFormatosDocumentosDokusi(tokenSesion, auditUser, documentID);

			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.recuperarFormatosDocumentosDokusi - Return: " + resultado);

			Formato formato = Y41bDokusiTransformer.transformRecuperarFormatosDocumentosDokusi(resultado);

			return formato;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_recuperarFirmarDocumentosDokusi) {
			Y41bDokusiAdapterServiceImpl.logger.error("Y41bDokusiAdapterServiceImpl.recuperarFirmarDocumentosDokusi",
					e_recuperarFirmarDocumentosDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_recuperarFirmarDocumentosDokusi);
		} finally {
			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.recuperarFirmarDocumentosDokusi - Final");
		}
	}

	/**
	 * Metodo que genera un nuevo documento copia que comparte el contenido del
	 * original.
	 * 
	 * @param documento
	 *            Objeto con los atributos necesarios para compartir el
	 *            documento en dokusi
	 * 
	 * @return Documento.
	 */
	public Documento shareDocumentContents(Documento documento) {
		Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.shareDocumentContents - Inicio");
		try {
			String tokenSesion = Y41bUtils.getInstance().getTokenAplicacion(
					Boolean.parseBoolean(this.appConfiguration.getProperty("xlnets.loginApp.isSimulated")));
			String auditUser = Y41bDokusiHelper.getAuditUser(documento.getProcedure().getIdProcedimiento(),
					documento.getUser().getDni(), documento.getUser().getNombre());

			String documentID = Y41bDokusiHelper.crearXMLdocumentID(documento.getOidDokusi());

			String respuesta;
			if (Y41bConstantes.ID_PROCEDIMIENTO_ESTABLECIMIENTOS
					.equalsIgnoreCase(documento.getProcedure().getIdProcedimiento())) {
				respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
						Y41bConstantes.ID_PROCEDIMIENTO_ADHALTA);
			} else {
				respuesta = plateaService.obtenerDatosCatalogoProcedimientos(tokenSesion,
						documento.getProcedure().getIdProcedimiento());
			}

			ProcedureDocument procedureDocument = ProcedureDocument.Factory.parse(respuesta.toString());

			String attributeList = Y41bDokusiHelper.crearXMLAttributeList(documento, procedureDocument);

			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.shareDocumentContents - Inicio llamada Servicio");
			String resultado = this.dokusiService.shareDocumentContents(tokenSesion, auditUser, documentID,
					attributeList);
			Y41bDokusiAdapterServiceImpl.logger
					.info("Y41bDokusiAdapterServiceImpl.shareDocumentContents - Fin llamada Servicio");

			Y41bDokusiAdapterServiceImpl.logger
					.debug("Y41bDokusiAdapterServiceImpl.shareDocumentContents - Return: " + resultado);

			documento.setOidDokusi(Y41bDokusiTransformer.transformShareDocumentContents(resultado));

			return documento;
		} catch (Y41bUDAException e) {
			throw e;
		} catch (Exception e_registrarDocumentoDokusi) {
			Y41bDokusiAdapterServiceImpl.logger.error("Y41bDokusiAdapterServiceImpl.shareDocumentContents",
					e_registrarDocumentoDokusi);
			throw new Y41bUDAException("error.dokusi.invocation", true, e_registrarDocumentoDokusi);

		} finally {
			Y41bDokusiAdapterServiceImpl.logger.info("Y41bDokusiAdapterServiceImpl.shareDocumentContents - Final");
		}
	}

}