package com.ejie.y41e.control;

import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.ejie.x38.control.bind.annotation.RequestJsonBody;
import com.ejie.x38.dto.JQGridRequestDto;
import com.ejie.x38.dto.JQGridResponseDto;
import com.ejie.x38.dto.TableRowDto;
import com.ejie.y41e.model.Y41eEncuesta;
import com.ejie.y41e.model.Y41eEnvio;
import com.ejie.y41e.model.Y41eGrafico;
import com.ejie.y41e.model.Y41eListGrafico;
import com.ejie.y41e.model.Y41eResultados;
import com.ejie.y41e.model.excel.Y41eExcelModel;
import com.ejie.y41e.service.Y41eExcelReportService;
import com.ejie.y41e.service.Y41eResultadosService;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * Y41eResultadosController
 *  
 */

@Controller
@RequestMapping (value = "/resultados")

public class Y41eResultadosController  {

    private static final Logger LOGGER = LoggerFactory.getLogger(Y41eResultadosController.class);

    @Autowired
    private Y41eResultadosService y41eResultadosService;
    @Autowired
    private Y41eExcelReportService y41eExcelReportService;


    /**
     * Metodo de presentacion del RUP_TABLE.
     * 
     * @param model Model
     * @return String
     */
    @RequestMapping(value = "/encuesta/maint", method = RequestMethod.GET)
    public String getFormEditEncuesta(Model model) {
        Y41eResultadosController.LOGGER.info("[GET - View] : resulEncuesta");
        return "resulEncuesta";
    }

    /**
     * Operacion de filtrado del componente RUP_TABLE.
     * 
     * @param filterResultados Resultados
     *            Bean que contiene los parametros de filtrado a emplear.
     * @param jqGridRequestDto
     *            Dto que contiene los parametros de configuracion propios del
     *            RUP_TABLE a aplicar en el filtrado.
     * @return JQGridResponseDto<Resultados>
     *            Dto que contiene el resultado del filtrado realizado por el 
     *            componente RUP_TABLE.
     */
    @RequestMapping(value = "/encuesta/filter", method = RequestMethod.POST)
    @ResponseBody public JQGridResponseDto<Y41eEncuesta> filterEncuesta(
            @RequestJsonBody(param="filter") Y41eEncuesta filterEncuesta,
            @RequestJsonBody JQGridRequestDto jqGridRequestDto) {
        Y41eResultadosController.LOGGER.info("[POST - filter] : Obtener Resultados encuesta");
        return this.y41eResultadosService.filterEncuestas(filterEncuesta, jqGridRequestDto, false);
    }

    @RequestMapping(value = "/encuesta/grafico", method = RequestMethod.POST)
    public String getGraficos(@RequestJsonBody() Y41eResultados resultado, Model model) {
        Y41eResultadosController.LOGGER.info("[GET - View] : getGraficos");

        List<Y41eGrafico> listadoGraficos = this.y41eResultadosService.getGraficos(resultado);
        Y41eListGrafico listGrafico = new Y41eListGrafico();
        listGrafico.setListGrafico(listadoGraficos);
        ObjectMapper objectMapper = new ObjectMapper();
        String listGraficoJson="";
        try {
            listGraficoJson = objectMapper.writeValueAsString(listadoGraficos);
        } catch (JsonProcessingException e) {
            Y41eResultadosController.LOGGER.error("Error al serializar los gráficos", e);
        }
        model.addAttribute("listGrafico", listGrafico);
        model.addAttribute("listGraficoJson", listGraficoJson);
        return "graficosEncuesta";
    }

    /**
     * Metodo de presentacion del RUP_TABLE.
     * 
     * @param model Model
     * @return String
     */
    @RequestMapping(value = "/destinatario/maint", method = RequestMethod.GET)
    public String getFormEditDestinatario(Model model) {
        Y41eResultadosController.LOGGER.info("[GET - View] : resulDestinatario");
        return "resulDestinatario";
    }

    /**
     * Operacion de filtrado del componente RUP_TABLE.
     * 
     * @param filterResultados Resultados
     *            Bean que contiene los parametros de filtrado a emplear.
     * @param jqGridRequestDto
     *            Dto que contiene los parametros de configuracion propios del
     *            RUP_TABLE a aplicar en el filtrado.
     * @return JQGridResponseDto<Resultados>
     *            Dto que contiene el resultado del filtrado realizado por el 
     *            componente RUP_TABLE.
     */
    @RequestMapping(value = "/destinatario/filter", method = RequestMethod.POST)
    @ResponseBody public JQGridResponseDto<Y41eEnvio> filterDestinatario(
            @RequestJsonBody(param="filter") Y41eEnvio filter,
            @RequestJsonBody JQGridRequestDto jqGridRequestDto) {
        Y41eResultadosController.LOGGER.info("[POST - filter] : Obtener Resultados destinatario");
        return this.y41eResultadosService.filterDestinatario(filter, jqGridRequestDto, false);
    }



    /**
     * Genera el excel con los resultados del destinatario
     * @param id Integer
     * @param request HttpServletRequest
     * @param response HttpServletResponse
     * @param model Model
     */
    @RequestMapping(value = "/destinatario/excel", method = RequestMethod.POST)
    public void excelDestinatario(@RequestParam() Long id, HttpServletRequest request, 
            HttpServletResponse response, Model model) {
        Y41eResultadosController.LOGGER.info("[POST - filter] : excelDestinatario");
        Y41eEnvio envio = new Y41eEnvio(id);
        final Y41eExcelModel excelModel= this.y41eResultadosService.getExcelResultadosDestinatario(envio);
        this.y41eExcelReportService.generarExcel("resultados", excelModel, response);
    }

    /**
     * Operacion CRUD Read. Devuelve el bean correspondiente al identificador indicado.
     * 
     * @param idEnvio Long
     * @param idPregunta Long
     * @return Resultados 
     *            Objeto correspondiente al identificador indicado.
     */
    @RequestMapping(value = "/{idEnvio}", method = RequestMethod.GET)
    @ResponseBody public Y41eResultados get(@PathVariable Long idEnvio, @PathVariable Long idPregunta) {
        Y41eResultados resultados = new Y41eResultados();
        resultados.getEnvio().setIdEnvio(idEnvio);
        resultados.getPregunta().setIdPregunta(idPregunta);
        resultados = this.y41eResultadosService.find(resultados);
        Y41eResultadosController.LOGGER.info("[GET - findBy_PK] : Obtener Resultados por PK");
        return resultados;
    }

    /**
     * Devuelve una lista de beans correspondientes a los valores de filtrados
     * indicados en el objeto pasado como parametro.
     *
     * @param filterResultados Resultados
     *            Objeto que contiene los parametros de filtrado utilizados en
     *            la busqueda.
     * @return List<Resultados> 
     *            Lista de objetos correspondientes a la busqueda realizada.
     */
    @RequestMapping(method = RequestMethod.GET)
    @ResponseBody public List<Y41eResultados> getAll(@ModelAttribute Y41eResultados filterResultados) {
        Y41eResultadosController.LOGGER.info("[GET - find_ALL] : Obtener Resultados por filtro");
        return this.y41eResultadosService.findAll(filterResultados, null);
    }

    /**
     * Operacion CRUD Edit. Modificacion del bean indicado.
     *
     * @param resultados Resultados 
     *            Bean que contiene la informacion a modificar.
     * @return Resultados 
     *            Bean resultante de la modificacion.
     */
    @RequestMapping(method = RequestMethod.PUT)
    @ResponseBody public Y41eResultados edit(@RequestBody Y41eResultados resultados) {
        Y41eResultados resultadosAux = this.y41eResultadosService.update(resultados);
        Y41eResultadosController.LOGGER.info("[PUT] : Resultados actualizado correctamente");
        return resultadosAux;
    }

    /**
     * Operacion CRUD Create. Creacion de un nuevo registro a partir del bean
     * indicado.
     *
     * @param resultados Resultados 
     *            Bean que contiene la informacion con la que se va a crear el
     *            nuevo registro.
     * @return Resultados
     *            Bean resultante del proceso de creacion.
     */
    @RequestMapping(method = RequestMethod.POST)
    @ResponseBody public Y41eResultados add(@RequestBody Y41eResultados resultados) {
        Y41eResultados resultadosAux = this.y41eResultadosService.add(resultados);
        Y41eResultadosController.LOGGER.info("[POST] : Resultados insertado correctamente");
        return resultadosAux;
    }

    /**
     * Operacion CRUD Delete. Borrado del registro correspondiente al
     * identificador especificado.
     *
     * @param idEnvio Long
     *            Identificador del objeto que se desea eliminar.
     * @param idPregunta Long
     * @return Resultados
     *            Bean eliminado.
     */
    @RequestMapping(value = "/{idEnvio}", method = RequestMethod.DELETE)
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody public Y41eResultados remove(@PathVariable Long idEnvio, @PathVariable Long idPregunta) {
        Y41eResultados resultados = new Y41eResultados();
        resultados.getEnvio().setIdEnvio(idEnvio);
        resultados.getPregunta().setIdPregunta(idPregunta);
        this.y41eResultadosService.remove(resultados);
        Y41eResultadosController.LOGGER.info("[DELETE] : Resultados borrado correctamente");
        return resultados;
    }

    /**
     * Operacion de busqueda del componente RUP_TABLE.
     * 
     * @param filterResultados Resultados
     *            Bean que contiene los parametros de filtrado a emplear.
     * @param searchResultados Resultados
     *            Bean que contiene los parametros de busqueda a emplear.
     * @param jqGridRequestDto
     *            Dto que contiene los parametros de configuracion propios del
     *            RUP_TABLE a aplicar en la búsqueda.
     * @return TableRowDto<Resultados> 
     *            Dto que contiene el resultado de la busqueda realizada por el
     *            componente RUP_TABLE. 
     */
    @RequestMapping(value = "/search", method = RequestMethod.POST)
    @ResponseBody public List<TableRowDto<Y41eResultados>> search(
            @RequestJsonBody(param="filter") Y41eResultados filterResultados,
            @RequestJsonBody(param="search") Y41eResultados searchResultados,
            @RequestJsonBody JQGridRequestDto jqGridRequestDto) {
        Y41eResultadosController.LOGGER.info("[POST - search] : Buscar Resultadoss");
        return this.y41eResultadosService.search(filterResultados, searchResultados, jqGridRequestDto, false);
    }

    /**
     * Borrado multiple de registros
     * 
     * @param filterResultados Resultados
     *            Bean que contiene los parametros de filtrado a emplear.
     * @param jqGridRequestDto
     *            Dto que contiene los parametros de configuracion propios del
     *            RUP_TABLE a aplicar en la busqueda.
     * @return List<String>
     *            Lista de los identificadores de los registros eliminados.
     * 
     */
    @RequestMapping(value = "/deleteAll", method = RequestMethod.POST)
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody public List<String> removeMultiple(
            @RequestJsonBody(param="filter") Y41eResultados filterResultados,
            @RequestJsonBody JQGridRequestDto jqGridRequestDto) {
        Y41eResultadosController.LOGGER.info("[POST - search] : [POST - removeMultiple] : Eliminar multiples Resultadoss");
        this.y41eResultadosService.removeMultiple(filterResultados, jqGridRequestDto, false);
        Y41eResultadosController.LOGGER.info("All entities correctly deleted!");

        return jqGridRequestDto.getMultiselection().getSelectedIds();
    }

}
