package com.ejie.y41e.control;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.ejie.x38.control.bind.annotation.RequestJsonBody;
import com.ejie.x38.dto.JQGridRequestDto;
import com.ejie.x38.dto.JQGridResponseDto;
import com.ejie.y41e.common.Y41eConstants;
import com.ejie.y41e.model.Y41ePregunta;
import com.ejie.y41e.service.Y41ePreguntaService;
import com.ejie.y41e.service.Y41eRelBloquePreguntaService;

/**
 * Y41ePreguntaController
 *  
 */

@Controller
@RequestMapping (value = "/pregunta")

public class Y41ePreguntaController  {

    private static final Logger LOGGER = LoggerFactory.getLogger(Y41ePreguntaController.class);

    @Autowired
    private Y41ePreguntaService y41ePreguntaService;
    @Autowired
    private Y41eRelBloquePreguntaService y41eRelBloquePreguntaService;


    /**
     * Comprueba que no existe ninguna encuesta enviada o respondida para el id de la pregunta.
     * 
     * @param idPregunta Long
     * @return String el area a la que pertence la encuesta 
     *            
     */
    @RequestMapping(value = "/validarPregunta/{idPregunta}", method = RequestMethod.GET)
    @ResponseBody public String validarPregunta(@PathVariable Long idPregunta) {
        Y41ePreguntaController.LOGGER.info("validarPregunta");
        Integer encuestaEnviada = this.y41ePreguntaService.validarPreguntaEnviada(idPregunta);
        if(encuestaEnviada>0){
            return Y41eConstants.ERROR_ENCUESTA_ENVIADA;
        }
        Integer tieneEncuesta = this.y41ePreguntaService.validarPreguntaEncuesta(idPregunta);
        if(tieneEncuesta>0){
            return Y41eConstants.ERROR_TIENE_ENCUESTA;
        }
        return "";
    }

    /**
     * Comprueba que no existe ninguna encuesta enviada o respondida para el id de la pregunta.
     * 
     * @param idPregunta Long
     * @return String el area a la que pertence la encuesta 
     *            
     */
    @RequestMapping(value = "/validarDesasociar/{idBloque}", method = RequestMethod.GET)
    @ResponseBody public String validarDesasociar(@PathVariable Long idBloque) {
        Y41ePreguntaController.LOGGER.info("validarDesasociar");
        Integer encuestaEnviada = this.y41ePreguntaService.validarDesasociarEnviada(idBloque);
        if(encuestaEnviada>0){
            return Y41eConstants.ERROR_ENCUESTA_ENVIADA;
        }
        Integer tieneEncuesta = this.y41ePreguntaService.validarDesasociarEncuesta(idBloque);
        if(tieneEncuesta>0){
            return Y41eConstants.ERROR_TIENE_ENCUESTA;
        }
        return "";
    }

    /**
     * Operacion CRUD Read. Devuelve el bean correspondiente al identificador indicado.
     * 
     * @param idPregunta Long
     * @return Y41ePregunta 
     *            Objeto correspondiente al identificador indicado.
     */
    @RequestMapping(value = "/{idPregunta}", method = RequestMethod.GET)
    @ResponseBody public Y41ePregunta get(@PathVariable Long idPregunta) {
        Y41ePregunta y41ePregunta = new Y41ePregunta();
        y41ePregunta.setIdPregunta(idPregunta);
        y41ePregunta = this.y41ePreguntaService.find(y41ePregunta);
        Y41ePreguntaController.LOGGER.info("[GET - findBy_PK] : Obtener Y41ePregunta por PK");
        return y41ePregunta;
    }

    /**
     * Devuelve una lista de beans correspondientes a los valores de filtrados
     * indicados en el objeto pasado como parametro.
     *
     * @param filterY41ePregunta Y41ePregunta
     *            Objeto que contiene los parametros de filtrado utilizados en
     *            la busqueda.
     * @return List<Y41ePregunta> 
     *            Lista de objetos correspondientes a la busqueda realizada.
     */
    @RequestMapping(method = RequestMethod.GET)
    @ResponseBody public List<Y41ePregunta> getAll(@ModelAttribute Y41ePregunta filterY41ePregunta) {
        Y41ePreguntaController.LOGGER.info("[GET - find_ALL] : Obtener Y41ePregunta por filtro");
        return this.y41ePreguntaService.findAll(filterY41ePregunta, null);
    }

    /**
     * Operacion CRUD Edit. Modificacion del bean indicado.
     *
     * @param y41ePregunta Y41ePregunta 
     *            Bean que contiene la informacion a modificar.
     * @return Y41ePregunta 
     *            Bean resultante de la modificacion.
     */
    @RequestMapping(method = RequestMethod.PUT)
    @ResponseBody public Y41ePregunta edit(@RequestBody Y41ePregunta y41ePregunta) {
        if(y41ePregunta.getTipoPregunta().getIdTipoPregunta()==Integer.valueOf(Y41eConstants.TIPO_RESPUESTA_TEXTO).intValue()){
            y41ePregunta.setAviso(Y41eConstants.NO);
        }
        Y41ePregunta y41ePreguntaAux = this.y41ePreguntaService.update(y41ePregunta);
        Y41ePreguntaController.LOGGER.info("[PUT] : Y41ePregunta actualizado correctamente");
        return y41ePreguntaAux;
    }

    /**
     * Operacion CRUD Create. Creacion de un nuevo registro a partir del bean
     * indicado.
     *
     * @param y41ePregunta Y41ePregunta 
     *            Bean que contiene la informacion con la que se va a crear el
     *            nuevo registro.
     * @return Y41ePregunta
     *            Bean resultante del proceso de creacion.
     */
    @RequestMapping(method = RequestMethod.POST)
    @ResponseBody public Y41ePregunta add(@RequestBody Y41ePregunta y41ePregunta) {
        if(y41ePregunta.getTipoPregunta().getIdTipoPregunta()==Integer.valueOf(Y41eConstants.TIPO_RESPUESTA_TEXTO).intValue()){
            y41ePregunta.setAviso(Y41eConstants.NO);
        }
        Y41ePregunta y41ePreguntaAux = this.y41ePreguntaService.add(y41ePregunta);
        Y41ePreguntaController.LOGGER.info("[POST] : Y41ePregunta insertado correctamente");
        return y41ePreguntaAux;
    }

    /**
     * Añade la pregunta y se la asigna al bloque
     *
     * @param y41ePregunta Y41ePregunta 
     *            Bean que contiene la informacion con la que se va a crear el
     *            nuevo registro.
     * @return Y41ePregunta
     *            Bean resultante del proceso de creacion.
     */
    @RequestMapping(value = "/addPreguntaBloque", method = RequestMethod.POST)
    @ResponseBody public void addPreguntaBloque(@RequestBody Y41ePregunta y41ePregunta) {
        Y41ePreguntaController.LOGGER.info("[POST] : addPreguntaBloque");
        this.y41ePreguntaService.addPreguntaBloque(y41ePregunta);
    }

    /**
     * Asocia las preguntas seleccionadas al bloque
     * @param filterY41ePregunta Y41ePregunta
     * @param selectedIds List<String>
     * @param selectedAll Boolean
     * @param locale Locale
     * @return String
     * @throws Exception e
     */
    @RequestMapping(value = "/asosiarPreguntas", method = RequestMethod.POST)
    @ResponseBody() public void asosiarPreguntas(
            @RequestJsonBody(param="filter") Y41ePregunta filterY41ePregunta,
            @RequestJsonBody(param = "selectedIds") List<String> selectedIds,
            @RequestJsonBody(param = "selectedAll") Boolean selectedAll) {
        Y41ePreguntaController.LOGGER.info("[POST] : asosiarPreguntas");
        filterY41ePregunta.setSelectedAll(selectedAll);
        filterY41ePregunta.setSelectedIds(selectedIds);
        List<Y41ePregunta> listaPreguntasSeleccionadas = this.y41ePreguntaService.findAllLike(filterY41ePregunta, null, false);
        y41eRelBloquePreguntaService.asosiarPreguntas(filterY41ePregunta.getIdBloque(), listaPreguntasSeleccionadas);
    }

    /**
     * Operacion CRUD Delete. Borrado del registro correspondiente al
     * identificador especificado.
     *
     * @param idPregunta Long
     *            Identificador del objeto que se desea eliminar.
     * @return Y41ePregunta
     *            Bean eliminado.
     */
    @RequestMapping(value = "/{idPregunta}", method = RequestMethod.DELETE)
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody public Y41ePregunta remove(@PathVariable Long idPregunta) {
        Y41ePregunta y41ePregunta = new Y41ePregunta();
        y41ePregunta.setIdPregunta(idPregunta);
        this.y41ePreguntaService.remove(y41ePregunta);
        Y41ePreguntaController.LOGGER.info("[DELETE] : Y41ePregunta borrado correctamente");
        return y41ePregunta;
    }

    /**
     * Metodo de presentacion del RUP_TABLE.
     * 
     * @param model Model
     * @return String
     */
    @RequestMapping(value = "/maint", method = RequestMethod.GET)
    public String getFormEdit(Model model) {
        Y41ePreguntaController.LOGGER.info("[GET - View] : y41epregunta");
        return "pregunta";
    }

    /**
     * Operacion de filtrado del componente RUP_TABLE.
     * 
     * @param filterY41ePregunta Y41ePregunta
     *            Bean que contiene los parametros de filtrado a emplear.
     * @param jqGridRequestDto
     *            Dto que contiene los parametros de configuracion propios del
     *            RUP_TABLE a aplicar en el filtrado.
     * @return JQGridResponseDto<Y41ePregunta>
     *            Dto que contiene el resultado del filtrado realizado por el 
     *            componente RUP_TABLE.
     */
    @RequestMapping(value = "/filter", method = RequestMethod.POST)
    @ResponseBody public JQGridResponseDto<Y41ePregunta> filter(
            @RequestJsonBody(param="filter") Y41ePregunta filterY41ePregunta,
            @RequestJsonBody JQGridRequestDto jqGridRequestDto) {
        Y41ePreguntaController.LOGGER.info("[POST - filter] : Obtener Y41ePreguntas");
        return this.y41ePreguntaService.filter(filterY41ePregunta, jqGridRequestDto, false);
    }

    /**
     * Obtiene el suggest de bloques
     * @param q String
     * @param idEncuesta Long
     * @param locale Locale
     * @return List<idBloque>
     */
    @RequestMapping(value = "/suggest", method = RequestMethod.GET)
    @ResponseBody() public List<Y41ePregunta> getSuggestPregunta(@RequestParam() String q, Long idBloque, String tipo) {
        Y41ePreguntaController.LOGGER.info("[GET - getSuggestPregunta]");
        Y41ePregunta pregunta = new Y41ePregunta();
        pregunta.setActiva(Y41eConstants.SI);
        pregunta.setIdBloque(idBloque);
        if(Y41eConstants.DESC_PUBLICA.equals(tipo)){
            pregunta.setDescPubl(q);
        }else{
            pregunta.setDescInt(q);
        }
        return this.y41ePreguntaService.findAllSuggest(pregunta, false);
    }
}
