package com.ejie.y41e.control;

import java.util.List;
import java.util.Locale;
import java.util.Properties;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.ejie.x38.control.bind.annotation.RequestJsonBody;
import com.ejie.x38.dto.JQGridRequestDto;
import com.ejie.x38.dto.JQGridResponseDto;
import com.ejie.y41e.common.Y41eConstants;
import com.ejie.y41e.model.Y41eEncuesta;
import com.ejie.y41e.model.excel.Y41eExcelModel;
import com.ejie.y41e.service.Y41eEncuestaService;
import com.ejie.y41e.service.Y41eExcelReportService;
import com.ejie.y41e.utils.Y41eUtils;

/**
 * Y41eEncuestaController
 *  
 */

@Controller
@RequestMapping (value = "/encuesta")

public class Y41eEncuestaController  {

    private static final Logger LOGGER = LoggerFactory.getLogger(Y41eEncuestaController.class);

    @Autowired
    private Y41eEncuestaService y41eEncuestaService;
    @Autowired
    private Y41eExcelReportService y41eExcelReportService;
    @Autowired()
    private Properties appConfiguration;

    /**
     * Metodo de presentacion del RUP_TABLE.
     * 
     * @param model Model
     * @return String
     */
    @RequestMapping(value = "/busqEncuestas/maint", method = RequestMethod.GET)
    public String getBusqueda(Model model) {
        Y41eEncuestaController.LOGGER.info("[GET - View] : busqEncuestas");
        Locale locale = LocaleContextHolder.getLocale();
        model.addAttribute("LOCALE_CAMEL", StringUtils.capitalize(locale.getLanguage()));
        model.addAttribute("MODO", Y41eConstants.DETALLE);

        return "busqEncuestas";
    }

    /**
     * Detalle de la encuesta
     * 
     * @param model Model
     * @return String
     */
    @RequestMapping(value = "/detalleEncuesta/maint", method = RequestMethod.GET)
    public String getDetalleEncuesta(Model model) {
        Y41eEncuestaController.LOGGER.info("[GET - View] : detalleEncuesta");
        Locale locale = LocaleContextHolder.getLocale();
        model.addAttribute("LOCALE_CAMEL", StringUtils.capitalize(locale.getLanguage()));
        model.addAttribute("MODO", Y41eConstants.ALTA);

        return "detalleEncuesta";
    }

    /**
     * Operacion de filtrado del componente RUP_TABLE.
     * 
     * @param filterY41eEncuesta Y41eEncuesta
     *            Bean que contiene los parametros de filtrado a emplear.
     * @param jqGridRequestDto
     *            Dto que contiene los parametros de configuracion propios del
     *            RUP_TABLE a aplicar en el filtrado.
     * @return JQGridResponseDto<Y41eEncuesta>
     *            Dto que contiene el resultado del filtrado realizado por el 
     *            componente RUP_TABLE.
     */
    @RequestMapping(value = "/busqEncuestas/filter", method = RequestMethod.POST)
    @ResponseBody public JQGridResponseDto<Y41eEncuesta> filter(
            @RequestJsonBody(param="filter") Y41eEncuesta filterY41eEncuesta,
            @RequestJsonBody JQGridRequestDto jqGridRequestDto) {
        Y41eEncuestaController.LOGGER.info("[POST - filter] : Obtener Y41eEncuestas");
        return this.y41eEncuestaService.filter(filterY41eEncuesta, jqGridRequestDto, false);
    }

    /**
     * Operacion CRUD Read. Devuelve el bean correspondiente al identificador indicado.
     * 
     * @param idEncuesta Long
     * @return Y41eEncuesta 
     *            Objeto correspondiente al identificador indicado.
     */
    @RequestMapping(value = "/{idEncuesta}", method = RequestMethod.GET)
    @ResponseBody public Y41eEncuesta get(@PathVariable Long idEncuesta) {
        Y41eEncuesta y41eEncuesta = new Y41eEncuesta();
        y41eEncuesta.setIdEncuesta(idEncuesta);
        y41eEncuesta = this.y41eEncuestaService.find(y41eEncuesta);
        String urlEncuesta = this.appConfiguration.getProperty("url.encuesta");
        y41eEncuesta.setUrl(urlEncuesta.replace("{0}", y41eEncuesta.getUrl()));
        Y41eEncuestaController.LOGGER.info("[GET - findBy_PK] : Obtener Y41eEncuesta por PK");
        return y41eEncuesta;
    }

    /**
     * Devuelve una lista de beans correspondientes a los valores de filtrados
     * indicados en el objeto pasado como parametro.
     *
     * @param filterY41eEncuesta Y41eEncuesta
     *            Objeto que contiene los parametros de filtrado utilizados en
     *            la busqueda.
     * @return List<Y41eEncuesta> 
     *            Lista de objetos correspondientes a la busqueda realizada.
     */
    @RequestMapping(method = RequestMethod.GET)
    @ResponseBody public List<Y41eEncuesta> getAll(@ModelAttribute Y41eEncuesta filterY41eEncuesta) {
        Y41eEncuestaController.LOGGER.info("[GET - find_ALL] : Obtener Y41eEncuesta por filtro");
        return this.y41eEncuestaService.findAll(filterY41eEncuesta, null);
    }

    /**
     * Operacion CRUD Edit. Modificacion del bean indicado.
     *
     * @param y41eEncuesta Y41eEncuesta 
     *            Bean que contiene la informacion a modificar.
     * @return Y41eEncuesta 
     *            Bean resultante de la modificacion.
     */
    @RequestMapping(method = RequestMethod.PUT)
    @ResponseBody public Y41eEncuesta edit(@RequestBody Y41eEncuesta y41eEncuesta) {
        Y41eEncuesta y41eEncuestaAux = this.y41eEncuestaService.update(y41eEncuesta);
        Y41eEncuestaController.LOGGER.info("[PUT] : Y41eEncuesta actualizado correctamente");
        return y41eEncuestaAux;
    }

    /**
     * Operacion CRUD Create. Creacion de un nuevo registro a partir del bean
     * indicado.
     *
     * @param y41eEncuesta Y41eEncuesta 
     *            Bean que contiene la informacion con la que se va a crear el
     *            nuevo registro.
     * @return Y41eEncuesta
     *            Bean resultante del proceso de creacion.
     */
    @RequestMapping(method = RequestMethod.POST)
    @ResponseBody public Y41eEncuesta add(@RequestBody Y41eEncuesta y41eEncuesta) {
        y41eEncuesta.setUrl(Y41eUtils.generarCodigo());
        Y41eEncuesta y41eEncuestaAux = this.y41eEncuestaService.add(y41eEncuesta);

        String urlEncuesta = this.appConfiguration.getProperty("url.encuesta");
        y41eEncuesta.setUrl(urlEncuesta.replace("{0}", y41eEncuesta.getUrl()));

        Y41eEncuestaController.LOGGER.info("[POST] : Y41eEncuesta insertado correctamente");
        return y41eEncuestaAux;
    }

    /**
     * Operacion CRUD Delete. Borrado del registro correspondiente al
     * identificador especificado.
     *
     * @param idEncuesta Long
     *            Identificador del objeto que se desea eliminar.
     * @return Y41eEncuesta
     *            Bean eliminado.
     */
    @RequestMapping(value = "/{idEncuesta}", method = RequestMethod.DELETE)
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody public Y41eEncuesta remove(@PathVariable Long idEncuesta) {
        Y41eEncuesta y41eEncuesta = new Y41eEncuesta();
        y41eEncuesta.setIdEncuesta(idEncuesta);
        this.y41eEncuestaService.remove(y41eEncuesta);
        Y41eEncuestaController.LOGGER.info("[DELETE] : Y41eEncuesta borrado correctamente");
        return y41eEncuesta;
    }

    /**
     * Comprueba que no existe ninguna encuesta enviada o respondida para el id de la encuesta.
     * 
     * @param idEncuesta Long
     * @return String el area a la que pertence la encuesta 
     *            
     */
    @RequestMapping(value = "/validarEncuesta/{idEncuesta}", method = RequestMethod.GET)
    @ResponseBody public String validarEncuesta(@PathVariable Long idEncuesta) {
        Y41eEncuestaController.LOGGER.info("validarEncuesta");
        return this.y41eEncuestaService.validarEncuesta(idEncuesta);
    }

    /**
     * Copia la encuesta que recibe por parámetro
     *
     * @param y41eEncuesta Y41eEncuesta 
     *            Bean que contiene la informacion con la que se va a crear el
     *            nuevo registro.
     * @return Y41eEncuesta
     *            Bean resultante del proceso de copiar.
     */
    @RequestMapping(value = "/copiarEncuesta", method = RequestMethod.POST)
    @ResponseBody public Y41eEncuesta copiarEncuesta(@RequestBody Y41eEncuesta y41eEncuesta) {
        Y41eEncuestaController.LOGGER.info("[POST] : copiarEncuesta");
        return this.y41eEncuestaService.copiarEncuesta(y41eEncuesta);
    }

    /**
     * Genera el excel de la encuesta
     * @param id Integer
     * @param request HttpServletRequest
     * @param response HttpServletResponse
     * @param model Model
     */
    @RequestMapping(value = "/excel", method = RequestMethod.POST)
    public void excelEncuesta(@RequestParam() Long id, HttpServletRequest request, 
            HttpServletResponse response, Model model) {
        Y41eEncuestaController.LOGGER.info("[POST - filter] : excelEncuesta");
        Y41eEncuesta y41eEncuesta = new Y41eEncuesta();
        y41eEncuesta.setIdEncuesta(id);
        final Y41eExcelModel excelModel= this.y41eEncuestaService.getExcelEncuesta(y41eEncuesta);
        this.y41eExcelReportService.generarExcel("encuesta", excelModel, response);
    }
}
