package com.ejie.y41e.control;

import java.util.List;
import java.util.Locale;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.ejie.x38.control.bind.annotation.RequestJsonBody;
import com.ejie.x38.dto.JQGridRequestDto;
import com.ejie.x38.dto.JQGridResponseDto;
import com.ejie.y41e.common.Y41eConstants;
import com.ejie.y41e.model.Y41eBloque;
import com.ejie.y41e.service.Y41eBloqueService;
import com.ejie.y41e.service.Y41eRelEncuestaBloqueService;

/**
 * Y41eBloqueController
 *  
 */

@Controller
@RequestMapping (value = "/bloque")

public class Y41eBloqueController  {

    private static final Logger LOGGER = LoggerFactory.getLogger(Y41eBloqueController.class);

    @Autowired
    private Y41eBloqueService y41eBloqueService;
    @Autowired
    private Y41eRelEncuestaBloqueService y41eRelEncuestaBloqueService;


    /**
     * Comprueba que no existe ninguna encuesta enviada o respondida para el id del bloque.
     * 
     * @param idBloque Long
     * @return String el area a la que pertence la encuesta 
     *            
     */
    @RequestMapping(value = "/validarBloque/{idBloque}", method = RequestMethod.GET)
    @ResponseBody public String validarBloque(@PathVariable Long idBloque) {
        Y41eBloqueController.LOGGER.info("validarBloque");
        Integer encuestaEnviada = this.y41eBloqueService.validarBloqueEnviada(idBloque);
        if(encuestaEnviada>0){
            return Y41eConstants.ERROR_ENCUESTA_ENVIADA;
        }
        Integer tieneEncuesta = this.y41eBloqueService.validarBloqueEncuesta(idBloque);
        if(tieneEncuesta>0){
            return Y41eConstants.ERROR_TIENE_ENCUESTA;
        }
        return "";
    }

    /**
     * Asocia los bloques seleccionados a la encuesta
     * @param filterY41eBloque Y41eBloque
     * @param selectedIds List<String>
     * @param selectedAll Boolean
     * @param locale Locale
     * @return String
     * @throws Exception e
     */
    @RequestMapping(value = "/asosiarBloques", method = RequestMethod.POST)
    @ResponseBody() public void asosiarBloques(
            @RequestJsonBody(param="filter") Y41eBloque filterY41eBloque,
            @RequestJsonBody(param = "selectedIds") List<String> selectedIds,
            @RequestJsonBody(param = "selectedAll") Boolean selectedAll) {
        Y41eBloqueController.LOGGER.info("[POST] : asosiarBloques");
        filterY41eBloque.setSelectedAll(selectedAll);
        filterY41eBloque.setSelectedIds(selectedIds);
        List<Y41eBloque> listaBloquesSeleccionados = this.y41eBloqueService.findAllLike(filterY41eBloque, null, false);
        y41eRelEncuestaBloqueService.asosiarBloques(filterY41eBloque.getIdEncuesta(), listaBloquesSeleccionados);
    }

    /**
     * Operacion CRUD Read. Devuelve el bean correspondiente al identificador indicado.
     * 
     * @param idBloque Long
     * @return Y41eBloque 
     *            Objeto correspondiente al identificador indicado.
     */
    @RequestMapping(value = "/{idBloque}", method = RequestMethod.GET)
    @ResponseBody public Y41eBloque get(@PathVariable Long idBloque) {
        Y41eBloque y41eBloque = new Y41eBloque();
        y41eBloque.setIdBloque(idBloque);
        y41eBloque = this.y41eBloqueService.find(y41eBloque);
        Y41eBloqueController.LOGGER.info("[GET - findBy_PK] : Obtener Y41eBloque por PK");
        return y41eBloque;
    }

    /**
     * Devuelve una lista de beans correspondientes a los valores de filtrados
     * indicados en el objeto pasado como parametro.
     *
     * @param filterY41eBloque Y41eBloque
     *            Objeto que contiene los parametros de filtrado utilizados en
     *            la busqueda.
     * @return List<Y41eBloque> 
     *            Lista de objetos correspondientes a la busqueda realizada.
     */
    @RequestMapping(method = RequestMethod.GET)
    @ResponseBody public  List<Y41eBloque> getAll(@ModelAttribute Y41eBloque filterY41eBloque) {
        Y41eBloqueController.LOGGER.info("[GET - find_ALL] : Obtener Y41eBloque por filtro");
        return this.y41eBloqueService.findAll(filterY41eBloque, null);
    }

    /**
     * Operacion CRUD Edit. Modificacion del bean indicado.
     *
     * @param y41eBloque Y41eBloque 
     *            Bean que contiene la informacion a modificar.
     * @return Y41eBloque 
     *            Bean resultante de la modificacion.
     */
    @RequestMapping(method = RequestMethod.PUT)
    @ResponseBody public Y41eBloque edit(@RequestBody Y41eBloque y41eBloque) {
        Y41eBloque y41eBloqueAux = this.y41eBloqueService.update(y41eBloque);
        Y41eBloqueController.LOGGER.info("[PUT] : Y41eBloque actualizado correctamente");
        return y41eBloqueAux;
    }

    /**
     * Operacion CRUD Create. Creacion de un nuevo registro a partir del bean
     * indicado.
     *
     * @param y41eBloque Y41eBloque 
     *            Bean que contiene la informacion con la que se va a crear el
     *            nuevo registro.
     * @return Y41eBloque
     *            Bean resultante del proceso de creacion.
     */
    @RequestMapping(method = RequestMethod.POST)
    @ResponseBody public Y41eBloque add(@RequestBody Y41eBloque y41eBloque) {
        Y41eBloque y41eBloqueAux = this.y41eBloqueService.add(y41eBloque);
        Y41eBloqueController.LOGGER.info("[POST] : Y41eBloque insertado correctamente");
        return y41eBloqueAux;
    }

    /**
     * Operacion CRUD Delete. Borrado del registro correspondiente al
     * identificador especificado.
     *
     * @param idBloque Long
     *            Identificador del objeto que se desea eliminar.
     * @return Y41eBloque
     *            Bean eliminado.
     */
    @RequestMapping(value = "/{idBloque}", method = RequestMethod.DELETE)
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody public Y41eBloque remove(@PathVariable Long idBloque) {
        Y41eBloque y41eBloque = new Y41eBloque();
        y41eBloque.setIdBloque(idBloque);
        this.y41eBloqueService.remove(y41eBloque);
        Y41eBloqueController.LOGGER.info("[DELETE] : Y41eBloque borrado correctamente");
        return y41eBloque;
    }

    /**
     * Metodo de presentacion del RUP_TABLE.
     * 
     * @param model Model
     * @return String
     */
    @RequestMapping(value = "/maint", method = RequestMethod.GET)
    public String getFormEdit(Model model) {
        Y41eBloqueController.LOGGER.info("[GET - View] : y41ebloque");
        Locale locale = LocaleContextHolder.getLocale();
        model.addAttribute("LOCALE_CAMEL", StringUtils.capitalize(locale.getLanguage()));
        return "bloque";
    }

    /**
     * Operacion de filtrado del componente RUP_TABLE.
     * 
     * @param filterY41eBloque Y41eBloque
     *            Bean que contiene los parametros de filtrado a emplear.
     * @param jqGridRequestDto
     *            Dto que contiene los parametros de configuracion propios del
     *            RUP_TABLE a aplicar en el filtrado.
     * @return JQGridResponseDto<Y41eBloque>
     *            Dto que contiene el resultado del filtrado realizado por el 
     *            componente RUP_TABLE.
     */
    @RequestMapping(value = "/filter", method = RequestMethod.POST)
    @ResponseBody public JQGridResponseDto<Y41eBloque> filter(
            @RequestJsonBody(param="filter") Y41eBloque filterY41eBloque,
            @RequestJsonBody JQGridRequestDto jqGridRequestDto) {
        Y41eBloqueController.LOGGER.info("[POST - filter] : Obtener Y41eBloques");
        return this.y41eBloqueService.filter(filterY41eBloque, jqGridRequestDto, false);
    }

    /**
     * Añade el bloque y lo asigna a la encuesta
     *
     * @param y41ebloque Y41eBloque 
     *            Bean que contiene la informacion con la que se va a crear el
     *            nuevo registro.
     */
    @RequestMapping(value = "/addBloqueEncuesta", method = RequestMethod.POST)
    @ResponseBody public void addBloqueEncuesta(@RequestBody Y41eBloque y41ebloque) {
        Y41eBloqueController.LOGGER.info("[POST] : addBloqueEncuesta");
        this.y41eBloqueService.addBloqueEncuesta(y41ebloque);
    }

    /**
     * Obtiene el suggest de bloques
     * @param q String
     * @param idEncuesta Long
     * @param locale Locale
     * @return List<Y41eBloque>
     */
    @RequestMapping(value = "/suggest", method = RequestMethod.GET)
    @ResponseBody() public List<Y41eBloque> getSuggestBloque(@RequestParam() String q, Long idEncuesta, String tipo) {
        Y41eBloqueController.LOGGER.info("[GET - getSuggestBloque]");
        Y41eBloque bloque = new Y41eBloque();
        bloque.setActiva(Y41eConstants.SI);
        bloque.setIdEncuesta(idEncuesta);
        if(Y41eConstants.DESC_PUBLICA.equals(tipo)){
            bloque.setDescPubl(q);
        }else{
            bloque.setDescInt(q);
        }
        return this.y41eBloqueService.findAllSuggest(bloque, false);
    }

}
