package com.ejie.y41e.control;

import java.util.Calendar;
import java.util.Locale;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import com.ejie.y41e.common.Y41eConstants;
import com.ejie.y41e.common.Y41eMimeMessagePreparator;
import com.ejie.y41e.model.Y41eBloque;
import com.ejie.y41e.model.Y41eConfiguracion;
import com.ejie.y41e.model.Y41eEncuesta;
import com.ejie.y41e.model.Y41ePregunta;
import com.ejie.y41e.service.Y41eBloqueService;
import com.ejie.y41e.service.Y41eConfiguracionService;
import com.ejie.y41e.service.Y41eEncuestaService;
import com.ejie.y41e.service.Y41ePreguntaService;
import com.ejie.y41e.service.Y41eResultadosService;

/**
 * Y41eEncuestaWebController
 *  
 */
@Controller
@RequestMapping (value = "/")

public class Y41eEncuestaWebController  {

    private static final Logger LOGGER = LoggerFactory.getLogger(Y41eEncuestaWebController.class);
    private static final String ERROR = "errorEncuesta";
    private static final String AGRADECIMIENTO = "agradecimiento";
    private static final String MENSAJE_ERROR_EU = "mensajeErrorEu";
    private static final String MENSAJE_ERROR_ES = "mensajeErrorEs";
    private static final String MENSAJE_NO_ACTIVA = "mensaje.encuestaNoActiva";
    private static final String MENSAJE_NO_DISPONIBLE = "mensaje.encuestaNoDisponible";
    private static final Locale LOCALE_ES = new Locale(Y41eConstants.LANG_CASTELLANO);
    private static final Locale LOCALE_EU = new Locale(Y41eConstants.LANG_EUSKERA);

    @Autowired
    private Y41eEncuestaService y41eEncuestaService;
    @Autowired
    private Y41eResultadosService y41eResultadosService;
    @Autowired
    private Y41eConfiguracionService y41eConfiguracionService;
    @Autowired
    private Y41eBloqueService y41eBloqueService;
    @Autowired
    private Y41ePreguntaService y41ePreguntaService;
    @Autowired()
    private JavaMailSender mailSender;
    @Autowired()
    private ReloadableResourceBundleMessageSource appMessageSource;


    /**
     * Obtiene la encuesta y la muestra 
     * @param param String
     * @param model Model
     * @return String
     */
    @RequestMapping(value = "/{param}", method = RequestMethod.GET)
    public String getEncuesta(@PathVariable String param, Model model, HttpServletRequest request) {
        try{
            String retorno = "";
            Y41eEncuestaWebController.LOGGER.info("Obtener encuestas");
            Y41eEncuesta encuesta = y41eEncuestaService.getEncuestaWsByParam(param);

            //nulo, 0 o que llega...?
            if ( encuesta == null || Y41eConstants.CERO.equals(encuesta.getIdEncuesta())){
                encuesta = y41eEncuestaService.getEncuestaByParam(param);
            }

            retorno = this.validarEncuesta(encuesta, model);
            if(StringUtils.isNotBlank(retorno)){
                return retorno;
            }

            encuesta = y41eEncuestaService.cargarEncuesta(encuesta);
            model.addAttribute("encuesta", encuesta);
            request.getSession().setAttribute("idEnvio", encuesta.getIdEnvio());

        }catch (Exception e) {
            Y41eEncuestaWebController.LOGGER.error("Error al cargar los datos de la encuesta " + e);
            model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_EU, 
                    this.appMessageSource.getMessage("mensaje.errorCargarEncuesta", null, Y41eEncuestaWebController.LOCALE_EU));
            model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_ES, 
                    this.appMessageSource.getMessage("mensaje.errorCargarEncuesta", null, Y41eEncuestaWebController.LOCALE_ES));
            return Y41eEncuestaWebController.ERROR;
        }


        return "encuesta";
    }

    /**
     * Guardar los resultados de la encuesta
     * @param encuesta Y41eEncuesta
     * @param model Model
     * @param request HttpServletRequest
     * @return String
     */
    @RequestMapping(value = "/enviar", method = RequestMethod.POST)
    public String enviarEncuesta(@ModelAttribute Y41eEncuesta encuesta, 
            Model model, HttpServletRequest request) {
        Y41eEncuestaWebController.LOGGER.info("enviar encuesta");
        Long idEnvio = (Long) request.getSession().getAttribute("idEnvio");

        try{
            //guardar resultados
            y41eResultadosService.guardarRespuestas(encuesta, idEnvio);
            //obtener mensaje agradecimiento
            Y41eConfiguracion configuracion = new Y41eConfiguracion();
            configuracion.setIdConfig(Integer.valueOf(Y41eConstants.CONFIG_AGRADECIMIENTO));
            configuracion = this.y41eConfiguracionService.find(configuracion);
            model.addAttribute("mensajeAgradEu", configuracion.getMensajeEu());
            model.addAttribute("mensajeAgradEs", configuracion.getMensajeEs());
        }catch (Exception e) {
            Y41eEncuestaWebController.LOGGER.error("Error guardar la encuesta " + e);
            model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_EU, 
                    this.appMessageSource.getMessage("mensaje.errorGuardarEncuesta", null, Y41eEncuestaWebController.LOCALE_EU));
            model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_ES, 
                    this.appMessageSource.getMessage("mensaje.errorGuardarEncuesta", null, Y41eEncuestaWebController.LOCALE_ES));
            return Y41eEncuestaWebController.ERROR;
        }
        try{
            //mirar las respuesta para ver si hay que enviar mails
            this.enviarAvisosPreguntas(encuesta);
        }catch (Exception e) {
            Y41eEncuestaWebController.LOGGER.error("Error guardar la encuesta " + e);
        }
        return Y41eEncuestaWebController.AGRADECIMIENTO;
    }


    /**
     * Por cada respuesta comprueba si hay que enviar email y lo envía
     * @param encuesta Y41eEncuesta
     */
    private void enviarAvisosPreguntas(Y41eEncuesta encuesta) {
        //obtener encuesta
        Y41eEncuesta datosEncuesta = y41eEncuestaService.find(encuesta);
        //obtener datos de configuración
        Y41eConfiguracion configRemitente = new Y41eConfiguracion(Integer.valueOf(Y41eConstants.CONFIG_REMITENTE));
        configRemitente = this.y41eConfiguracionService.find(configRemitente);
        Y41eConfiguracion configAviso = new Y41eConfiguracion(Integer.valueOf(Y41eConstants.CONFIG_AVISO));
        configAviso = this.y41eConfiguracionService.find(configAviso);
        StringBuilder asunto = new StringBuilder();
        asunto.append(configAviso.getAsuntoEu()).append(" / ").append(configAviso.getAsuntoEs());
        StringBuilder texto = new StringBuilder();
        for(Y41eBloque bloque: encuesta.getListaBloques()){
            Y41eBloque datosBloque = y41eBloqueService.find(bloque);
            for(Y41ePregunta pregunta: bloque.getListaPreguntas()){
                Y41ePregunta preguntaAviso = y41ePreguntaService.findPreguntaAviso(pregunta.getIdPregunta(), pregunta.getResultado().getRespuesta().getIdTipoRespuesta());
                if(preguntaAviso!=null){
                    Y41eMimeMessagePreparator mensaje = new Y41eMimeMessagePreparator();
                    mensaje.setRemitente(configRemitente.getEmail());
                    mensaje.getDestinatarios().clear();
                    mensaje.getDestinatarios().add(configAviso.getEmail());
                    mensaje.setAsunto(asunto.toString());
                    texto.append(configAviso.getMensajeEu().replace("{preg_eu}", preguntaAviso.getDescIntEu())
                            .replace("{encu_eu}", datosEncuesta.getDescIntEu())
                            .replace("{bloque_eu}", datosBloque.getDescIntEu())
                            .replace("{valor}", preguntaAviso.getResultado().getRespuesta().getDescEu()))
                    .append("<br>")
                    .append("<br>")
                    .append(configAviso.getMensajeEs().replace("{preg_es}", preguntaAviso.getDescIntEs())
                            .replace("{encu_es}", datosEncuesta.getDescIntEu())
                            .replace("{bloque_es}", datosBloque.getDescIntEu())
                            .replace("{valor}", preguntaAviso.getResultado().getRespuesta().getDescEs()));
                    mensaje.setTexto(texto.toString());
                    this.mailSender.send(mensaje);
                    //reiniciar campos
                    texto.delete(0, texto.length());
                }
            }
        }
    }

    /**
     * Valida los datos de la encuesta
     * @param encuesta Y41eEncuesta
     * @param model Model
     * @return String
     */
    private String validarEncuesta(Y41eEncuesta encuesta, Model model) {
        String retorno = "";
        //comprobar si existe la encuesta
        if(encuesta==null){
            model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_EU, 
                    this.appMessageSource.getMessage("mensaje.encuestaNoExiste", null, Y41eEncuestaWebController.LOCALE_EU));
            model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_ES, 
                    this.appMessageSource.getMessage("mensaje.encuestaNoExiste", null, Y41eEncuestaWebController.LOCALE_ES));
            return Y41eEncuestaWebController.ERROR;
        }
        //mirar si ya ha sido respondida
        if(Y41eConstants.SI.equals(encuesta.getRespondida())){
            model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_EU, 
                    this.appMessageSource.getMessage("mensaje.encuestaRespondida", null, Y41eEncuestaWebController.LOCALE_EU));
            model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_ES, 
                    this.appMessageSource.getMessage("mensaje.encuestaRespondida", null, Y41eEncuestaWebController.LOCALE_ES));
            return Y41eEncuestaWebController.ERROR;
        }
        //mirar si la encuesta está activa
        if(Y41eConstants.NO.equals(encuesta.getActiva())){
            model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_EU, 
                    this.appMessageSource.getMessage(Y41eEncuestaWebController.MENSAJE_NO_ACTIVA, null, Y41eEncuestaWebController.LOCALE_EU));
            model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_ES, 
                    this.appMessageSource.getMessage(Y41eEncuestaWebController.MENSAJE_NO_ACTIVA, null, Y41eEncuestaWebController.LOCALE_ES));
            return Y41eEncuestaWebController.ERROR;
        }
        retorno = this.validarFechasVigencia(encuesta, model);

        if(StringUtils.isNotBlank(retorno)){
            return retorno;
        }
        retorno = this.validarPeriodoVigencia(encuesta, model);
        return retorno;
    }


    /**
     * Valida el periodo de vigencia
     * @param encuesta Y41eEncuesta
     * @param model Model
     * @return String
     */
    private String validarPeriodoVigencia(Y41eEncuesta encuesta, Model model) {
        String retorno = "";
        Calendar fechaActual = Calendar.getInstance();
        Calendar fechaEnvio = Calendar.getInstance();

        if (null != encuesta.getFecEnvio()){
            fechaEnvio.setTime(encuesta.getFecEnvio());
            //mirar periodo de vigencia
            if(encuesta.getPeriodoResp()!=null && encuesta.getPeriodoResp()>0){
                fechaEnvio.add(Calendar.DATE, encuesta.getPeriodoResp().intValue());
                if(fechaActual.after(fechaEnvio)){
                    model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_EU, 
                            this.appMessageSource.getMessage(Y41eEncuestaWebController.MENSAJE_NO_DISPONIBLE, null, Y41eEncuestaWebController.LOCALE_EU));
                    model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_ES, 
                            this.appMessageSource.getMessage(Y41eEncuestaWebController.MENSAJE_NO_DISPONIBLE, null, Y41eEncuestaWebController.LOCALE_ES));
                    return Y41eEncuestaWebController.ERROR;
                }
            }
        }

        return retorno;
    }


    /**
     * Valida las fechas de vigencia
     * @param encuesta Y41eEncuesta
     * @param model Model
     * @return String
     */
    private String validarFechasVigencia(Y41eEncuesta encuesta, Model model) {
        String retorno = "";
        Calendar fechaActual = Calendar.getInstance();
        //mirar fecha vigencia inicio
        if(encuesta.getFecVigInicio() !=null){
            Calendar fechaVigenciaInicio = Calendar.getInstance();
            fechaVigenciaInicio.setTime(encuesta.getFecVigInicio());
            fechaVigenciaInicio.set(Calendar.HOUR, 0);
            fechaVigenciaInicio.set(Calendar.MINUTE, 0);
            fechaVigenciaInicio.set(Calendar.SECOND, 0);
            if(fechaActual.before(fechaVigenciaInicio)){
                model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_EU, 
                        this.appMessageSource.getMessage(Y41eEncuestaWebController.MENSAJE_NO_DISPONIBLE, null, Y41eEncuestaWebController.LOCALE_EU));
                model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_ES, 
                        this.appMessageSource.getMessage(Y41eEncuestaWebController.MENSAJE_NO_DISPONIBLE, null, Y41eEncuestaWebController.LOCALE_ES));
                return Y41eEncuestaWebController.ERROR;
            }
        }
        //mirar fecha vigencia fin
        if(encuesta.getFecVigFin()!=null){
            Calendar fechaVigenciaFin = Calendar.getInstance();
            fechaVigenciaFin.setTime(encuesta.getFecVigFin());
            fechaVigenciaFin.set(Calendar.HOUR, 23);
            fechaVigenciaFin.set(Calendar.MINUTE, 59);
            fechaVigenciaFin.set(Calendar.SECOND, 59);
            if(fechaActual.after(fechaVigenciaFin)){
                model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_EU, 
                        this.appMessageSource.getMessage(Y41eEncuestaWebController.MENSAJE_NO_DISPONIBLE, null, Y41eEncuestaWebController.LOCALE_EU));
                model.addAttribute(Y41eEncuestaWebController.MENSAJE_ERROR_ES, 
                        this.appMessageSource.getMessage(Y41eEncuestaWebController.MENSAJE_NO_DISPONIBLE, null, Y41eEncuestaWebController.LOCALE_ES));
                return Y41eEncuestaWebController.ERROR;
            }
        }
        return retorno;
    }


}
