package com.ejie.y41e.service;
import java.util.List;
import java.util.Locale;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.x38.dto.JQGridRequestDto;
import com.ejie.x38.dto.JQGridResponseDto;
import com.ejie.y41e.common.Y41eConstants;
import com.ejie.y41e.dao.Y41eEncuestaDao;
import com.ejie.y41e.dao.Y41eEnviosDao;
import com.ejie.y41e.dao.Y41eGenericoDao;
import com.ejie.y41e.dao.Y41eResultadosDao;
import com.ejie.y41e.model.Y41eEncuesta;
import com.ejie.y41e.model.Y41eEnvio;
import com.ejie.y41e.model.Y41eGrafico;
import com.ejie.y41e.model.Y41eResultados;
import com.ejie.y41e.model.excel.Y41eExcelCelda;
import com.ejie.y41e.model.excel.Y41eExcelEstilo;
import com.ejie.y41e.model.excel.Y41eExcelHoja;
import com.ejie.y41e.model.excel.Y41eExcelModel;
import com.ejie.y41e.utils.Y41eUtils;

/**
 * ResultadosServiceImpl
 *  
 */

@Service(value = "y41eResultadosService")
public class Y41eResultadosServiceImpl extends Y41eGenericoServiceImpl<Y41eResultados> implements Y41eResultadosService {

    @Autowired
    private Y41eResultadosDao resultadosDao;
    @Autowired
    private Y41eEncuestaDao y41eEncuestaDao;
    @Autowired
    private Y41eEnviosDao y41eEnviosDao;
    @Autowired()
    private ReloadableResourceBundleMessageSource appMessageSource;

    @Override()
    protected Y41eGenericoDao<Y41eResultados> getDao() {
        return this.resultadosDao;
    }

    /**
     * Obtiene resultados por encuesta
     * @param filterEncuesta Y41eEncuesta
     * @param jqGridRequestDto JQGridRequestDto
     * @param startsWith boolean
     * @return JQGridResponseDto<Y41eEncuesta>
     */
    @Override
    public JQGridResponseDto<Y41eEncuesta> filterEncuestas(Y41eEncuesta filterEncuesta,
            JQGridRequestDto jqGridRequestDto, boolean startsWith){
        List<Y41eEncuesta> listaEncuestas = this.resultadosDao.findAllLikeEncuestas(filterEncuesta, jqGridRequestDto, false);
        Long recordNum = this.resultadosDao.findAllLikeCountEncuestas(filterEncuesta, false);
        return new JQGridResponseDto<Y41eEncuesta>(jqGridRequestDto, recordNum, listaEncuestas);
    }

    /**
     * Obtiene una lista de resultados para pintar los graficos
     * @param resultado Y41eResultados
     * @return List<Y41eGrafico>
     */
    @Override
    public List<Y41eGrafico> getGraficos(Y41eResultados resultado){
        //Obtener las preguntas de la encuesta
        List<Y41eGrafico> listaGraficos = y41eEncuestaDao.getGraficosEncuesta(resultado);
        //por cada pregunta obtener el numero de respuestas y porcentaje
        for(Y41eGrafico grafico : listaGraficos){
            //Si no tiene tipoGrafico se buscan los textos de las respuestas
            if(StringUtils.isBlank(grafico.getPregunta().getTipoPregunta().getTipoGrafico())){
                grafico.setResultados(this.resultadosDao.getResultadosPreguntasTexto(resultado, grafico.getBloque().getIdBloque(), 
                        grafico.getPregunta().getIdPregunta()));
            }else{
                //Si no los tipos de respuesta con su número de resuestas y porcentajes
                grafico.setResultados(this.resultadosDao.getResultadosPreguntas(resultado, grafico.getBloque().getIdBloque(),
                        grafico.getPregunta().getIdPregunta()));
            }
        }

        return listaGraficos;
    }

    /**
     * Obtiene los resultados por destinatario
     * @param filter Y41eEnvio
     * @param jqGridRequestDto JQGridRequestDto
     * @param startsWith boolean
     * @return JQGridResponseDto<Y41eEnvio>
     */
    @Override
    public JQGridResponseDto<Y41eEnvio> filterDestinatario(Y41eEnvio filter, JQGridRequestDto jqGridRequestDto,
            boolean startsWith){
        List<Y41eEnvio> listaEncuestas = this.resultadosDao.findAllLikeDestinatario(filter, jqGridRequestDto, false);
        Long recordNum = this.resultadosDao.findAllLikeCountDestinatarios(filter, false);
        return new JQGridResponseDto<Y41eEnvio>(jqGridRequestDto, recordNum, listaEncuestas);
    }

    /**
     * Obtiene el excel de resultados por destinatario
     * @param envio Y41eEnvio
     * @return ExcelModel
     */
    @Override
    public Y41eExcelModel getExcelResultadosDestinatario(Y41eEnvio envio){
        Locale locale = LocaleContextHolder.getLocale();
        //obtener datos
        List<Y41eResultados> resultados = resultadosDao.getResultadosDestinatario(envio);
        final Y41eExcelModel excelModel = new Y41eExcelModel();
        final Y41eExcelHoja excelHoja= new Y41eExcelHoja(this.appMessageSource.getMessage("comun.resultados", null, locale), 0);

        //imagen
        Y41eExcelCelda imagenCelda = new Y41eExcelCelda();
        imagenCelda.setTipoCelda(Y41eConstants.TIPO_CELDA_IMAGEN);
        imagenCelda.setRutaImagen(Y41eConstants.RUTA_IMAGEN);
        imagenCelda.setPosicionX(0);
        imagenCelda.setPosicionX(0);
        excelHoja.addCelda("imagen", imagenCelda);

        long idBloqueActual=0;
        int posY = Y41eConstants.TRECE;
        int cont = 1;
        Y41eExcelCelda excelCelda;
        for(int i=0; i<resultados.size();i++){
            Y41eResultados resultado = resultados.get(i);

            if(i==0){
                //texto destinatario
                excelCelda = new Y41eExcelCelda(Y41eUtils.getLiteralEuEs(this.appMessageSource, "comun.destinatario", null, Y41eConstants.LITERAL_SEPARATOR), 
                        Y41eConstants.CERO, Y41eConstants.SIETE, Y41eExcelEstilo.NEGRITA);
                excelHoja.addCelda("celdaDestinatarioTexto", excelCelda);
                //destinatario
                excelCelda = new Y41eExcelCelda(resultado.getEnvio().getIdUsuario(), Y41eConstants.CUATRO, Y41eConstants.SIETE, "");
                excelHoja.addCelda("destinatario", excelCelda);
                //texto fecha respuesta
                excelCelda = new Y41eExcelCelda(Y41eUtils.getLiteralEuEs(this.appMessageSource, "comun.fechaRespuesta", null, Y41eConstants.LITERAL_SEPARATOR), 
                        Y41eConstants.CERO, Y41eConstants.OCHO, Y41eExcelEstilo.NEGRITA);
                excelHoja.addCelda("celdaFechaTexto", excelCelda);
                //fecha respuesta 
                excelCelda = new Y41eExcelCelda(Y41eUtils.obtFechaFormateada(resultado.getEnvio().getFecRespuesta(), new Locale(Y41eConstants.LANG_EUSKERA)),
                        Y41eConstants.CUATRO, Y41eConstants.OCHO, "");
                excelHoja.addCelda("fecha", excelCelda);
                // encuesta euskera
                excelCelda = new Y41eExcelCelda(resultado.getEnvio().getEncuesta().getDescPublEu(), Y41eConstants.CERO, Y41eConstants.DIEZ, Y41eExcelEstilo.TITLE);
                excelHoja.addCelda("encuestaEu", excelCelda);
                // encuesta castellano
                excelCelda = new Y41eExcelCelda(resultado.getEnvio().getEncuesta().getDescPublEs(), Y41eConstants.CERO, Y41eConstants.ONCE, Y41eExcelEstilo.TITLE);
                excelHoja.addCelda("encuestaEs", excelCelda);
            }

            if(idBloqueActual!=resultado.getBloque().getIdBloque()){
                //bloque euskera
                excelCelda = new Y41eExcelCelda(resultado.getBloque().getDescPublEu(), Y41eConstants.CERO, posY, Y41eExcelEstilo.NEGRITA, Y41eConstants.OCHO);
                if(resultado.getBloque().getDescPublEu().length()>Y41eConstants.NOVENTAYNUEVE){
                    excelCelda.setAlto(Y41eConstants.EXCEL_HEIGTH);
                }
                excelHoja.addCelda(String.valueOf("bloqueEu"+resultado.getBloque().getIdBloque()), excelCelda);
                posY++;
                //bloque csatellano
                excelCelda = new Y41eExcelCelda(resultado.getBloque().getDescPublEs(), Y41eConstants.CERO, posY, Y41eExcelEstilo.CONTENT, Y41eConstants.OCHO);
                if(resultado.getBloque().getDescPublEs().length()>Y41eConstants.NOVENTAYNUEVE){
                    excelCelda.setAlto(Y41eConstants.EXCEL_HEIGTH);
                }
                excelHoja.addCelda(String.valueOf("bloqueEs"+resultado.getBloque().getIdBloque()), excelCelda);

                posY++;
                posY++;
                cont = 1;
            }
            //pregunta euskera
            excelCelda = new Y41eExcelCelda(cont+ ". " + resultado.getPregunta().getDescPublEu(), Y41eConstants.UNO, posY, Y41eExcelEstilo.NEGRITA, Y41eConstants.SIETE);
            if(resultado.getPregunta().getDescPublEu().length()>Y41eConstants.OCHENTA){
                excelCelda.setAlto(Y41eConstants.EXCEL_HEIGTH);
            }
            excelHoja.addCelda(String.valueOf("preguntaEu"+resultado.getBloque().getIdBloque()+resultado.getPregunta().getIdPregunta()), excelCelda);
            posY++;
            //pregunta euskera
            excelCelda = new Y41eExcelCelda(resultado.getPregunta().getDescPublEs(), Y41eConstants.UNO, posY, Y41eExcelEstilo.CONTENT, Y41eConstants.SIETE);
            if(resultado.getPregunta().getDescPublEs().length()>Y41eConstants.OCHENTA){
                excelCelda.setAlto(Y41eConstants.EXCEL_HEIGTH);
            }
            excelHoja.addCelda(String.valueOf("preguntaEs"+resultado.getBloque().getIdBloque()+resultado.getPregunta().getIdPregunta()), excelCelda);
            posY++;

            //resultado
            if(StringUtils.isNotBlank(resultado.getTexto())){
                excelCelda = new Y41eExcelCelda(resultado.getTexto(), Y41eConstants.UNO, posY, Y41eExcelEstilo.CONTENT);
                if(resultado.getTexto().length()>Y41eConstants.OCHENTA){
                    excelCelda.setAlto(Y41eConstants.SIETE);
                    excelCelda.setAlto(Y41eConstants.EXCEL_HEIGTH);
                }
                excelHoja.addCelda(String.valueOf("resultado"+resultado.getBloque().getIdBloque()+resultado.getPregunta().getIdPregunta()+ resultado.getRespuesta().getIdTipoRespuesta()),  excelCelda);
            }else{
                excelHoja.addCelda(String.valueOf("resultado"+resultado.getBloque().getIdBloque()+resultado.getPregunta().getIdPregunta()+ resultado.getRespuesta().getIdTipoRespuesta()),  new Y41eExcelCelda(resultado.getRespuesta().getDescLocale(), Y41eConstants.UNO, posY, null));
            }

            posY++;
            idBloqueActual= resultado.getBloque().getIdBloque();
            posY++;
            cont++;
        }
        excelModel.getListaHojas().add(excelHoja);
        return excelModel;
    }

    /**
     * Guardar las respuestas del usuario
     * @param encuesta Y41eEncuesta
     * @param idEnvio Long
     */
    @Override
    @Transactional(rollbackFor = Throwable.class)
    public void guardarRespuestas(Y41eEncuesta encuesta, Long idEnvioRecibido){
        Long idEnvio = 0L;
        // insercion en la 10 si el idEnvioRecibido es 0
        if (Y41eConstants.CERO == idEnvioRecibido.intValue()){
            idEnvio = this.y41eEnviosDao.addEnvioSondeo(encuesta.getIdEncuesta());
        }else{
            idEnvio = idEnvioRecibido;
        }
        //guardar resultados
        this.resultadosDao.guardarRespuestas(encuesta, idEnvio);
        //actualizar fecha de respuesta
        this.y41eEnviosDao.actualizarFechaRespuesta(idEnvio);
    }

}

