package com.ejie.y41e.service;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.y41e.common.Y41eConstants;
import com.ejie.y41e.dao.Y41eGenericoDao;
import com.ejie.y41e.dao.Y41ePreguntaDao;
import com.ejie.y41e.dao.Y41eRelBloquePreguntaDao;
import com.ejie.y41e.model.Y41eBloque;
import com.ejie.y41e.model.Y41ePregunta;
import com.ejie.y41e.model.Y41eRelBloquePregunta;

/**
 * Y41ePreguntaServiceImpl
 *  
 */

@Service(value = "y41ePreguntaService")
public class Y41ePreguntaServiceImpl extends Y41eGenericoServiceImpl<Y41ePregunta> implements Y41ePreguntaService {

    @Autowired
    private Y41ePreguntaDao y41ePreguntaDao;
    @Autowired
    private Y41eRelBloquePreguntaDao y41eRelBloquePreguntaDao;

    @Override()
    protected Y41eGenericoDao<Y41ePregunta> getDao() {
        return this.y41ePreguntaDao;
    }

    /**
     * Comprueba que no existe ninguna encuesta enviada o respondida para el id de la pregunta.
     * @param idPregunta Long
     * @return El área o áreas a la que pertenece la encuesta
     */
    @Override
    public Integer validarPreguntaEnviada(Long idPregunta){
        return this.y41ePreguntaDao.validarPreguntaEnviada(idPregunta);
    }

    /**
     * Comprueba que el bloque no esté asociado a una encuesta enviada.
     * @param idBloque Long
     * @return Integer
     */
    @Override
    public Integer validarDesasociarEnviada(Long idBloque){
        return this.y41ePreguntaDao.validarDesasociarEnviada(idBloque);
    }

    /**
     * Comprueba que el bloque no esté asociado a una encuesta.
     * @param idBloque Long
     * @return Integer
     */
    @Override
    public Integer validarDesasociarEncuesta(Long idBloque){
        return this.y41ePreguntaDao.validarDesasociarEncuesta(idBloque);
    }

    /**
     * Comprueba que la pregunta pertenece a una encuesta.
     * @param idPregunta Long
     * @return El área o áreas a la que pertenece la encuesta
     */
    @Override
    public Integer validarPreguntaEncuesta(Long idPregunta){
        return this.y41ePreguntaDao.validarPreguntaEncuesta(idPregunta);
    }

    /**
     * Añade la pregunta y la asigna al bloque indicado
     * @param pregunta Y41ePregunta
     * @return
     */
    @Override
    @Transactional(rollbackFor = Throwable.class)
    public void addPreguntaBloque(Y41ePregunta pregunta){
        y41ePreguntaDao.add(pregunta);
        if(StringUtils.isNotEmpty(pregunta.getRespuestasAviso())){
            this.y41ePreguntaDao.addRespuestasAviso(pregunta);
        }
        Long orden = y41eRelBloquePreguntaDao.getSigOrden(pregunta.getIdBloque());
        Y41eBloque bloque = new Y41eBloque(pregunta.getIdBloque());
        Y41eRelBloquePregunta relBloquePregunta = new Y41eRelBloquePregunta(bloque, pregunta, orden);
        y41eRelBloquePreguntaDao.add(relBloquePregunta);

    }

    /**
     * Obtiene el suggest de bloques
     * @param pregunta Y41ePregunta
     * @param startsWith boolean
     * @return List<Y41ePregunta>
     */
    @Override
    public List<Y41ePregunta> findAllSuggest(Y41ePregunta pregunta, boolean startsWith){
        return this.y41ePreguntaDao.findAllSuggest(pregunta, startsWith);
    }

    @Override
    public Y41ePregunta add(Y41ePregunta bean){
        Y41ePregunta preguntaRetorno = this.y41ePreguntaDao.add(bean);
        if(StringUtils.isNotEmpty(bean.getRespuestasAviso())){
            this.y41ePreguntaDao.addRespuestasAviso(bean);
        }
        return preguntaRetorno;
    }


    /* (non-Javadoc)
     * @see com.ejie.y41e.service.Y41eGenericoServiceImpl#find(java.lang.Object)
     */
    @Override
    public Y41ePregunta find(Y41ePregunta bean){
        Y41ePregunta preguntaRetorno = this.y41ePreguntaDao.find(bean);
        if(Y41eConstants.SI.equals(preguntaRetorno.getAviso())){
            preguntaRetorno.setRespuestasAviso(this.y41ePreguntaDao.getRespuestasAviso(bean));
        }
        return preguntaRetorno;
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.service.Y41eGenericoService#update(java.lang.Object)
     */
    @Override
    public Y41ePregunta update(Y41ePregunta bean){
        Y41ePregunta preguntaRetorno = this.y41ePreguntaDao.update(bean);
        this.y41ePreguntaDao.removeRespuestasAviso(bean);
        if(Y41eConstants.SI.equals(preguntaRetorno.getAviso())){
            this.y41ePreguntaDao.addRespuestasAviso(bean);
        }
        return preguntaRetorno;
    }

    /**
     * Devuelve la pregunta si tiene configurado avisos
     * @param idPregunta Long
     * @param idTipoRespuesta Long
     * @return Y41ePregunta
     */
    @Override
    public Y41ePregunta findPreguntaAviso(Long idPregunta, Long idTipoRespuesta){
        return this.y41ePreguntaDao.findPreguntaAviso(idPregunta, idTipoRespuesta);
    }

}

