package com.ejie.y41e.service;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Service;

import com.ejie.y41e.common.Y41eConstants;
import com.ejie.y41e.common.Y41eMimeMessagePreparator;
import com.ejie.y41e.dao.Y41eConfiguracionDao;
import com.ejie.y41e.model.Y41eConfiguracion;
import com.ejie.y41e.model.Y41eEncuesta;
import com.ejie.y41e.model.Y41eEnvio;
import com.ejie.y41e.model.webservice.Y41eEncuestaWs;
import com.ejie.y41e.model.webservice.Y41eResultadoWs;
import com.ejie.y41e.model.webservice.Y41eUsuarioWs;
import com.ejie.y41e.utils.Y41eUtils;

/**
 * Y41eEncuestaWSServiceImpl
 *  
 */

@Service(value = "y41eEncuestaWSService")
public class Y41eEncuestaWSServiceImpl  implements Y41eEncuestaWSService {

    private static final Logger LOGGER = LoggerFactory.getLogger(Y41eEncuestaWSServiceImpl.class);
    private static final Locale LOCALE_ES = new Locale(Y41eConstants.LANG_CASTELLANO);
    private static final Locale LOCALE_EU = new Locale(Y41eConstants.LANG_EUSKERA);

    @Autowired()
    private ReloadableResourceBundleMessageSource appMessageSource;
    @Autowired
    private Y41eConfiguracionDao y41eConfiguracionDao;
    @Autowired()
    private JavaMailSender mailSender;
    @Autowired()
    private Properties appConfiguration;

    /**
     * Valida los campos obligatorios
     * 
     * @param encuesta Y41eEncuestaWs
     * @return Y41eResultadoWs
     */
    @Override
    public Y41eResultadoWs validarCamposObligatorios(Y41eEncuestaWs encuesta){
        Y41eEncuestaWSServiceImpl.LOGGER.info("validarCamposObligatorios");
        Y41eResultadoWs rdo = new Y41eResultadoWs();
        if (encuesta == null) {
            rdo.setCodigo(Y41eConstants.RETORNO_WS_ENCUESTA_REQUIRED);
            rdo.setDescripcionEs(appMessageSource.getMessage("retornoWs.entidadRequired", null, LOCALE_ES));
            rdo.setDescripcionEu(appMessageSource.getMessage("retornoWs.entidadRequired", null, LOCALE_EU));
            return rdo;
        }
        // id encuesta obligatorio
        if (StringUtils.isBlank(encuesta.getIdEncuesta())) {
            rdo.setCodigo(Y41eConstants.RETORNO_WS_ID_ENCUESTA_REQUIRED);
            rdo.setDescripcionEs(appMessageSource.getMessage("retornoWs.idEncuestaRequired", null, LOCALE_ES));
            rdo.setDescripcionEu(appMessageSource.getMessage("retornoWs.idEncuestaRequired", null, LOCALE_EU));
            return rdo;
        }
        try {
            Long.valueOf(encuesta.getIdEncuesta());
        }catch (Exception e) {
            Y41eEncuestaWSServiceImpl.LOGGER.error("El id de la encuesta no es un número " + e);
            rdo.setCodigo(Y41eConstants.RETORNO_WS_ID_ENCUESTA_NUMERO);
            rdo.setDescripcionEs(appMessageSource.getMessage("retornoWs.errorIdEncuestaNumero", null, LOCALE_ES));
            rdo.setDescripcionEu(appMessageSource.getMessage("retornoWs.errorIdEncuestaNumero", null, LOCALE_EU));
            return rdo;
        }

        //usuario obligatorio
        if(encuesta.getListaUsuarios().isEmpty()) {
            rdo.setCodigo(Y41eConstants.RETORNO_WS_USUARIOS_REQUIRED);
            rdo.setDescripcionEs(appMessageSource.getMessage("retornoWs.listaUsuariosRequired", null, LOCALE_ES));
            rdo.setDescripcionEu(appMessageSource.getMessage("retornoWs.listaUsuariosRequired", null, LOCALE_EU));
            return rdo;
        }else{
            // validar ususarios
            rdo = this.validarUsuarios(encuesta.getListaUsuarios());
        }

        return rdo;
    }

    /**
     * Crea una lista de envios con los datos de entrada
     * @param encuesta Y41eEncuesta
     * @param listaUsuarios  List<Y41eUsuarioWs>
     * @return List<Y41eEnvio>
     */
    @Override
    public List<Y41eEnvio> crearEnvios(Y41eEncuesta encuesta, List<Y41eUsuarioWs> listaUsuarios){
        List<Y41eEnvio> listaEnvios = new ArrayList<Y41eEnvio>();
        for(Y41eUsuarioWs usuario : listaUsuarios){
            Y41eEnvio envio = new Y41eEnvio();
            envio.getEncuesta().setIdEncuesta(encuesta.getIdEncuesta());
            envio.setIdUsuario(usuario.getIdUsuario());
            envio.setDataUsuario(usuario.getDataUsuario());
            envio.setEmail(usuario.getEmail());
            envio.setParam(Y41eUtils.generarCodigo());
            listaEnvios.add(envio);
        }
        return listaEnvios;
    }

    /**
     * Valida los datos de la lista de usuarios
     * @param listaUsuarios
     * @return
     */
    private Y41eResultadoWs validarUsuarios(List<Y41eUsuarioWs> listaUsuarios){
        Y41eResultadoWs rdo = new Y41eResultadoWs();
        for (Y41eUsuarioWs usuario : listaUsuarios) {
            if (StringUtils.isBlank(usuario.getIdUsuario())){
                rdo.setCodigo(Y41eConstants.RETORNO_WS_ID_USUARIO_REQUIRED);
                rdo.setDescripcionEs(appMessageSource.getMessage("retornoWs.usuarioRequired", null, LOCALE_ES));
                rdo.setDescripcionEu(appMessageSource.getMessage("retornoWs.usuarioRequired", null, LOCALE_EU));
                return rdo;
            }
            if (StringUtils.isBlank(usuario.getEmail())){
                rdo.setCodigo(Y41eConstants.RETORNO_WS_EMAIL_REQUIRED);
                rdo.setDescripcionEs(appMessageSource.getMessage("retornoWs.emailRequired", null, LOCALE_ES));
                rdo.setDescripcionEu(appMessageSource.getMessage("retornoWs.emailRequired", null, LOCALE_EU));
                return rdo;
            }
        }
        return rdo;
    }

    /**
     * Realiza el envio de los mails
     * @param encuesta Y41eEncuesta
     * @param listaEnvios List<Y41eEnvio>
     */
    @Override
    public void notificarEnvios(Y41eEncuesta encuesta, List<Y41eEnvio> listaEnvios){

        Y41eConfiguracion configRemitente = new Y41eConfiguracion(Integer.valueOf(Y41eConstants.CONFIG_REMITENTE));
        configRemitente = this.y41eConfiguracionDao.find(configRemitente);

        Y41eConfiguracion configMensaje = new Y41eConfiguracion(Integer.valueOf(Y41eConstants.CONFIG_ENCUESTA));
        configMensaje = this.y41eConfiguracionDao.find(configMensaje);

        final String urlInternet = this.appConfiguration.getProperty("url.internet");

        String urlUsuario="";

        //Por cada usuario se genera un mensaje
        int cont = 1;
        Y41eMimeMessagePreparator[] listadoMimeMessagePreparatorArray;
        StringBuilder asunto = new StringBuilder();
        asunto.append(configMensaje.getAsuntoEu()).append(" / ").append(configMensaje.getAsuntoEs());
        List<Y41eMimeMessagePreparator> listadoMimeMessagePreparator = new ArrayList<Y41eMimeMessagePreparator>();
        StringBuilder texto = new StringBuilder();
        for(Y41eEnvio envio : listaEnvios){
            urlUsuario= urlInternet.replace("{0}", envio.getParam());
            Y41eMimeMessagePreparator mensaje = new Y41eMimeMessagePreparator();
            mensaje.setRemitente(configRemitente.getEmail());
            mensaje.getDestinatarios().clear();
            mensaje.getDestinatarios().add(envio.getEmail());
            mensaje.setAsunto(asunto.toString());
            texto.append(configMensaje.getMensajeEu().replace("{encuestaEu}", encuesta.getDescPublEu()).replace("{urlEu}", urlUsuario))
            .append("<br>")
            .append(configMensaje.getMensajeEs().replace("{encuestaEs}", encuesta.getDescPublEs()).replace("{urlEs}", urlUsuario));
            mensaje.setTexto(texto.toString());
            listadoMimeMessagePreparator.add(mensaje);
            //reiniciar campos
            texto.delete(0, texto.length());
            urlUsuario="";
            if(cont==100){
                //enviar
                listadoMimeMessagePreparatorArray = new Y41eMimeMessagePreparator[listadoMimeMessagePreparator.size()];
                listadoMimeMessagePreparatorArray = listadoMimeMessagePreparator.toArray(listadoMimeMessagePreparatorArray);
                try{
                    this.mailSender.send(listadoMimeMessagePreparatorArray);
                }catch(Exception e){
                    Y41eEncuestaWSServiceImpl.LOGGER.error("Error al enviar los mails " + e);
                }
                listadoMimeMessagePreparator.clear();
                cont = 1;
            }

            cont++;
        }
        //enviar
        listadoMimeMessagePreparatorArray = new Y41eMimeMessagePreparator[listadoMimeMessagePreparator.size()];
        listadoMimeMessagePreparatorArray = listadoMimeMessagePreparator.toArray(listadoMimeMessagePreparatorArray);
        this.mailSender.send(listadoMimeMessagePreparatorArray);
    }
}

