package com.ejie.y41e.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.y41e.common.Y41eConstants;
import com.ejie.y41e.model.Y41eBloque;
import com.ejie.y41e.model.Y41eEncuesta;
import com.ejie.y41e.model.Y41eRelEncuestaBloque;
import com.ejie.y41e.utils.Y41eSqlUtils;

/**
 * RelEncuestaBloqueDaoImpl  
 *  
 */

@Repository
@Transactional
public class Y41eRelEncuestaBloqueDaoImpl extends Y41eGenericoDaoImpl<Y41eRelEncuestaBloque> implements Y41eRelEncuestaBloqueDao {


    public static final String TABLA="Y41E03S01";
    public static final String ID_ENCUESTA_003="ID_ENCUESTA_003";
    public static final String ID_BLOQUE_003="ID_BLOQUE_003";
    public static final String ORDEN_BLOQ_003="ORDEN_BLOQ_003";
    protected static final String[] ORDER_BY_WHITE_LIST = new String[] { ID_ENCUESTA_003, ID_BLOQUE_003, ORDEN_BLOQ_003};

    /**
     * Y41eRelEncuestaBloqueDaoImpl
     */
    public Y41eRelEncuestaBloqueDaoImpl() {
        super(Y41eRelEncuestaBloque.class);
    }

    /*
     * ROW_MAPPERS
     */
    private final RowMapper<Y41eRelEncuestaBloque> rwMap = new RowMapper<Y41eRelEncuestaBloque>() {
        @Override
        public Y41eRelEncuestaBloque mapRow(ResultSet resultSet, int rowNum) throws SQLException {
            Y41eRelEncuestaBloque relEncuestaBloque = new Y41eRelEncuestaBloque();
            relEncuestaBloque.getEncuesta().setIdEncuesta(resultSet.getLong(ID_ENCUESTA_003));
            relEncuestaBloque.getBloque().setIdBloque(resultSet.getLong(ID_BLOQUE_003));
            relEncuestaBloque.getBloque().setDescPublEs(resultSet.getString("DESC_PUBL_ES_002"));
            relEncuestaBloque.getBloque().setDescPublEu(resultSet.getString("DESC_PUBL_EU_002"));
            relEncuestaBloque.getBloque().setDescIntEs(resultSet.getString("DESC_INT_ES_002"));
            relEncuestaBloque.getBloque().setDescIntEu(resultSet.getString("DESC_INT_EU_002"));
            relEncuestaBloque.getBloque().setNumPreguntas(resultSet.getLong("PREGUNTAS"));
            relEncuestaBloque.getBloque().setIsEnlazada(resultSet.getString("IS_BLOQUE_ENLAZADA"));
            relEncuestaBloque.setOrdenBloq(resultSet.getLong(ORDEN_BLOQ_003));
            return relEncuestaBloque; 
        } 
    };

    private final RowMapper<Y41eRelEncuestaBloque> rwMapPK = new RowMapper<Y41eRelEncuestaBloque>() {
        @Override
        public Y41eRelEncuestaBloque mapRow(ResultSet resultSet, int rowNum) throws SQLException {
            Y41eRelEncuestaBloque relEncuestaBloque = new Y41eRelEncuestaBloque();
            relEncuestaBloque.getEncuesta().setIdEncuesta(resultSet.getLong(ID_ENCUESTA_003));
            relEncuestaBloque.getBloque().setIdBloque(resultSet.getLong(ID_BLOQUE_003));
            return relEncuestaBloque; 
        }
    };

    private final RowMapper<Y41eBloque> rwMapBloque = new RowMapper<Y41eBloque>() {
        @Override
        public Y41eBloque mapRow(ResultSet resultSet, int rowNum) throws SQLException {
            Y41eBloque bloque = new Y41eBloque();
            bloque.setIdBloque(resultSet.getLong(ID_BLOQUE_003));
            bloque.setDescPublEs(resultSet.getString("DESC_PUBL_ES_002"));
            bloque.setDescPublEu(resultSet.getString("DESC_PUBL_EU_002"));
            bloque.setObligatorio(resultSet.getString("OBLIGATORIO_002"));
            bloque.setIsEnlazada(resultSet.getString("IS_ENLAZADA"));
            bloque.setIdPreguntaPadre(resultSet.getLong("ID_PREGUNTA_014"));
            return bloque; 
        } 
    };


    /*
     * OPERACIONES CRUD
     */

    /**
     * Inserts a single row in the Y41eRelEncuestaBloque table.
     *
     * @param relBloquePregunta Y41eRelEncuestaBloque
     * @return Y41eRelEncuestaBloque
     */
    @Override
    public Y41eRelEncuestaBloque add(Y41eRelEncuestaBloque relBloquePregunta) {
        StringBuilder insert = new StringBuilder(Y41eSqlUtils.INSERT)
                .append(TABLA).append(" (")
                .append(ID_ENCUESTA_003).append(",")
                .append(ID_BLOQUE_003).append(",")
                .append(ORDEN_BLOQ_003).append(")")
                .append("VALUES (?,?,?)");
        this.getJdbcTemplate().update(insert.toString(), relBloquePregunta.getEncuesta().getIdEncuesta(), relBloquePregunta.getBloque().getIdBloque(), 
                relBloquePregunta.getOrdenBloq());
        return relBloquePregunta;
    }

    /**
     * Updates a single row in the Y41eRelEncuestaBloque table.
     *
     * @param relBloquePregunta Y41eRelEncuestaBloque
     * @return Y41eRelEncuestaBloque
     */
    @Override
    public Y41eRelEncuestaBloque update(Y41eRelEncuestaBloque relBloquePregunta) {
        StringBuilder update = new StringBuilder(Y41eSqlUtils.UPDATE)
                .append(TABLA).append(" SET ")
                .append(ORDEN_BLOQ_003).append("=? ")
                .append(Y41eSqlUtils.WHERE)
                .append(ID_ENCUESTA_003).append("=?")
                .append(Y41eSqlUtils.AND)
                .append(ID_BLOQUE_003).append("=?");
        this.getJdbcTemplate().update(update.toString(), relBloquePregunta.getOrdenBloq(), relBloquePregunta.getEncuesta().getIdEncuesta(), relBloquePregunta.getBloque().getIdBloque());
        return relBloquePregunta;
    }

    /**
     * Removes a single row in the Y41eRelEncuestaBloque table.
     *
     * @param relBloquePregunta Y41eRelEncuestaBloque
     * @return
     */
    @Override
    public void remove(Y41eRelEncuestaBloque relBloquePregunta) {
        StringBuilder delete = new StringBuilder(Y41eSqlUtils.DELETE).append(TABLA).append(Y41eSqlUtils.WHERE)
                .append(ID_ENCUESTA_003).append("=?")
                .append(Y41eSqlUtils.AND)
                .append(ID_BLOQUE_003).append("=?");
        this.getJdbcTemplate().update(delete.toString(), relBloquePregunta.getEncuesta().getIdEncuesta(), relBloquePregunta.getBloque().getIdBloque());
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getSelect()
     */
    @Override
    protected String getSelect() {
        StringBuilder select = new StringBuilder("SELECT ")
                .append(ID_ENCUESTA_003).append(",")
                .append(ID_BLOQUE_003).append(",")
                .append(ORDEN_BLOQ_003).append(",")
                .append(" DESC_PUBL_ES_002, DESC_PUBL_EU_002, DESC_INT_ES_002, DESC_INT_EU_002,")
                .append(" PREGUNTAS, NVL2(IS_BLOQUE_ENLAZADA, 'S', 'N') IS_BLOQUE_ENLAZADA");
        return select.toString();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getFrom()
     */
    @Override
    protected String getFrom() {
        StringBuilder from = new StringBuilder(" FROM ").append(TABLA)
                .append(" JOIN Y41E02S01 ON ID_BLOQUE_002 = ").append(ID_BLOQUE_003)
                .append(" LEFT JOIN (SELECT COUNT(1) PREGUNTAS, ID_BLOQUE_005 FROM Y41E05S01 GROUP BY ID_BLOQUE_005)")
                .append(" ON ID_BLOQUE_005=").append(ID_BLOQUE_003)
                .append(Y41eSqlUtils.LEFT_JOIN).append("(")
                .append(Y41eSqlUtils.SELECT_COUNT).append(" IS_BLOQUE_ENLAZADA, ID_ENLAZADA_014, ID_BLOQUE_014, ID_ENCUESTA_014")
                .append(Y41eSqlUtils.FROM).append("Y41E14S01")
                .append(Y41eSqlUtils.WHERE).append("TIPO_014='").append(Y41eConstants.TIPO_BLOQUE).append("'")
                .append(Y41eSqlUtils.GROUP_BY).append("ID_ENLAZADA_014, ID_BLOQUE_014, ID_ENCUESTA_014) T3 ON T3.ID_ENLAZADA_014=ID_BLOQUE_005")
                .append(Y41eSqlUtils.AND).append("T3.ID_ENCUESTA_014=").append(ID_ENCUESTA_003);
        return from.toString();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getRwMap()
     */
    @Override
    protected RowMapper<Y41eRelEncuestaBloque> getRwMap() {
        return this.rwMap;
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getPK()
     */
    @Override
    protected String getPK() {
        StringBuilder selectPk = new StringBuilder(ID_ENCUESTA_003).append(",").append(ID_BLOQUE_003);
        return selectPk.toString();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getRwMapPK()
     */
    @Override
    protected RowMapper<Y41eRelEncuestaBloque> getRwMapPK() {
        return this.rwMapPK;
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getWherePK(java.lang.Object, java.util.List)
     */
    @Override
    protected String getWherePK(Y41eRelEncuestaBloque bean, List<Object> params) {
        StringBuilder wherePK = new StringBuilder();
        wherePK.append(Y41eSqlUtils.WHERE)
        .append(ID_ENCUESTA_003).append("=?")
        .append(Y41eSqlUtils.AND)
        .append(ID_BLOQUE_003).append("=?");
        params.add(bean.getEncuesta().getIdEncuesta());
        params.add(bean.getBloque().getIdBloque());
        return wherePK.toString();

    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getWhere(java.lang.Object, java.util.List)
     */
    @Override
    protected String getWhere(Y41eRelEncuestaBloque bean, List<Object> params) {
        StringBuilder where = new StringBuilder(Y41eRelEncuestaBloqueDaoImpl.STRING_BUILDER_INIT);

        if (bean  != null){
            where.append(Y41eSqlUtils.generarWhereIgual(ID_ENCUESTA_003, bean.getEncuesta().getIdEncuesta(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ID_BLOQUE_003, bean.getBloque().getIdBloque(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ORDEN_BLOQ_003, bean.getOrdenBloq(), params));
        }

        return where.toString();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getWhereLike(java.lang.Object, java.lang.Boolean, java.util.List, java.lang.Boolean)
     */
    @Override
    protected String getWhereLike(Y41eRelEncuestaBloque bean, Boolean startsWith, List<Object> params, Boolean search) {
        StringBuilder where = new StringBuilder(Y41eRelEncuestaBloqueDaoImpl.STRING_BUILDER_INIT);

        if (bean  != null){
            where.append(Y41eSqlUtils.generarWhereIgual(ID_ENCUESTA_003, bean.getEncuesta().getIdEncuesta(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ID_BLOQUE_003, bean.getBloque().getIdBloque(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ORDEN_BLOQ_003, bean.getOrdenBloq(), params));
        }
        return where.toString();
    }

    /**
     * Obtiene el siguiente orden 
     * @param idEncuesta Long
     * @return Long
     */
    @Override
    public Long getSigOrden(Long idEncuesta){
        StringBuilder sqlOrden = new StringBuilder(Y41eSqlUtils.SELECT)
                .append(" NVL(MAX(")
                .append(ORDEN_BLOQ_003)
                .append("), 0 )+1")
                .append(Y41eSqlUtils.FROM)
                .append(TABLA)
                .append(Y41eSqlUtils.WHERE)
                .append(ID_ENCUESTA_003)
                .append("=?");
        return this.getJdbcTemplate().queryForObject(sqlOrden.toString(), Long.class, idEncuesta);
    }

    /**
     * Asocia los bloques seleccionados a la encuesta
     * @param idEncuesta Long
     * @param listaBloquesSeleccionados List<Y41eBloque>
     */
    @Override
    public void asosiarBloques(Long idEncuesta, List<Y41eBloque> listaBloquesSeleccionados){
        //obtener el máximo orden
        Long orden = this.getSigOrden(idEncuesta);
        StringBuilder sqlInsert = new StringBuilder(Y41eRelBloquePreguntaDaoImpl.STRING_BUILDER_INIT);
        sqlInsert.append(Y41eSqlUtils.INSERT)
        .append(TABLA).append(" (")
        .append(ID_ENCUESTA_003).append(",")
        .append(ID_BLOQUE_003).append(",")
        .append(ORDEN_BLOQ_003).append(")")
        .append("VALUES (?,?,?)");

        final List<Object[]> paramsInsert = new ArrayList<Object[]>();
        List<Object> auxInsert = new ArrayList<Object>();
        for(Y41eBloque bloque:listaBloquesSeleccionados){
            auxInsert.clear();
            auxInsert.add(idEncuesta);
            auxInsert.add(bloque.getIdBloque());
            auxInsert.add(orden);
            paramsInsert.add(auxInsert.toArray().clone());
            orden++;
        }
        this.getJdbcTemplate().batchUpdate(sqlInsert.toString(), paramsInsert);
    }

    /**
     * Ordena las bloques según el orden pasado
     * @param ordenIds String
     * @param idEncuesta String
     */
    @Override
    public void moverBloque(String ordenIds, String idEncuesta){
        StringBuilder sqlOrden = new StringBuilder(Y41eRelBloquePreguntaDaoImpl.STRING_BUILDER_INIT);
        sqlOrden.append(Y41eSqlUtils.UPDATE)
        .append(TABLA).append(" SET ")
        .append(ORDEN_BLOQ_003).append("=? ")
        .append(Y41eSqlUtils.WHERE)
        .append(ID_ENCUESTA_003).append("=?")
        .append(Y41eSqlUtils.AND)
        .append(ID_BLOQUE_003).append("=?");

        final List<Object[]> paramsUpdate = new ArrayList<Object[]>();
        List<Object> auxUpdate = new ArrayList<Object>();

        //en el atributo orden están los ids de las preguntas
        //hay que separarlos por ; y luego obtener el id por ___
        String[] listaIds = ordenIds.split(";");
        int orden = 1;
        for (String id : listaIds){
            auxUpdate.clear();
            auxUpdate.add(orden);
            auxUpdate.add(idEncuesta);
            auxUpdate.add(id.substring(id.indexOf("-")+1).split(Y41eConstants.SEPARADOR_OPTION)[1]);
            paramsUpdate.add(auxUpdate.toArray().clone());
            orden++;
        }
        this.getJdbcTemplate().batchUpdate(sqlOrden.toString(), paramsUpdate);
    }

    /**
     * Copiar los bloques de una encuesta a otra
     * @param idEncuestaOrigen Long
     * @param idEncuestaCopiar Long
     */
    @Override
    public void copiarBloques(Long idEncuestaOrigen, Long idEncuestaCopiar){
        StringBuilder sqlCopiar = new StringBuilder(Y41eRelBloquePreguntaDaoImpl.STRING_BUILDER_INIT);
        sqlCopiar.append(Y41eSqlUtils.INSERT).append(TABLA)
        .append(Y41eSqlUtils.SELECT).append(idEncuestaCopiar).append(", ")
        .append(ID_BLOQUE_003).append(",")
        .append(ORDEN_BLOQ_003)
        .append(Y41eSqlUtils.FROM).append(TABLA)
        .append(Y41eSqlUtils.WHERE)
        .append(ID_ENCUESTA_003).append("=?");

        this.getJdbcTemplate().update(sqlCopiar.toString(), idEncuestaOrigen);
    }

    /**
     * Obtiene los bloques de la encuesta en orden
     * @param encuesta Y41eEncuesta
     * @return List<Y41eBloque>
     */
    @Override
    public List<Y41eBloque> getBloquesWeb(Y41eEncuesta encuesta){
        StringBuilder sqlBloque = new StringBuilder(Y41eRelBloquePreguntaDaoImpl.STRING_BUILDER_INIT);
        sqlBloque.append(Y41eSqlUtils.SELECT)
        .append(ID_BLOQUE_003).append(",")
        .append(" DESC_PUBL_ES_002, DESC_PUBL_EU_002, OBLIGATORIO_002,")
        .append("NVL2(ID_PREGUNTA_014, 'S', 'N') IS_ENLAZADA, ID_PREGUNTA_014")
        .append(Y41eSqlUtils.FROM)
        .append(" Y41E02S01")
        .append(Y41eSqlUtils.JOIN).append(TABLA).append(" ON ")
        .append("ID_BLOQUE_002=").append(ID_BLOQUE_003)
        .append(Y41eSqlUtils.LEFT_JOIN).append(" Y41E14S01 ON ").append(ID_ENCUESTA_003).append("=ID_ENCUESTA_014")
        .append(Y41eSqlUtils.AND).append("TIPO_014='").append(Y41eConstants.TIPO_BLOQUE).append("'")
        .append(Y41eSqlUtils.AND).append(ID_BLOQUE_003).append("=ID_ENLAZADA_014")
        .append(Y41eSqlUtils.WHERE).append(ID_ENCUESTA_003).append("=?")
        .append(Y41eSqlUtils.ORDER_BY).append(ORDEN_BLOQ_003).append(" ASC");

        return this.getJdbcTemplate().query(sqlBloque.toString(), this.rwMapBloque, encuesta.getIdEncuesta());
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getOrderBy()
     */
    @Override
    protected String[] getOrderBy() {
        return Y41eRelEncuestaBloqueDaoImpl.ORDER_BY_WHITE_LIST;
    }


    /**
     * StringBuilder initilization value
     */
    public static final int STRING_BUILDER_INIT = 4096;

}